/***************************************************************************
 *   Copyright (C) 2004 by  ͤ                                            *
 *   tasuku@linux-life.net                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "canna.h"
#include "worditem.h"

#include <dlfcn.h>
#include <stdio.h>
#include <pwd.h>
#include <unistd.h>
#include <stdlib.h>
#include <kdebug.h>
#include <qtextstream.h>
#include <qfile.h>
#include <qtextcodec.h>
#include <qdir.h>
#include <kfiledialog.h>
#include <klocale.h>
#include <qregexp.h>
#include <kmessagebox.h>
#include <kconfig.h>

#define BUFSIZE 1024

/*
extern "C"{
	int RkGetWordTextDic( int, const char*, const char*, char*, int );
	int RkInitialize( char* );
	int RkFinalize( void );
	int RkMountDic( int, char*, int );
	int RkDefineDic( int, char*, char* );
	int RkDeleteDic( int, char*, char* );
	char* RkwGetServerName( void );
}
*/

Canna::Canna( QString name ){
	m_dicname = name;

	m_libcanna = dlopen( "libcanna.so.1", RTLD_LAZY );

	RKFunc.Initialize = ( int(*)(char*) )dlsym( m_libcanna, "RkInitialize" );
	RKFunc.Finalize = ( void(*)(void) )dlsym( m_libcanna, "RkFinalize" );
	RKFunc.GetWordTextDic = ( int(*)(int, const char *, const char *, char *, int) )dlsym( m_libcanna, "RkGetWordTextDic" );
	RKFunc.MountDic = ( int(*)( int, const char*, int ) )dlsym( m_libcanna, "RkMountDic" ) ;
	RKFunc.DefineDic = ( int (*)( int, const char*, const char* ) )dlsym( m_libcanna, "RkDefineDic" );
	RKFunc.DeleteDic = (int (*)( int, const char*, const char* ) )dlsym( m_libcanna, "RkDeleteDic" );
	RKFunc.RemoveDic = (int (*)(int, const char *, int) )dlsym( m_libcanna, "RkRemoveDic" );


	int ret = RKFunc.Initialize( getenv( "CANNA_SERVER" ) );
	if( ret < 0 ){
		KMessageBox::error( (QWidget*)0, i18n( "Can't connect %1." ).arg( getenv( "CANNA_SERVER" ) ) );
		return;
	}

	QString dirname;
	char buf[BUFSIZE];

	dirname = QString(":user/");
	dirname += searchuname();

	ret = RKFunc.GetWordTextDic( 0, dirname, m_dicname.local8Bit(), buf, BUFSIZE-1 );
	if( ret < 0 ){
		switch( ret ){
			case -2:
				KMessageBox::error( (QWidget*)0, i18n( "Dictionary \"%1\" does not exist." ).arg( name ) );
				return;
			default:
				KMessageBox::error( (QWidget*)0, i18n( "GetWordTextDic() = %1." ).arg( ret ) );
				return;
		}
	}
	QString from, to, type;
	while( ret > 0 ){
		QString str( escape( QString::fromLocal8Bit(buf) ) );
		QTextStream line( str, IO_ReadOnly );
		line >> from;
		while( !line.atEnd() ){
			QString temp;
			line >> temp;
			if( temp[0] == '#' ){
				type = temp;
			}else{
				to = unescape( temp );
				Word word;
				word.type = CannaDic::instance().code2type( type );
				word.from = from;
				word.to = to;
				word.freq = 1;
				push_back( word );
			}
		}
		ret = RKFunc.GetWordTextDic( 0, "", "", buf, BUFSIZE-1 );
	}
	RKFunc.Finalize();
}

Canna::~Canna()
{
	if( m_libcanna != NULL ){
		dlclose( m_libcanna );
	}
}

bool Canna::mkdic( QString& dicname )
{
	struct _canna_api RKFunc;
	QStringList list = Canna::lsdic();
	for( QStringList::iterator it = list.begin(); it != list.end(); ++it ){
		if( (*it) == dicname ){
			KMessageBox::error( (QWidget*)0, i18n( "%1 is already exist." ).arg( dicname ) );
			return false;
		}
	}

	void *libcanna;
	libcanna = dlopen( "libcanna.so.1", RTLD_LAZY );

	RKFunc.Initialize = ( int(*)(char*) )dlsym( libcanna, "RkInitialize" );
	RKFunc.Finalize = ( void(*)(void) )dlsym( libcanna, "RkFinalize" );
	RKFunc.CreateDic = (int (*)(int, const char *, int) )dlsym( libcanna, "RkCreateDic" );

	int ret = RKFunc.Initialize( getenv( "CANNA_SERVER" ) );
	if( ret < 0 ){
		KMessageBox::error( (QWidget*)0, i18n( "Failed mkdic(%1)" ).arg( ret ) );
		return false;
	}

	ret = RKFunc.CreateDic( 0, dicname.local8Bit(), 0 );
	if( ret < 0 ){
		KMessageBox::error( (QWidget*)0, i18n( "Failed mkdic(%1)" ).arg( ret ) );
		RKFunc.Finalize();
		return false;
	}
	RKFunc.Finalize();

	if( !checkEditDotCanna() ) return false;

	QString dotCanna( Canna::dotCanna() );
	if( dotCanna.isEmpty() ){
		return false;
	}
	QFile fin( dotCanna );
	if ( !fin.open( IO_ReadOnly ) ){
		KMessageBox::error( (QWidget*)0, i18n( "KannaDic can't open %1" ).arg( dotCanna ) );
		return false;
	}
	dotCanna = "";
	QTextStream in( &fin );
	bool isDic = false;

	while( !in.atEnd() ){
		QString str( in.readLine() );
		if( isDic && str == " )" ){
			dotCanna +=  QString(" :user \"%1\"\n").arg( dicname );
			isDic = false;
		}
		if( str == "(use-dictionary" ){
			isDic = true;
		}
		dotCanna += str + "\n";
	}
	fin.close();

	QFile fout( QDir::home().absPath() + "/.canna" );
	if ( !fout.open( IO_WriteOnly ) ){
		KMessageBox::error( (QWidget*)0, i18n( "KannaDic can't open %1" ).arg( QDir::home().absPath() + "/.canna" ) );
		return false;
	}
	QTextStream out( &fout );
	out << dotCanna;
	fout.close();
	return true;
}

bool Canna::checkEditDotCanna()
{
	int result = KMessageBox::questionYesNo( (QWidget*)0, i18n( "Do I edit .canna?" ), i18n("Question"), KStdGuiItem::yes(), KStdGuiItem::no(), "AutoEditDotFile" );
	return ( result == KMessageBox::Yes );
}

bool Canna::checkRestartXIM()
{
	int result = KMessageBox::questionYesNo( (QWidget*)0, i18n( "Do I restart your XIM?" ), i18n("Question"), KStdGuiItem::yes(), KStdGuiItem::no(), "AutoRestartXIM" );
	return ( result == KMessageBox::Yes );
}

QString Canna::dotCanna()
{
	if( QFile::exists( QDir::home().absPath() + "/.canna" ) ){
		return QDir::home().absPath() + "/.canna";
	}else if( QFile::exists( "/usr/local/lib/canna/default.canna" ) ){
		return "/usr/local/lib/canna/default.canna";
	}else if( QFile::exists( "/usr/lib/canna/default.canna" ) ){
		return "/usr/lib/canna/default.canna";
	}else if( QFile::exists( "/var/lib/canna/default.canna" ) ){
		return "/var/lib/canna/default.canna";
	}else{
		return KFileDialog::getOpenFileName( QString::null, I18N_NOOP( "*.canna|canna configuration file" ) );
	}
}

QStringList Canna::lsdic()
{
	QStringList list;
	char buf[BUFSIZE];
	int ret;
	struct _canna_api RKFunc;

	void *libcanna;
	libcanna = dlopen( "libcanna.so.1", RTLD_LAZY );

	RKFunc.Initialize = ( int(*)(char*) )dlsym( libcanna, "RkInitialize" );
	RKFunc.Finalize = ( void(*)(void) )dlsym( libcanna, "RkFinalize" );
	RKFunc.ListDic = ( int(*)(int, char *, char *, int) )dlsym( libcanna, "RkListDic" );
	ret = RKFunc.Initialize( getenv( "CANNA_SERVER" ) );
	if( ret < 0 ){
		KMessageBox::error( (QWidget*)0, i18n( "Failed lsdic(%1)" ).arg( ret ) );
		return list;
	}

	ret = RKFunc.ListDic(0, searchuname(), buf, BUFSIZE - 1 );
	if( ret < 0 ){
		KMessageBox::error( (QWidget*)0, i18n( "Failed lsdic(%1)" ).arg( ret ) );
		RKFunc.Finalize();
		return list;
	}
	char* p = buf;
	for( int i = 0; i < ret; i++){
		list.push_back( QString::fromLocal8Bit(p) );
		p += strlen((char *)p) + 1;
	}
	RKFunc.Finalize();
	dlclose( libcanna );
	return list;
}

void Canna::rmdic()
{
	proc = NULL;
	m_blXIMRestart = checkRestartXIM();
	ximStop();
}

void Canna::ximStart()
{
	if( m_blXIMRestart ){
		KConfig *conf = KGlobal::config();
		conf->setGroup("Canna");
		QStringList list = QStringList::split( " ", conf->readEntry( "XIMStartCommand" ) );
		if( list.count() == 0 ){
			emit deleted( true );
		}else{
			proc = new QProcess( this );
			for( QStringList::iterator it = list.begin(); it != list.end(); ++it ){
				proc->addArgument( *it );
			}
			proc->start();
		}
	}
}

void Canna::ximStop()
{
	if( m_blXIMRestart ){
		KConfig *conf = KGlobal::config();
		conf->setGroup("Canna");
		QStringList list = QStringList::split( " ", conf->readEntry( "XIMStopCommand" ) );
		if( list.count() == 0 ){
			slotRemove();
		}else{
			proc = new QProcess( this );
			for( QStringList::iterator it = list.begin(); it != list.end(); ++it ){
				proc->addArgument( *it );
			}
			connect( proc, SIGNAL( processExited() ), this, SLOT( slotRemove() ) );
			proc->start();
		}
	}else{
		slotRemove();
	}
}

void Canna::slotRemove()
{
	kdDebug() << "Canna::slotRemove()" << endl;
	if( proc ){
		if( !proc->normalExit() ){
			QString err;
			while( proc->canReadLineStderr() ){
				err += proc->readLineStderr();
			}
			kdDebug() << err << endl;
			KMessageBox::error( (QWidget*)0, err );
			emit deleted( false );
			return;
		}
	}

	int ret = RKFunc.Initialize( getenv( "CANNA_SERVER" ) );
	if( ret < 0 ){
		KMessageBox::error( (QWidget*)0, i18n( "Can't connect %1." ).arg( getenv( "CANNA_SERVER" ) ) );
		ximStart();
		emit deleted( false );
		return;
	}

	ret = RKFunc.RemoveDic(0, m_dicname.local8Bit(), 0);
	RKFunc.Finalize();
	if( ret < 0){
		KMessageBox::error( (QWidget*)0, i18n( "Failed rmdic(%1)" ).arg( ret ) );
		ximStart();
		emit deleted( false );
		return;
	}

	if( checkEditDotCanna() ){
		QString dotCanna( Canna::dotCanna() );
		if( dotCanna.isEmpty() ){
			ximStart();
			emit deleted( false );
			return;
		}
		QFile fin( dotCanna );
		if ( !fin.open( IO_ReadOnly ) ){
			KMessageBox::error( (QWidget*)0, i18n( "KannaDic can't open %1" ).arg( dotCanna ) );
			ximStart();
			emit deleted( false );
			return;
		}
		dotCanna = "";
		QTextStream in( &fin );
		bool isDic = false;
	//	QString dicname = QTextCodec::codecForName("eucJP")->fromUnicode( m_dicname );
		while( !in.atEnd() ){
			QString str( in.readLine() );
			if( isDic && str == " )" ){
				isDic = false;
			}
			if( isDic ){
				QRegExp regexp( ":user +\"(.*)\"", false, false );
				if( regexp.search( str ) != -1 ){
					if( regexp.cap(1) == m_dicname ){
						continue;
					}
				}
			}
			if( str == "(use-dictionary" ){
				isDic = true;
			}
			dotCanna += str + "\n";
		}
		fin.close();

		QFile fout( QDir::home().absPath() + "/.canna" );
		if ( !fout.open( IO_WriteOnly ) ){
			KMessageBox::error( (QWidget*)0, i18n( "KannaDic can't open %1" ).arg( QDir::home().absPath() + "/.canna" ) );
			ximStart();
			emit deleted( false );
			return;
		}
		QTextStream out( &fout );
		out << dotCanna;
		fout.close();
	}

	ximStart();
	emit deleted( true );
}

// void Canna::slotRemoved()
// {
// 	kdDebug() << "Canna::slotRemoved()" << endl;
// 	if( proc->normalExit() ){
// 	kdDebug() << "true" << endl;
// 		emit deleted( true );
// 	}else{
// 	kdDebug() << "false" << endl;
// 		QString err;
// 		while( proc->canReadLineStderr() ){
// 			err += proc->readLineStderr();
// 		}
// 		KMessageBox::error( (QWidget*)0, err );
// 		emit deleted( false );
// 	}
// 	kdDebug() << "Canna::slotRemoved()" << endl;
// }

bool Canna::isAvailable()
{
	void *libcanna = dlopen( "libcanna.so.1", RTLD_LAZY );
	if( libcanna != NULL ){
		dlclose( libcanna );
		return true;
	}
	return false;
}

bool Canna::expand( KListView* lvw )
{
	WordItem* word;
	m_count = size();
	m_counter = 0;
	m_progress = 0;
	for( iterator it = begin(); it != end(); ++it ){
		word = new WordItem( lvw, (*it) );
		lvw->insertItem( word );
		countup();
	}
	emit progress( 100 );
	return true;
}

bool Canna::save( KListView* lvw )
{
	QValueList<Word> view;
	QListViewItemIterator it( lvw );
	while( it.current() ){
		WordItem* word = (WordItem*)it.current();
		view.push_back( word->word() );
		++it;
	}
	qHeapSort( view );
	qHeapSort( *this );

	mount();

	m_add.clear();
	m_del.clear();
	m_count = size() + view.size();
	m_counter = 0;
	m_progress = 0;
	iterator before, after;
	for( before = begin(), after = view.begin();
		before != end() && after != view.end();){
//		kdDebug() << (*before).to << '\t' << (*after).to << endl;
		if( *before == *after ){
			countup();
			++before;
			countup();
			++after;
			continue;
		}
		if( *before < *after ){
			delWord( *before );
			countup();
			++before;
		}else{
			addWord( *after );
			countup();
			++after;
		}
	}
	for(; before != end(); ++before ){
		delWord( *before );
		countup();
	}
	for(; after != view.end(); ++after ){
		addWord( *after );
		countup();
	}

	unmount();

	for( iterator it = m_del.begin(); it != m_del.end(); ++it )
	{
		remove( *it );
	}

	for( iterator it = m_add.begin(); it != m_add.end(); ++it )
	{
		push_back( *it );
	}
	return true;
}

void Canna::countup()
{
	int percent = ++m_counter*100/m_count;
	if( percent > m_progress + 1 ){
		m_progress = percent;
		emit progress( percent );
	}
}

bool Canna::mount()
{
	int ret;
	ret = RKFunc.Initialize( getenv( "CANNA_SERVER" ) );
	if( ret < 0 ){
		KMessageBox::error( (QWidget*)0, i18n( "mount initialize error %1" ).arg( ret ) );
		return false;
	}
	ret = RKFunc.MountDic( 0, m_dicname, 0 );
	if( ret < 0 ){
		KMessageBox::error( (QWidget*)0, i18n( "mount initialize error %1" ).arg( ret ) );
		RKFunc.Finalize();
		return false;
	}
	return true;
}

void Canna::unmount()
{
	RKFunc.Finalize();
}

bool Canna::addWord( Word word )
{
	int ret;

	QString from, to, type;
	from = word.from.local8Bit();
	type = word.type.code;

	to = word.to.local8Bit();
	to.replace( "&", "&amp;" );
	to.replace( "\\", "&slash;" );
	to.replace( " ", "&space;" );
	to.replace( "&slash;", "\\\\" );
	to.replace( "&space;", "\\ " );
	to.replace( "&amp;", "&" );

	QString str = QString( "%1 %2 %3" ).arg( from ).arg( type ).arg( to );

	ret = RKFunc.DefineDic( 0, m_dicname, str );
	if( ret < 0 ){
		kdDebug() << "addword define error " << QString(str) << endl;
		unmount();
		mount();
		return false;
	}
	m_add.push_back( word );
	return true;
}

bool Canna::delWord( Word& word )
{
	int ret;

	QString from, to, type;
	from = word.from.local8Bit();
	type = word.type.code;

	to = word.to.local8Bit();
	to.replace( "&", "&amp;" );
	to.replace( "\\", "&slash;" );
	to.replace( " ", "&space;" );
	to.replace( "&slash;", "\\\\" );
	to.replace( "&space;", "\\ " );
	to.replace( "&amp;", "&" );

	QString str = QString( "%1 %2 %3" ).arg( from ).arg( type ).arg( to );

	ret = RKFunc.DeleteDic( 0, m_dicname, str );
	if( ret < 0 ){
		kdDebug() << "addword define error " << QString(str) << endl;
		unmount();
		mount();
		return false;
	}
	m_del.push_back( word );
	return true;
}

char* Canna::searchuname()
{
    char *username = NULL;

    struct passwd *pass = getpwuid( getuid() ) ;
    if ( pass ) username = pass->pw_name ;

    if ( username == NULL ) {
	if( (username = getlogin()) == NULL ) {
	    if( (username = getenv( "LOGNAME" )) == NULL ) {
		  username = getenv( "USER" ) ;
	    }
	}
    }
    if ( username == NULL ) {
	exit(-2);
    }
    return( username ) ;
}

QString Canna::escape( QString from )
{
	QString to( from );
	to.replace( "&", "&amp;" );
	to.replace( "\\\\", "&slash;" );
	to.replace( "\\ ", "&space;" );
	return to;

}

QString Canna::unescape( QString from )
{
	QString to( from );
	to.replace( "&slash;", "\\" );
	to.replace( "&space;", " " );
	to.replace( "&amp;", "&" );
	return to;

}
