/***************************************************************************
 *   Copyright (C) 2008 by Joerg Zopes   *
 *   joerg.zopes@linux-specialist.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "wifisetup.h"
#include "iconsadd.h"

wifiSetup::wifiSetup(QWidget *parent, const char *name, const char * iface) : QDialog(parent, name, true)
{ interface = (QString)iface;
  if (interface.find('\n') != -1) interface.replace(QChar('\n'),"");

basePath = (QString)getenv("HOME") + "/.kndiswrapper/" + interface;
QDir tempDir(basePath);
QString comment = tr("Here you can setup various parameter for your wireless network\n\n\
The mode of operation can be set to Ad-hoc for a peer-to-peer network without an Access-Point \
or to Managed for a network with an Access-Point.\n\nSpecify the essid for your network (either by \
scanning or entering manually). This information is needed to make your pc a part of a network cell.\n\
If you use a managed network and do not use WPA as authentication you can leave this field empty. \
In this case the wifi-adapter uses the access-point with the best signal quality\n\n\
Within a network you have to specify the authentication-mode. Here you can choose between WEP and WPA. \
WEP (Wired Equivalent Privacy) is a system for encryption of networks with optional authentication. \
Usually when using WEP, the mode WEP-Open is approriate (no authentication).\n\n\
WEP is regarded insecure. Therefore WPA (Wifi Protected Access) was implemented. But not every hardware support WPA. \
WPA can only be uses in Managed mode.\n\nYou can enter the key in HEX, ASCII or a Passphrase.\nIf you use WEB as authentication mode \
you have to enter a 10 digit HEX value for a 64-bit key, a 26 digit HEX value for a 128-bit key, a 32 digit HEX value for a 156-bit key and 58 digits for a 256-bit key. \
You can alternatively enter a 5 characters for a 64-bit key, 13 characters for a 128-bit key, 16 characters for a 156-bit key and \
29 characters for a 256-bit key. But keep in mind that not every hardware supports keys up to 256 bit.\n\n\
If you use WPA you can enter the key in HEX. Therefore you have to enter a 64-digit Hex-key or you can alternatively enter a passphrase (8 - 63 characters)");

        this->setFixedSize(600,630);
        this->setCaption("Wifi-Setup");


        if (tempDir.exists(basePath,true) == false) tempDir.mkdir(basePath,true);
        deviceLabel = new QLabel(tr("Device : ") + interface,this,"deviceLabel");
        deviceLabel->setFont(QFont("Helvetic",20,QFont::Bold,true));
        deviceLabel->setFrameStyle(QFrame::Panel|QFrame::Raised);
        deviceLabel->setBackgroundColor(QColor("#d0d0d0"));

        mainFrame = new QFrame(this,"mainFrame");
        mainFrame->setFrameStyle(QFrame::Panel|QFrame::Raised);
        commentFrame = new QFrame(this,"commentFrame");
        commentFrame->setFrameStyle(QFrame::Panel|QFrame::Sunken);
        ipSetupFrame = new QFrame(this,"ipSetupFrame");
        ipSetupFrame->setFrameStyle(QFrame::Panel|QFrame::Sunken);
        profileFrame = new QFrame(this,"profileFrame");
        profileFrame->setFrameStyle(QFrame::Panel|QFrame::Raised);
        profileFrame->setBackgroundColor(QColor("#d0d0d0"));

        commentEdit = new QMultiLineEdit(this,"commentEdit");
        commentEdit->setText(tr(comment));
        commentEdit->setFont(QFont("Helvetic",10,QFont::Normal,true));
        commentEdit->setReadOnly(true);
        QToolTip::add(commentEdit,tr("Read this for further information about how to setup your wireless lan"));

        confLabel = new QLabel(tr("Profile : "),this,"confLabel");
        confLabel->setFont(QFont("Helvetic",12,QFont::Normal,false));
        confLabel->setBackgroundColor(QColor("#d0d0d0"));
        QToolTip::add(confLabel,tr("If multiple configurations are found, you can select the active configuration"));

        confComboBox = new QComboBox(this,"QComboBox");
        confComboBox->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(confComboBox,tr("If multiple configurations are found, you can select the active configuration"));

        delButton = new QPushButton("X",this,"delButton");
        delButton->setPixmap(QPixmap(stop_xpm));
        QToolTip::add(delButton,tr("Removes the actual configuration file"));

        buttonGroup = new QButtonGroup(tr("Key-Format"),this,"buttonGroup");
        QToolTip::add(buttonGroup,tr("Select the way you want to enter your key"));
        buttonGroup->setFrameStyle(QFrame::Panel|QFrame::Sunken);
        hexButton = new QRadioButton("hex",this,"hexButton");
        asciiButton = new QRadioButton("ascii",this,"asciiButton");
        passphraseButton = new QRadioButton("Passphrase",this,"passphraseButton");
        buttonGroup->insert(hexButton);
        buttonGroup->insert(asciiButton);
        buttonGroup->insert(passphraseButton);
        hexButton->setChecked(true);

        configGroup = new QButtonGroup(tr("Config"),this,"configGroup");
        configGroup->setFrameStyle(QFrame::Panel|QFrame::Raised);
        QToolTip::add(configGroup,tr("Select whether you want to config interface with a static IP or via DHCP"));
        ipConfig = new QRadioButton(tr("Static IP"),this,"ipConfig");
        dhcpConfig = new QRadioButton(tr("DHCP"),this,"dhcpConfig");
        configGroup->insert(ipConfig);
        configGroup->insert(dhcpConfig);
        ipConfig->setChecked(true);

        keyLengthLabel = new QLabel("0",this,"keyLengthLabel");
        keyLengthLabel->setFrameStyle(QFrame::Panel|QFrame::Sunken);
        keyLengthLabel->setFont(QFont("Helvetic",10,QFont::Normal,false));
        keyLengthLabel->setBackgroundColor(QColor(red));
        QToolTip::add(keyLengthLabel,tr("Shows the number of character and whether the key is valid (green) or not (red)"));

        keyEdit = new QLineEdit("",this,"keyEdit");
        keyEdit->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(keyEdit,tr("Enter the hex-key or passphrase"));
        connect(keyEdit,SIGNAL(textChanged(const QString &)),this,SLOT(slot_keyTextChanged(const QString &)));

        modeLabel = new QLabel(tr("Mode of Operation"),this,"modeLabel");
        essidLabel = new QLabel(tr("ESSID"),this,"essidLabel");
        cryptLabel = new QLabel(tr("Authentication"),this,"cryptLabel");
        essidComboBox = new QComboBox(this,"essidComboBox");
        QToolTip::add(essidComboBox,tr("Select a network or enter the ESSID manually"));
        cryptComboBox = new QComboBox(this,"cryptComboBox");
        QToolTip::add(cryptComboBox,tr("Select which encryption mode you want to use"));
        modeLabel->setFont(QFont("Helvetic",12,QFont::Normal,false));
        essidLabel->setFont(QFont("Helvetic",12,QFont::Normal,false));
        essidComboBox->setFont(QFont("Helvetic",12,QFont::Normal,false));
        cryptLabel->setFont(QFont("Helvetic",12,QFont::Normal,false));
        cryptComboBox->setFont(QFont("Helvetic",12,QFont::Normal,false));

        modeComboBox = new QComboBox(this,"modeComboBox");
        modeComboBox->insertItem("Managed");
        modeComboBox->insertItem("Ad-hoc");
        connect(modeComboBox,SIGNAL(activated(int)),this,SLOT(slot_modeComboBoxActivated(int)));

        cryptComboBox->insertItem("WEP-Open");
        cryptComboBox->insertItem("WEP-Shared Key");
        cryptComboBox->insertItem("WPA-PSK (WPA-Home)");

        scanButton = new QPushButton(tr("ESSID-Scan"),this,"scanButton");
        scanButton->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(scanButton,tr("Scan for available networks"));

        editButton = new QPushButton("...",this,"editButton");
        editButton->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(editButton,tr("Enter the name of your network (ESSID) manually"));

        scanProcess = new QProcess(this);
        find_wpa_supplicantProcess = new QProcess(this);
        find_iwlistProcess = new QProcess(this);
        find_iwconfigProcess = new QProcess(this);
        find_dhclientProcess = new QProcess(this);
        get_hostname = new QProcess(this);

        findResult = tr("Required Programms:\n\n");

        ipAddrLabel = new QLabel(tr("IP-Addr : "),this,"ipAddrLabel");
        ipAddrLabel->setFont(QFont("Helvetic",12,QFont::Normal,false));
        netMaskLabel = new QLabel(tr("Netmask : "),this,"netMaskLabel");
        netMaskLabel->setFont(QFont("Helvetic",12,QFont::Normal,false));
        defaultRouteLabel = new QLabel(tr("Default GW (opt.) : "),this,"defaultRouteLabel");
        defaultRouteLabel->setFont(QFont("Helvetic",12,QFont::Normal,false));
        DNSLabel = new QLabel(tr("DNS Entry (opt)"),this,"DNSLabel");
        DNSLabel->setFont(QFont("Helvetic",12,QFont::Normal,false));
        preConfScriptLabel = new QLabel(tr("pre Conf. Script"),this,"preConfScriptLabel");
        preConfScriptLabel->setFont(QFont("Helvetic",12,QFont::Normal,false));
        postConfScriptLabel = new QLabel(tr("post Conf. Script"),this,"postConfScriptLabel");
        postConfScriptLabel->setFont(QFont("Helvetic",12,QFont::Normal,false));

        ipAddrEdit = new QLineEdit("",this,"ipAddrEdit");
        ipAddrEdit->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(ipAddrEdit,tr("Enter the static IP-Addr of the interface (192.168.2.1)"));
        netMaskEdit = new QLineEdit("",this,"netMaskEdit");
        netMaskEdit->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(netMaskEdit,tr("Enter the desired Netmask (255.255.255.0)"));
        defaultRouteEdit = new QLineEdit("",this,"defaultRouteEdit");
        defaultRouteEdit->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(defaultRouteEdit,tr("(Optional) The IP Address of the Gateway (e.g. DSL Route to the internet)\nIf you have a working default Gateway you can ommit this field"));
        DNSEdit = new QLineEdit("",this,"DNSEdit");
        DNSEdit->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(DNSEdit,tr("Optional a DNS Entry for the /etc/resolv.conf file.\nIf you have a working DNS entry you can ommit this field."));
        preConfScriptEdit = new QLineEdit(this,"preConfScriptEdit");
        preConfScriptEdit->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(preConfScriptEdit,tr("A script, that should be startet before the network configuration is done."));
        postConfScriptEdit = new QLineEdit(this,"postConfScriptEdit");
        postConfScriptEdit->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(postConfScriptEdit,tr("A script, that should be startet after the network configurationis done."));

        preConfButton = new QPushButton("...",this,"preConfButton");
        preConfButton->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(preConfButton,tr("Select a script to run"));
        postConfButton = new QPushButton("...",this,"postConfButton");
        postConfButton->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(postConfButton,tr("Select a script to run"));

        saveButton = new QPushButton(QPixmap(saveas_xpm),tr("save as"),this,"saveButton");
        saveButton->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(saveButton,tr("Save the actual configuration"));

        closeButton = new QPushButton(QPixmap(close_xpm),tr("close"),this,"closeButton");
        closeButton->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(closeButton,tr("close this dialog window without doing anything"));

        submitButton = new QPushButton(QPixmap(activate_xpm),tr("Activate"),this,"submitButton");
        submitButton->setFont(QFont("Helvetic",12,QFont::Normal,false));
        QToolTip::add(submitButton,tr("Try to activate the new network configuration"));
        submitButton->setEnabled(false);


        connect(scanProcess,SIGNAL(processExited()),this,SLOT(slot_scanProcessExited()));
        connect(scanButton,SIGNAL(clicked()),this,SLOT(slot_scanButtonClicked()));
        connect(editButton,SIGNAL(clicked()),this,SLOT(slot_editButtonClicked()));
        connect(cryptComboBox,SIGNAL(activated(int)),this,SLOT(slot_cryptComboBoxActivated(int)));
        connect(configGroup,SIGNAL(clicked(int)),this,SLOT(slot_configGroupClicked(int)));
        connect(buttonGroup,SIGNAL(clicked(int)),this,SLOT(slot_modeComboBoxActivated(int)));
        connect(closeButton,SIGNAL(clicked()),this,SLOT(slot_closeButtonClicked()));
        connect(saveButton,SIGNAL(clicked()),this,SLOT(slot_saveButtonClicked()));
        connect(find_wpa_supplicantProcess,SIGNAL(processExited()),this,SLOT(slot_find_wpa_supplicantProcessExited()));
        connect(find_iwlistProcess,SIGNAL(processExited()),this,SLOT(slot_find_iwlistProcessExited()));
        connect(find_iwconfigProcess,SIGNAL(processExited()),this,SLOT(slot_find_iwconfigProcessExited()));
        connect(find_dhclientProcess,SIGNAL(processExited()),this,SLOT(slot_find_dhclientProcessExited()));
        connect(submitButton,SIGNAL(clicked()),this,SLOT(slot_submitButtonClicked()));
        connect(get_hostname,SIGNAL(processExited()),this,SLOT(slot_getHostnameProcessExited()));
        connect(confComboBox,SIGNAL(activated(int)),this,SLOT(loadStage2(int)));
        connect(delButton,SIGNAL(clicked()),this,SLOT(slot_delButtonClicked()));
        connect(preConfButton,SIGNAL(clicked()),this,SLOT(slot_preConfButtonClicked()));
        connect(postConfButton,SIGNAL(clicked()),this,SLOT(slot_postConfButtonClicked()));

        initWidget();
}

void wifiSetup::initWidget(){
    deviceLabel->setGeometry(205,5,this->width()-210,40);
    mainFrame->setGeometry(205,50,this->width()-210,this->height()-100);
    commentFrame->setGeometry(5,5,mainFrame->x()-10,this->height()-10);
    profileFrame->setGeometry(mainFrame->x()+5,mainFrame->y()+5,mainFrame->width()-10,40);
    commentEdit->setGeometry(commentFrame->x()+5,commentFrame->y()+5,commentFrame->width()-10,commentFrame->height()-10);
    confLabel->setGeometry(mainFrame->x()+10,mainFrame->y()+15,120,20);
    confComboBox->setGeometry(confLabel->x()+confLabel->width()+5,confLabel->y(),this->width()-390,confLabel->height());
    delButton->setGeometry(confComboBox->x()+confComboBox->width()+5,confComboBox->y()-5,30,30);
    modeLabel->setGeometry(confLabel->x(),confLabel->y()+confLabel->height()+25,confLabel->width()+20,20);
    modeComboBox->setGeometry(modeLabel->x()+modeLabel->width()+5,modeLabel->y(),110,modeLabel->height());
    essidLabel->setGeometry(modeLabel->x(),modeLabel->y()+modeLabel->height()+15,60,modeLabel->height());
    essidComboBox->setGeometry(essidLabel->x()+essidLabel->width()+5,essidLabel->y(),mainFrame->width()-essidLabel->width()-140,essidLabel->height());
    editButton->setGeometry(essidComboBox->x()+essidComboBox->width()+5,essidLabel->y(),20,essidLabel->height());
    scanButton->setGeometry(editButton->x()+editButton->width()+5,essidComboBox->y(),90,essidComboBox->height());
    cryptLabel->setGeometry(essidLabel->x(),essidLabel->y()+essidLabel->height()+15,120,essidLabel->height());
    cryptComboBox->setGeometry(cryptLabel->x()+cryptLabel->width()+5,cryptLabel->y(),mainFrame->width()-cryptLabel->width()-20,essidComboBox->height());
    buttonGroup->setGeometry(cryptLabel->x(),cryptLabel->y()+cryptLabel->height()+15,mainFrame->width()-15,90);
    hexButton->setGeometry(buttonGroup->x()+15,buttonGroup->y()+20,(buttonGroup->width()-50)/3,cryptLabel->height());
    asciiButton->setGeometry(hexButton->x()+hexButton->width()+5,buttonGroup->y()+20,(buttonGroup->width()-15)/3,cryptLabel->height());
    passphraseButton->setGeometry(asciiButton->x()+asciiButton->width()+5,buttonGroup->y()+20,(buttonGroup->width()-15)/3,cryptLabel->height());
    keyEdit->setGeometry(buttonGroup->x()+5,asciiButton->y()+35,buttonGroup->width()-50,hexButton->height());
    keyLengthLabel->setGeometry(keyEdit->x()+keyEdit->width()+5,keyEdit->y(),30,keyEdit->height());
    ipSetupFrame->setGeometry(buttonGroup->x(),buttonGroup->y()+buttonGroup->height()+5,mainFrame->width()-15,mainFrame->height()-(buttonGroup->y()+buttonGroup->height())+35);
    configGroup->setGeometry(ipSetupFrame->x()+5,ipSetupFrame->y()+5,ipSetupFrame->width()-10,40);
    ipConfig->setGeometry(configGroup->x()+15,configGroup->y()+15,(configGroup->width()-25) / 2,hexButton->height());
    dhcpConfig->setGeometry(ipConfig->x()+ipConfig->width()+5,ipConfig->y(),ipConfig->width(),ipConfig->height());
    ipAddrLabel->setGeometry(ipSetupFrame->x()+5,configGroup->y()+configGroup->height()+15,cryptLabel->width(),cryptLabel->height());
    ipAddrEdit->setGeometry(ipAddrLabel->x()+ipAddrLabel->width()+5,ipAddrLabel->y(),ipSetupFrame->width()-15-ipAddrLabel->width(),ipAddrLabel->height());
    netMaskLabel->setGeometry(ipAddrLabel->x(),ipAddrLabel->y()+ipAddrLabel->height()+15,ipAddrLabel->width(),ipAddrLabel->height());
    netMaskEdit->setGeometry(netMaskLabel->x()+netMaskLabel->width()+5,netMaskLabel->y(),ipAddrEdit->width(),ipAddrEdit->height());
    defaultRouteLabel->setGeometry(netMaskLabel->x(),netMaskLabel->y()+netMaskLabel->height()+15,netMaskLabel->width(),netMaskLabel->height());
    defaultRouteEdit->setGeometry(netMaskLabel->x()+netMaskLabel->width()+5,defaultRouteLabel->y(),ipAddrEdit->width(),ipAddrEdit->height());
    DNSLabel->setGeometry(defaultRouteLabel->x(),defaultRouteLabel->y()+defaultRouteLabel->height()+15,defaultRouteLabel->width(),defaultRouteLabel->height());
    DNSEdit->setGeometry(DNSLabel->x()+DNSLabel->width()+5,DNSLabel->y(),defaultRouteEdit->width(),defaultRouteEdit->height());
    preConfScriptLabel->setGeometry(DNSLabel->x(),DNSLabel->y()+DNSLabel->height()+15,DNSLabel->width(),DNSLabel->height());
    preConfScriptEdit->setGeometry(preConfScriptLabel->x()+preConfScriptLabel->width()+5,preConfScriptLabel->y(),DNSEdit->width()-40,DNSEdit->height());
    preConfButton->setGeometry(preConfScriptEdit->x()+preConfScriptEdit->width()+5,preConfScriptEdit->y(),30,20);
    postConfScriptLabel->setGeometry(preConfScriptLabel->x(),preConfScriptLabel->y()+preConfScriptLabel->height()+15,preConfScriptLabel->width(),preConfScriptLabel->height());
    postConfScriptEdit->setGeometry(postConfScriptLabel->x()+postConfScriptLabel->width()+5,postConfScriptLabel->y(),preConfScriptEdit->width(),preConfScriptEdit->height());
    postConfButton->setGeometry(postConfScriptEdit->x()+postConfScriptEdit->width()+5,postConfScriptEdit->y(),30,20);
    saveButton->setGeometry(mainFrame->x()+5,ipSetupFrame->y()+ipSetupFrame->height()+15,(mainFrame->width()-20) / 3,editButton->height() + 10);
    submitButton->setGeometry(saveButton->x()+saveButton->width()+5,saveButton->y(),saveButton->width(),saveButton->height());
    closeButton->setGeometry(submitButton->x()+submitButton->width()+5,submitButton->y(),submitButton->width(),submitButton->height());

    find_wpa_supplicantProcess->clearArguments();
    find_wpa_supplicantProcess->addArgument("whereis");
    find_wpa_supplicantProcess->addArgument("wpa_supplicant");
    find_wpa_supplicantProcess->start();
}

wifiSetup::~wifiSetup()
{
}


void wifiSetup::slot_scanButtonClicked(){
    enableControls(false);
    scanProcess->clearArguments();
    scanProcess->addArgument("/usr/sbin/iwlist");
    scanProcess->addArgument(interface);
    scanProcess->addArgument("scanning");
    scanProcess->start();
}

void wifiSetup::slot_scanProcessExited(){
    QString buffer="";
    int start,end;
    int hiddenCount=0;
    QString entry;

    while (scanProcess->canReadLineStdout()) buffer = buffer + scanProcess->readLineStdout() + "\n";
    while (scanProcess->canReadLineStderr()) buffer = buffer + scanProcess->readLineStderr() + "\n";
    if (buffer.find("No scan results") != -1){
        if (QMessageBox::information(this,tr("Question"),tr("No network essid found. Do you want to enter a network essid manually"),QMessageBox::Yes,QMessageBox::No) == QMessageBox::Yes){
            essidComboBox->clear();
            buffer="";
            buffer = QInputDialog::getText(tr("ESSID"),tr("Enter the ESSID of your network:"),QLineEdit::Normal,buffer,0,this,"inputDialog");
            if (buffer != ""){
                essidComboBox->insertItem(buffer);
                essidComboBox->setCurrentItem(0);
            }
            if ((essidComboBox->currentText() == "") && (cryptComboBox->currentItem() == 2)){
              submitButton->setEnabled(false);
            } else {
              submitButton->setEnabled(true);
            }
        }
    } else {
        essidComboBox->clear();
        while (buffer.find("ESSID:") >= 0){
            start = buffer.find("ESSID:");
            start = start + 6;
            end = buffer.find("\n",start);
            end = end-start;
            entry=buffer.mid(start+1,end-2);
            buffer=buffer.mid(start+end+7);
            if (entry != ""){
                essidComboBox->insertItem(entry);
            } else {
                hiddenCount++;
            }
        }
        if ((essidComboBox->count() == 0) && (cryptComboBox->currentItem() == 2)){
          submitButton->setEnabled(false);
        } else {
          submitButton->setEnabled(true);
        }
        if (hiddenCount > 0) QMessageBox::information(this,tr("Info"),tr("There was at least one hidden cell found!"));
        essidComboBox->setCurrentItem(0);
    }
    enableControls(true);
}

void wifiSetup::enableControls(bool mode){
    scanButton->setEnabled(mode);
    modeComboBox->setEnabled(mode);
    essidComboBox->setEnabled(mode);
    cryptComboBox->setEnabled(mode);
    editButton->setEnabled(mode);
}

void wifiSetup::slot_editButtonClicked(){
    QString buffer;
    essidComboBox->clear();
    buffer = QInputDialog::getText(tr("ESSID"),tr("Enter the ESSID of your network:"),QLineEdit::Normal,buffer,0,this,"inputDialog");
    if (buffer != ""){
        essidComboBox->insertItem(buffer);
        essidComboBox->setCurrentItem(0);
    }
    if ((essidComboBox->currentText() == "") && (cryptComboBox->currentItem() == 2)){
      submitButton->setEnabled(false);
    } else {
      submitButton->setEnabled(true);
    }
}

void wifiSetup::slot_modeComboBoxActivated(int index){
    if (index == 1){ // is Ad-hoc
        if (cryptComboBox->currentItem() == 2) { // WPA-PSK
            cryptComboBox->setCurrentItem(0);
            hexButton->setEnabled(true);
            asciiButton->setEnabled(true);
            passphraseButton->setEnabled(false);
            if (passphraseButton->isChecked() == true) hexButton->setChecked(true);
        }
    }
    slot_keyTextChanged(keyEdit->text());
}

void wifiSetup::slot_cryptComboBoxActivated(int index){

    if (index == 2) { // is WPA-PSK
        if (modeComboBox->currentItem() == 1){ // Ad-hoc
            modeComboBox->setCurrentItem(0);
            hexButton->setEnabled(true);
            asciiButton->setEnabled(false);
            passphraseButton->setEnabled(true);
            if (asciiButton->isChecked() == true) hexButton->setChecked(true);
        } else {
            hexButton->setEnabled(true);
            asciiButton->setEnabled(false);
            passphraseButton->setEnabled(true);
            if (asciiButton->isChecked() == true) hexButton->setChecked(true);
        }
    } else {
        hexButton->setEnabled(true);
        asciiButton->setEnabled(true);
        passphraseButton->setEnabled(false);
        if (passphraseButton->isChecked() == true) hexButton->setChecked(true);
    }
    if ((essidComboBox->currentText().length() == 0) && (cryptComboBox->currentItem() == 2)){
      submitButton->setEnabled(false);
    } else {
      submitButton->setEnabled(true);
    }
    slot_keyTextChanged(keyEdit->text());
}

void wifiSetup::slot_keyTextChanged(const QString & text){
    int len=text.length();
    bool valid=false;

// ASCII Key (nur WEP)
    if (asciiButton->isChecked() == true){
        if ((len == 5) || (len == 13)){
            valid=true;
        } else {
            valid=false;
        }
    } else {

    // HEX Key (WEP oder WPA)
        if (hexButton->isChecked() == true){
            if (cryptComboBox->currentItem() == 2){
      // WPA
                len == 64?valid=true:valid=false;
            } else {
      // WEP
                ((len == 10) || (len == 26))?valid=true:valid=false;
            }
            if (valid) valid=checkValidHex(keyEdit->text());
        } else {

      // Passphrase
            if (passphraseButton->isEnabled() == true){
                ((len >= 8) && (len <= 63))?valid=true:valid=false;
            }
        }
    }

    valid==true?keyLengthLabel->setBackgroundColor(QColor(green)):keyLengthLabel->setBackgroundColor(QColor(red));
    valid==true?submitButton->setEnabled(true):submitButton->setEnabled(false);
    keyLengthLabel->setText(QString::number(text.length()));
}

void wifiSetup::slot_configGroupClicked(int index){
    bool mode;

    index == 0?mode=true:mode=false;
    ipAddrEdit->setEnabled(mode);
    netMaskEdit->setEnabled(mode);
    defaultRouteEdit->setEnabled(mode);
    DNSEdit->setEnabled(mode);

}

void wifiSetup::slot_closeButtonClicked(){
    this->done(0);
}

void wifiSetup::loadConfig(){
QDir tempDir((QString)getenv("HOME") + "/.kndiswrapper/" + interface);
    tempDir.setNameFilter("*.conf");
QStringList fileList = tempDir.entryList();
QString entry;

    for ( QStringList::Iterator it = fileList.begin(); it != fileList.end(); ++it ) {
        entry = *it;
        entry = entry.mid(0,entry.find(".conf"));
        confComboBox->insertItem(entry);
    }
    if (confComboBox->count() == 0) confComboBox->insertItem("wlan0");
    confComboBox->setCurrentItem(0);
    loadStage2(confComboBox->currentItem());
}

void wifiSetup::loadStage2(int entry){
QString oldPath=(QString)getenv("HOME") + "/.kndiswrapper/" + confComboBox->currentText() + ".conf";
int count=0;
QString line;
    entry = entry; // just to keep the compiler happy .... entry is not used.

    filePath=(QString)getenv("HOME") + "/.kndiswrapper/" + interface + "/" + confComboBox->currentText() + ".conf";

    if (QFile::exists(oldPath)) system((QString)"mv " + oldPath + " " + filePath);
    QFile loadFile(filePath);
    if (loadFile.open(IO_ReadOnly)){
        QTextStream stream(&loadFile);
        while (!stream.eof()){
            line = stream.readLine();
            switch (count){
                case 0  : line == "Managed"?modeComboBox->setCurrentItem(0):modeComboBox->setCurrentItem(1);
                            break;
                case 1  : essidComboBox->clear();
                          essidComboBox->insertItem(line);
                          essidComboBox->setCurrentItem(0);
                            break;
                case 2  : if (line == "WEP-Open") cryptComboBox->setCurrentItem(0);
                          if (line == "WEP-Shared Key") cryptComboBox->setCurrentItem(1);
                          if (line == "WPA-PSK (WPA-Home)") {
                            cryptComboBox->setCurrentItem(2);
                            asciiButton->setEnabled(false);
                            passphraseButton->setEnabled(true);
                          } else {
                            passphraseButton->setEnabled(false);
                          }
                            break;
                case 3  : if (line == "hex") hexButton->setChecked(true);
                          if (line == "ascii") asciiButton->setChecked(true);
                          if (line == "passphrase") passphraseButton->setChecked(true);
                            break;
                case 4  : keyEdit->setText(line);
                            break;
                case 5  : if (line == "dhcp") {
                            dhcpConfig->setChecked(true);
                            ipAddrEdit->setEnabled(false);
                            netMaskEdit->setEnabled(false);
                            defaultRouteEdit->setEnabled(false);
                            DNSEdit->setEnabled(false);
                          } else {
                            ipConfig->setChecked(true);
                            ipAddrEdit->setEnabled(true);
                            netMaskEdit->setEnabled(true);
                            defaultRouteEdit->setEnabled(true);
                            DNSEdit->setEnabled(true);
                          }
                            break;
                case 6  : ipAddrEdit->setText(line);
                            break;
                case 7  : netMaskEdit->setText(line);
                            break;
                case 8  : defaultRouteEdit->setText(line);
                            break;
                case 9  : DNSEdit->setText(line);
                            break;
                case 10  : preConfScriptEdit->setText(line);
                            break;
                case 11  : postConfScriptEdit->setText(line);
                            break;
            }
            count++;
        }
        loadFile.close();
        if ((essidComboBox->currentText() == "") && (cryptComboBox->currentItem() == 2)){
          submitButton->setEnabled(false);
        } else {
          submitButton->setEnabled(true);
        }
    }
}

void wifiSetup::slot_saveButtonClicked(){
QString buffer = confComboBox->currentText();

    buffer = QInputDialog::getText(tr("Save as"),tr("Save configuration as (press Enter to save it with its actual name):"),QLineEdit::Normal,buffer,0,this,"inputDialog");
    if (buffer != ""){
      bool exist=false;
      for (int i=0;i<confComboBox->count();i++){
        if (buffer == confComboBox->text(i)) exist=true;
      }
      if (exist == false) {
        confComboBox->insertItem(buffer);
        confComboBox->setCurrentItem(confComboBox->count()-1);
      }
      filePath = (QString)getenv("HOME") + "/.kndiswrapper/" + interface + "/" + buffer + ".conf";
      QFile saveFile(filePath);
      if (saveFile.open(IO_WriteOnly)){
        QTextStream stream(&saveFile);
        stream << modeComboBox->currentText() << "\n";
        stream << essidComboBox->currentText() << "\n";
        stream << cryptComboBox->currentText() << "\n";
        if (hexButton->isChecked()) stream << "hex" << "\n";
        if (asciiButton->isChecked()) stream << "ascii" << "\n";
        if (passphraseButton->isChecked()) stream << "passphrase" << "\n";
        stream << keyEdit->text() << "\n";
        if (ipConfig->isChecked()) stream << "static" << "\n";
        if (dhcpConfig->isChecked()) stream << "dhcp" << "\n";
        stream << ipAddrEdit->text() << "\n";
        stream << netMaskEdit->text() << "\n";
        stream << defaultRouteEdit->text() << "\n";
        stream << DNSEdit->text() << "\n";
        stream << preConfScriptEdit->text() << "\n";
        stream << postConfScriptEdit->text() << "\n";
        saveFile.close();
        QMessageBox::information(this,tr("Info"),tr("Configuration for ") + confComboBox->currentText() + tr(" saved!"));
    } else {
        QMessageBox::information(this,tr("WARNING"),tr("Couldn't save the configuration. Unknown Error!"));
    }
  }
}

void wifiSetup::slot_find_wpa_supplicantProcessExited(){
    QString line;
    int start,end;

    while (find_wpa_supplicantProcess->canReadLineStdout()){
        line = line + find_wpa_supplicantProcess->readLineStdout();
    }

    if (line != "wpa_supplicant:"){
        start=line.find("/");
        end=line.find(" ",start);
        end = end - start;
        line=line.mid(start,end);
        wpa_supplicantPath = line;
        if (wpa_supplicantPath == "") {
            findResult = findResult + tr("wpa_supplicant not found. No WPA encryption available\n");
            cryptComboBox->removeItem(2);
        } else {
            findResult = findResult + tr("wpa_supplicant found : ") + line + "\n";
        }
    } else {
        findResult = findResult + tr("wpa_supplicant not found. No WPA encryption available\n");
        cryptComboBox->removeItem(2);
    }

    find_iwlistProcess->clearArguments();
    find_iwlistProcess->addArgument("whereis");
    find_iwlistProcess->addArgument("iwlist");
    find_iwlistProcess->start();
}

void wifiSetup::slot_find_iwlistProcessExited(){
    QString line;
    int start,end;

    while (find_iwlistProcess->canReadLineStdout()){
        line = line + find_iwlistProcess->readLineStdout();
    }

    if (line != "iwlist:"){
        start=line.find("/");
        end=line.find(" ",start);
        end = end - start;
        line=line.mid(start,end);
        iwlistPath = line;
        if (iwlistPath == "") {
            findResult = findResult + ("iwlist not found. No network scanning possible.\n");
            scanButton->setEnabled(false);
        } else {
            findResult = findResult + tr("iwlist found : ") + line + "\n";
        }
    } else {
        findResult = findResult + ("iwlist not found. No network scanning possible.\n");
        scanButton->setEnabled(false);
    }

    find_iwconfigProcess->clearArguments();
    find_iwconfigProcess->addArgument("whereis");
    find_iwconfigProcess->addArgument("iwconfig");
    find_iwconfigProcess->start();
}

void wifiSetup::slot_find_iwconfigProcessExited(){
    QString line;
    int start,end;

    while (find_iwconfigProcess->canReadLineStdout()){
        line = line + find_iwconfigProcess->readLineStdout();
    }

    if (line != "iwconfig:"){
        start=line.find("/");
        end=line.find(" ",start);
        end = end - start;
        line=line.mid(start,end);
        iwconfigPath = line;
        if (iwconfigPath == "") {
            findResult = findResult + tr("iwconfig not found. No configuration of wlan possible.\n");
            submitButton->setEnabled(false);
        } else {
            findResult = findResult + tr("iwconfig found : ") + line + "\n";
        }
    } else {
        findResult = findResult + tr("iwconfig not found. No configuration of wlan possible.\n");
        submitButton->setEnabled(false);
    }

    get_hostname->clearArguments();
    get_hostname->addArgument("hostname");
    get_hostname->start();
}

void wifiSetup::slot_getHostnameProcessExited(){

    while (get_hostname->canReadLineStdout()){
        hostname = get_hostname->readLineStdout();
    }

    find_dhclientProcess->clearArguments();
    find_dhclientProcess->addArgument("whereis");
    find_dhclientProcess->addArgument("dhcpcd");
    find_dhclientProcess->start();
}

void wifiSetup::slot_find_dhclientProcessExited(){
    QString line;
    int start,end;

    while (find_dhclientProcess->canReadLineStdout()){
        line = line + find_dhclientProcess->readLineStdout();
    }

    if (line != "dhclient:"){
        start=line.find("/");
        end=line.find(" ",start);
        end = end - start;
        line=line.mid(start,end);
        dhclientPath = line;
        if (dhclientPath == "") {
            findResult = findResult + tr("dhcpcd not found. No dhcp configuration possible. Static IP only\n");
            dhcpConfig->setEnabled(false);
        } else {
            findResult = findResult + tr("dhcpcd found : ") + line + "\n";
        }
    } else {
        dhcpConfig->setEnabled(false);
        findResult = findResult + tr("dhcpcd not found. No dhcp configuration possible. Static IP only\n");
    }
    if (findResult.find(tr("not found")) != -1)  QMessageBox::information(this,tr("Warning"),findResult);

    loadConfig();
}

bool wifiSetup::checkValidHex(QString value){
    unsigned int pos=0;
    bool valid=true;
    QString checkString="01234567890abcdefABCDEF";

    while (pos < value.length()){
        if (checkString.find(value.mid(pos,1)) == -1) valid=false;
        pos++;
    }
    return valid;
}

bool wifiSetup::checkIP(QString ipAddress){
bool flag=false;
int ip1=256;
int ip2=256;
int ip3=256;
int ip4=256;
QString part;

    part = ipAddress.mid(0,ipAddress.find("."));
    ip1 = part.toInt();
    ipAddress=ipAddress.mid(ipAddress.find(".")+1);

    part = ipAddress.mid(0,ipAddress.find("."));
    ip2 = part.toInt();
    ipAddress=ipAddress.mid(ipAddress.find(".")+1);

    part = ipAddress.mid(0,ipAddress.find("."));
    ip3 = part.toInt();
    ipAddress=ipAddress.mid(ipAddress.find(".")+1);

    part = ipAddress.mid(0,ipAddress.find("."));
    ip4 = part.toInt();
    ipAddress=ipAddress.mid(ipAddress.find(".")+1);

    if (
        ((ip1 >=1) && (ip1 <=254)) &&
        ((ip2 >=0) && (ip2 <=255)) &&
        ((ip3 >=0) && (ip3 <=255)) &&
        ((ip4 >=1) && (ip4 <=254))
       ) flag = true;

  return flag;
}


bool wifiSetup::checkNetMask(){
bool flag=false;
int oct1=256;
int oct2=256;
int oct3=256;
int oct4=256;
QString netMask = netMaskEdit->text();
QString part;

    part = netMask.mid(0,netMask.find("."));
    oct1 = part.toInt();
    netMask=netMask.mid(netMask.find(".")+1);

    part = netMask.mid(0,netMask.find("."));
    oct2 = part.toInt();
    netMask=netMask.mid(netMask.find(".")+1);

    part = netMask.mid(0,netMask.find("."));
    oct3 = part.toInt();
    netMask=netMask.mid(netMask.find(".")+1);

    part = netMask.mid(0,netMask.find("."));
    oct4 = part.toInt();
    netMask=netMask.mid(netMask.find(".")+1);

    if (
        ((oct1 >=128) && (oct1 <=255)) &&
        ((oct2 >=0) && (oct2 <=255)) &&
        ((oct3 >=0) && (oct3 <=255)) &&
        ((oct4 >=0) && (oct4 <=255))
       ) flag = true;

  return flag;
}


void wifiSetup::slot_submitButtonClicked(){
bool valid = true;
QString essidString = essidComboBox->currentText();
    if (essidString == "") essidString="any";

    QString home = (QString)getenv("HOME") + "/.kndiswrapper";

    if (keyLengthLabel->backgroundColor() == QColor(green)){
        QString scriptPath=home + "/ipup-" + interface + ".sh";
        QFile file(scriptPath);
        if (file.open(IO_WriteOnly)){
            QTextStream stream(&file);
            stream << "#/bin/bash\n# Startup-Script for Interface " << interface << "\n\n";
            stream << "echo \"- Shutting down interface " << interface << " ....\"\n";
            stream << "ifconfig " << interface << " down\n\n";
            if (modeComboBox->currentItem() == 0){
          // managed
                if (cryptComboBox->currentItem() != 2){
            // WEP
                    stream << "if [ -e \"/var/run/wpa_supplicant/kndis-" << interface << ".pid\" ]; then\n  echo \"- Stopping pre-existing wpa_supplicant\"\n  killproc wpa_supplicant\n  sleep 2\n  if [ -e /var/run/wpa_supplicant ];\n  then\n    echo \"- removing staled pid-file and folder ....\"\n    rm -rf /var/run/wpa_supplicant\n    sleep 1\n  fi\nfi\n";
                    stream << "echo \"- Setting up " << interface << " via iwconfig ....\"\n";
                    stream << "sleep 1\n";
                    stream << "iwconfig " << interface << " essid \"" << essidString << "\" mode " << modeComboBox->currentText();
                    cryptComboBox->currentItem() == 0?stream << " key open ":stream << " key restricted ";
                    asciiButton->isChecked() == true?stream << "s:" + keyEdit->text():stream << keyEdit->text();
                    stream << " commit\n\n";
                } else {
            // WPA-PSK
                    stream << "echo \"- Creating conf-file for wpa_supplicant ....\"\n# .... just for info. The Conf-File was already created by the gui\n";
                    stream << "sleep 1\n\n";
                    QString wpaConfFilename=(QString)getenv("HOME") + "/.kndiswrapper/wpa_supplicant-" + interface + ".conf";
                    QFile wpaConfFile(wpaConfFilename);
                    if (wpaConfFile.open(IO_WriteOnly)){
                        QTextStream wpaStream(&wpaConfFile);
                        wpaStream << "ctrl_interface=/var/run/wpa_supplicant\nnetwork={\n  scan_ssid=1\n  ssid=\"" << essidString << "\"\n";
                        wpaStream << "  key_mgmt=WPA-PSK\n";
                        if (hexButton->isChecked()){
                            wpaStream << "  psk=" << keyEdit->text() << "\n}\n";
                        } else {
                            wpaStream << "  psk=\"" << keyEdit->text() << "\"\n}\n";
                        }
                        wpaConfFile.close();
                    }
                    stream << "if [ -e \"/var/run/wpa_supplicant/kndis-" << interface << ".pid\" ]; then\n  echo \"- Stopping pre-existing wpa_supplicant\"\n  killproc wpa_supplicant\n  sleep 2\n  if [ -e /var/run/wpa_supplicant ];\n  then\n    echo \"- removing staled pid-file and folder ....\"\n    rm -rf /var/run/wpa_supplicant\n    sleep 1\n  fi\nfi\n";
                    stream << "echo \"- Starting up wpa_supplicant ....\"\n";
                    stream << "sleep 1\n";
                    stream << "wpa_supplicant -i" << interface << " -c" << wpaConfFilename << " -Dwext -P /var/run/wpa_supplicant/kndis-" << interface << ".pid -B\n\n";
                }
            } else {
          // ad-hoc
                stream << "echo \"- Setting up " << interface << " via iwconfig ....\"\n";
                stream << "sleep 1\n";
                stream << "iwconfig " << interface << " essid \"" << essidString << "\" mode " << modeComboBox->currentText();
                cryptComboBox->currentItem() == 0?stream << " key open ":stream << " key restricted ";
                asciiButton->isChecked() == true?stream << "s:" + keyEdit->text():stream << keyEdit->text();
                stream << " commit\n\n";
            }
        // IP-Config oder dhcp
            if (dhcpConfig->isChecked()){
                stream << "echo \"- terminating existing dhcpcd .... if exist\"\n";
                stream << "killproc dhcpcd\n";
                stream << "rm " << home << "/dhcpcd-" << interface << ".*\n";
                stream << "sleep 1\n\n";
                stream << "echo \"- Setting up interface " << interface << " via dhcpcd .... timeout 30 sec.\"\n";
                stream << "sleep 1\n";
                stream << "ip link set " << interface << " up\n";
                stream << "sleep 1\n";
                stream << "dhcpcd -L " << home << " -C -D -K -R -N -t 999999 -h " << hostname << " " << interface << " &\n";
                stream << "for ((i=0;i<30;i++)); do\n  x=\".\"\n  if [ -e " << home << "/dhcpcd-" << interface << ".pid ]; then continue\n  else\n    sleep 1\n  fi\ndone\n";
                stream << "if [ ! -e " << home << "/dhcpcd-" << interface << ".pid ]; then\n  echo \"- no IP address yet ..... backgrounding.\"\nelse\necho \"- interface set via dhcp\"\nfi\n\n";
            } else {
                if (checkIP(ipAddrEdit->text()) == false){
                  valid=false;
                  QMessageBox::information(this,"ERROR","The IP-Address you entered is not valid!");
                }
                if ((valid) && (checkNetMask() == false)){
                  valid=false;
                  QMessageBox::information(this,"ERROR","The NetMask you entered is not valid!");
                }
                if (defaultRouteEdit->text().length() > 0){
                  if ((valid) && (checkIP(defaultRouteEdit->text()) == false)){
                    valid=false;
                    QMessageBox::information(this,"ERROR","The default Route-Address you entered is not valid!");
                  }
                }

                if (DNSEdit->text().length() > 0){
                  if ((valid) && (checkIP(DNSEdit->text()) == false)){
                    valid=false;
                    QMessageBox::information(this,"ERROR","The DNS-Address you entered is not valid!");
                  }
                }
                stream << "echo \"- Setting up interface " << interface << " via ifconfig ....\"\n";
                stream << "sleep 1\n";
                stream << "ifconfig " << interface << " " << ipAddrEdit->text() << " netmask " << netMaskEdit->text() << " up\n\n";
                if (defaultRouteEdit->text() != "") {
                    stream << "echo \"- Adding default gateway " << defaultRouteEdit->text() << "\"\n";
                    stream << "route add default gw " << defaultRouteEdit->text() << "\n\n";
                }
                if (DNSEdit->text() != ""){
                    stream << "rc=`cat /etc/resolv.conf | grep \"nameserver " << DNSEdit->text() << "\"`\n";
                    stream << "if [ -z \"$rc\" ];\nthen\n  echo \"nameserver " << DNSEdit->text() << "\" >> /etc/resolv.conf\nfi";
                }
            }


            file.close();
            system ("chmod 777 " + scriptPath);
            if (valid){
              wifiStartup * startup = new wifiStartup(this,"startup",scriptPath);
              startup->show();
            }
        }
    } else {
        QMessageBox::information(this,tr("WARNING"),tr("invalid Key. Can not create config file"));
    }
}

void wifiSetup::slot_delButtonClicked(){
  if (confComboBox->currentText() != ""){
    int rc = QMessageBox::question(this,tr("WARNING"),tr("Do you realy want to remove the config-file for ") + confComboBox->currentText(),QMessageBox::Yes,QMessageBox::No);
    if (rc == QMessageBox::Yes){
      QFile tempFile(filePath);
      tempFile.remove();
      confComboBox->clear();
      loadConfig();
    }
  }
}

void wifiSetup::slot_preConfButtonClicked(){
QString Buffer = QFileDialog::getOpenFileName("/","all (*)",this,tr("Select a File"),tr("Select the pre config script.") );
  preConfScriptEdit->setText(Buffer);
}

void wifiSetup::slot_postConfButtonClicked(){
QString Buffer = QFileDialog::getOpenFileName("/","all (*)",this,tr("Select a File"),tr("Select the post config script." ));
  postConfScriptEdit->setText(Buffer);
}

#include "wifisetup.moc"
