"""
Virtualization installation functions.  

Copyright 2007-2008 Red Hat, Inc.
Michael DeHaan <mdehaan@redhat.com>
Lee Faus <lfaus@redhat.com>
Rob Crittenden <rcritten@redhat.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301  USA
"""

import os, sys, time, stat
import tempfile
import random
from optparse import OptionParser
import exceptions
import errno
import re
import virtinst

VMWARE_DIR = "/var/lib/vmware"

# FIXME: are other settings ok?
TEMPLATE = """
#!/usr/bin/vmware
config.version = "8"
virtualHW.version = "4"
scsi0.present = "TRUE"
scsi0.virtualDev = "lsilogic"
scsi0:0.present = "TRUE"
scsi0:0.writeThrough = "TRUE"
ide1:0.present = "TRUE"
ide1:0.deviceType = "atapi-cdrom"
ide1:0.startConnected = "FALSE"
ide1:0.autodetect = "TRUE"
Ethernet0.present = "TRUE"
Ethernet0.AddressType = "static"
Ethernet0.Address = "%(MAC_ADDRESS)s"
Ethernet0.virtualDev = "e1000"
Ethernet0.connectionType = "custom"
Ethernet0.vnet = "vmnet1"
guestOS = "%(GUESTOS)s"
priority.grabbed = "normal"
priority.ungrabbed = "normal"
powerType.powerOff = "hard"
powerType.powerOn = "hard"
powerType.suspend = "hard"
powerType.reset = "hard"
floppy0.present = "FALSE"
scsi0:0.filename = "%(VMDK_IMAGE)s"
displayName = "%(IMAGE_NAME)s"
memsize = "%(MEMORY)s"
annotation = "%(ANNOTATION)s"
"""

# To set the CDROM as an ISO image at install time set:
# ide1:0.filename = "/path/to/iso"
# ide1:0.deviceType = "cdrom-image"

class VirtCreateException(exceptions.Exception):
    pass

def random_mac():
    """
    http://www.vmware.com/support/esx21/doc/esx21admin_MACaddress.html
    Generate a random MAC address.
    Uses OUI 00-50-56, allocated to VMWare, Inc.
    The last 3 fields must be in the range 00:00:00 - 3F:FF:FF
    return: MAC address string
    """
    mac = [ 0x00, 0x50, 0x56,
        random.randint(0x00, 0x3f),
        random.randint(0x00, 0xff),
        random.randint(0x00, 0xff) ]
    return ':'.join(map(lambda x: "%02x" % x, mac))

def make_disk(disksize,image):
    cmd = "vmware-vdiskmanager -c -a lsilogic -s %sGb -t 0 %s" % (disksize, image)
    print "- %s" % cmd
    rc = os.system(cmd)
    if rc != 0:
       raise VirtCreateException("command failed")

def make_vmx(path,vmdk_image,image_name,mac_address,memory,guestos,annotation):
    template_params =  {
        "VMDK_IMAGE"  : vmdk_image,
        "IMAGE_NAME"  : image_name,
        "MAC_ADDRESS" : mac_address.lower(),
        "MEMORY"      : memory,
        "GUESTOS"     : guestos,
        "ANNOTATION"  : annotation
    }
    templated = TEMPLATE % template_params
    fd = open(path,"w+")
    fd.write(templated)
    fd.close()

def start_vm(vmx_file):
    os.chmod(vmx_file,0755)
    cmd = "vmrun start %s nogui" % vmx_file
    print "- %s" % cmd
    rc = os.system(cmd)
    if rc != 0:
       raise VirtCreateException("vm start failed")

def start_install(name=None, ram=None, disks=None, mac=None,
                  uuid=None,  
                  extra=None,
                  vcpus=None, 
                  profile_data=None, bridge=None, arch=None, no_gfx=False, fullvirt=True):

    mac = None
    if not profile_data.has_key("interfaces"):
        print "- vmware installation requires a system, not a profile"
        return 1
    for iname in profile_data["interfaces"]:
        intf = profile_data["interfaces"][iname]
        mac = intf["mac_address"]
    if mac is None:
        print "- no MAC information available in this record, cannot install"
        return 1

    print "DEBUG: name=%s" % name
    print "DEBUG: ram=%s" % ram
    print "DEBUG: mac=%s" % mac
    print "DEBUG: disks=%s" % disks
    print "DEBUG: arch=%s" % arch
    # starts vmware using PXE.  disk/mem info come from Cobbler
    # rest of the data comes from PXE which is also intended
    # to be managed by Cobbler.

    # Store everything in one-place so that removing a VM from the vmware
    # client will remove all files
    if not os.path.exists(VMWARE_DIR):
        os.makedirs(VMWARE_DIR)

    # FIXME. It can support multiple files but either the template would need
    # to be modified or we would need to append the data after the template
    # is created.
    if len(disks) != 1:
       raise VirtCreateException("vmware support is limited to 1 virtual disk")

    diskname = disks[0][0]
    disksize = disks[0][1]

    if not os.path.exists(VMWARE_DIR + "/" + name):
        os.makedirs(VMWARE_DIR + "/" + name)

    image = "%s/%s/%s.vmdk" % (VMWARE_DIR, name, name)
    print "- saving virt disk image as %s" % image
    make_disk(disksize,image)

    vmx = "%s/%s/%s.vmx" % (VMWARE_DIR, name, name)
    print "- saving vmx file as %s" % vmx

    if arch == "x86_64":
        guestos="other26xlinux-64"
    else:
        guestos="linux"

    annotation="Created by koan on %s|0AProfile %s|0ADistro %s" % (time.strftime("%a, %d %b %Y %H:%M:%S"), profile_data['profile_name'], profile_data['distro_name'])

    make_vmx(vmx,image,name,mac,ram,guestos,annotation)
    start_vm(vmx)

    return "use VMware Workstation to connect to instance %s" % name
