/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#ifndef GroupsView_H_Included
#define GroupsView_H_Included

#include <qwidget.h>

#include "AutoPtr.h"


class QPushButton;
class QCheckBox;
class QListView;
class QListViewItem;
class QMutex;

class Process;

class GroupsView : public QWidget
  {
    Q_OBJECT

    private:
      enum E_Action
        {
          c_NoAction,
          c_ListInstalled,
          c_ListAvailable,
          c_Install,
          c_Remove,
          c_GetInfo
        };

      enum E_List
        {
          c_NoList,
          c_Installed,
          c_Available
        };

      enum E_ListSection
        {
          c_NoSection,
          c_SectionInstalled,
          c_SectionAvailable
        };

      enum E_InfoSection
        {
          c_NoInfoSection,
          c_RequiredGroups,
          c_MandatoryPkgs,
          c_DefaultPkgs,
          c_OptionalPkgs
        };

      enum E_Step
        {
          c_Step_1,
          c_Step_2
        };

      struct Section
        {
          typedef std::map<QString, bool> Map;
          typedef AutoPtr<Section>        Ptr;
          typedef std::list<Ptr>          List;

          QString       m_strName;
          QStringList   m_itemList;
          Map           m_itemMap;

                        Section(QString strName = "")
                          : m_strName(strName)
                        {
                        }

          void          addItem(QString strItem);
        };

      struct GroupInfo
        {
          typedef AutoPtr<GroupInfo>            Ptr;
          typedef AutoPtr<Section>              SectionPtr;
          typedef std::map<QString, SectionPtr> SectionMap;

          QString         m_strName,
                          m_strDescription;
          Section::List   m_sectionList;
          SectionMap      m_sectionMap;
          bool            m_bChecked;



                          GroupInfo(QString strName, QString strDescription = "")

                            : m_strName       (strName),
                              m_strDescription(strDescription),
                              m_bChecked      (false)

                          {
                          }

          void            addItemToSection(QString strSection, QString strName);

          SectionPtr      getSection(QString strSection) const;
        };


      typedef std::map<QString, GroupInfo::Ptr>   GroupInfoMap;


    private:
      QPushButton     * m_pButtonListAvail,
                      * m_pButtonListInstalled,
                      * m_pButtonInstall,
                      * m_pButtonUpdate,
                      * m_pButtonRemove,
                      * m_pButtonSelectAll,
                      * m_pButtonDeselectAll;
      QCheckBox       * m_pCheckCache;
      QListView       * m_pListView;
      Process         * m_pProcess;
      QMutex          * m_pMutex;
      E_Action          m_currentAction;
      E_List            m_currentList;
      E_Step            m_currentStep;
      E_ListSection     m_currentSection;
      QString           m_strCurrentInfoSection;
      GroupInfoMap      m_groupInfoMap;
      GroupInfo::Ptr    m_ptrCurrentGroup;
      int               m_numCheckedItems;
      bool              m_bActionsEnabled;


    public:
                        GroupsView        (QWidget * pParent);
      virtual         ~ GroupsView        ();

      void              stopAction        ();
      void              enableActions     (bool bEnable = true);


    private:
      bool              startAction       ();
      void              startStep2        ();
      GroupInfo::Ptr    findGroupInfo     (QString strGroup) const;
      void              addGroup          (QString strGroup);
      void              addSubItems       ();
      QString           getGroupName      (const QListViewItem * pItem) const;
      GroupInfo::Ptr    getGroupInfo      (const QListViewItem * pItem) const;


    signals:
      void              yumInfo           (QString strInfo);
      void              yumError          (QString strError);

      void              actionStarted     (Process * pProcess);
      void              actionTerminated  (Process * pProcess);


    private slots:
      void              slotListItemSelected(QListViewItem * pItem);


    public slots:
      void              slotLaunchFinished();
      void              slotProcessExited ();

      void              slotReadyStdout   ();
      void              slotReadyStderr   ();

      void              slotListAvailable ();
      void              slotListInstalled ();

      void              slotInstallGroup  ();
      void              slotRemoveGroup   ();
  };


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

inline GroupsView::GroupInfo::Ptr
  GroupsView::getGroupInfo(const QListViewItem * pItem) const
{
    return findGroupInfo(getGroupName(pItem));
}

#endif
