/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <qregexp.h>
#include <qfile.h>

#include "StrTok.h"
#include "KYumConf.h"


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

ConfigItem::ConfigItem()
{
    m_bIsGroup   = false;
    m_bIsComment = true;
    m_strName    = "";
    m_strValue   = "";
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool ConfigItem::readFrom(QString strItem)
{
    int   pos1,
          pos2;

    if ( strItem.endsWith("\n") )
        strItem.remove(strItem.length() - 1, 1);

    for ( pos1=0;
          pos1 < (int) strItem.length() && strItem[pos1].isSpace();
          pos1++ )
      ;

    if ( pos1 < (int) strItem.length() )
    {
        m_bIsComment = ('#' == strItem[pos1]);

        if ( m_bIsComment )
            pos1++;

        while ( pos1 < (int) strItem.length() && strItem[pos1].isSpace() )
            pos1++;
    }

    if ( pos1 < (int) strItem.length() && '#' != strItem[pos1] )
    {
        char chSep;

        if ( '[' == strItem[pos1] )
        {
            setGroup(true);
            chSep = ']';
            pos1++;
        }
        else
        {
            setGroup(false);
            chSep = '=';
        }

        for ( pos2 = pos1;
              pos2 < (int) strItem.length() && chSep != strItem[pos2];
              pos2++ )
          ;

        if ( pos2 == (int) strItem.length() )
        {
            if ( isComment() )
                setValue(strItem.mid(pos1));
            else
                setValue("");

            setName("");
        }
        else
        {
            int end = pos2;

            while ( strItem[end - 1].isSpace() )
              end--;

            setName(strItem.mid(pos1, end - pos1).lower());

            if ( !isGroup() )
            {
                while ( ++pos2 < (int) strItem.length() && strItem[pos2].isSpace() )
                  ;

                setValue(strItem.mid(pos2));
            }
        }
    }
    else
        setName("");

    return isValid();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

ConfigItem &  ConfigItem::operator = (const ConfigItem & item)
{
    setName(item.getName());
    setValue(item.getValue());
    setGroup(item.isGroup());

    return *this;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

CfgGroup::CfgGroup(const CfgGroup & group)
{
    *this = group;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void CfgGroup::setValue(QString strName, QString strValue)
{
    CfgItemPtr ptrItem = getItem(strName = strName.lower());

    if ( ptrItem.isNull() )
    {
        m_itemNames.push_back(strName);
        ptrItem = new ConfigItem;

        ptrItem->setName(strName);
        m_itemMap[strName] = ptrItem;
    }

    ptrItem->setValue(strValue);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString CfgGroup::getValue(QString strName) const
{
    QString     strValue;
    CfgItemPtr  ptrItem = getItem(strName);

    if ( ptrItem.isValid() )
        strValue = ptrItem->getValue();

    return strValue;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

ConfigItem::Ptr CfgGroup::getItem(QString strName) const
{
    CfgItemPtr ptrItem;

    try
    {
        CfgItemMap::const_iterator it = m_itemMap.find(strName.lower());

        if ( m_itemMap.end() != it )
            ptrItem = it->second;
    }

    catch(...)
    {
    }

    return ptrItem;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void CfgGroup::deleteItem(QString strName)
{
    try
    {
        strName = strName.lower();

        CfgItemMap::iterator itMap = m_itemMap.find(strName);

        if ( m_itemMap.end() != itMap )
        {
            m_itemMap.erase(itMap);

            QStringList::iterator it;

            for ( it = m_itemNames.begin(); it != m_itemNames.end(); it++ )
            {
                if ( *it == strName )
                {
                    m_itemNames.erase(it);
                    break;
                }
            }
        }
    }

    catch(...)
    {
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString CfgGroup::toString() const
{
    QString                     strResult;
    QStringList::const_iterator it;

    strResult  = "[";
    strResult += getTag();
    strResult += "]\n";

    for ( it = m_itemNames.begin(); it != m_itemNames.end(); it++ )
    {
        if ( !(*it).startsWith("#") )
        {
            QString strValue = getValue(*it);

            if ( !strValue.isEmpty() )
            {
                strResult += *it;
                strResult += '=';
                strResult += getValue(*it);
                strResult += '\n';
            }
        }
    }

    return strResult;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void CfgGroup::setOptions(QString strOptions)
{
    int pos1 = 0, pos2 = 0;

    do
    {
        pos2 = strOptions.find('\n', pos1);

        if ( pos2 < 0 )
            pos2 = strOptions.length();

        if ( pos1 < pos2 )
        {
            ConfigItem  cfg;
            QString     strLine = strOptions.mid(pos1, pos2 - pos1);

            cfg.readFrom(strLine);

            if ( !cfg.isGroup() && !cfg.getName().isEmpty() )
            {
                setValue(cfg.getName(), cfg.getValue());
            }

            pos1 = pos2 + 1;

            while ( strOptions[pos1].isSpace() )
              pos1++;
        }
    }
    while ( pos1+1 < (int) strOptions.length() );
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString CfgGroup::getOptions(const char * pExcluding[]) const
{
    std::map<QString, bool>  excludeMap;

    if ( pExcluding )
    {
        for ( int i=0; pExcluding[i]; i++ )
            excludeMap[QString(pExcluding[i])] = true;
    }

    QStringList::const_iterator it;
    QString                     strResult;

    for ( it = m_itemNames.begin(); it != m_itemNames.end(); it++ )
    {
        QString strName = *it;

        if ( !strName.startsWith("#") )
        {
            if ( excludeMap.find(strName) == excludeMap.end() )
            {
                QString strValue = getValue(strName);

                if ( !strValue.isEmpty() )
                {
                    strResult += strName;
                    strResult += '=';
                    strResult += strValue;
                    strResult += '\n';
                }
            }
        }
    }

    return strResult;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void CfgFile::setValue(QString strGroup, QString strName, QString strValue)
{
    CfgGroupPtr ptrGroup = getGroup(strGroup = strGroup.lower());

    if ( ptrGroup.isNull() )
    {
        m_groupNames.push_back(strName);
        ptrGroup = new CfgGroup;

        ptrGroup->setTag(strGroup);
        m_groupMap[strGroup] = ptrGroup;
    }

    ptrGroup->setValue(strName, strValue);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString CfgFile::getValue(QString strGroup, QString strName) const
{
    QString     strValue;
    CfgGroupPtr ptrGroup = getGroup(strGroup);

    if ( ptrGroup.isValid() )
        strValue = ptrGroup->getValue(strName);

    return strValue;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

ConfigItem::Ptr CfgFile::getItem(QString strGroup, QString strName) const
{
    CfgItemPtr  ptrItem;
    CfgGroupPtr ptrGroup = getGroup(strGroup);

    if ( ptrGroup.isValid() )
        ptrItem = ptrGroup->getItem(strName);

    return ptrItem;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

CfgGroup::Ptr CfgFile::getGroup(QString strGroup) const
{
    CfgGroupPtr ptrGroup;

    try
    {
        CfgGroupMap::const_iterator it = m_groupMap.find(strGroup.lower());

        if ( m_groupMap.end() != it )
            ptrGroup = it->second;
    }

    catch(...)
    {
    }

    return ptrGroup;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

CfgGroup & CfgGroup::operator = (const CfgGroup & group)
{
    m_itemMap   = group.m_itemMap;
    m_itemNames = group.m_itemNames;
    m_strTag    = group.m_strTag;

    return *this;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void CfgFile::addGroup(CfgGroupPtr ptrGroup)
{
    try
    {
        QString                     strGroup = ptrGroup->getTag().lower();
        CfgGroupMap::const_iterator it = m_groupMap.find(strGroup);

        if ( m_groupMap.end() == it )
            m_groupNames.push_back(strGroup);

        m_groupMap[strGroup] = ptrGroup;
    }

    catch(...)
    {
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString CfgFile::toString() const
{
    QString                     strResult;
    QStringList::const_iterator it;

    for ( it = m_groupNames.begin(); it != m_groupNames.end(); it++ )
    {
       CfgGroupPtr ptrGroup = getGroup(*it);

       if ( ptrGroup.isValid() )
       {
          strResult += ptrGroup->toString();
          strResult += '\n';
       }
    }

    return strResult;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void CfgFile::setOptions(QString strOptions)
{
    int pos1 = 0, pos2 = 0;

    do
    {
        pos2 = strOptions.find('\n', pos1);

        if ( pos2 < 0 )
            pos2 = strOptions.length();

        if ( pos1 < pos2 )
        {
            ConfigItem  cfg;
            CfgGroupPtr ptrGroup;
            QString     strLine = strOptions.mid(pos1, pos2 - pos1);

            cfg.readFrom(strLine);

            if ( !cfg.getName().isEmpty() )
            {
                if ( cfg.isGroup() )
                {
                    ptrGroup = new CfgGroup;
                    ptrGroup->setTag(cfg.getName());

                    addGroup(ptrGroup);
                }
                else if ( ptrGroup.isValid() )
                {
                    ptrGroup->setValue(cfg.getName(), cfg.getValue());
                }
            }

            pos1 = pos2 + 1;

            while ( strOptions[pos1].isSpace() )
              pos1++;
        }
    }
    while ( pos1+1 < (int) strOptions.length() );
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void CfgFile::setOptions(QString strGroup, QString strOptions)
{
    CfgGroupPtr ptrGroup = getGroup(strGroup);

    if ( !ptrGroup.isValid() )
    {
        ptrGroup = new CfgGroup;
        ptrGroup->setTag(strGroup);
        addGroup(ptrGroup);
    }

    ptrGroup->setOptions(strOptions);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString CfgFile::getOptions() const
{
    QStringList::const_iterator it;
    QString                     strResult;

    for ( it = m_groupNames.begin(); it != m_groupNames.end(); it++ )
    {
        CfgGroupPtr ptrGroup = getGroup(*it);

        if ( ptrGroup.isValid() )
        {
            strResult += '[';
            strResult += *it;
            strResult += "]\n";

            strResult += ptrGroup->getOptions();
        }
    }

    return strResult;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString CfgFile::getOptions(QString strGroup, const char * pExcluding[]) const
{
    CfgGroupPtr ptrGroup = getGroup(strGroup);
    QString     strResult;

    if ( ptrGroup.isValid() )
        strResult = ptrGroup->getOptions(pExcluding);

    return strResult;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool CfgFile::parse(QFile & file)
{
    bool        bSuccess = true;
    CfgGroupPtr ptrGroup = 0;

    while ( !file.atEnd() )
    {
        QString     strLine;
        ConfigItem  cfgItem;

        file.readLine(strLine, Q_ULONG(1024));
        cfgItem.readFrom(strLine);

        if ( cfgItem.isComment() )
        {
            // ignore
        }
        else if ( cfgItem.isGroup() )
        {
            if ( ptrGroup.isValid() && !ptrGroup->getTag().isEmpty() )
                addGroup(ptrGroup);

            ptrGroup = new CfgGroup(cfgItem.getName());
        }
        else if ( ptrGroup.isValid() )
            ptrGroup->setValue(cfgItem.getName(), cfgItem.getValue());
    }

    if ( ptrGroup.isValid() && !ptrGroup->getTag().isEmpty() )
        addGroup(ptrGroup);

    return bSuccess;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool CfgFile::parse(const QString & strCfg)
{
    bool        bSuccess = true;
    CfgGroupPtr ptrGroup = 0;
    StrTok      strTok(strCfg);
    QString     strLine;
    ConfigItem  cfgItem;
    QRegExp     sep(" *\n");

    while ( strTok.nextToken(sep, strLine) )
    {
        cfgItem.readFrom(strLine);

        if ( cfgItem.isComment() )
        {
            // ignore
        }
        else if ( cfgItem.isValid() )
        {
            if ( cfgItem.isGroup() )
            {
                if ( ptrGroup.isValid() && !ptrGroup->getTag().isEmpty() )
                    addGroup(ptrGroup);

                ptrGroup = new CfgGroup(cfgItem.getName());
            }
            else if ( ptrGroup.isValid() )
                ptrGroup->setValue(cfgItem.getName(), cfgItem.getValue());
        }
    }

    if ( ptrGroup.isValid() && !ptrGroup->getTag().isEmpty() )
        addGroup(ptrGroup);

    return bSuccess;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void CfgFile::deleteGroup(QString strGroup)
{
    QStringList::iterator it;
    QString               strResult;

    for ( it = m_groupNames.begin(); it != m_groupNames.end(); it++ )
    {
        if ( strGroup == *it )
        {
            m_groupNames.erase(it);
            break;
        }
    }

    m_groupMap.erase(strGroup);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QStringList & CfgFile::getGroupList(QStringList & groups) const
{
    return groups = m_groupNames;
}

#include "CfgFile.moc"
