/*
 * D-BUS Support API
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-dbus.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-status.h>
#include <string.h>
#include <stdio.h>

/**
 * galago_dbus_message_iter_append_string_or_nil
 * @iter: The message iterator.
 * @str:  The string, or NULL.
 *
 * Appends a string to a message, or an empty string if the passed string
 * is NULL.
 */
void
galago_dbus_message_iter_append_string_or_nil(DBusMessageIter *iter,
											  const char *str)
{
	g_return_if_fail(iter != NULL);

	if (str == NULL)
		str = "";

	dbus_message_iter_append_basic(iter, DBUS_TYPE_STRING, &str);
}

/**
 * galago_dbus_message_iter_get_string_or_nil
 * @iter: The iterator.
 *
 * Returns a string that the iterator points to. If the returned string
 * is an empty string (""), this returns NULL.
 *
 * Returns: The string or NULL.
 */
const char *
galago_dbus_message_iter_get_string_or_nil(DBusMessageIter *iter)
{
	const char *str;

	g_return_val_if_fail(iter != NULL, NULL);

	dbus_message_iter_get_basic(iter, &str);

	if (*str == '\0')
		str = NULL;

	return str;
}

/**
 * galago_dbus_message_iter_append_object
 * @iter:   The D-BUS message iterator.
 * @object: The object to append.
 *
 * Appends an object to a D-BUS message.
 */
void
galago_dbus_message_iter_append_object(DBusMessageIter *iter,
									   const GalagoObject *object)
{
	GalagoObjectClass *klass;

	g_return_if_fail(iter != NULL);
	g_return_if_fail(object != NULL && GALAGO_IS_OBJECT(object));

	klass = GALAGO_OBJECT_GET_CLASS(object);

	if (klass->dbus_message_append != NULL)
	{
		DBusMessageIter struct_iter;

		dbus_message_iter_open_container(iter, DBUS_TYPE_STRUCT, NULL,
										 &struct_iter);
		klass->dbus_message_append(&struct_iter, object);
		dbus_message_iter_close_container(iter, &struct_iter);
	}
	else
	{
		g_warning("Class type %s passed to "
				  "galago_dbus_message_iter_append_object does not "
				  "implement dbus_message_append!",
				  g_type_name(G_OBJECT_CLASS_TYPE(klass)));
	}
}

/**
 * galago_dbus_message_iter_append_object_list
 * @iter: The D-BUS message iterator.
 * @type: The type of the objects inside.
 * @list: The list of objects to append.
 *
 * Appends a list of objects to a D-BUS message.
 */
void
galago_dbus_message_iter_append_object_list(DBusMessageIter *iter, GType type,
											GList *list)
{
	const char *dbus_signature;
	DBusMessageIter array_iter;
	GList *l;

	g_return_if_fail(iter != NULL);

	dbus_signature = galago_object_type_get_dbus_signature(type);
	g_return_if_fail(dbus_signature != NULL);

	dbus_message_iter_open_container(iter, DBUS_TYPE_ARRAY,
									 dbus_signature, &array_iter);

	for (l = list; l != NULL; l = l->next)
	{
		galago_dbus_message_iter_append_object(&array_iter,
											   GALAGO_OBJECT(l->data));
	}

	dbus_message_iter_close_container(iter, &array_iter);
}

/**
 * galago_dbus_message_iter_get_object
 * @iter: The D-BUS message iterator.
 * @type: The GType.
 *
 *
 * Returns an object of the specified type from a D-BUS message.
 *
 * Returns: The object.
 */
void *
galago_dbus_message_iter_get_object(DBusMessageIter *iter, GType type)
{
	GalagoObject *object;
	GalagoObjectClass *klass;
	DBusMessageIter temp_iter;

	g_return_val_if_fail(iter != NULL, NULL);
	g_return_val_if_fail(dbus_message_iter_get_arg_type(iter) !=
						 DBUS_TYPE_INVALID, NULL);

	klass = g_type_class_ref(type);

	if (klass->dbus_message_get == NULL)
	{
		g_warning("Class type %s passed to "
				  "galago_dbus_message_iter_get_object does not "
				  "implement dbus_message_get!",
				  g_type_name(type));

		g_type_class_unref(klass);
		return NULL;
	}

	dbus_message_iter_recurse(iter, &temp_iter);
	object = klass->dbus_message_get(&temp_iter);

	g_type_class_unref(klass);
	return object;
}

/**
 * galago_dbus_message_iter_get_object_list
 * @iter: The D-BUS message iterator.
 * @type: The GType.
 *
 * Returns a list of objects of the specified type from a D-BUS message.
 * The returned list must be destroyed.
 *
 * Returns: The list.
 */
GList *
galago_dbus_message_iter_get_object_list(DBusMessageIter *iter, GType type)
{
	GList *list = NULL;
	DBusMessageIter array_iter;

	g_return_val_if_fail(iter != NULL, NULL);
	g_return_val_if_fail(dbus_message_iter_get_arg_type(iter) ==
						 DBUS_TYPE_ARRAY,
						 NULL);

	dbus_message_iter_recurse(iter, &array_iter);

	while (dbus_message_iter_get_arg_type(&array_iter) != DBUS_TYPE_INVALID)
	{
		list = g_list_append(list,
			galago_dbus_message_iter_get_object(&array_iter, type));

		dbus_message_iter_next(&array_iter);
	}

	return list;
}

static void
galago_dbus_message_iter_append_value_list(DBusMessageIter *iter,
										   GalagoValue *value)
{
	GList *list = NULL;
	GList *l;

	switch (galago_value_get_subtype(value))
	{
		case GALAGO_VALUE_TYPE_OBJECT:
			for (l = galago_value_get_list(value); l != NULL; l = l->next)
			{
				GalagoValue *temp_value = (GalagoValue *)l->data;

				list = g_list_append(list,
					galago_value_get_object(temp_value));
			}

			galago_dbus_message_iter_append_object_list(iter,
				galago_value_get_gtype(value), list);
			g_list_free(list);
			break;

		default:
			g_warning("Unsupported list type %d appended to message",
					  galago_value_get_type(value));
			break;
	}
}

static void
galago_dbus_message_iter_append_value_array(DBusMessageIter *iter,
											GalagoValue *value)
{
	const void *array;
	gsize array_size;
	DBusMessageIter array_iter;
	const char *array_type_str = NULL;
	char array_type = 0;

	galago_value_get_array(value, &array, &array_size);

	switch (galago_value_get_subtype(value))
	{
		case GALAGO_VALUE_TYPE_CHAR:
		case GALAGO_VALUE_TYPE_UCHAR:
			array_type_str = DBUS_TYPE_BYTE_AS_STRING;
			array_type = DBUS_TYPE_BYTE;
			break;

		case GALAGO_VALUE_TYPE_BOOLEAN:
			array_type_str = DBUS_TYPE_BOOLEAN_AS_STRING;
			array_type = DBUS_TYPE_BOOLEAN;
			break;

		case GALAGO_VALUE_TYPE_SHORT:
		case GALAGO_VALUE_TYPE_INT:
		case GALAGO_VALUE_TYPE_LONG:
			array_type_str = DBUS_TYPE_INT32_AS_STRING;
			array_type = DBUS_TYPE_INT32;
			break;

		case GALAGO_VALUE_TYPE_USHORT:
		case GALAGO_VALUE_TYPE_UINT:
		case GALAGO_VALUE_TYPE_ULONG:
			array_type_str = DBUS_TYPE_UINT32_AS_STRING;
			array_type = DBUS_TYPE_UINT32;
			break;

		case GALAGO_VALUE_TYPE_STRING:
			array_type_str = DBUS_TYPE_STRING_AS_STRING;
			array_type = DBUS_TYPE_STRING;
			break;

		default:
			g_warning("Invalid array type %d appended to message",
					  galago_value_get_subtype(value));
			return;
	}

	dbus_message_iter_open_container(iter, DBUS_TYPE_ARRAY, array_type_str,
									 &array_iter);
	dbus_message_iter_append_fixed_array(&array_iter, array_type,
										 &array, array_size);
	dbus_message_iter_close_container(iter, &array_iter);
}

void
galago_dbus_message_iter_append_value(DBusMessageIter *iter,
									  GalagoValue *value)
{
	g_return_if_fail(iter  != NULL);
	g_return_if_fail(value != NULL);

#define CHECK_APPEND_VALUE(type, dbustype, galagoname, vartype) \
	case type: \
	{ \
		vartype var = galago_value_get_##galagoname(value); \
		dbus_message_iter_append_basic(iter, (dbustype), &var); \
		break; \
	}

	switch (galago_value_get_type(value))
	{
		CHECK_APPEND_VALUE(GALAGO_VALUE_TYPE_CHAR,
						   DBUS_TYPE_BYTE,
						   char, char);
		CHECK_APPEND_VALUE(GALAGO_VALUE_TYPE_UCHAR,
						   DBUS_TYPE_BYTE,
						   uchar, unsigned char);
		CHECK_APPEND_VALUE(GALAGO_VALUE_TYPE_BOOLEAN,
						   DBUS_TYPE_BOOLEAN,
						   boolean, gboolean);
		CHECK_APPEND_VALUE(GALAGO_VALUE_TYPE_SHORT,
						   DBUS_TYPE_INT32,
						   short, short);
		CHECK_APPEND_VALUE(GALAGO_VALUE_TYPE_USHORT,
						   DBUS_TYPE_UINT32,
						   ushort, unsigned short);
		CHECK_APPEND_VALUE(GALAGO_VALUE_TYPE_INT,
						   DBUS_TYPE_INT32,
						   int, int);
		CHECK_APPEND_VALUE(GALAGO_VALUE_TYPE_UINT,
						   DBUS_TYPE_UINT32,
						   uint, unsigned int);
		CHECK_APPEND_VALUE(GALAGO_VALUE_TYPE_LONG,
						   DBUS_TYPE_INT32,
						   long, long);
		CHECK_APPEND_VALUE(GALAGO_VALUE_TYPE_ULONG,
						   DBUS_TYPE_UINT32,
						   ulong, unsigned long);

		case GALAGO_VALUE_TYPE_STRING:
			galago_dbus_message_iter_append_string_or_nil(iter,
				galago_value_get_string(value));
			break;

		case GALAGO_VALUE_TYPE_OBJECT:
			galago_dbus_message_iter_append_object(iter,
				galago_value_get_object(value));
			break;

		case GALAGO_VALUE_TYPE_LIST:
			galago_dbus_message_iter_append_value_list(iter, value);
			break;

		case GALAGO_VALUE_TYPE_ARRAY:
			galago_dbus_message_iter_append_value_array(iter, value);
			break;

		default:
			g_warning("Invalid type %d appended to message",
					  galago_value_get_type(value));
			break;
	}

#undef CHECK_APPEND_VALUE
}

GalagoValue *
galago_dbus_message_iter_get_value(DBusMessageIter *iter)
{
	GalagoValue *value = NULL;

	g_return_val_if_fail(iter != NULL, NULL);

#define CHECK_GET_VALUE(dbustype, galagotype, galagoname, vartype) \
	case dbustype: \
	{ \
		vartype var; \
		value = galago_value_new(galagotype, NULL, NULL); \
		dbus_message_iter_get_basic(iter, &var); \
		galago_value_set_##galagoname(value, var); \
		break; \
	}

	switch (dbus_message_iter_get_arg_type(iter))
	{
		CHECK_GET_VALUE(DBUS_TYPE_BYTE,
						GALAGO_VALUE_TYPE_CHAR,
						char, char);
		CHECK_GET_VALUE(DBUS_TYPE_INT32,
						GALAGO_VALUE_TYPE_INT,
						int, int);
		CHECK_GET_VALUE(DBUS_TYPE_UINT32,
						GALAGO_VALUE_TYPE_UINT,
						uint, unsigned int);
		CHECK_GET_VALUE(DBUS_TYPE_BOOLEAN,
						GALAGO_VALUE_TYPE_BOOLEAN,
						boolean, gboolean);

		case DBUS_TYPE_STRING:
		{
			const char *str;
			value = galago_value_new(GALAGO_VALUE_TYPE_STRING, NULL, NULL);
			str = galago_dbus_message_iter_get_string_or_nil(iter);
			galago_value_set_string(value, str);
			break;
		}

		default:
			g_warning("Unsupported type %d retrieved from message",
					  dbus_message_iter_get_arg_type(iter));
			break;
	}

	return value;

#undef CHECK_GET_VALUE
}

/**
 * galago_dbus_message_new_method_call
 * @object:   The object.
 * @name:     The message name.
 * @reply:    TRUE if a reply is expected.
 * @ret_iter: The returned message iterator, if not NULL.
 *
 * Creates a D-BUS message to the specified object.
 *
 * Returns: The D-BUS message.
 */
DBusMessage *
galago_dbus_message_new_method_call(const GalagoObject *object,
									const char *name, gboolean reply,
									DBusMessageIter *iter)
{
	DBusMessage *message;
	GalagoObjectClass *klass;
	const char *obj_path;
	const char *iface;

	g_return_val_if_fail(object != NULL, NULL);
	g_return_val_if_fail(name   != NULL, NULL);
	g_return_val_if_fail(*name  != '\0', NULL);
	g_return_val_if_fail(GALAGO_IS_OBJECT(object), NULL);

	klass = GALAGO_OBJECT_GET_CLASS(object);

	obj_path = galago_object_get_dbus_path(object);
	iface    = klass->dbus_interface;

	if (obj_path == NULL)
	{
		g_error("No object path was registered for class '%s'. "
				"Please report this.",
				g_type_name(G_OBJECT_CLASS_TYPE(klass)));

		return NULL;
	}

	if (iface == NULL)
	{
		g_error("No D-BUS interface was registered for class '%s'. "
				"Please report this.",
				g_type_name(G_OBJECT_CLASS_TYPE(klass)));

		return NULL;
	}

	message = dbus_message_new_method_call(GALAGO_DBUS_SERVICE,
										   obj_path, iface, name);

	g_return_val_if_fail(message != NULL, NULL);

	dbus_message_set_no_reply(message, !reply);

	if (iter != NULL)
		dbus_message_iter_init_append(message, iter);

	return message;
}

/**
 * galago_dbus_message_new_method_call_vargs
 * @object:   The object.
 * @name:     The message name.
 * @reply:    TRUE if a reply is expected.
 * @args:     The va_list of parameters.
 *
 *
 * Creates a D-BUS message to the specified object with a va_list of
 * parameters.
 *
 * Returns: The D-BUS message.
 */
DBusMessage *
galago_dbus_message_new_method_call_vargs(const GalagoObject *object,
										  const char *name,
										  gboolean reply,
										  va_list args)
{
	DBusMessage *message;
	DBusMessageIter iter;
	GalagoValue *value;

	g_return_val_if_fail(object != NULL, NULL);
	g_return_val_if_fail(name   != NULL, NULL);
	g_return_val_if_fail(*name  != '\0', NULL);
	g_return_val_if_fail(GALAGO_IS_OBJECT(object), NULL);

	message = galago_dbus_message_new_method_call(object, name, reply, &iter);

	g_return_val_if_fail(message != NULL, NULL);

	while ((value = (GalagoValue *)va_arg(args, GalagoValue *)) != NULL)
	{
		galago_dbus_message_iter_append_value(&iter, value);
		galago_value_destroy(value);
	}

	return message;
}

/**
 * galago_dbus_message_new_method_call_args
 * @object:   The object.
 * @name:     The message name.
 * @reply:    TRUE if a reply is expected.
 * @...:      The list of parameters.
 *
 * Creates a D-BUS message to the specified object with a list of parameters.
 *
 * The parameters passed are in GalagoValue, pointer-to-data form.
 *
 * Returns: The D-BUS message.
 */
DBusMessage *
galago_dbus_message_new_method_call_args(const GalagoObject *object,
										 const char *name,
										 gboolean reply, ...)
{
	va_list args;
	DBusMessage *message;

	g_return_val_if_fail(object != NULL, NULL);
	g_return_val_if_fail(name   != NULL, NULL);
	g_return_val_if_fail(*name  != '\0', NULL);
	g_return_val_if_fail(GALAGO_IS_OBJECT(object), NULL);

	va_start(args, reply);
	message = galago_dbus_message_new_method_call_vargs(object, name,
														reply, args);
	va_end(args);

	return message;
}

/**
 * galago_dbus_send_message
 * @object:   The object.
 * @name:     The message name.
 * @...:      The list of parameters.
 *
 * Sends a new D-BUS message with an object, name, and parameters.
 *
 * The parameters passed are in GalagoValue, pointer-to-data form.
 */
void
galago_dbus_send_message(const GalagoObject *object, const char *name, ...)
{
	va_list args;
	DBusMessage *message;

	g_return_if_fail(object != NULL);
	g_return_if_fail(name   != NULL);
	g_return_if_fail(*name  != '\0');
	g_return_if_fail(GALAGO_IS_OBJECT(object));

	if (!galago_is_connected())
		return;

	va_start(args, name);
	message = galago_dbus_message_new_method_call_vargs(object, name,
														FALSE, args);
	va_end(args);

	g_return_if_fail(message != NULL);

	dbus_connection_send(galago_get_dbus_conn(), message, NULL);
	dbus_message_unref(message);
}

static void *
get_ret_val_from_iter(DBusMessageIter *iter, GalagoValue *value)
{
	void *retval = NULL;

	switch (galago_value_get_type(value))
	{
		case GALAGO_VALUE_TYPE_LIST:
			switch (galago_value_get_subtype(value))
			{
				case GALAGO_VALUE_TYPE_OBJECT:
					retval = galago_dbus_message_iter_get_object_list(iter,
						galago_value_get_gtype(value));
					break;

				default:
					g_warning("Unsupported list type %d returned from "
							  "message",
							  galago_value_get_subtype(value));
					break;
			}
			break;

		case GALAGO_VALUE_TYPE_OBJECT:
			retval = galago_dbus_message_iter_get_object(iter,
				galago_value_get_gtype(value));
			break;

		default:
			switch (dbus_message_iter_get_arg_type(iter))
			{
				case DBUS_TYPE_INT32:
				case DBUS_TYPE_UINT32:
				case DBUS_TYPE_BOOLEAN:
					dbus_message_iter_get_basic(iter, &retval);
					break;

				case DBUS_TYPE_STRING:
					retval = (gpointer)g_strdup(
						galago_dbus_message_iter_get_string_or_nil(iter));
					break;

				default:
					g_warning("Unsupported type %d retrieved from message",
							  dbus_message_iter_get_arg_type(iter));
					break;
			}
			break;
	}

	return retval;
}

static GList *
galago_dbus_send_message_with_reply_list_vargs(const GalagoObject *object,
											   const char *name,
											   GList *return_types,
											   va_list args)
{
	DBusMessage *message;
	DBusMessage *reply = NULL;
	DBusMessageIter iter;
	DBusError error;
	GList *ret_list = NULL, *l;

	galago_goto_if_fail(object != NULL,           exit);
	galago_goto_if_fail(name   != NULL,           exit);
	galago_goto_if_fail(*name  != '\0',           exit);
	galago_goto_if_fail(GALAGO_IS_OBJECT(object), exit);
	galago_goto_if_fail(return_types != NULL,     exit);

	dbus_error_init(&error);

	if (!galago_is_connected())
		goto exit;

	message = galago_dbus_message_new_method_call_vargs(object, name,
														TRUE, args);

	galago_goto_if_fail(message != NULL, exit);

	reply = dbus_connection_send_with_reply_and_block(
		galago_get_dbus_conn(), message, -1, &error);

	dbus_message_unref(message);

	if (dbus_error_is_set(&error))
	{
		if (!dbus_error_has_name(&error, GALAGO_DBUS_ERROR_OBJECT_NOT_FOUND))
		{
			g_warning("Error sending %s.%s: %s",
					  g_type_name(G_OBJECT_TYPE(object)),
					  name, error.message);
		}

		goto exit;
	}

	dbus_message_iter_init(reply, &iter);

	for (l = return_types; l != NULL; l = l->next)
	{
		ret_list = g_list_append(ret_list,
			get_ret_val_from_iter(&iter, (GalagoValue *)l->data));

		dbus_message_iter_next(&iter);
	}

exit:
	dbus_error_free(&error);

	if (reply != NULL)
		dbus_message_unref(reply);

	if (return_types != NULL)
	{
		g_list_foreach(return_types, (GFunc)galago_value_destroy, NULL);
		g_list_free(return_types);
	}

	return ret_list;
}

/**
 * galago_dbus_send_message_with_reply_list
 * @object:       The object.
 * @name:         The message name.
 * @return_types: A list of expected return types, in order.
 * @...:          The list of parameters.
 *
 * Sends a new D-BUS message with an object, name, and parameters, and
 * waits for a reply with a variable list of return types.
 *
 * The parameters passed are in GalagoValue, pointer-to-data form.
 *
 * Returns: The returned data in a list. The list will need to be g_freed,
 *          as may some of its data.
 */
GList *
galago_dbus_send_message_with_reply_list(const GalagoObject *object,
										 const char *name,
										 GList *return_types, ...)
{
	va_list args;
	GList *list;

	va_start(args, return_types);
	list = galago_dbus_send_message_with_reply_list_vargs(object, name,
														  return_types, args);
	va_end(args);

	return list;
}

/**
 * galago_dbus_send_message_with_reply
 * @object:      The object.
 * @name:        The message name.
 * @return_type: The expected return type.
 * @...:         The list of parameters.
 *
 * Sends a new D-BUS message with an object, name, and parameters, and
 * waits for a reply.
 *
 * The parameters passed are in GalagoValue, pointer-to-data form.
 *
 * Returns: The returned data. This may need to be g_freed.
 */
void *
galago_dbus_send_message_with_reply(const GalagoObject *object,
									const char *name,
									GalagoValue *return_type, ...)
{
	va_list args;
	void *retval = NULL;
	GList *list;

	va_start(args, return_type);
	list = galago_dbus_send_message_with_reply_list_vargs(
		object, name, g_list_append(NULL, return_type), args);
	va_end(args);

	if (list != NULL)
	{
		retval = list->data;

		g_list_free(list);
	}

	return retval;
}

/**
 * galago_dbus_object_push_full
 * @object: The object to push.
 *
 * Pushes an object and all its internal data to the server.
 *
 * This is meant for internal use.
 */
void
galago_dbus_object_push_full(GalagoObject *object)
{
	GalagoObjectClass *klass;

	g_return_if_fail(object != NULL);
	g_return_if_fail(GALAGO_IS_OBJECT(object));

	klass = GALAGO_OBJECT_GET_CLASS(object);

	if (klass->dbus_push_full != NULL)
		klass->dbus_push_full(object);
	else
	{
		g_warning("Class type %s passed to galago_dbus_object_push_full "
				  "does not implement dbus_push_full!",
				  g_type_name(G_OBJECT_CLASS_TYPE(klass)));
	}
}
