/**
 * @file list-people.c Displays a list of all people.
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include <libgalago/galago.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static void
display_person(GalagoPerson *person)
{
	GList *l;
	const char *display_name, *uid, *me_text = "";
	const char *displayed_uid;

	uid = galago_person_get_id(person);
	display_name = galago_person_get_display_name(person);

	if (galago_person_is_me(person))
		me_text = " (me) ";

	if (uid == NULL)
		displayed_uid = "no uid";
	else
		displayed_uid = uid;

	if (display_name == NULL)
		printf("%s<%s>:\n", me_text, displayed_uid);
	else
		printf("%s %s<%s>\n", display_name, me_text, displayed_uid);

	for (l = galago_person_get_accounts(person, TRUE); l != NULL; l = l->next)
	{
		GalagoAccount *account = (GalagoAccount *)l->data;
		GalagoService *service = galago_account_get_service(account);

		printf("\t%s (%s)\n",
			   galago_account_get_username(account),
			   galago_service_get_name(service));
	}

	printf("\n");
}

int
main(int argc, char **argv)
{
	char *person_uid = NULL;

	if (!galago_init("list-people-test", GALAGO_INIT_CLIENT) ||
		!galago_is_connected())
	{
		fprintf(stderr, "Unable to connect to the Galago service.\n");
		exit(1);
	}

	if (argc > 1)
		person_uid = argv[1];

	if (person_uid == NULL)
	{
		GList *l;

		for (l = galago_get_people(GALAGO_REMOTE, TRUE);
			 l != NULL;
			 l = l->next)
		{
			display_person((GalagoPerson *)l->data);
		}
	}
	else
	{
		GalagoPerson *person;

		person = galago_get_person(person_uid, GALAGO_REMOTE, TRUE);

		if (person == NULL)
			printf("%s is not a known person UID\n", person_uid);
		else
			display_person(person);
	}

	return 0;
}
