/****************************************************************************
** Class Encode::FFmpeg implementation ...
**
**   Created : Wed Jun 17 07:53:05 2008
**        by : Varol Okan
** Copyright : (c) Varol Okan
**   License : GPL v 2.0
**
** Here we have some misc functions which are needed by a few classes 
** but can not really be assigned to any of those classes.
**
****************************************************************************/

#include <QImage>

extern "C" {
#define __STDC_CONSTANT_MACROS
#define __STDC_LIMIT_MACROS
#include <ffmpeg/avformat.h>
}

#define AUDIO_BUF_SIZE 50000
#define VIDEO_BUF_SIZE 1835008

#include "ffmpeg_enc.h"

namespace Encoder
{

FFmpeg::FFmpeg  ( )
      : Encoder ( )
{
  m_pOutputCtx   = NULL;
  m_pVideoStream = NULL;
  m_pAudioStream = NULL;
  m_pAudioBuffer = NULL;
  m_pVideoBuffer = NULL;
  m_pFrame       = NULL;
  m_pSamples     = NULL;

  // must be called before using avcodec lib
  avcodec_init ( );

  // register all the codecs
  avcodec_register_all ( );
  av_register_all ( );
}

FFmpeg::~FFmpeg ( )
{
  endStream ( );
}

bool FFmpeg::initStream ( QString qsFileName, enVideo videoFormat, enAudio audioFormat, uint iFrames )
{
  Encoder::initStream ( qsFileName, videoFormat, audioFormat, iFrames );
  endStream ( );
  AVOutputFormat *pOutputFormat = guess_format ( "dvd", NULL, NULL );
  if ( audioFormat == afUndef )
    pOutputFormat = guess_format ( NULL, (const char *)qsFileName.toUtf8 ( ), NULL );

  if ( ! pOutputFormat )
    return false;

  pOutputFormat->video_codec = CODEC_ID_MPEG2VIDEO;
  if ( audioFormat == afUndef )
    pOutputFormat->audio_codec = CODEC_ID_NONE;
  else if (audioFormat == afAC3 )
    pOutputFormat->audio_codec = CODEC_ID_AC3;
  else
    pOutputFormat->audio_codec = CODEC_ID_MP2;

  m_pOutputCtx = av_alloc_format_context ( );
  if ( ! m_pOutputCtx )
    return false;

  m_pOutputCtx->oformat = pOutputFormat;
  snprintf ( m_pOutputCtx->filename, sizeof ( m_pOutputCtx->filename ), "%s", (const char *)qsFileName.toUtf8 ( ) );

  // add video and audio streams
  if ( ! addVideoStream ( pOutputFormat->video_codec, videoFormat, m_iVideoBitrate ) )
    return false;
  if ( ! addAudioStream ( pOutputFormat->audio_codec ) )
    return false;

  if ( av_set_parameters ( m_pOutputCtx, NULL ) < 0 )
    return false;

  dump_format ( m_pOutputCtx, 0, (const char*)qsFileName.toUtf8 ( ), 1 );
  m_pOutputCtx->packet_size = 2048;

  // open the audio and video codecs and allocate the necessary encode buffers
  if ( m_pVideoStream  )
    OpenVideoEncoder ( );
  if ( m_pAudioStream  )
    OpenAudioEncoder ( );

  // open the output file
  if ( url_fopen ( &m_pOutputCtx->pb, (const char *)qsFileName.toUtf8 ( ), URL_WRONLY ) < 0 )  {
    printf ( "Could not open '%s'\n", (const char *)qsFileName.toUtf8 ( ) );
    return false;
  }
  // write the stream header
  m_pOutputCtx->packet_size =  2048;
  m_pOutputCtx->mux_rate = 10080000;
  av_write_header ( m_pOutputCtx );
  return true;
}

bool FFmpeg::addImage ( QImage *pImage, int iFrames )
{
  if ( ! pImage )
    return false;

  AVCodecContext *pVideo = m_pVideoStream->codec;

  // Allocate buffer for FFMPeg ...
  int iWidth, iHeight;
  int iSize = pImage->width ( ) * pImage->height ( );
  uint8_t *pBuffer = new uint8_t [ ( ( iSize * 3 ) / 2 ) + 100 ]; // 100 bytes extra buffer
  iWidth  = pImage->width  ( );
  iHeight = pImage->height ( );

  m_pFrame->data[0] = pBuffer;
  m_pFrame->data[1] = m_pFrame->data[0] + iSize;
  m_pFrame->data[2] = m_pFrame->data[1] + iSize / 4;
  m_pFrame->linesize[0] = iWidth;
  m_pFrame->linesize[1] = iWidth / 2;
  m_pFrame->linesize[2] = m_pFrame->linesize[1];

  // Copy data over from the QImage. Convert from 32bitRGB to YUV420P
  RGBtoYUV420P ( pImage->bits ( ), pBuffer, pImage->depth ( ) / 8, true, iWidth, iHeight );

  double fDuration = ((double) m_pVideoStream->pts.val ) * m_pVideoStream->time_base.num / m_pVideoStream->time_base.den + ((double) iFrames) * pVideo->time_base.num / pVideo->time_base.den;
  while ( true )  {
    double fAudioPts = m_pAudioStream ? ((double) m_pAudioStream->pts.val) * m_pAudioStream->time_base.num / m_pAudioStream->time_base.den : 0.0;
    double fVideoPts = ((double) m_pVideoStream->pts.val) * m_pVideoStream->time_base.num / m_pVideoStream->time_base.den;

    if ( ( ! m_pAudioStream || fAudioPts >= fDuration ) &&
         ( ! m_pVideoStream || fVideoPts >= fDuration ) )
      break;

    // write interleaved audio and video frames
    if ( m_pAudioStream && ( fAudioPts < fVideoPts ) )  {
      if ( ! writeAudioFrame ( ) )
        return false;
    }
    else {
      if ( ! writeVideoFrame ( ) )
        return false;
    }
  }

  delete pBuffer;
  m_pFrame->data[0] = NULL;
  return true;
}

/**
 * RGBtoYUV420P function is from Gnomemeeting
 */
#define rgbtoyuv(r, g, b, y, u, v) \
  y=(uint8_t)(((int)30*r   +(int)59*g +(int)11*b)/100); \
  u=(uint8_t)(((int)-17*r  -(int)33*g +(int)50*b+12800)/100); \
  v=(uint8_t)(((int)50*r   -(int)42*g -(int)8*b+12800)/100); \

void FFmpeg::RGBtoYUV420P ( const uint8_t *pRGB, uint8_t *pYUV, uint iRGBIncrement, bool bSwapRGB, int iWidth, int iHeight,  bool bFlip )
{
  const unsigned iPlaneSize = iWidth * iHeight;
  const unsigned iHalfWidth = iWidth >> 1;

  // get pointers to the data
  uint8_t *yplane  = pYUV;
  uint8_t *uplane  = pYUV + iPlaneSize;
  uint8_t *vplane  = pYUV + iPlaneSize + (iPlaneSize >> 2);
  const uint8_t *pRGBIndex = pRGB;
  int iRGBIdx[3];
  iRGBIdx[0] = 0;
  iRGBIdx[1] = 1;
  iRGBIdx[2] = 2;
  if ( bSwapRGB )  {
    iRGBIdx[0] = 2;
    iRGBIdx[2] = 0;
  }

  for (int y = 0; y < (int) iHeight; y++) {
    uint8_t *yline  = yplane + (y * iWidth);
    uint8_t *uline  = uplane + ((y >> 1) * iHalfWidth);
    uint8_t *vline  = vplane + ((y >> 1) * iHalfWidth);

    if ( bFlip ) // Flip horizontally
      pRGBIndex = pRGB + ( iWidth * ( iHeight -1 -y ) * iRGBIncrement );

    for ( int x=0; x<iWidth; x+=2 ) {
      rgbtoyuv ( pRGBIndex[iRGBIdx[0]], pRGBIndex[iRGBIdx[1]], pRGBIndex[iRGBIdx[2]], *yline, *uline, *vline );
      pRGBIndex += iRGBIncrement;
      yline++;
      rgbtoyuv ( pRGBIndex[iRGBIdx[0]], pRGBIndex[iRGBIdx[1]], pRGBIndex[iRGBIdx[2]], *yline, *uline, *vline );
      pRGBIndex += iRGBIncrement;
      yline++;
      uline++;
      vline++;
    }
  }
}

void FFmpeg::endStream ( )
{
  CloseVideoEncoder ( );
  CloseAudioEncoder ( );

  if ( ! m_pOutputCtx )
    return;

  // write the trailer
  av_write_trailer ( m_pOutputCtx );

  // free the streams
  unsigned int t;
  for  ( t=0; t<m_pOutputCtx->nb_streams; t++ )  {
    av_freep ( &m_pOutputCtx->streams[t]->codec );
    av_freep ( &m_pOutputCtx->streams[t] );
  }

  // close the output file
#if LIBAVFORMAT_VERSION_INT >= (52<<16)
  url_fclose (  m_pOutputCtx->pb );
#else
  url_fclose ( &m_pOutputCtx->pb );
#endif

  // free the stream
  av_free ( m_pOutputCtx );
  m_pOutputCtx = NULL;
}

bool FFmpeg::addVideoStream ( int iCodecID, enVideo videoFormat, int iBitrate )
{
  if ( iCodecID == CODEC_ID_NONE )  {
    m_pVideoStream = NULL;
    return true;
  }

  m_pVideoStream = av_new_stream ( m_pOutputCtx, 0 );
  if ( ! m_pVideoStream )
    return false;

  AVCodecContext *pVideo  = m_pVideoStream->codec;
  pVideo->codec_id        = (CodecID)iCodecID;
  pVideo->codec_type      = CODEC_TYPE_VIDEO;
  pVideo->bit_rate        = iBitrate * 1024;
  pVideo->width           = 720;
  pVideo->sample_aspect_ratio.den = 4;
  pVideo->sample_aspect_ratio.num = 3;

//  pVideo->dtg_active_format = FF_DTG_AFD_4_3; only used for decoding

  if ( videoFormat == vfPAL )  {
    pVideo->height        = 576;
    pVideo->time_base.den =  25;
    pVideo->time_base.num =   1;
    pVideo->gop_size      =  15;
  }
  else  {
    pVideo->height        =   480;
    pVideo->time_base.den = 30000;
    pVideo->time_base.num =  1001;
    pVideo->gop_size      =    18;
  }
  pVideo->pix_fmt         = PIX_FMT_YUV420P;
  pVideo->rc_buffer_size  = VIDEO_BUF_SIZE;
  pVideo->rc_max_rate     = 9 * 1024 * 1024;
  pVideo->rc_min_rate     = 0;
  return true;
}

bool FFmpeg::addAudioStream ( int iCodecID )
{
  if ( iCodecID == CODEC_ID_NONE ) {
    m_pAudioStream = NULL;
    return true;
  }

  m_pAudioStream = av_new_stream ( m_pOutputCtx, 1 );
  if ( ! m_pAudioStream )
    return false;

  AVCodecContext *pAudio = m_pAudioStream->codec;
  pAudio->codec_id       = (CodecID)iCodecID;
  pAudio->codec_type     = CODEC_TYPE_AUDIO;
  pAudio->bit_rate       = 64000;
  pAudio->sample_rate    = 48000;
  pAudio->channels       = 2;
  return true;
}

bool FFmpeg::OpenAudioEncoder ( )
{
  AVCodecContext *pAudio = m_pAudioStream->codec;

  // find the audio encoder and open it
  AVCodec *pCodec = avcodec_find_encoder ( pAudio->codec_id );
  if ( ! pCodec )  {
    printf ( "Audio codec not found" );
    return false;
  }
  if ( avcodec_open ( pAudio, pCodec ) < 0 )  {
    printf ("Could not open audio codec" );
    return false;
  }

  m_pAudioBuffer = (uint8_t *) av_malloc ( AUDIO_BUF_SIZE );

  int audioInputFrameSize = pAudio->frame_size;
  // ugly hack for PCM codecs (will be removed ASAP with new PCM
  // support to compute the input frame size in samples
  if ( pAudio->frame_size <= 1 )  {
    audioInputFrameSize = AUDIO_BUF_SIZE / pAudio->channels;
    switch ( m_pAudioStream->codec->codec_id ) {
    case CODEC_ID_PCM_S16LE:
    case CODEC_ID_PCM_S16BE:
    case CODEC_ID_PCM_U16LE:
    case CODEC_ID_PCM_U16BE:
      audioInputFrameSize >>= 1;
    break;
    default:
      break;
    }
  }
  m_pSamples = (int16_t*) av_malloc ( audioInputFrameSize * 2 * pAudio->channels );
  int16_t *q = m_pSamples;
  for (int j = 0; j < audioInputFrameSize * pAudio->channels; j++)
    *q++ = 0;
  return true;
}

void FFmpeg::CloseAudioEncoder ( )
{
  if ( ! m_pAudioStream )
    return;
  avcodec_close ( m_pAudioStream->codec );

  if ( m_pSamples )
    av_free ( m_pSamples );
  m_pSamples = NULL;

  if ( m_pAudioBuffer )
    av_free ( m_pAudioBuffer );

  m_pAudioBuffer = NULL;
  m_pAudioStream = NULL;
}

AVFrame *FFmpeg::allocPicture ( int iPixFormat, int iWidth, int iHeight )
{
  AVFrame *pFrame = avcodec_alloc_frame ( );
  if (  !  pFrame )
    return NULL;

  int iSize = avpicture_get_size ( iPixFormat, iWidth, iHeight );
  uint8_t *pBuffer = (uint8_t*) av_malloc ( iSize );
  if (  !  pBuffer )  {
      av_free ( pFrame );
      return NULL;
  }

  avpicture_fill ( (AVPicture *)pFrame, pBuffer, iPixFormat, iWidth, iHeight );
  return pFrame;
}

bool FFmpeg::OpenVideoEncoder()
{
  AVCodecContext *pVideo = m_pVideoStream->codec;

  // find the video encoder and open it
  AVCodec *pCodec = avcodec_find_encoder ( pVideo->codec_id );
  if ( ! pCodec )  {
    printf ( "Video codec not found" );
    return false;
  }

  if ( avcodec_open ( pVideo, pCodec ) < 0 )  {
    printf ( "Could not open video codec" );
    return false;
  }

  m_pVideoBuffer = (uint8_t*) av_malloc ( VIDEO_BUF_SIZE );

  // allocate the encoded raw picture
  m_pFrame = allocPicture ( pVideo->pix_fmt, pVideo->width, pVideo->height );
  if ( ! m_pFrame )  {
    printf ( "Could not allocate picture" );
    return false;
  }

  // The following settings will prevent warning messages from FFmpeg
  float fMuxPreload  = 0.5f;
  float fMuxMaxDelay = 0.7f;
  //For svcd you might set it to:
  //mux_preload= (36000+3*1200) / 90000.0; //0.44
  m_pOutputCtx->preload   = (int)( fMuxPreload  * AV_TIME_BASE );
  m_pOutputCtx->max_delay = (int)( fMuxMaxDelay * AV_TIME_BASE );

  return true;
}

void FFmpeg::CloseVideoEncoder ( )
{
  if ( ! m_pVideoStream )
    return;

  avcodec_close ( m_pVideoStream->codec );

  if ( m_pFrame )  {
    if ( m_pFrame->data[0] )
      av_free ( m_pFrame->data[0] );
    av_free ( m_pFrame );
    m_pFrame = NULL;
  }

  if ( m_pVideoBuffer )
    av_free ( m_pVideoBuffer );
  m_pVideoBuffer = NULL;
  m_pVideoStream = NULL;
}

bool FFmpeg::writeAudioFrame ( )
{
  AVPacket pkt;
  av_init_packet ( &pkt );

  AVCodecContext *pAudio = m_pAudioStream->codec;
  pkt.size   = avcodec_encode_audio ( pAudio, m_pAudioBuffer, AUDIO_BUF_SIZE, m_pSamples );
  pkt.pts    = av_rescale_q ( pAudio->coded_frame->pts, pAudio->time_base, m_pAudioStream->time_base);
  pkt.flags |= PKT_FLAG_KEY;
  pkt.stream_index = m_pAudioStream->index;
  pkt.data         = m_pAudioBuffer;

  // write the compressed frame in the media file
  if ( av_write_frame ( m_pOutputCtx, &pkt ) != 0 )  {
    printf ( "Error while writing audio frame" );
    return false;
  }
  return true;
}

bool FFmpeg::writeVideoFrame ( )
{
  AVCodecContext *pVideo = m_pVideoStream->codec;

  // encode the image
  int  out_size = avcodec_encode_video ( pVideo, m_pVideoBuffer, VIDEO_BUF_SIZE, m_pFrame );
  if ( out_size < 0 )
    return false;
  // if zero size, it means the image was buffered
  if ( out_size > 0 ) {
    AVPacket pkt;
    av_init_packet ( &pkt );

    pkt.pts= av_rescale_q ( pVideo->coded_frame->pts, pVideo->time_base, m_pVideoStream->time_base );
    if ( pVideo->coded_frame->key_frame )
      pkt.flags     |= PKT_FLAG_KEY;

    pkt.stream_index = m_pVideoStream->index;
    pkt.data         = m_pVideoBuffer;
    pkt.size         = out_size;

    // write the compressed frame in the media file
    int  ret = av_write_frame ( m_pOutputCtx, &pkt );
    if ( ret != 0 ) {
      printf ( "Error while writing video frame" );
      return false;
    }
  }
  return true;
}

}; // end of namespace Encoder
