/****************************************************************************
** Implementation for DialogVFolder
**
**   Created : Tue Dec 25 22:05:28 2005
**        by : Varol Okan using XEmacs
** Copyright : (c) Varol Okan
**   License : GPL v 2.0
**
** Handling of VirtualFolder from within DialogFiles.
**
****************************************************************************/

#include <qimage.h>
#include <qlabel.h>
#include <qtimer.h>
#include <qpixmap.h>
#include <qfileinfo.h>
#include <qpushbutton.h>
#include <qmessagebox.h>

#include "global.h"
#include "qdvdauthor.h"
#include "dialogvfolder.h"
#include "filepreviewdialog.h"
#include "qplayer/mediacreator.h"

namespace Input 
{

DialogVFolder::VFolderItem::VFolderItem ( Cache::Thumbs::Entry *p, QListView *pListView, QString qsLabel  )
  : QListViewItem ( pListView, qsLabel )
{
  bAlternateColor = true;
  pEntry          = p;
}

void DialogVFolder::VFolderItem::paintCell ( QPainter *p, const QColorGroup & cg, int column, int width, int align)
{
  QColorGroup theColors = cg;
  if ( isSelected ( ) )   {
    theColors.setColor(QColorGroup::Base, theColors.highlight() );
    theColors.setColor(QColorGroup::Text, theColors.highlightedText() );
  }
  QColor colorAlternate ( 250, 250, 250 );
  if ( bAlternateColor )   // every second file we change the color slightly
       colorAlternate  = QColor ( 235, 235, 235 );
  theColors.setColor ( QColorGroup::Base, colorAlternate );
  QListViewItem::paintCell ( p, theColors, column, width, align );
}

DialogVFolder::DialogVFolder ( QWidget *pParent, const char *pName, bool bModal, WFlags flags )
  : uiDialogVFolder ( pParent, pName, bModal, flags )
{
  m_bIsImageFolder = false;
  m_pNextItem      = NULL;
}

DialogVFolder::~DialogVFolder ( )
{
}

void DialogVFolder::initMe ( QString qsVirtualFolder, bool bImageFolder )
{
  m_pLabelHeader->setText ( "<p align=\"center\"><b>" + qsVirtualFolder + "</b></p>" );
  Cache::Thumbs::VirtualFolder *pVFolder = Global::pThumbsCache->findVirtualFolder ( qsVirtualFolder, bImageFolder );

  m_bCanClose      = true;
  m_iThumbSize     = 65;
  m_qsFolderName   = qsVirtualFolder;
  m_bIsImageFolder = bImageFolder;
  m_pListView->setSortColumn ( -1 );
  if ( ! pVFolder )
    return;

  QImage    theImage;
  QPixmap   thePixmap;
  QFileInfo fileInfo;
  QListViewItem *pItem = NULL;
  Cache::Thumbs::Entry *pEntry = NULL;
  Cache::Thumbs::VirtualFolder::VFile *pVFile = NULL;
  QValueList<Cache::Thumbs::VirtualFolder::VFile *>::iterator it;

  it = pVFolder->listOfFiles.end ( );
  while ( it != pVFolder->listOfFiles.begin ( ) ) {
    pVFile = *--it;
    fileInfo.setFile ( pVFile->qsFileName );
    pEntry = Global::pThumbsCache->find ( pVFile->iFileHashValue, bImageFolder );
    pItem  = new VFolderItem ( pEntry, m_pListView, fileInfo.fileName( ) );
    if ( pEntry ) {
      if ( ! pEntry->arrayOfThumbs )
             pEntry->loadImages  ( );
      theImage = pEntry->arrayOfThumbs[0]->smoothScale ( m_iThumbSize, m_iThumbSize, QImage::ScaleMin );
      thePixmap.convertFromImage ( theImage );
      pItem->setPixmap ( 0, thePixmap );
    }
  }
  QTimer::singleShot ( 10, this, SLOT ( slotAlternateColors ( ) ) );

  connect ( m_pButtonAdd,    SIGNAL ( clicked ( ) ), this, SLOT ( slotAdd    ( ) ) );
  connect ( m_pButtonDelete, SIGNAL ( clicked ( ) ), this, SLOT ( slotDelete ( ) ) );
  connect ( m_pButtonUp,     SIGNAL ( clicked ( ) ), this, SLOT ( slotUp     ( ) ) );
  connect ( m_pButtonDown,   SIGNAL ( clicked ( ) ), this, SLOT ( slotDown   ( ) ) );
}

void DialogVFolder::slotAdd ( )
{
  uint t;
  QFileInfo      fileInfo;
  QImage         theImage;
  QPixmap        thePixmap;
  QString        qsMovieFilter;
  QStringList    list;
  QListViewItem *pItem;
  if ( m_bIsImageFolder )  {
    qsMovieFilter =  Global::pApp->getMovieFilter ( );
    list          =  FilePreviewDialog::getOpenFileNames ( NULL, Global::qsCurrentPath, QString ("Movies ( ") + qsMovieFilter + QString (" );;All ( * )"));
  }
  else  {
    qsMovieFilter =  Global::pApp->getImageFilter ( );
    list          =  FilePreviewDialog::getOpenFileNames ( NULL, Global::qsCurrentPath, QString ("Images ( ") + qsMovieFilter + QString (" );;All ( * )"));
  }
  if ( list.count ( ) < 1 )
    return;

  Cache::Thumbs::Entry *pEntry = NULL;
  for ( t=0; t<list.count ( ); t++ ) {
    pEntry = Global::pThumbsCache->find ( list[t], m_bIsImageFolder );
    fileInfo.setFile ( list[t] );
    if ( pEntry )  {
      pItem  = new VFolderItem ( pEntry, m_pListView, fileInfo.fileName ( ) );
      if ( ! pEntry->arrayOfThumbs )
	pEntry->loadImages ( );
      theImage = pEntry->arrayOfThumbs[0]->smoothScale ( m_iThumbSize, m_iThumbSize, QImage::ScaleMin );
      thePixmap.convertFromImage ( theImage );
      pItem->setPixmap ( 0, thePixmap );
    }
    else {
      // No cache entry exists. So we have to create one.
      if ( m_bIsImageFolder )
        pEntry = Global::pThumbsCache->append ( list[t], (QImage *)NULL );
      else
        pEntry = Global::pThumbsCache->append ( list[t], this );
      pItem    = new VFolderItem ( pEntry, m_pListView, fileInfo.fileName ( ) );
      theImage = QImage ( ).fromMimeSource( "please_wait.jpg" );
      theImage = theImage.smoothScale ( m_iThumbSize, m_iThumbSize, QImage::ScaleMin );
      thePixmap.convertFromImage ( theImage );
      pItem->setPixmap ( 0, thePixmap );
      m_bCanClose = false;
    }
  }
  QTimer::singleShot ( 10, this, SLOT ( slotAlternateColors ( ) ) );
}

void DialogVFolder::timerEvent ( QTimerEvent *pEvent )
{
  // called back after the thumbs have been rendered in the CacheThumbs class.
  if ( (const uint)pEvent->timerId ( ) == MEDIASCANNER_EVENT + 5 ) { // TYPE_MULTI_THUMB
    QImage  theImage;
    QPixmap thePixmap;
    VFolderItem *pItem;
    Cache::Thumbs::Entry *pEntry = NULL;
    QValueList<Cache::Thumbs::Entry *>list = Global::pThumbsCache->getScannedThumbs ( );
    QValueList<Cache::Thumbs::Entry *>::iterator it;
    it = list. begin ( );
    while ( it != list.end ( ) ) {
      pEntry = *it++;
      // Next we have to find the corresponding Item in the ListView 
      pItem = (VFolderItem *)m_pListView->firstChild ( );
      while ( pItem ) {
	if  ( pItem->pEntry == pEntry ) {
	  pEntry->iScanStatus = 0;
	  if ( ! pEntry->arrayOfThumbs )
	         pEntry->loadImages  ( );
	  theImage = pEntry->arrayOfThumbs[0]->smoothScale ( m_iThumbSize, m_iThumbSize, QImage::ScaleMin );
	  thePixmap.convertFromImage ( theImage );
	  pItem->setPixmap ( 0, thePixmap );
	  break;
	}
	pItem = (VFolderItem *)pItem->nextSibling ( );
      }
    }
  }

  uiDialogVFolder::timerEvent ( pEvent );
  m_bCanClose = true;
}

void DialogVFolder::slotDelete ( )
{
  QListViewItem *pItem = m_pListView->selectedItem ( );
  m_pNextItem  = pItem->nextSibling ( );

  if ( pItem )
    delete pItem;

  QTimer::singleShot ( 10, this, SLOT ( slotAlternateColors ( ) ) );
}

void DialogVFolder::slotUp ( )
{
  QListViewItem *pItem = m_pListView->selectedItem ( );
  if ( ! pItem )
    return;
  QListViewItem *pPrevItem = pItem->itemAbove ( );
  if ( ! pPrevItem )
    return;
  pPrevItem = pPrevItem->itemAbove ( );

  if ( ! pPrevItem ) {
    m_pListView->takeItem    ( pItem );
    m_pListView->insertItem  ( pItem );
    m_pListView->setSelected ( pItem, TRUE );
  }
  else
    pItem->moveItem  ( pPrevItem );
  QTimer::singleShot ( 10, this, SLOT ( slotAlternateColors ( ) ) );
}

void DialogVFolder::slotDown ( )
{
  QListViewItem *pItem = m_pListView->selectedItem ( );
  if ( ! pItem )
    return;
  QListViewItem *pNextItem = pItem->itemBelow ( );
  if ( ! pNextItem )
    return;
  pItem->moveItem ( pNextItem );

  QTimer::singleShot ( 10, this, SLOT ( slotAlternateColors ( ) ) );
}

void DialogVFolder::slotAlternateColors ( )
{
  VFolderItem *pItem = (VFolderItem *)m_pListView->firstChild ( );
  bool bAlternate = true;
  while ( pItem ) {
    bAlternate = ! bAlternate;
    pItem->bAlternateColor = bAlternate;
    pItem->repaint ( );
    pItem = (VFolderItem *)pItem->nextSibling ( );
  }

  if ( m_pNextItem )
       m_pNextItem->setSelected ( TRUE );
  m_pNextItem = NULL;
}

QValueList<Cache::Thumbs::Entry *> &DialogVFolder::getEntries ( )
{
  return m_returnList;
}

void DialogVFolder::accept ( )
{
  if ( ! m_bCanClose ) {
    QMessageBox::warning ( this, tr ( "Can't close." ), tr ( "Can not close dialog while waiting for MediaScanner to finish.\nPlease wait for previews to be generated and try again." ), QMessageBox::Ok, QMessageBox::NoButton );
    return;
  }
  
  m_returnList.clear ( );
  VFolderItem *pItem = (VFolderItem *)m_pListView->firstChild ( );
  while ( pItem ) {
    m_returnList.append  ( pItem->pEntry );
    pItem = (VFolderItem *)pItem->nextSibling ( );
  }      

  uiDialogVFolder::accept ( );
}

void DialogVFolder::reject ( )
{
  if ( ! m_bCanClose ) {
    QMessageBox::warning ( this, tr ( "Can't close." ), tr ( "Can not close dialog while waiting for MediaScanner to finish.\nPlease wait for previews to be generated and try again." ), QMessageBox::Ok, QMessageBox::NoButton );
    return;
  }
  uiDialogVFolder::reject ( );
}

}; // end namespace Input
