/* Copyright (C) 2006 P.L. Lucas
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA. 
 */

#include "syntax.h"
#include <iostream>
#include <QtXml/QXmlSimpleReader>
#include <QtXml/QXmlDefaultHandler>
#include <QTextBlockUserData>

//QMap<QString, QVector<Syntax::Rule> > Syntax::instances;

QList<Syntax::Rule*> Syntax::rules;

/*** Xml Handler ***/
class SyntaxXmlHandler:public QXmlDefaultHandler
{
private:
  Syntax *syntax;
  QString type_name, text;
  struct Tag
  {
  	QString tag, type;
  	QStringList items;
  	QList<Tag> childs;
  };
  QList<Tag> stack;
public:
  	// Constructor
  	SyntaxXmlHandler(Syntax *s): QXmlDefaultHandler(), syntax(s) {};


	bool startElement(const QString &/*namespaceURI*/, const QString &/*localName*/,
			const QString &qname, const QXmlAttributes &atts)
	{
		Tag tag;
		tag.tag=qname;
		
		if(qname == "list")
		{// List block. Get the type name.
			tag.type = atts.value("name").trimmed();
		}
		//else if(qname == "item")
		//{// Item. Next string is an item.
		//}
		else if(qname == "comment")
		{// Comments.
			syntax->setComment(atts.value("start"), "$", atts.value("name"));
		}
	
		
		stack.append(tag);
		return true;
	}


	bool characters(const QString &ch)
	{
		text+=ch;
		return true;
	}
  
	bool endElement( const QString & /*namespaceURI*/, const QString & /*localName*/, const QString & qname )
	{
		Tag tag;
		
		if(stack.isEmpty()) return true;
		
		tag=stack.last();
		stack.removeLast();
		
		if(tag.tag!=qname)
		{
			printf("Error reading XML syntax\n");
			return false;
		}
		
		if(qname == "list")
		{// List block. Get the type name.
			if(stack.last().tag=="list")
			{
				stack.last().childs.append(tag);
			}
			else
			{
				syntax->setItem(tag.items.join("|"), tag.type);
				for(int i=0;i<tag.childs.size();i++)
				{
					syntax->setItem(tag.childs[i].items.join("|"), tag.childs[i].type,tag.type);
				}
			}
		}	
		else if(qname == "item")
		{// Item. Next string is an item.
			if(! text.trimmed().isEmpty() )
				stack.last().items << text.trimmed();
		}
		
		text="";
		
		return true;
	}
};

/*** Block data ***/

class BlockData:public QTextBlockUserData
{
	public:
		BlockData() {braket_start_pos=braket_end_pos=-1;}
		~BlockData (){}
		int braket_start_pos, braket_end_pos;
};



/*** Syntax ***/

Syntax::Syntax(QTextEdit *parent): QSyntaxHighlighter(parent)
{
  QTextCharFormat f;
  
  	QFont text_edit_font;
  	QString font_name=get_config("textEditFont");
	QString font_size=get_config("textEditFontSize");
	if(font_name.isEmpty())
	{
		font_name=text_edit_font.family();
	}
	if(font_size.isEmpty())
	{
		font_size=QString::number(text_edit_font.pointSize());
	}
	text_edit_font.setFamily(font_name);
	text_edit_font.setPointSize(font_size.toInt());

  f.setFont(text_edit_font);
  f.setFontWeight(QFont::Bold);
  _format["keywords"] = f;
  _format["commands"] = f;
  f.setFontWeight(QFont::Normal);
  f.setForeground(Qt::darkGreen);
  _format["builtin"] = f;
  f.setForeground(Qt::blue);
  _format["functions"] = f;
  // operators
  f.setForeground(Qt::black);
  _format["variables"] = f;
  f.setForeground(Qt::darkMagenta);
  _format["numbers"] = f;
  f.setForeground(Qt::red);
  _format["strings"] = f;
  // delimiters
  f.setForeground(Qt::darkGray);
  _format["singleLine"] = f;
  //Brackets matched
  f.setForeground(Qt::black);
  //f.setFontWeight(QFont::Bold);
  f.setBackground(Qt::yellow);
  _format["bracket match"]=f;
  
  active_ok=true;
  
  	braketsMacth_ok=false;
  
}

static void clear_block_data(QTextDocument *doc);

Syntax::~Syntax()
{
	//foreach(Rule *value, rules_map)
	//{
	//	delete value;
	//}
	
	clear_block_data(document());
	//This line is added because sometimes Qt try rehighlight text at destroy
	setDocument(NULL);
}

void Syntax::load(const QString &path)
{
  //rules = &(instances[path]);
  if(rules.isEmpty())
  {
    // Load from file
    FILE *fl;

    fl = fopen(path.toLocal8Bit().constData(), "rt");
    if(!fl)
    {
      std::cerr << "[Syntax::load] Can not load the syntax file" << std::endl;
      return;
    }
    
    QFile file(path);
    QXmlSimpleReader parser;
    QXmlInputSource source(&file);
    SyntaxXmlHandler handler(this);

    file.open(fl, QIODevice::ReadOnly);
    
    parser.setContentHandler(&handler);
    parser.setErrorHandler(&handler);

    parser.parse(&source);
    
    file.close();
    
    fclose(fl);
    
    std::cout << "[Sytax::load] "
	      << path.toLocal8Bit().constData()
	      << " loaded"
	      << std::endl;
  }
}

// void Syntax::setItem(const QString &item, const QString &type)
// {
//   Rule r;
//   if(!item.isEmpty())
//   {
//     r.pattern = QRegExp(item);
//     r.type = type;
//     r.format = _format[type];
//     rules.push_back(r);
//   }
// }

void Syntax::setItem(const QString &item, const QString &type, const QString parent)
{
	Rule *r;
	if(!item.isEmpty())
	{
		r=new Rule;
		r->pattern = QRegExp(item);
		r->type = type;
		r->format = _format[type];
		rules_map[type]=r;
		if(parent.isEmpty() || !rules_map.contains(parent))
			rules.push_back(r);
		else
			rules_map[parent]->rules.push_back(r);
	}
}

void Syntax::setComment(const QString &start, const QString &end, const QString &type)
{
	Rule *r;
	if(!type.isEmpty())
	{
		r=new Rule;
		r->pattern = QRegExp(/*QString("^") +*/ start + ".*" + end);
		r->type = type;
		r->format = _format[type];
		rules_map[type]=r;
		rules.push_back(r);
	}
}

void Syntax::setType(const QString &type, const QTextCharFormat &f)
{
  _format[type] = f;
}

void Syntax::highlightBlock(const QString &str)
{
  //Para aumentar el rendimiento se hace una tabla i_aux con la posición de lo
  //que ha encontrado cada expresión regular rules.at(n)->pattern.
  //Se aplicará el formato debido a la Rule que tenga la i_aux más pequeña
  if( !str.isEmpty() && !rules.isEmpty() && active_ok )
  {
	setFormat(0, str.length(), _format["variables"]);
  	
  	int i=0, len=0; //Actual position
	int *i_aux, n_min; //Auxiliar position and minimal position
	
	i_aux=new int[rules.size()];
	QRegExp *re=new QRegExp[rules.size()];
	
	for(int n=0;n<rules.size();n++)
	{
		re[n]=rules.at(n)->pattern;
		i_aux[n] = re[n].indexIn( str, i);
		//printf("%s %d %d\n", rules.at(n)->type.toLocal8Bit().data(), n, i_aux[n]);
	}
	
	while(i >= 0)
	{
		n_min=-1;
		for(int n=0;n<rules.size();n++)
		{
			if(i_aux[n]<0) continue;
			if(i_aux[n]<i ) i_aux[n] = re[n].indexIn( str, i);
			//printf("%s n=%d i_aux=%d n_min=%d i=%d\n", rules.at(n)->type.toLocal8Bit().data(), n, i_aux[n], n_min, i);
			if( n_min<0 || i_aux[n_min]<0 || (i_aux[n]>=0 && i_aux[n]<i_aux[n_min]) )
			{
				n_min=n;
				if(i_aux[n]==i) break;
			}
		}
		//printf("n_min=%d elegido\n", n_min);
		if(n_min>=0) i=i_aux[n_min];
		else break;
		if( i<0 ) break;
		len = re[n_min].matchedLength();
		
		//QStringList list=re[n_min].capturedTexts ();
		//printf("\n");
		//for(int n=0;n<list.size();n++)
		//{
		//	printf("%d >%s<\n", n, list.at(n).toLocal8Bit().data() );
		//}
		//printf("Aplicando %s i=%d len=%d\n", rules.at(n_min)->type.toLocal8Bit().data(), i, len);
		if(len<1) break;
		//QTextCharFormat i_format=format(i);
		//if( !(i_format==strings) )
		
		if(rules.at(n_min)->rules.isEmpty())
		{
			setFormat(i, len, rules.at(n_min)->format);
			
			//Do brackets macth
			if( braketsMacth_ok && currentBlockUserData()!= NULL )
			{
				BlockData *dat=(BlockData *)currentBlockUserData();
				if(dat->braket_start_pos>=0)
					setFormat(dat->braket_start_pos, 1, _format["bracket match"]);
				if(dat->braket_end_pos>=0)
					setFormat(dat->braket_end_pos, 1, _format["bracket match"]);
			}
		}
		else
		{
			//Rules can contains another rules
			QString text=str.mid(i,len);
			//printf("text=%s\n", text.toLocal8Bit().data() );
			bool format_ok=true;
			for(int n=0;n<rules.at(n_min)->rules.size(); n++)
			{
				if(rules.at(n_min)->rules.at(n)->pattern.exactMatch(text))
				{
					setFormat(i, len, rules.at(n_min)->rules.at(n)->format);
					format_ok=false;
					break;
				}
			}
			if(format_ok) setFormat(i, len, rules.at(n_min)->format);
		}
		i+=len;
		//printf("i=%d\n",i);
	}
	
	delete [] i_aux;
	delete [] re;
  }
}


static int forward_search(QTextBlock & block, int pos, char bracket_start, char bracket_end)
{
	int i=pos,  open=0;
	
	while(block.isValid())
	{
		if(!block.text().isEmpty())
		{
			QString str=block.text();
			int len=str.length();
			
			//This line is added to check lower position
			if(i<0) i=0;
			
			for(;i<len;i++) //i<len checks upper position
			{
				QChar ch=str.at(i);
				
				if(ch==bracket_end)
				{
					open--;
					if(open==0) return i;
				}
				else if(ch==bracket_start) open++;
			}
		}
		block=block.next();
		
		i=0;
	}
	
	return -1;
}

static int backward_search(QTextBlock & block, int pos, char bracket_start, char bracket_end)
{
	int i=pos,  open=0;
	
	while(block.isValid())
	{
		if(!block.text().isEmpty())
		{
			QString str=block.text();
			int len=str.length();
			
			//This line is added to check upper position
			if(i>=len) i=len-1;
			
			for(;i>=0;i--) //i>=0 checks lower position
			{
				QChar ch=str.at(i);
				
				if(ch==bracket_start)
				{
					open--;
					if(open==0) return i;
				}
				else if(ch==bracket_end) open++;
			}
		}
		block=block.previous();
		
		if(block.isValid() && !block.text().isEmpty()) i=block.length()-1;
	}
	
	return -1;
}

static void set_block_data(QTextBlock & block0, QTextBlock & block1, int start, int end)
{
	BlockData *udat=new BlockData();
	udat->braket_start_pos=start;
	block0.setUserData(udat);
	
	if(block0==block1)
	{
		udat->braket_end_pos=end;
	}
	else
	{
		udat=new BlockData();
		udat->braket_end_pos=end;
		block1.setUserData(udat);
	}
}

static void clear_block_data(QTextDocument *doc)
{
	QTextBlock block=doc->findBlock(0);
	
	while( block.isValid() )
	{
#if QT_VERSION < 0x040400
		QTextBlockUserData *userdata=block.userData();
#endif
		block.setUserData(NULL);
#if QT_VERSION < 0x040400
		if(userdata!=NULL) delete userdata;
#endif
		block=block.next();
	}
}

void Syntax::braketsMacth(int pos)
{
	QTextDocument *doc=document();
	QTextBlock block0=doc->findBlock(pos), block1;
	if(!block0.isValid() || block0.text().length()<=0) 
	{
		return;
	}
	
	pos=pos-block0.position();
	if (block0.text().size()<=pos) pos=block0.text().size()-1;
	if(pos<0) pos=0;
	
	QChar ch=block0.text().at(pos);
	
	block1=block0;
	
	int i=-1;
	if(ch=='(')
	{
		i=forward_search(block1,pos,'(', ')');
	}
	else if(ch==')')
	{
		i=backward_search(block1,pos,'(', ')');
	}
	else if(ch=='[')
	{
		i=forward_search(block1,pos,'[', ']');
	}
	else if(ch==']')
	{
		i=backward_search(block1,pos,'[', ']');
	}
	else if(ch=='{')
	{
		i=forward_search(block1,pos,'{', '}');
	}
	else if(ch=='}')
	{
		i=backward_search(block1,pos,'{', '}');
	}
	else
	{
		if( braketsMacth_ok )
		{
			//clear_block_data(doc);
			braketsMacth_ok=false;
			rehighlight();
		}
		
		return;
	}
	
	if(i>=0)
	{
		set_block_data(block0, block1, pos, i);
		braketsMacth_ok=true;
		rehighlight();
		clear_block_data(doc);
	}
}


void Syntax::setActive(bool active)
{
	active_ok=active;
}

