/* Copyright (C) 2006 P.L. Lucas
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA. 
 */

#include "table.h"
#include <QTextStream>
#include <QRegExp>
#include <QMenu>
#include <QClipboard>
#include <QMessageBox>
#include <QFile>
#include <QMenuBar>

Table::Table(QWidget * parent):BaseWidget(parent)
{
	widget_type=TABLE;

	table_form=new TableForm();
	table_form->setupUi(centralWidget());
	setWindowIcon( QIcon( QString( ICON_PATH )+"/table.png" ) );
	table_form->reloadButton->setIcon( QIcon( QString( ICON_PATH )+"/reload.png" ) );
	table_form->reloadButton->setToolTip(tr("<b>Reload matrix.</b><br> Some operations change matrix, use reload to view changes."));
	
	model=new ComplexNumberTableModel;
	table_form->table_widget->setModel(model);
	table_form->table_widget->setItemDelegate( new LineEditDelegate(this) );
	
	connect(model,SIGNAL(cellChanged( int, int, QString)),this,SLOT(cellChanged( int, int, QString)));
	connect(table_form->rows_spinbox,SIGNAL( editingFinished() ),this,SLOT(rows_changed()));
	
	connect(table_form->cols_spinbox,SIGNAL( editingFinished ()),this,SLOT(cols_changed()));
	
	connect(table_form->changeOrderButton,SIGNAL( clicked() ),this,SLOT( rows_changed() ));
	connect(table_form->changeOrderButton,SIGNAL( clicked() ),this,SLOT( cols_changed() ));
	connect(table_form->changeOrderButton,SIGNAL( clicked() ),this,SLOT( windowActivated() ));
	
	connect(table_form->reloadButton,SIGNAL( clicked() ),this,SLOT( windowActivated() ));
	
	build_menu();
	setContextMenuPolicy ( Qt::DefaultContextMenu );
	menuBar()->addMenu(menu);
}

void Table::setOctaveConnection(OctaveConnection *octave_connection)
{
	this->octave_connection=octave_connection;
	connect(octave_connection,SIGNAL(line_ready(QString)),this,SLOT(line_ready(QString)));
}

void Table::setMatrix(QString matrix)
{
	this->matrix=matrix;
	setWindowTitle("Table: "+matrix);
}

void Table::windowActivated()
{
	windowActivated(this);
}

void Table::windowActivated(QWidget *w)
{
	if(w!=this) return;
	QString command;
	/*
	command+="function qtoctave_show_matrix(m);";
	command+="fprintf(stderr,\"~~matrix:"+matrix+" #%d %d\\n\",rows(m),columns(m));";
	command+="for i=1:rows(m);";
		command+="fprintf(stderr,\"~~matrix:"+matrix+" %d\",i);";
		command+="for j=1:columns(m);";
			command+="if(imag(m(i,j))==0);";
				command+="fprintf(stderr,\" %g\",m(i,j));";
			command+="elseif(imag(m(i,j))>0);";
				command+="fprintf(stderr,\" %g+%gi\",m(i,j),imag(m(i,j)));";
			command+="else;";
				command+="fprintf(stderr,\" %g%gi\",m(i,j),imag(m(i,j)));";
			command+="endif;";
		command+="endfor;";
		command+="fprintf(stderr,\"\\n\");";
	command+="endfor;endfunction;";
	command+="eval(\"qtoctave_show_matrix("+matrix+")\", \"qtoctave_show_matrix("+matrix+"=[0])\")";
	*/
	
	command+="if( 0==0 )\n";
		command+="qtoctave__file_name=tmpnam();\n";
		//Next line: If matrix doesn't exit, it will build it
		command+="eval(\""+matrix+";\",\""+matrix+"=[0]\");";
		command+="save(\"-text\", qtoctave__file_name, \""+matrix+"\");\n";
// 		command+="__out=fopen(__file_name,\"w\");";
// 		command+="for __l=1:rows("+matrix+")\n";
// 		command+="for __c=1:columns("+matrix+")\n";
// 		command+="fprintf(__out,\"(%g,%g)\","+matrix+"(__l,__c), imag("+matrix+"(__l,__c)) );";
// 		command+="end;";
// 		command+="fprintf(__out,\"\\n\");";
// 		command+="end;";
// 		command+="fclose(__out);";
		command+="fprintf(stderr,\"~~matrix:"+matrix+" \\\"%s\\\" %d %d "+QString::number((long)this)+"\\n\", qtoctave__file_name, rows("+matrix+"),columns("+matrix+"));\n";
		//command+="disp(\"Guardados los datos\");";
		command+="clear qtoctave__file_name;\n";
	command+="endif\n";
	
	octave_connection->command_enter(command,false);
}


void Table::reloadCell(int row, int col)
{
	QString command;
		
	command+="fprintf(stderr,\"~~matrixCell:"+matrix+" %d %d %g %g\\n\", "
	+QString::number(row+1)+", "+QString::number(col+1)+", "
	"real("+matrix+"("+QString::number(row+1)+","+QString::number(col+1)+")), "
	"imag("+matrix+"("+QString::number(row+1)+","+QString::number(col+1)+")));\n";
	
	octave_connection->command_enter(command,false);
}


#include <QTime>

void Table::line_ready(QString line)
{
	if(line.startsWith("~~matrixCell:"+matrix))
	{
		line=line.right(line.length() - (13+matrix.length()) ).trimmed();
		//printf("line:>%s<\n",line.toLocal8Bit().data());
		
		double real, imag;
		int row, col;
		sscanf(line.toLocal8Bit().data(), "%d %d %lf %lf",&row, &col, &real, &imag);
		
		model->setRawData(row-1,col-1, real, imag);
		//model->update();
		model->update(row-1,col-1);
		
		return;
	}
	else if( ! line.startsWith("~~matrix:"+matrix) ) return;
	//line=line.right(line.length() - (10+matrix.length()) );
	//printf("line:>%s<\n",line.toLocal8Bit().data());
	
	QTime time;
	
	time.start();
	
	line=line.trimmed();
	
	//Read line-event data:
	//~~matrix:matrix-name "file_name" rows columns
	
	QRegExp mre("~~matrix:([A-Za-z0-9_]+) \"([^\"]+)\" (\\d+) (\\d+) (\\d+)");
	
	int _rows=-1, _columns=-1;
	QString file_name;
	
	mre.indexIn ( line ) ;
	{
		QStringList list = mre.capturedTexts();
		//for(int i=0;i<list.size();i++)
		//	printf("Texto(%d): %s\n", i, list.at(i).toLocal8Bit().data());
		if( ((long)this)!=list.at(5).toInt() ) return;
		file_name=list.at(2);
		_rows=list.at(3).toInt();
		_columns=list.at(4).toInt();
		model->resize(_rows,_columns);
		table_form->rows_spinbox->setValue(_rows);
		table_form->cols_spinbox->setValue(_columns);
		//printf("Table: %s %d %d\n", file_name.toLocal8Bit().data(), _rows, _columns);
	}
	
	{
		QFile file(file_name);
		if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) return;
		
		//disconnect(model,SIGNAL(cellChanged( int, int)),this,SLOT(cellChanged( int, int, QString)));
		
		int col=0, row=0;
		char ch;
		QString number, real, imag;
		number.reserve(64);
		real.reserve(32);
		imag.reserve(32);
		bool data_available_ok=false;
		
		while (!file.atEnd()) {
		
			if(!file.getChar(&ch)) break;

			switch(ch)
			{
				case ' ':
				if( !number.isEmpty() )
				{
					real=number;
					data_available_ok=true;
				}
				number.clear();
				break;
				
				case ',':
				real=number;
				number.clear();
				break;
				
				case ')':
				imag=number;
				number.clear();
				data_available_ok=true;
				break;
				
				case '(':
				real.clear();
				imag.clear();
				number.clear();
				break;
				
				case '\n':
				if( !number.isEmpty() )
				{
					real=number;
					data_available_ok=true;
				}
				else {row++;col=0;}
				//real.clear();
				//imag.clear();
				number.clear();
				break;
				
				case '#':
					if(!file.getChar(&ch)) break;
					while(ch!='\n') if(!file.getChar(&ch)) break;
				continue;
				break;
				
				default:
				number.append(ch);
			};
			
			if( data_available_ok )
			{
				//printf("(%s,%s)\n", real.toLocal8Bit().data(), imag.toLocal8Bit().data());
				if( imag.isEmpty() ) model->setRawData(row,col, real.toDouble() );
				else model->setRawData(row,col, real.toDouble(), imag.toDouble() );
				col++;
				if(ch=='\n') {row++;col=0;}
				data_available_ok=false;
			}
		}
		
		file.close();
		
		file.remove();
		
		//connect(model,SIGNAL(cellChanged( int, int, QString)),this,SLOT(cellChanged( int, int, QString)));
		
		printf("Datos cargados %dms\n", time.elapsed() );
		
		model->update();
	}
}

void Table::cellChanged ( int row, int col, QString value )
{
	QString command(matrix+"("), aux;
	command+=aux.setNum(row+1)+",";
	command+=aux.setNum(col+1)+")="+value+";";
	octave_connection->command_enter(command);
	
	reloadCell(row, col);
}

void Table::change_rows()
{
	int rows=table_form->rows_spinbox->value();
	int old_rows=model->rowCount();
	int old_cols=model->columnCount();
	
	if(rows<old_rows)
	{
		QString command;
		QTextStream(&command) 	<< matrix << "=" << matrix << "(1:" << rows 
					<< ",1:" << old_cols << ");";
		octave_connection->command_enter(command);
	}
	else if(rows>old_rows)
	{
		QString command;
		QTextStream(&command) 	<< matrix << "(" << rows 
					<< "," << old_cols << ")=0;";
		octave_connection->command_enter(command);
	}
}

void Table::rows_changed()
{
	change_rows();
	windowActivated(this);
}

void Table::cols_changed()
{
	change_cols();
	windowActivated(this);
}

void Table::change_cols()
{
	int cols=table_form->cols_spinbox->value();
	int old_rows=model->rowCount();
	int old_cols=model->columnCount();
	
	if(cols<old_cols)
	{
		QString command;
		QTextStream(&command) 	<< matrix << "=" << matrix << "(1:" << old_rows 
					<< ",1:" << cols << ");";
		octave_connection->command_enter(command);
	}
	else if(cols>old_cols)
	{
		QString command;
		QTextStream(&command) 	<< matrix << "(" << old_rows 
					<< "," << cols << ")=0;";
		octave_connection->command_enter(command);
	}
}

void Table::order_changed()
{
	change_rows();
	change_cols();
	windowActivated(this);
}

QString Table::getMatrix()
{
	return matrix;
}

void Table::build_menu()
{
	menu=new QMenu("Table", this);
	QAction *copyAction=new QAction("Copy", menu);
	menu->addAction(copyAction);
	connect(copyAction, SIGNAL(triggered()), this, SLOT(copy_cb()) );
	QAction *copyMatrixAction=new QAction("Copy as Octave matrix", menu);
	menu->addAction(copyMatrixAction);
	connect(copyMatrixAction, SIGNAL(triggered()), this, SLOT(copy_matrix_cb()) );
	QAction *pasteAction=new QAction("Paste", menu);
	menu->addAction(pasteAction);
	connect(pasteAction, SIGNAL(triggered()), this, SLOT(paste_cb()) );
	QAction *deleteRowsAction=new QAction("Delete rows", menu);
	menu->addAction(deleteRowsAction);
	connect(deleteRowsAction, SIGNAL(triggered()), this, SLOT(delete_rows_cb()) );
	QAction *deleteColumnsAction=new QAction("Delete columns", menu);
	menu->addAction(deleteColumnsAction);
	connect(deleteColumnsAction, SIGNAL(triggered()), this, SLOT(delete_columns_cb()) );
	QAction *insertColumnRightAction=new QAction("Insert column (right)", menu);
	menu->addAction(insertColumnRightAction);
	connect(insertColumnRightAction, SIGNAL(triggered()), this, SLOT(insert_column_right_cb()) );
	QAction *insertColumnLeftAction=new QAction("Insert column (left)", menu);
	menu->addAction(insertColumnLeftAction);
	connect(insertColumnLeftAction, SIGNAL(triggered()), this, SLOT(insert_column_left_cb()) );
	QAction *insertRowUpAction=new QAction("Insert row (up)", menu);
	menu->addAction(insertRowUpAction);
	connect(insertRowUpAction, SIGNAL(triggered()), this, SLOT(insert_row_up_cb()) );
	QAction *insertRowDownAction=new QAction("Insert row (down)", menu);
	menu->addAction(insertRowDownAction);
	connect(insertRowDownAction, SIGNAL(triggered()), this, SLOT(insert_row_down_cb()) );

	/* Gráficas */
	menu->addSeparator();

	QMenu *plotMenu = new QMenu("Plot", menu);
	menu->addMenu(plotMenu);
	QMenu *plot2dMenu = new QMenu("2D", plotMenu);
	plotMenu->addMenu(plot2dMenu);

	QAction *plotAction = new QAction("Plot", plot2dMenu);
	plot2dMenu->addAction(plotAction);
	connect(plotAction, SIGNAL(triggered()), this, SLOT(plotPlot()));

	QAction *polarAction = new QAction("Polar", plot2dMenu);
	plot2dMenu->addAction(polarAction);
	connect(polarAction, SIGNAL(triggered()), this, SLOT(plotPolar()));

	QAction *logxyAction = new QAction("Log scale for the x and y axis", plot2dMenu);
	plot2dMenu->addAction(logxyAction);
	connect(logxyAction, SIGNAL(triggered()), this, SLOT(plotLogXandY()));

	QAction *logyAction = new QAction("Log scale for the y axis", plot2dMenu);
	plot2dMenu->addAction(logyAction);
	connect(logyAction, SIGNAL(triggered()), this, SLOT(plotLogY()));

	QAction *logxAction = new QAction("Log scale for the x axis", plot2dMenu);
	plot2dMenu->addAction(logxAction);
	connect(logxAction, SIGNAL(triggered()), this, SLOT(plotLogX()));

	QAction *barAction = new QAction("Bar graph", plot2dMenu);
	plot2dMenu->addAction(barAction);
	connect(barAction, SIGNAL(triggered()), this, SLOT(plotBar()));
}

void Table::insert_row_up_cb()
{
	QString _command;
	QTextStream command(&_command);
	
	int row=table_form->table_widget->currentIndex().row()+1;
	
	command << matrix << "=[" << matrix << "(1:" << row-1 << ",:); zeros(1,columns(" << matrix << ")); "
			<< matrix << "(" << row << ":rows(" << matrix << "),:)]";
	
	octave_connection->command_enter(_command);
	
	windowActivated();
}

void Table::insert_row_down_cb()
{
	QString _command;
	QTextStream command(&_command);
	
	int row=table_form->table_widget->currentIndex().row()+1;
	
	command << matrix << "=[" << matrix << "(1:" << row << ",:); zeros(1,columns(" << matrix << ")); "
			<< matrix << "(" << row+1 << ":rows(" << matrix << "),:)]";
	
	octave_connection->command_enter(_command);
	
	windowActivated();
}

void Table::insert_column_left_cb()
{
	QString _command;
	QTextStream command(&_command);

	int col=table_form->table_widget->currentIndex().column()+1;
	
	command << matrix << "=[" << matrix << "(:,1:" << col-1 << "), zeros(rows(" << matrix << "),1), "
			<< matrix << "(:," << col << ":columns(" << matrix << "))]";
	
	octave_connection->command_enter(_command);
	
	windowActivated();
}

void Table::insert_column_right_cb()
{
	QString _command;
	QTextStream command(&_command);
	
	int col=table_form->table_widget->currentIndex().column()+1;
	
	command << matrix << "=[" << matrix << "(:,1:" << col << "), zeros(rows(" << matrix << "),1), "
			<< matrix << "(:," << col+1 << ":columns(" << matrix << "))]";
	
	octave_connection->command_enter(_command);
	
	windowActivated();
}

void Table::delete_columns_cb()
{
	int result=QMessageBox::question (this, "Warning", "Comlumns will be deleted. Continue?", QMessageBox::Ok, QMessageBox::Cancel);
	
	if(result!=QMessageBox::Ok) return;
	
	
	QString _command;
	QTextStream command(&_command);
	
	command << matrix << "=[";
	QModelIndexList ranges=table_form->table_widget->selectionModel ()->selectedIndexes ();
	int column=0;
	int start, end;
	bool ok=false;
	start=-1;
	end=-1;
	while(column < model->columnCount ())
	{
		ok=true;
		
		for(int i=0;i<ranges.size();i++)
		{
			if( ranges.at(i).column()  == column )
			{
				ok=false;
				break;
			}
		}
		if(ok && start<0) start=column;
		if(!ok)
		{
			if(start>=0)
			{
				end=column;
				command << matrix << "(1:rows(" << matrix << "),"<< start+1 << ":" << end <<"), ";
				start=-1;
			}
		}
		column++;
	}
	if(ok && start>=0)
	{
		end=column;
		command << matrix << "(1:rows(" << matrix << "),"<< start+1 << ":" << end <<") ";
	}
	command << "];";
	
	octave_connection->command_enter(_command);
	
	windowActivated();
}


void Table::delete_rows_cb()
{
	int result=QMessageBox::question (this, "Warning", "Rows will be deleted. Continue?", QMessageBox::Ok, QMessageBox::Cancel);
	
	if(result!=QMessageBox::Ok) return;
	
	
	QString _command;
	QTextStream command(&_command);
	
	command << matrix << "=[";
	QModelIndexList ranges=table_form->table_widget->selectionModel ()->selectedIndexes ();
	int row=0;
	int start, end;
	bool ok=false;
	start=-1;
	end=-1;
	while(row < model->rowCount ())
	{
		ok=true;
		
		for(int i=0;i<ranges.size();i++)
		{
			if( ranges.at(i).row() == row )
			{
				ok=false;
				break;
			}
		}
		if(ok && start<0) start=row;
		if(!ok)
		{
			if(start>=0)
			{
				end=row;
				command << matrix << "(" << start+1 << ":" << end << ",1:columns(" << matrix << "));\n";
				start=-1;
			}
		}
		row++;
	}
	if(ok && start>=0)
	{
		end=row;
		command << matrix << "(" << start+1 << ":" << end << ",1:columns(" << matrix << "))\n";
	}
	command << "];";
	
	octave_connection->command_enter(_command);
	
	windowActivated();
}

void Table::paste_cb()
{
	QClipboard *clipboard = QApplication::clipboard();
	QString text=clipboard->text();
	
	QString _command;
	QTextStream command(&_command);
	
	int row=table_form->table_widget->currentIndex().row()+1, col=table_form->table_widget->currentIndex().column()+1;
	
	QRegExp rx("([0-9ieEj\\.\\-\\+]+|\\n)");
	QString value;
	int pos = 0;
	while ((pos = rx.indexIn(text, pos)) != -1) {
		value=rx.cap(0);
		pos += rx.matchedLength();
		if(value=="\n") {row++;col=table_form->table_widget->currentIndex().column()+1;}
		else
		{
			command << matrix << "(" << row << "," << col << ")=" << value << ";\n";
			col++;
		}
	}
	octave_connection->command_enter(_command);
	
	windowActivated();
}

void Table::copy_matrix_cb()
{
	QString str("[");
	QModelIndexList ranges=table_form->table_widget->selectionModel ()->selectedIndexes ();
	
	QMap<int, QMap<int,QString> > values;
	
	for(int i=0;i<ranges.size();i++)
	{
		values[ranges.at(i).row()][ranges.at(i).column()] = model->data(ranges.at(i)).toString();
	}
	
	QList<int> rows=values.keys();
	for(int i=0;i<rows.size();i++)
	{
		QList<int> cols=values[rows.at(i)].keys();
		str+=values[rows.at(i)][cols.at(0)];
		for(int j=1;j<cols.size();j++)
		{
			str+=", "+values[rows.at(i)][cols.at(j)];
		}
		str+=";\n";
	}
	
	str+="]";
	QClipboard *clipboard = QApplication::clipboard();
	clipboard->setText(str);
}

void Table::copy_cb()
{
	QString str;
	QModelIndexList ranges=table_form->table_widget->selectionModel ()->selectedIndexes ();
	
	QMap<int, QMap<int,QString> > values;
	
	for(int i=0;i<ranges.size();i++)
	{
		values[ranges.at(i).row()][ranges.at(i).column()] = model->data(ranges.at(i)).toString();
	}
	
	QList<int> rows=values.keys();
	for(int i=0;i<rows.size();i++)
	{
		QList<int> cols=values[rows.at(i)].keys();
		str+=values[rows.at(i)][cols.at(0)];
		for(int j=1;j<cols.size();j++)
		{
			str+=" "+values[rows.at(i)][cols.at(j)];
		}
		str+="\n";
	}

	QClipboard *clipboard = QApplication::clipboard();
	clipboard->setText(str);
}

void Table::contextMenuEvent ( QContextMenuEvent * event )
{
	QPoint p(event->globalX(),event->globalY());
	menu->popup(p);
}

void Table::plot(TablePlot::Type type)
{
  TablePlot *dialog = new TablePlot(this, table_form->table_widget, model, type);
  
  if(dialog->exec() == QDialog::Accepted)
  {
    try
    {
      this->octave_connection->command_enter(dialog->command());
    }catch(const char *str)
    {
      //QMessageBox errorMsg(QMessageBox::Warning, "Error", str, QMessageBox::Ok, this);
      //errorMsg.exec();
      
      QMessageBox::warning (NULL, "Error", str);
    }
  }

  delete dialog;
}

void Table::plotPlot()
{
  plot(TablePlot::PLOT);
}

void Table::plotPolar()
{
  plot(TablePlot::POLAR);
}

void Table::plotLogXandY()
{
  plot(TablePlot::LOGLOG);
}

void Table::plotLogY()
{
  plot(TablePlot::SEMILOGY);
}

void Table::plotLogX()
{
  plot(TablePlot::SEMILOGX);
}

void Table::plotBar()
{
  plot(TablePlot::BAR);
}


BaseWidget *Table::copyBaseWidget(QWidget * parent )
{
	Table *bw=new Table(parent);
	bw->setSession(session);
	bw->setOctaveConnection(octave_connection);
	bw->setMatrix( getMatrix() );
	
	return bw;
}

void Table::toXML(QXmlStreamWriter &xml)
{
	xml.writeStartElement("matrix");
	
	xml.writeAttribute("value", getMatrix());
	
	xml.writeEndElement();
}


////////////////////////////////////////////////////////////////////////////////

ComplexNumberTableModel::ComplexNumberTableModel(QObject *parent):QAbstractTableModel(parent)
{
	rows=cols=0;
	real=imag=NULL;
}

////////////////////////////////////////////////////////////////////////////////

int ComplexNumberTableModel::rowCount ( const QModelIndex & /*parent*/ ) const
{
	return rows;
}

////////////////////////////////////////////////////////////////////////////////

int ComplexNumberTableModel::columnCount ( const QModelIndex & /*parent*/ ) const
{
	return cols;
}

////////////////////////////////////////////////////////////////////////////////

QVariant ComplexNumberTableModel::data ( const QModelIndex & index, int role ) const
{
	if (!index.isValid())
		return QVariant();
	
	if (index.row() > rows || index.column() > cols)
		return QVariant();
	
	if (role == Qt::DisplayRole)
	{
		int row=index.row(), col=index.column();
		
		return data(row,col);
	}
	else
		return QVariant();
}

////////////////////////////////////////////////////////////////////////////////

QString ComplexNumberTableModel::data(int row, int col) const
{
	QString value = QString::number(real[row][col]);
	if(imag[row][col]!=0) 
	{
		if(imag[row][col]>0) value+="+"+QString::number(imag[row][col])+"i";
		else value+=QString::number(imag[row][col])+"i";
	}
	return value;
}

////////////////////////////////////////////////////////////////////////////////

void ComplexNumberTableModel::resize( int _rows, int _columns)
{
	if(_rows==rows && _columns==cols) 
	{
		update();
		return;
	}
	
	if(real!=NULL)
	{
		for(int i=0;i<rows;i++) delete [] real[i];
		delete [] real;
	}
	if(imag!=NULL)
	{
		for(int i=0;i<rows;i++) delete [] imag[i];
		delete [] imag;
	}
	
	if(rows>0)
	{
		beginRemoveRows(QModelIndex(), 0, rows-1);
		endRemoveRows();
	}
	if(cols>0)
	{
		beginRemoveColumns(QModelIndex(), 0, cols-1);
		endRemoveColumns();
	}
	
	rows=_rows; cols=_columns;
	
	real=new double*[rows];
	imag=new double*[rows];
	
	for(int i=0;i<rows;i++)
	{
		real[i]=new double[cols]; imag[i]=new double[cols];
	}
	
	if(rows>0)
	{
		beginInsertRows(QModelIndex(), 0, rows-1);
		endInsertRows();
	}
	if(cols>0)
	{
		beginInsertColumns(QModelIndex(), 0, cols-1);
		endInsertColumns();
	}
}

////////////////////////////////////////////////////////////////////////////////

ComplexNumberTableModel::~ComplexNumberTableModel()
{
	if(real!=NULL)
	{
		for(int i=0;i<rows;i++) delete [] real[i];
		delete [] real;
	}
	if(imag!=NULL)
	{
		for(int i=0;i<rows;i++) delete [] imag[i];
		delete [] imag;
	}
}

////////////////////////////////////////////////////////////////////////////////

bool ComplexNumberTableModel::setData ( const QModelIndex & index, const QVariant & value, int role )
{
	if (index.isValid() && role == Qt::EditRole)
	{
		emit dataChanged(index,index);
		emit cellChanged(index.row(), index.column(), value.toString() );
		return true;
	}
	return false;
}

////////////////////////////////////////////////////////////////////////////////

Qt::ItemFlags ComplexNumberTableModel::flags ( const QModelIndex & /*index*/ ) const
{
	return Qt::ItemIsEditable|Qt::ItemIsSelectable|Qt::ItemIsEnabled;
}

////////////////////////////////////////////////////////////////////////////////

void ComplexNumberTableModel::update()
{
	//beginInsertRows(QModelIndex(), 0, rows-1);
	//endInsertRows();
	//beginInsertColumns(QModelIndex(), 0, rows-1);
	//endInsertColumns();
	QModelIndex cell0=index(0, 0), cell1;
	int n_rows=rows-1, n_cols=cols-1;
	if(n_rows<0) n_rows=0;
	if(n_cols<0) n_cols=0;
	cell1=index(n_rows, n_cols);
	emit dataChanged(cell0,cell1);
}

////////////////////////////////////////////////////////////////////////////////

void ComplexNumberTableModel::update(int row, int col) 
{
	QModelIndex cell=index(row, col);
	emit dataChanged(cell,cell);
}

////////////////////////////////////////////////////////////////////////////////

LineEditDelegate::LineEditDelegate(QObject *parent ):QItemDelegate(parent)
{
	
}

////////////////////////////////////////////////////////////////////////////////

void LineEditDelegate::setEditorData(QWidget *editor,
                                        const QModelIndex &index) const
{
	QString value = index.model()->data(index, Qt::DisplayRole).toString();
	
	QLineEdit *lineedit = static_cast<QLineEdit*>(editor);
	lineedit->setText(value);
}

#ifdef __DELEGATE_WIDGET__

////////////////////////////////////////////////////////////////////////////////

QWidget *LineEditDelegate::createEditor(QWidget *parent,
        const QStyleOptionViewItem &/* option */,
        const QModelIndex & index) const
{
	QLineEdit *editor = new QLineEdit(parent);
	editor->installEventFilter(const_cast<LineEditDelegate*>(this));

	return editor;
}



////////////////////////////////////////////////////////////////////////////////

void LineEditDelegate::setModelData(QWidget *editor, QAbstractItemModel *model,
                                       const QModelIndex &index) const
{
	QLineEdit *lineedit = static_cast<QLineEdit*>(editor);
	
	QString value = lineedit->text();
	
	model->setData(index, value);
}

////////////////////////////////////////////////////////////////////////////////

void LineEditDelegate::updateEditorGeometry(QWidget *editor,
        const QStyleOptionViewItem &option, const QModelIndex &/* index */) const
{
	editor->setGeometry(option.rect);
}

////////////////////////////////////////////////////////////////////////////////

bool LineEditDelegate::eventFilter( QObject * object, QEvent * event )
{
	if(event->type()==QEvent::KeyPress)
	{
		QKeyEvent *ekey=(QKeyEvent*)event;
		QWidget *editor=static_cast<QWidget*>(object);
		switch(ekey->key())
		{
			case Qt::Key_Return:
			case Qt::Key_Enter:
				emit commitData(editor);
			case Qt::Key_Escape:
				emit  closeEditor(editor,NoHint);
				event->setAccepted(true);
				return true;
			break;
		}
	}
	
	return false;
}

#endif
