/* # skkinput (Simple Kana-Kanji Input)
 * skkconfig.c --- Read dot.skkinput.
 * This file is part of skkinput.
 * Copyright (C) 1997
 * Takashi SAKAMOTO (sakamoto@yajima.kuis.kyoto-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/socket.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include "commondef.h"
#include "buffers.h"
#include "SeparateWin.h"
#include "config.h"
#include "skkkey.h"

/*
 * ¤Τġ
 */
enum {
  PARAM_BOOLEAN,
  PARAM_INTEGER,
  PARAM_STRING,
  PARAM_PATH,
  PARAM_INPUT_VECTOR,
  PARAM_ROM_KANA_RULE_LIST, 
#ifdef THIS_IS_NECESSARY
  PARAM_DEFINE_KEY_SKKMAP,
#endif
  PARAM_DEFINE_KEY_SKKJMAP,
  PARAM_DEFINE_KEY_ABBREVMAP,
  PARAM_SKKINPUTLISP,
#ifdef THIS_IS_NECESSARY
  PARAM_UNDEFINE_KEY_SKKMAP,
#endif
  PARAM_UNDEFINE_KEY_SKKJMAP,
  PARAM_UNDEFINE_KEY_ABBREVMAP,
  PARAM_END,
} ;

typedef struct {
  char *string ;
  int action ;
  void *argument ;
} SkkInpDef ;

struct skk_rom_kana_rule_list {
  struct skk_rom_kana_rule element ;
  struct skk_rom_kana_rule_list *next ;
} ;

/*
 * ץȥ
 */
/* skkldic.c */
extern int check_skkinput_jisyo_code( char *path ) ;
/* parseStr.c */
struct myChar *skip_empty_symbols( struct myChar *ptr ) ;
struct myChar *parseString
( struct myChar *string, struct myChar **dest, int *result ) ;
unsigned char *expand_file_name( unsigned char *path ) ;
int parseKeyString
( struct myChar *string, struct myChar *buffer, int bufsize ) ;

/* lispeval.c */
extern int skkinputlisp_weak_eval
( struct myChar *commandstring ) ;

/* keymap.c */
extern int skkinput_CompileDefaultKeyBind( void ) ;
extern int skkinput_DefineKey
( struct myChar *string, int length, int func_no, int map ) ;
extern int skkinput_UndefineKey
( struct myChar *string, int length, int map ) ;

/*
 * skkinput ưꤹѿ
 */
static int  skkserv_portnum ;	/* obsoleted, use skkserv_service */
extern char *skkserv_host ;
extern char *skkserv_service;
extern int skkserv_pf;
static char *skkserv_protocol_family ;
extern char *skkinput_local_jisyo_name ;
extern char *skkinput_backup_jisyo_name ;
extern char *skk_local_jisyo_name ;
extern char *skkinput_record_name ;
extern int  skk_egg_like_newline ;
extern int  skkinput_dabbrev_like_completion ;
extern int  skkinput_chatadaptermode ;
extern int  skkinput_search_skk_jisyo ;
extern int  skkinput_keep_record ;
extern int  skkinput_date_ad ;
extern int  skkinput_number_style ;
extern int  skkinput_delete_implies_kakutei ;
extern int  skkinput_use_numeric_conversion ;
extern int  skkinput_tab_width ;
extern int  skkinput_rjj_like_input ;

extern Boolean skkinput_jisyo_dirty ;

extern unsigned long skkinput_autosave_interval ;

/*
 * ϥ٥ȥ롣
 */
static unsigned char *skkinput_default_input_vector[ 128 ] = {
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  "",  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,
  NULL,  NULL,  NULL,  NULL,  "",  "",  "",  NULL,
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  "",  "",  NULL,  NULL,  NULL,  "",
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  "",  NULL,  "",  NULL,  NULL,
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,
} ;

/*
 * ºݤѴ˻Ȥϥ٥ȥ롣
 */
struct myChar *skkinput_input_vector[ 128 ] ;

/*
 * ϥ٥ȥ롣
 */
static unsigned char *skkinput_default_zenkaku_vector[ 128 ] = {
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL,  NULL, 
  "",  "",  "",  "",  "",  "",  "",  "",
  "",  "",  "",  "",  "",  "",  "",  "",
  "",  "",  "",  "",  "",  "",  "",  "", 
  "",  "",  "",  "",  "",  "",  "",  "",
  "",  "",  "",  "",  "",  "",  "",  "",
  "",  "",  "",  "",  "",  "",  "",  "", 
  "",  "",  "",  "",  "",  "",  "",  "",  
  "",  "",  "",  "",  "",  "",  "",  "",  
  "",  "",  "",  "",  "",  "",  "",  "",  
  "",  "",  "",  "",  "",  "",  "",  "",  
  "",  "",  "",  "",  "",  "",  "",  "",  
  "",  "",  "",  "",  "",  "",  "",  NULL,
} ;

/*
 * ºݤѴ˻Ȥѥ٥ȥ롣
 */
struct myChar *skkinput_zenkaku_vector[ 128 ] ;

/*
 * ޻̾Ѵ롼ꥹȡ
 */
static struct skk_rom_kana_rule 
skkinput_default_rom_kana_rule_list[] = {
  { NULL, NULL, NULL, NULL },
  { NULL, NULL, NULL, NULL },
  { NULL, NULL, NULL, NULL },
} ;

static struct skk_rom_kana_rule_list *skkinput_rom_kana_rule_list_top ;

extern struct skk_rom_kana_rule *skkinput_rom_kana_rule_list ;

/*
 * .skkinput ˵ҤǤΰ
 */
SkkInpDef skkinput_initdef[] = {
  { "skk-port-num:",		
    PARAM_INTEGER,		&skkserv_portnum },
  { "skk-service:",
    PARAM_STRING,		&skkserv_service },
  { "skk-server-protocol-family:",
    PARAM_STRING,		&skkserv_protocol_family },
  { "skk-server-host:",	
    PARAM_STRING,		&skkserv_host },
  { "skkinput-jisyo:",
    PARAM_PATH,		 	&skkinput_local_jisyo_name },
  { "skkinput-backup-jisyo:",
    PARAM_PATH,		 	&skkinput_backup_jisyo_name },
  { "skk-jisyo:",
    PARAM_PATH,		 	&skk_local_jisyo_name },
  { "skkinput-record:",
    PARAM_PATH,		 	&skkinput_record_name },
  { "skkinput-egg-like-newline:",
    PARAM_BOOLEAN,		&skk_egg_like_newline },
  { "skkinput-dabbrev-like-completion:",
    PARAM_BOOLEAN,		&skkinput_dabbrev_like_completion },
  { "skkinput-chat-adapter-mode:",
    PARAM_BOOLEAN,		&skkinput_chatadaptermode },
  { "skkinput-keep-record:",
    PARAM_BOOLEAN,		&skkinput_keep_record },
  { "skkinput-search-skk-jisyo:",
    PARAM_BOOLEAN,		&skkinput_search_skk_jisyo },
  { "skkinput-date-ad:",
    PARAM_BOOLEAN,		&skkinput_date_ad },
  { "skkinput-number-style:",
    PARAM_INTEGER,		&skkinput_number_style },
  { "skkinput-delete-implies-kakutei:",
    PARAM_BOOLEAN,		&skkinput_delete_implies_kakutei },
  { "skkinput-use-numeric-conversion:",
    PARAM_BOOLEAN,		&skkinput_use_numeric_conversion },
  { "skkinput-zenkaku-vector:",
    PARAM_INPUT_VECTOR,		skkinput_zenkaku_vector },
  { "skkinput-input-vector:",
    PARAM_INPUT_VECTOR,		skkinput_input_vector },
  { "skkinput-rom-kana-rule-list:",
    PARAM_ROM_KANA_RULE_LIST,	&skkinput_rom_kana_rule_list_top },
#ifdef THIS_IS_NECESSARY
  { "define-key-skkmap:",
    PARAM_DEFINE_KEY_SKKMAP,	NULL },
  { "define-key-skkjmap:",
    PARAM_DEFINE_KEY_SKKJMAP,	NULL },
#else
  { "define-key-skkmap:",
    PARAM_DEFINE_KEY_SKKJMAP,	NULL },
#endif
  { "define-key-abbrevmap:",
    PARAM_DEFINE_KEY_ABBREVMAP,	NULL },
#ifdef THIS_IS_NECESSARY
  { "undefine-key-skkmap:",
    PARAM_UNDEFINE_KEY_SKKMAP,	NULL },
  { "undefine-key-skkjmap:",
    PARAM_UNDEFINE_KEY_SKKJMAP,	NULL },
#else
  { "undefine-key-skkmap:",
    PARAM_UNDEFINE_KEY_SKKJMAP,	NULL },
#endif
  { "undefine-key-abbrevmap:",
    PARAM_UNDEFINE_KEY_ABBREVMAP,NULL },
  { "tab-width:",
    PARAM_INTEGER,		&skkinput_tab_width },
  { "skkinput-lisp:",
    PARAM_SKKINPUTLISP,	 	NULL },
  { "skkinput-rjj-like-input:",
    PARAM_BOOLEAN,		&skkinput_rjj_like_input },
  { "skkinput-autosave-interval:",
    PARAM_INTEGER,		&skkinput_autosave_interval },
  { NULL,
    PARAM_END,		NULL },
} ;

/*
 * ꤵ줿ʸ󤬶ʸʤΤɤ֤ؿ
 *----
 * Ȱʲ϶ʸǤȤư
 */
static int is_empty_line( struct myChar *ptr )
{
  while( !IS_END_OF_STRING( *ptr ) &&
	 !IS_ASCII_EQUAL( *ptr, ';' ) ){
    if( !IS_ASCII_EQUAL( *ptr, 0x20 ) &&
	!IS_ASCII_EQUAL( *ptr, '\t' ) &&
	!IS_ASCII_EQUAL( *ptr, '\n' ) &&
	!IS_ASCII_EQUAL( *ptr, '\r' ) )
      return False ;
    ptr ++ ;
  }
  return True ;
}

/*
 * Boolean Ĥޤ "t"  "nil" Τɤ餬ꤵƤ뤫ؿ
 */
static int skkinput_analyze_boolean
( struct myChar *oneline, void *argument )
{
  oneline = skip_empty_symbols( oneline ) ;

  /* Ȥ٤ꤵƤ뤫ɤȽǤ롣*/
  if( IS_ASCII_EQUAL( *oneline, 't' ) && is_empty_line( oneline + 1 ) ){
    *( int *)argument = True ;
    return True ;
  } 
  /* Ȥ٤ꤵƤ뤫ɤȽǤ롣*/
  if( !myCharCharStrncmp( oneline, "nil", 3 ) ){
    if( is_empty_line( oneline + 3 ) ){
      *( int *)argument = False ;
      return True ;
    }
  }
  return False ;
}

static struct myChar *skip_number_digit( struct myChar *ptr )
{
  /* ³¤ꥹåפ롣*/
  while( !IS_END_OF_STRING( *ptr ) ){
    if( !IS_ASCII_CHARA( *ptr ) || ptr->chara < '0' || ptr->chara > '9' )
      break ;
    ptr ++ ;
  }
  return ptr ;
}

static struct myChar *analyze_integer_sub
( struct myChar *ptr, int *result, int *value )
{
  struct myChar *start ;
  struct myChar ch ;
  /* ޤʸ򥹥åפơȡ*/
  start = skip_empty_symbols( ptr ) ;
  /* 줫顢ʸȳФȴФޤ*/
  ptr = skip_number_digit( start ) ;
  /* ʸʤä顢ȡ*/
  if( start == ptr ){
    *result = False ;
    return ptr ;
  }
  /* äƤ顢ȡ*/
  ch     = *ptr ;
  MYCHAR_SET_END_OF_STRING( *ptr ) ;
  *value = myCharAtoi( start ) ;
  *ptr   = ch ;
  *result = True ;
  return ptr ;
}

/*
 * ʤꤵƤ뤫ؿ
 */
static int skkinput_analyze_integer
( struct myChar *oneline, void *argument )
{
  int result, value ;
  oneline = analyze_integer_sub( oneline, &result, &value ) ;
  if( !result )
    return False ;
  /* ͤμǤϤʤˤϥ顼*/
  if( !is_empty_line( oneline ) ){
    /* ͤϤʤ*/
    return False ;
  }
  *( int *)argument = value ;
  return True ;
}

/*
 * ʤʸꤵƤ뤫ؿ
 */
static int skkinput_analyze_string
( struct myChar *oneline, void *argument,
  int expandFilenameFlag, int convertBackslash )
{
  struct myChar *result ;
  unsigned char *asciiString ;
  int ret ;

  oneline = parseString( oneline, &result, &ret ) ;
  /* 顼Ƚꡣ*/
  if( ret < 0 )
    return False ;
  if( !is_empty_line( oneline ) ){
    /* ˥ߤդƤ뤼줸ܤ٥ӡ */
    free( result ) ;
    return False ;
  }
  /* ʸ򥳥ԡ */
  asciiString = myCharStringToAsciiString( result ) ;
  free( result ) ;
  if( expandFilenameFlag ){
    *( unsigned char **)argument = expand_file_name( asciiString ) ;
    free( asciiString ) ;
  } else {
    *( unsigned char **)argument = asciiString ;
  }
  return True ;
}

/*
 * Ϳ줿Ԥϥ٥ȥǤ뤫顢ᤷơ
 * skkinput_input_vector  skkinput_zenkaku_vector 褦
 * ؿ
 */
static int skkinput_analyze_input_vector
( struct myChar *oneline, void *argument )
{
  struct myChar *ptr ;
  struct myChar *istring ;
  int result, value, ret ;

  istring = NULL ;
  /* ǽζʸ򥹥åפޤ*/
  oneline = skip_empty_symbols( oneline ) ;
  /* ( 123  "" ) */ 
  if( !IS_ASCII_EQUAL( *oneline, '(' ) )
    goto err_ret ;
  oneline ++ ;
  /* ͤȴФޤ*/
  ptr = analyze_integer_sub( oneline, &result, &value ) ;
  /* ̤֤äƤʤä顢ܤǤ硣*/
  if( !result )
    goto err_ret ;
  /* input vector ʤͤä顢ΤƤޤ礦*/
  if( value > 128 || value < 0 )
    goto err_ret ;
  /* Τޤ޺ǸޤǹԤäƤޤä顢ܤǤ礦*/
  ptr = skip_empty_symbols( ptr ) ;
  if( !IS_ASCII_EQUAL( *ptr, '.' ) )
    goto err_ret ;
  /* ʸϤޤ*/
  ptr = parseString( ptr + 1, &istring, &ret ) ;
  if( ret < 0 )
    goto err_ret ;
  /* ³ʸФޤ*/
  ptr = skip_empty_symbols( ptr ) ;
  /* Ǹ ")" ĤƤʤСܤǤ*/
  if( !IS_ASCII_EQUAL( *ptr, ')' ) )
    goto err_ret ;
  ptr ++ ;
  /* θ˥ߤäƤϤޤ*/
  if( !is_empty_line( ptr ) )
    goto err_ret ;
  /* input vector 򹹿ޤ*/
  if( ( ( struct myChar ** )argument )[ value ] != NULL ){
    free( ( ( struct myChar ** )argument )[ value ] ) ;
  }
  ( ( struct myChar ** )argument )[ value ] = istring ;
  return True ;
  
err_ret:
  if( istring != NULL )
    free( istring ) ;
  return False ;
}

static void add_skkinput_rule_list
( struct skk_rom_kana_rule_list **top,
  struct skk_rom_kana_rule element )
{
  struct skk_rom_kana_rule_list *node ;
  if( *top == NULL ){
    *top = node = malloc( sizeof( struct skk_rom_kana_rule_list ) ) ;
  } else {
    node = *top ;
    while( node->next != NULL )
      node = node->next ;
    node->next = malloc( sizeof( struct skk_rom_kana_rule_list ) ) ;
    node = node->next ;
  }
  if( node == NULL ){
    fprintf( stderr, "Fatal: Memory Exhausted.\n" ) ;
    exit( 1 ) ;
  }
  node->next = NULL ;
  /* ¤Τ򥳥ԡ롣*/
  node->element = element ;
  return ;
}

static int skkinput_analyze_rule_list
( struct myChar *oneline, void *argument )
{
  struct myChar *ptr, *rstr[ 4 ] ;
  struct skk_rom_kana_rule node ;
  int i, ret ;

  /* ǽζʸ򥹥åפޤ*/
  oneline = skip_empty_symbols( oneline ) ;
  /* ( "nn" "" "" "" ) */ 
  if( !IS_ASCII_EQUAL( *oneline, '(' ) )
    return False ;
  oneline ++ ;
  ptr = oneline ;
  for( i = 0 ; i < 4 ; i ++ ){
    /* ʸϤޤ*/
    ptr = parseString( ptr, rstr + i, &ret ) ;
    if( ret < 0 ){
      while( i < 4 )
	rstr[ i ++ ] = NULL ;
      goto err_analyze ;
    }
  }
  /* ʸ󤬽ä顢")" ĤƤȦ*/
  ptr = skip_empty_symbols( ptr ) ;
  if( !IS_ASCII_EQUAL( *ptr, ')' ) )
    goto err_analyze ;
  /* ǡ")" ³϶ʸȦ(⤷ϥȤɤ) */
  ptr ++ ;
  if( !is_empty_line( ptr ) )
    goto err_analyze ;
  /* ǡȴФޤơȡ*/
  node.state   = rstr[ 0 ] ;
  node.next    = rstr[ 1 ] ;
  node.houtput = rstr[ 2 ] ;
  node.koutput = rstr[ 3 ] ;
  /* ߤξ̵֤ä顢㥨顼äϤåޤ*/
  if( node.state == NULL )
    goto err_analyze ;
  add_skkinput_rule_list( argument, node ) ;
  return True ;
err_analyze:
  for( i = 0 ; i < 4 ; i ++ ){
    if( rstr[ i ] != NULL )
      free( rstr[ i ] ) ;
  }
  return False ;
}

/* ( "@" . "j-self-insert" ) */
static int skkinput_analyze_define_key
( struct myChar *oneline,
  struct myChar **rkeystring,
  struct myChar **rkeyfunction,
  int *rlength )
{
  struct myChar *keyfunction, *ptr, *keystring ;
  int result ;

  keystring = keyfunction = NULL ;

  /* ǽζʸ򥹥åפޤ*/
  oneline = skip_empty_symbols( oneline ) ;
  /* ( "nn" "" "" "" ) */ 
  if( !IS_ASCII_EQUAL( *oneline, '(' ) )
    goto err_ret ;
  oneline ++ ;
  /* ʸϤޤ*/
  ptr = parseString( oneline, &keystring, &result ) ;
  /* Ԥġ*/
  if( keystring == NULL || result < 0 )
    goto err_ret ;
  *rlength = result ;
  /* Τޤ޺ǸޤǹԤäƤޤä顢ܤǤ礦*/
  ptr = skip_empty_symbols( ptr ) ;
  if( !IS_ASCII_EQUAL( *ptr, '.' ) )
    goto err_ret ;
  /* ʸϤޤ*/
  ptr = parseString( ptr + 1, &keyfunction, &result ) ;
  /* Ԥġ*/
  if( keyfunction == NULL || result < 0 )
    goto err_ret ;
  /* ³ʸФޤ*/
  ptr = skip_empty_symbols( ptr ) ;
  /* Ǹ ")" ĤƤʤСܤǤ*/
  if( !IS_ASCII_EQUAL( *ptr, ')' ) )
    goto err_ret ;
  ptr ++ ;
  /* θ˥ߤäƤϤޤ*/
  if( !is_empty_line( ptr ) )
    goto err_ret ;
  /* ̤ޤ*/
  *rkeystring   = keystring ;
  *rkeyfunction = keyfunction ;
  return True ;
err_ret:
  if( keystring != NULL )
    free( keystring ) ;
  if( keyfunction != NULL )
    free( keyfunction ) ;
  return False ;
}

/*
 * ؿ̾ؿֹѴԤؿ
 */
static int skkinput_string2function_number
( struct myChar *keyfunction )
{
  int func_no ;
  static unsigned char *function_string_table[] = {
    "self-insert-command",
    "j-self-insert",
    "j-self-zenkaku-insert",
    "j-display-code-for-char-at-point",
    "j-set-henkan-point",
    "j-set-henkan-point-subr",
    "j-insert-a",
    "j-insert-e",
    "j-insert-i",
    "j-insert-o",
    "j-insert-u",
    "j-kana-input",
    "j-start-henkan",
    "j-insert-comma",
    "j-insert-period",
    "j-purge-from-jisyo",
    "j-input-by-code-or-menu",
    "j-mode-off",
    "j-toggle-kana",
    "j-previous-candidate",
    "j-kakutei",
    "j-abbrev-input",
    "j-abbrev-period",
    "j-abbrev-comma",
    "j-zenkaku-eiji",
    "j-zenkaku-henkan",
    "j-today",
    "save-skkinput-local-jisyo",
    "vc-toggle-overthespot-like-input",
    "j-mode-off-and-self-insert",

    "j-kanainput-mode-on",
    "newline",
    "set-mark-command",
    "forward-char",
    "backward-char",
    "delete-char",
    "delete-backward-char",
    "j-try-comletion",
    "end-of-line",
    "beginning-of-line",
    "kill-line",
    "yank",
    "kill-region",
    "kill-ring-save",
    "exchange-point-and-mark",
    "previous-line",
    "next-line",
    "transpose-chars",
    "redraw",
    "prefix-char",
    "sendback-next-key",
    "sendback-key",
    "keyboard-quit",
    "close-skkinput",
    "vc-toggle-chatmode",
    "not-modified",
    "forward-word",
    "backward-word",
    "upcase-word",
    "downcase-word",
    "capitalize-word",
    "vc-toggle-eggnl",
    NULL,
  } ;
  /* ɤʸ˰פΤĴ٤롣*/
  func_no = 0 ;
  while( function_string_table[ func_no ] != NULL ){
    if( !myCharCharStrcmp
	( keyfunction, function_string_table[ func_no ] ) )
      return func_no ;
    func_no ++ ;
  }
  return ERR ;
}

/*
 * define-key-skkmap ᤹ؿ
 */
static int skkinput_analyze_define_key_skkmap
( struct myChar *oneline, int map )
{
  struct myChar *keyfunction ;
  struct myChar *keystring ;
  int func_no, ret = False, uselen ;

  /* ޤʸΥڥ᤹롣*/
  if( !skkinput_analyze_define_key
      ( oneline, &keystring, &keyfunction, &uselen ) )
    return False ;
  /* ɤδؿʤΤֹ򸫤Ĵ٤롣*/
  if( ( func_no = skkinput_string2function_number( keyfunction ) ) < 0 )
    goto err_ret ;
  
  ret = skkinput_DefineKey( keystring, uselen, func_no, map ) ;
err_ret:
  free( keystring ) ;
  free( keyfunction ) ;
  return ret ;
}

/*
 * undefine-key-skkmap ᤹ؿ
 */
static int skkinput_analyze_undefine_key_skkmap
( struct myChar *oneline, int map )
{
  struct myChar *keystring, *keybuf ;
  int ret = False, uselen ;

  keystring = keybuf = NULL ;

  /* ޤʸ᤹롣*/
  if( !skkinput_analyze_string( oneline, &keystring, False, False ) )
    goto err_ret ;
  /* ξ֤ keystring ˤϡȤΤޤޤʸ󤬥ԡƤ
   * 롣ᤷʤФʤʤ*/
  uselen = parseKeyString( keystring, NULL, 0 ) ;
  /* ˼ԤˤϼΤƤ롣*/
  if( uselen <= 0 )
    goto err_ret ;
  if( ( keybuf = malloc( sizeof( struct myChar ) * uselen ) ) == NULL )
    goto err_ret ;
  ( void )parseKeyString( keystring, keybuf, uselen ) ;
  ret = skkinput_UndefineKey( keybuf, uselen, map ) ;
  free( keybuf ) ;
err_ret:
  if( keystring != NULL )
    free( keystring ) ;
  return ret ;
}

/*
 * lisp ǽ񤫤줿Ԥ᤹ؿ
 */
static int skkinput_analyze_skkinputlisp
( struct myChar *oneline )
{
  if( !skkinputlisp_weak_eval( oneline ) )
    return False ;
  return True ;
}

/*
 * Ԥ¹Ԥؿ
 */
static int skkinput_analyze_line( struct myChar *oneline )
{
  int i, length ;
  struct myChar *wptr ;

  for( i = 0 ; skkinput_initdef[ i ].string != NULL ; i ++ ){
    /* ʸ󤬰פ뤫ɤȽǤ롣*/
    length = strlen( skkinput_initdef[ i ].string ) ;
    if( myCharCharStrncmp
	( oneline, skkinput_initdef[ i ].string, length ) )
      continue ;
    wptr = oneline + length ;

    /* פʸб륢¹Ԥ롣*/
    switch( skkinput_initdef[ i ].action ){
    case PARAM_BOOLEAN :
      return skkinput_analyze_boolean
	( wptr, skkinput_initdef[ i ].argument ) ;
    case PARAM_INTEGER :
      return skkinput_analyze_integer
	( wptr, skkinput_initdef[ i ].argument ) ;
    case PARAM_STRING :
      return skkinput_analyze_string
	( wptr, skkinput_initdef[ i ].argument, False, True ) ;
    case PARAM_PATH :
      return skkinput_analyze_string
	( wptr, skkinput_initdef[ i ].argument, True, True ) ;
    case PARAM_INPUT_VECTOR :
      return skkinput_analyze_input_vector
	( wptr, skkinput_initdef[ i ].argument ) ;
    case PARAM_ROM_KANA_RULE_LIST :
      return skkinput_analyze_rule_list
	( wptr, skkinput_initdef[ i ].argument ) ;
#ifdef THIS_IS_NECESSARY
    case PARAM_DEFINE_KEY_SKKMAP :
      return skkinput_analyze_define_key_skkmap( wptr, 0 ) ;
#endif
    case PARAM_DEFINE_KEY_SKKJMAP :
      return skkinput_analyze_define_key_skkmap( wptr, 1 ) ;
    case PARAM_DEFINE_KEY_ABBREVMAP :
      return skkinput_analyze_define_key_skkmap( wptr, 2 ) ;
#ifdef THIS_IS_NECESSARY
    case PARAM_UNDEFINE_KEY_SKKMAP :
      return skkinput_analyze_undefine_key_skkmap( wptr, 0 ) ;
#endif
    case PARAM_UNDEFINE_KEY_SKKJMAP :
      return skkinput_analyze_undefine_key_skkmap( wptr, 1 ) ;
    case PARAM_UNDEFINE_KEY_ABBREVMAP :
      return skkinput_analyze_undefine_key_skkmap( wptr, 2 ) ;
    case PARAM_SKKINPUTLISP :
      return skkinput_analyze_skkinputlisp( wptr ) ;
    default :
      return False ;
    }
  }
  return False ;
}

/*
 * skk-rom-kana-rule-list  skkinput-rom-kana-rule-list  
 * skkinput 褦Ѵؿ
 *---
 */
static struct skk_rom_kana_rule *makeSkkRomKanaRuleList
( struct skk_rom_kana_rule_list *top )
{
  int i ;
  struct skk_rom_kana_rule_list *node = top, *nNode ;
  struct skk_rom_kana_rule *rule_list = NULL ;

  if( node == NULL ){
    return skkinput_default_rom_kana_rule_list ;
  } else {
    for( i = 0 ; node != NULL ; i ++ )
      node = node->next ;
    /* 3 ĤϥǥեȤǤäƤΤ +3 */
    rule_list = malloc( sizeof( struct skk_rom_kana_rule ) * ( i + 3 ) ) ;
    if( rule_list == NULL ){
      return NULL ;
    }
    node = top ;
    for( i = 0 ; node != NULL ; i ++ ){
      rule_list[ i ] = node->element ;
#ifdef DEBUG
      if( rule_list[ i ].state == NULL ){
	fprintf( stderr, "( \"(null)\", " ) ;
      } else {
	fprintf( stderr, "( \"" ) ;
	myCharFputstring( stderr, rule_list[ i ].state ) ;
	fprintf( stderr, "\", " ) ;
	fflush( stderr ) ;
      }
      if( rule_list[ i ].next == NULL ){
	fprintf( stderr, "\"(null)\", " ) ;
      } else {
	fprintf( stderr, "\"" ) ;
	myCharFputstring( stderr, rule_list[ i ].next ) ;
	fprintf( stderr, "\", " ) ;
	fflush( stderr ) ;
      }
      if( rule_list[ i ].houtput == NULL ){
	fprintf( stderr, "\"(null)\", " ) ;
      } else {
	fprintf( stderr, "\"" ) ;
	myCharFputstring( stderr, rule_list[ i ].houtput ) ;
	fprintf( stderr, "\", " ) ;
	fflush( stderr ) ;
      }
      if( rule_list[ i ].koutput == NULL ){
	fprintf( stderr, "\"(null)\" )\n" ) ;
      } else {
	fprintf( stderr, "\"" ) ;
	myCharFputstring( stderr, rule_list[ i ].koutput ) ;
	fprintf( stderr, "\" )\n" ) ;
	fflush( stderr ) ;
      }
#endif
      nNode = node->next ;
      free( node ) ;
      node = nNode ;
    }
    rule_list[ i++ ] = skkinput_default_rom_kana_rule_list[ 0 ] ;
    rule_list[ i++ ] = skkinput_default_rom_kana_rule_list[ 1 ] ;
    rule_list[ i++ ] = skkinput_default_rom_kana_rule_list[ 2 ] ;
  }
  return rule_list ;
}

/*
 * եɤ߹ؿ
 *------
 * ֶϤʤΤϡ-option ǤȻפ顢ޤ DEFAULT 
 * ꡢˤեɤߡǸ option ǷꤷĤɡġ
 * ȡ ե뤬ˤΤ option ǻǤʤʤ
 *  ܰդʤġ
 */
int skkinput_readConfigFile( char *config_file )
{
  int chck = False, coding_system ;
  unsigned char *config_path ;
  /* ƣ patch ġֹֹ椬ľפȤΤФ patch*/
  unsigned long line = 1 ;
  FILE *fp ;
  struct myChar *oneline ;

  skkinput_rom_kana_rule_list_top = NULL ;
  /* ǥեȤΤΤѤ褦ؼ줿ν*/
  config_path = expand_file_name( config_file ) ;
  chck = True ;
  coding_system = check_skkinput_jisyo_code( config_path ) ;
  /* ե򳫤Ƥߤ롣*/
  if( ( fp = fopen( config_path, "rb" ) ) == NULL ){
    if( chck )
      free( config_path ) ;
    return False ;
  }
  /* ե뤬³¤ɤ߽Ф*/
  while( !feof( fp ) ){
    /* ɤ߽Ф*/
    if( ( oneline = mychar_readOneLine( fp, coding_system ) ) != NULL ){
      /* ȹԤäˤ̵뤹롣*/
      if( !is_empty_line( oneline ) ){
	if( !skkinput_analyze_line( oneline ) ){
	  fprintf( stderr, "Warning: I ignore line %ld.\n", line ) ;
	}
      }
      free( oneline ) ;
      oneline = NULL ;
    }
    line ++ ;
  }
  fclose( fp ) ;
  /* ƽλ롣*/
  if( chck )
    free( config_path ) ;
  /* rom-kana rule list ꤹ롣*/
  skkinput_rom_kana_rule_list = 
    makeSkkRomKanaRuleList( skkinput_rom_kana_rule_list_top ) ;

  /* 
   * if skkserv_portnum is defined use skkserv_portnum for 
   * backward compatibility.
   */
  if (skkserv_portnum != 0) {
    if (skkserv_service) free(skkserv_service);
    skkserv_service = malloc(8); /* 8 digit is enough */
    sprintf(skkserv_service, "%u", skkserv_portnum);
  }
  if (skkserv_protocol_family != NULL) {
    if (strcasecmp(skkserv_protocol_family, "inet4") == 0
	|| strcasecmp(skkserv_protocol_family, "ipv4") == 0) {
      skkserv_pf = PF_INET;
    } else if (strcasecmp(skkserv_protocol_family, "inet6") == 0
	       || strcasecmp(skkserv_protocol_family, "ipv6") == 0) {
#if defined(USE_INET6)
      skkserv_pf = PF_INET6;
#else
      fprintf( stderr, "Warning: IPv6 not supported.\n" ) ;
#endif
    }
  }
  return True ;
}

static void initialize_skkinputDefaultVectorSub
( struct myChar *myCharVector[], unsigned char *charVector[] )
{
  unsigned char **cptr = charVector ;
  struct myChar **wptr = myCharVector ;
  int i ;

  for( i = 0 ; i < 128 ; i ++, cptr ++ , wptr ++ ){
    if( *cptr != NULL ){
      *wptr = malloc( sizeof( struct myChar ) * 2 ) ;
      ( *wptr )->charset = CHARSET_JISX0208_1983 ;
      ( *wptr )->chara   = 
	( ( ( ( *cptr )[ 0 ] ) << 8 ) | ( ( *cptr )[ 1 ] ) ) & 0x7F7F ;
      MYCHAR_SET_END_OF_STRING( ( ( *wptr )[ 1 ] ) ) ;
    } else {
      *wptr = NULL ;
    }
  }
  return ;
}

static void initialize_skkinputDefaultVectors( void )
{
  initialize_skkinputDefaultVectorSub
    ( skkinput_input_vector, skkinput_default_input_vector ) ;
  initialize_skkinputDefaultVectorSub
    ( skkinput_zenkaku_vector, skkinput_default_zenkaku_vector ) ;
  return ;
}

/*
 * Υ޻̾Ѵ롼ꤹؿ
 *---
 *  char  string ǻꤷƤΤǡѿξǤ
 * ɡ󤫤 charset ѹƤޤäΤǡ줬Ǥʤʤ
 * Ƥޤäˤʤޤ
 */
static void initialize_skkinputDefaultRomKanaRuleList( void )
{
  struct myChar *ptr ;

  ptr = malloc( sizeof( struct myChar ) * 3 ) ;
  myCharCharStrcpy( ptr, "nn" ) ;
  skkinput_default_rom_kana_rule_list[ 0 ].state = ptr ;
  ptr = malloc( sizeof( struct myChar ) * 2 ) ;
  ptr->charset = CHARSET_JISX0208_1983 ;
  ptr->chara   = 0x2473 ;
  MYCHAR_SET_END_OF_STRING( *( ptr + 1 ) ) ;
  skkinput_default_rom_kana_rule_list[ 0 ].houtput = ptr ;
  ptr = malloc( sizeof( struct myChar ) * 2 ) ;
  ptr->charset = CHARSET_JISX0208_1983 ;
  ptr->chara   = 0x2573 ;
  MYCHAR_SET_END_OF_STRING( *( ptr + 1 ) ) ;
  skkinput_default_rom_kana_rule_list[ 0 ].koutput = ptr ;

  ptr = malloc( sizeof( struct myChar ) * 3 ) ;
  myCharCharStrcpy( ptr, "n'" ) ;
  skkinput_default_rom_kana_rule_list[ 1 ].state = ptr ;
  ptr = malloc( sizeof( struct myChar ) * 2 ) ;
  ptr->charset = CHARSET_JISX0208_1983 ;
  ptr->chara   = 0x2473 ;
  MYCHAR_SET_END_OF_STRING( *( ptr + 1 ) ) ;
  skkinput_default_rom_kana_rule_list[ 1 ].houtput = ptr ;
  ptr = malloc( sizeof( struct myChar ) * 2 ) ;
  ptr->charset = CHARSET_JISX0208_1983 ;
  ptr->chara   = 0x2573 ;
  MYCHAR_SET_END_OF_STRING( *( ptr + 1 ) ) ;
  skkinput_default_rom_kana_rule_list[ 1 ].koutput = ptr ;
  return ;
}

/*
 * skkinput ưҤ򤹤ѿؿ
 *------
 * ǥեȤͤƤޤؿǤ⤢롣
 */
void initSkkinputDousaketteiVariables( void )
{
  struct servent *servent ;

  /* ɽ꼭νߤΤʤȤƤ*/
  skkinput_local_jisyo_name  = DEFAULT_SKKLJISYO ;
  skkinput_backup_jisyo_name = DEFAULT_SKKBJISYO ;
  skkinput_record_name       = DEFAULT_SKKRECORD ;
  skk_local_jisyo_name	     = DEFAULT_SKKLOCALJISYO ;
  /* "/etc/services" ݡֹƤ*/
  servent = getservbyname( SKKSERV_SERVICE_NAME, SKKSERV_SERVICE_PROTO ) ;
  if( servent == NULL ){
    /* /etc/services ϤʤäΤǡconfig.h ΤȤ*/
    skkserv_service          = strdup(DEFAULT_SKKPORT);
  } else {
    /* /etc/services ꤬äġ*/
    skkserv_service          = strdup( servent->s_name ) ;
#if defined(DEBUG)
    fprintf
      ( stderr, "getservbyname: \"%s\", %d\n",
	servent->s_name, skkserv_portnum ) ;
#endif
  }
  /* Ķѿ SKKSERV ꤵƤС DEFAULT  skkserv *
   * host Ȥ롣*/
  if( ( skkserv_host = getenv( "SKKSERVER" ) ) == NULL )
    skkserv_host        = DEFAULT_SKKSERVER ;

  /* ǥեȤΥޥåפ롣*/
  skkinput_CompileDefaultKeyBind() ;
  /* ϥ٥ȥνԤ*/
  initialize_skkinputDefaultVectors() ;
  /* egg ߴ newline 򥵥ݡȤ뤫ݤ*/
  skk_egg_like_newline            = False ;
  /* dabbrev like  completion Ԥɤ*/
  skkinput_dabbrev_like_completion= False ;
  /* chat-adapter-mode Ĥޤ newline ƬǤʤ뤫*/
  skkinput_chatadaptermode        = False ;
  /* skk-local-jisyo 򸡺뤫ɤ*/
  skkinput_search_skk_jisyo       = True ;
  /* skkinput-record 뤫ɤ*/
  skkinput_keep_record            = True ;
  /* դɽ*/
  skkinput_date_ad                = False ;
  skkinput_number_style           = False ;
  /* ϳεǽޤǤΤݤ*/
  skkinput_delete_implies_kakutei = True ;
  /* ѴǽѤ롣*/
  skkinput_use_numeric_conversion = True ;
  skkinput_rom_kana_rule_list_top = NULL ;
  initialize_skkinputDefaultRomKanaRuleList() ;
  skkinput_rom_kana_rule_list     = skkinput_default_rom_kana_rule_list ;
  /* */
  skkinput_tab_width              = 8 ;
  skkinput_rjj_like_input         = False ;
  /* ȥ֤δ֤ꤹ롣ǥեȤ 10 äǤ롣*/
  skkinput_autosave_interval      = 10 * 1000 ;
  return ;
}

/*
 * Ѵץȥʤ󤫤˴طʤ SkkInputWidget 򵯤ˤäɬ
 * פ򤹤ؿ
 */
int skkinput_setSkkInputValues( Widget gw )
{
  Arg arg[ 5 ] ;
  Cardinal i ;
  i = 0 ;
  /* ξ(Modified ɤ)*/
  XtSetArg( arg[ i ], XtNjisyoDirty, skkinput_jisyo_dirty ) ;
  i ++ ;  
  /* egg-like-newline ɤ*/
  XtSetArg( arg[ i ], XtNeggLikeNewline, skk_egg_like_newline ) ;
  i ++ ;  
  /* chat adapter ⡼ɤɤ*/
  XtSetArg( arg[ i ], XtNchatAdapter, skkinput_chatadaptermode ) ;
  i ++ ;  
  XtSetValues( gw, arg, i ) ;
  return 0 ;
}
