/* === S Y N F I G ========================================================= */
/*!	\file canvasinterface.h
**	\brief Template Header
**
**	$Id: canvasinterface.h 1643 2008-02-09 19:12:18Z dooglus $
**
**	\legal
**	Copyright (c) 2002-2005 Robert B. Quattlebaum Jr., Adrian Bentley
**	Copyright (c) 2007, 2008 Chris Moore
**
**	This package is free software; you can redistribute it and/or
**	modify it under the terms of the GNU General Public License as
**	published by the Free Software Foundation; either version 2 of
**	the License, or (at your option) any later version.
**
**	This package is distributed in the hope that it will be useful,
**	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**	General Public License for more details.
**	\endlegal
*/
/* ========================================================================= */

/* === S T A R T =========================================================== */

#ifndef __SYNFIG_APP_CANVASINTERFACE_H
#define __SYNFIG_APP_CANVASINTERFACE_H

/* === H E A D E R S ======================================================= */

//#include <synfig/canvas.h>
#include <synfig/value.h>
#include <sigc++/signal.h>
#include <sigc++/object.h>
#include <list>
#include "selectionmanager.h"
#include "uimanager.h"
#include "value_desc.h"
#include "editmode.h"

/* === M A C R O S ========================================================= */

/* === T Y P E D E F S ===================================================== */

/* === C L A S S E S & S T R U C T S ======================================= */

namespace synfig { class ValueNode_DynamicList; class Waypoint; class GUIDSet; class Canvas; };

namespace synfigapp {

namespace Action { class ParamList; class Param; class EditModeSet; };

class Instance;
class ValueDesc;

class CanvasInterface : public etl::shared_object, public sigc::trackable
{
	friend class Instance;
	friend class Action::EditModeSet;

public:

	typedef EditMode Mode;

private:
	// Constructor is private to force the use of the "create()" constructor.
	CanvasInterface(etl::loose_handle<Instance> instance,etl::handle<synfig::Canvas> canvas);

private:
	etl::loose_handle<Instance> instance_;
	etl::handle<synfig::Canvas> canvas_;
	etl::handle<SelectionManager> selection_manager_;
	etl::handle<UIInterface> ui_interface_;
	synfig::Time cur_time_;
	Mode mode_;

	sigc::signal<void,synfig::Layer::Handle> signal_layer_raised_;
	sigc::signal<void,synfig::Layer::Handle> signal_layer_lowered_;
	sigc::signal<void,synfig::Layer::Handle,int> signal_layer_inserted_;
	sigc::signal<void,synfig::Layer::Handle,int,synfig::Canvas::Handle> signal_layer_moved_;
	sigc::signal<void,synfig::Layer::Handle> signal_layer_removed_;
	sigc::signal<void,synfig::Layer::Handle,bool> signal_layer_status_changed_;
	sigc::signal<void,synfig::Layer::Handle,synfig::String> signal_layer_new_description_;
	sigc::signal<void,synfig::Canvas::Handle> signal_canvas_added_;
	sigc::signal<void,synfig::Canvas::Handle> signal_canvas_removed_;

	sigc::signal<void,synfig::ValueNode::Handle> signal_value_node_added_;
	sigc::signal<void,synfig::ValueNode::Handle> signal_value_node_deleted_;
	sigc::signal<void,synfig::ValueNode::Handle,synfig::ValueNode::Handle> signal_value_node_replaced_;

	sigc::signal<void,synfig::Keyframe> signal_keyframe_added_;
	sigc::signal<void,synfig::Keyframe> signal_keyframe_removed_;
	sigc::signal<void,synfig::Keyframe> signal_keyframe_changed_;

	sigc::signal<void> signal_id_changed_;

	sigc::signal<void> signal_time_changed_;

	sigc::signal<void> signal_rend_desc_changed_;

	sigc::signal<void,Mode> signal_mode_changed_;

	//sigc::signal<void> signal_dirty_preview_;

	sigc::signal<void,synfig::Layer::Handle,synfig::String> signal_layer_param_changed_;

public:	// Signal Interface

	sigc::signal<void,synfig::Layer::Handle,int,synfig::Canvas::Handle>& signal_layer_moved() { return signal_layer_moved_; }

	sigc::signal<void,synfig::Layer::Handle,synfig::String>& signal_layer_new_description() { return signal_layer_new_description_; }

	//! Signal called when layer is raised.
	sigc::signal<void,synfig::Layer::Handle>& signal_layer_raised() { return signal_layer_raised_; }

	//! Signal called when layer is lowered.
	sigc::signal<void,synfig::Layer::Handle>& signal_layer_lowered() { return signal_layer_lowered_; }

	//! Signal called when layer has been inserted at a given position.
	sigc::signal<void,synfig::Layer::Handle,int>& signal_layer_inserted() { return signal_layer_inserted_; }

	//! Signal called when a layer has been removed from the canvas.
	sigc::signal<void,synfig::Layer::Handle>& signal_layer_removed() { return signal_layer_removed_; }

	//! Signal called when the layer's active status has changed.
	sigc::signal<void,synfig::Layer::Handle,bool>& signal_layer_status_changed() { return signal_layer_status_changed_; }

	//! Signal called when a canvas has been added.
	sigc::signal<void,etl::handle<synfig::Canvas> >& signal_canvas_added() { return signal_canvas_added_; }

	//! Signal called when a canvas has been removed.
	sigc::signal<void,etl::handle<synfig::Canvas> >& signal_canvas_removed() { return signal_canvas_removed_; }

	//! Signal called when a layer's parameter has been changed
	sigc::signal<void,synfig::Layer::Handle,synfig::String>& signal_layer_param_changed() { return signal_layer_param_changed_; }

	//! Signal called when the canvas's preview needs to be updated
	//sigc::signal<void>& signal_dirty_preview() { return signal_dirty_preview_; }
	sigc::signal<void>& signal_dirty_preview() { return get_canvas()->signal_dirty(); }

	sigc::signal<void,etl::handle<synfig::ValueNode>,etl::handle<synfig::ValueNode> >&
		signal_value_node_child_added() { return get_canvas()->signal_value_node_child_added(); }
	sigc::signal<void,etl::handle<synfig::ValueNode>,etl::handle<synfig::ValueNode> >&
		signal_value_node_child_removed() { return get_canvas()->signal_value_node_child_removed(); }

	//! Signal called when a ValueNode has changed
	sigc::signal<void,etl::handle<synfig::ValueNode> >& signal_value_node_added() { return signal_value_node_added_; }

	//! Signal called when a ValueNode has been deleted
	sigc::signal<void,etl::handle<synfig::ValueNode> >& signal_value_node_deleted() { return signal_value_node_deleted_; }

	//! Signal called when a ValueNode has been changed
	sigc::signal<void,etl::handle<synfig::ValueNode> >& signal_value_node_changed() { return get_canvas()->signal_value_node_changed(); }

	//! Signal called when a ValueNode has been renamed
	sigc::signal<void,etl::handle<synfig::ValueNode> >& signal_value_node_renamed() { return get_canvas()->signal_value_node_renamed(); }

	//! Signal called when the mode has changed
	sigc::signal<void,Mode> signal_mode_changed() { return signal_mode_changed_; }

	//! Signal called when a the ID has been changed
	sigc::signal<void>& signal_id_changed() { return signal_id_changed_; }

	//! Signal called whenever the time changes
	sigc::signal<void> signal_time_changed() { return signal_time_changed_; }

	//! Signal called whenever a data node has been replaced.
	/*!	Second ValueNode replaces first */
	sigc::signal<void,synfig::ValueNode::Handle,synfig::ValueNode::Handle>& signal_value_node_replaced()
		{ return signal_value_node_replaced_; }

	//! Signal called whenever the RendDesc changes
	sigc::signal<void>& signal_rend_desc_changed() { return signal_rend_desc_changed_; }

	sigc::signal<void,synfig::Keyframe>& signal_keyframe_added() { return signal_keyframe_added_; }

	sigc::signal<void,synfig::Keyframe>& signal_keyframe_removed() { return signal_keyframe_removed_; }

	sigc::signal<void,synfig::Keyframe>& signal_keyframe_changed() { return signal_keyframe_changed_; }

public:

	void auto_export(const ValueDesc& value_desc);

	void auto_export(synfig::ValueNode::Handle value_node);

	void set_meta_data(const synfig::String& key,const synfig::String& data);

	//! Changes the current SelectionManager object
	void set_selection_manager(const etl::handle<SelectionManager> &sm) { selection_manager_=sm; }

	//! Disables the selection manager
	void unset_selection_manager() { selection_manager_=new NullSelectionManager(); }

	//! Returns a handle to the current SelectionManager
	const etl::handle<SelectionManager> &get_selection_manager()const { return selection_manager_; }

	//! Changes the current UIInterface object
	void set_ui_interface(const etl::handle<UIInterface> &uim) { ui_interface_=uim; }

	//! Disables the UIInterface
	void unset_ui_interface() { ui_interface_=new DefaultUIInterface(); }

	//! Returns a handle to the current UIInterface
	const etl::handle<UIInterface> &get_ui_interface() { return ui_interface_; }

	//! Returns the Canvas associated with this interface
	etl::handle<synfig::Canvas> get_canvas()const { return canvas_; }

	//! Returns the Instance associated with this interface
	etl::loose_handle<Instance> get_instance()const { return instance_; }

	//! Changes the name of the canvas. Undoable.
	bool set_name(const synfig::String &x);

	//! Changes the description of the canvas. Undoable.
	bool set_description(const synfig::String &x);

	//! Changes the ID of the canvas. Undoable.
	bool set_id(const synfig::String &x);

	//! Convenience function to retrieve the name of the canvas
	synfig::String get_name()const { return get_canvas()->get_name(); }

	//! Convenience function to retrieve the description of the canvas
	synfig::String get_description()const { return get_canvas()->get_description(); }

	//! Convenience function to retrieve the ID of the canvas
	synfig::String get_id()const { return get_canvas()->get_id(); }

	//! Sets the current time
	void set_time(synfig::Time x);

	//! Retrieves the current time
	synfig::Time get_time()const;

	//! Changes the current time to the next keyframe
	void jump_to_next_keyframe();

	//! Changes the current time to the next keyframe
	void jump_to_prev_keyframe();

	void seek_frame(int frames);

	void seek_time(synfig::Time time);

	//! \writeme
	void refresh_current_values();

	//! Sets the current editing mode
	/*! \see Mode */
	void set_mode(Mode x);

	//! Retrieves the current editing mode
	/*! \see Mode */
	Mode get_mode()const;

	//! Creates a new layer, of type \c id at the top of the layer stack
	synfig::Layer::Handle add_layer(synfig::String id) { return add_layer_to(id,get_canvas()); }

	synfig::Layer::Handle add_layer_to(synfig::String id,synfig::Canvas::Handle canvas, int depth=0);

	bool convert(ValueDesc value_desc, synfig::String type);
	//! Adds the given ValueNode to the canvas.
	bool add_value_node(synfig::ValueNode::Handle value_node, synfig::String name);


	Action::ParamList generate_param_list(const synfigapp::ValueDesc &);

	Action::ParamList generate_param_list(const std::list<synfigapp::ValueDesc> &);

	void set_rend_desc(const synfig::RendDesc &rend_desc);

	bool import(const synfig::String &filename, bool copy=false);


	void waypoint_duplicate(synfigapp::ValueDesc value_desc,synfig::Waypoint waypoint);
	void waypoint_duplicate(synfig::ValueNode::Handle value_node,synfig::Waypoint waypoint);

	void waypoint_remove(synfigapp::ValueDesc value_desc,synfig::Waypoint waypoint);
	void waypoint_remove(synfig::ValueNode::Handle value_node,synfig::Waypoint waypoint);

	bool change_value(synfigapp::ValueDesc value_desc,synfig::ValueBase new_value);


	int find_important_value_descs(std::vector<synfigapp::ValueDesc>& out);
	static int find_important_value_descs(synfig::Canvas::Handle canvas,std::vector<synfigapp::ValueDesc>& out,synfig::GUIDSet& guid_set);

	~CanvasInterface();

	static etl::handle<CanvasInterface> create(etl::loose_handle<Instance> instance,etl::handle<synfig::Canvas> canvas);
}; // END of class CanvasInterface

/*!	\class PushMode
**	\brief Class that changes the mode of a CanvasInterface, and restores it on destruction.
*/
class PushMode
{
	CanvasInterface* canvas_interface_;
	CanvasInterface::Mode old_mode_;
public:
	PushMode(etl::loose_handle<CanvasInterface> c, CanvasInterface::Mode mode):
		canvas_interface_(c.get()), old_mode_(canvas_interface_->get_mode())
	{ canvas_interface_->set_mode(mode); }

	~PushMode() { canvas_interface_->set_mode(old_mode_); }
}; // END of class PushMode

}; // END of namespace studio

/* === E N D =============================================================== */

#endif
