/*  $Id: xfcesmart.c 3315 2007-10-08 08:53:04Z afb $
 *
 *  xfce4-smartpm-plugin - a package manager applet for the xfce4 panel
 *  Copyright (c) 2007 Anders F Bjorklund <afb@users.sourceforge.net>
 *
 *  Based on: ksmarttray by Gustavo Niemeyer <niemeyer@conectiva.com>
 *            mailwatch by Brian Tarricone <bjt23@cornell.edu>,
 *                         Jasper Huijsmans <jasper@xfce.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>

#include <libxfcegui4/libxfcegui4.h>
#include <libxfce4util/libxfce4util.h>
#include <libxfce4panel/xfce-panel-plugin.h>

#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#else
#define WIFEXITED(status) TRUE
#define WEXITSTATUS(status) status
#endif

#include "xfcesmart.h"

#define SMART_PATH    "/usr/local/bin:/usr/bin:/bin:/opt/gnome/bin"
#define SMART_LANG    "C" /* we're looking for certain phrases... */

enum
{
  PACKAGE_MANAGER_SMART	= 0, /* use Smart Package Manager */
  PACKAGE_MANAGER_YUM	= 1, /* use Yum+Extender (Fedora) */
  PACKAGE_MANAGER_APT	= 2, /* use APT+Synaptic (Ubuntu) */
};

#define SMART_WEBSITE "http://smartpm.org/"
#define SMART_PROGRAM "smart" /* regular bin */
#define SMART_UPDATE  "smart-update" /* suid */
#define SMART_SUIDBIN "smart-root" /* Fedora */
#define SMART_SUDOBIN "gksudo" /* Ubuntu/etc */

#define YUM_WEBSITE   "http://www.yum-extender.org/"
#define YUM_PROGRAM   "yumex"
#define YUM_UPDATE    "yum"

#define APT_WEBSITE   "http://www.nongnu.org/synaptic/"
#define APT_PROGRAM   "synaptic"
#define APT_UPDATE    "apt-get"

#define TIMEOUT_TIME  1000 /* milliseconds */

#define PROGRESS_TIME 5*60 /* seconds */

#define UNUSED(x)     (void) x

/* prototypes */
static void
smart_construct (XfcePanelPlugin *plugin);

static void
smart_set_available(SmartPlugin *smart, gint packages);

static int
smart_update_channels( SmartPlugin *smart);

static void
smart_set_interval(SmartPlugin *smart, gint interval);

static int
smart_check_upgrades( SmartPlugin *smart);

static void
smart_set_manager(SmartPlugin *smart, gint manager);

static void
smart_sudo_smart_gui(SmartPlugin *smart, gchar *args);

static gboolean
smart_size_changed (XfcePanelPlugin *plugin,
                     gint             size,
                     SmartPlugin    *smart);

/* register the plugin */
XFCE_PANEL_PLUGIN_REGISTER_EXTERNAL (smart_construct);


static void
smart_set_icon(SmartPlugin *smart, GdkPixbuf *pix, GdkPixbuf *overlay)
{
    GdkPixbuf       *pb = gdk_pixbuf_copy(pix);
    gint w, h, ow, oh;

    w = gdk_pixbuf_get_width(pb);
    h = gdk_pixbuf_get_height(pb);
    if(overlay) {
        ow = gdk_pixbuf_get_width(overlay);
        oh = gdk_pixbuf_get_height(overlay);
        gdk_pixbuf_composite(overlay, pb, w - ow, h - oh, ow, oh, w - ow, h - oh,
                             1.0, 1.0, GDK_INTERP_BILINEAR, 255);
    }
    
    xfce_scaled_image_set_from_pixbuf(XFCE_SCALED_IMAGE(smart->image), pb);
    g_object_unref(G_OBJECT(pb));
}

static void
smart_set_available(SmartPlugin *smart, gint packages)
{
  if (packages == 0)
  {
    char *s = _("No upgrades");
    gtk_tooltips_set_tip(smart->tooltip, smart->button, s, NULL);
    if (smart->show_application)
      smart_set_icon(smart, smart->application_icon, smart->updated_icon);
    else
      smart_set_icon(smart, smart->nopackage_icon, NULL);
  }
  else if (packages > 0)
  {
    gchar *s = g_strdup_printf((packages==1)?_("%d upgrade available"):
                               _("%d upgrades available"), packages);
    gtk_tooltips_set_tip(smart->tooltip, smart->button, s, NULL);
    g_free(s);
    if (smart->show_application)
      smart_set_icon(smart, smart->application_icon, smart->available_icon);
    else
      smart_set_icon(smart, smart->newpackage_icon, NULL);
  }
  else /* -1 == unknown */
  {
    if (smart->show_application)
      smart_set_icon(smart, smart->application_icon, NULL);
    else
      smart_set_icon(smart, smart->nopackage_icon, NULL);
  }
  smart->updates_available = packages;
}

static void
smart_update_channels_watch_func(GPid pid, gint status, gpointer user_data)
{
  SmartPlugin *smart = user_data;
  char buf[1024], *p;
  int packages;
  FILE *file;
  gchar *msg;
  gchar *s;
  
  packages = -1;
  
  file = fdopen(smart->stdout, "r");
  if (file != NULL)
  {
    while ( (p = fgets(buf, sizeof(buf), file)) != NULL )
    {
      switch (smart->package_manager)
      {
        case PACKAGE_MANAGER_SMART:
          /* "Channels have no new packages." */
          if (strstr(p, "Channels have ") != NULL)
          {
            p += strlen("Channels have ");
            if (p[0] == 'n' && p[1] == 'o')
              packages = 0;
            else
              packages = atoi(p);
          }
          break;
        case PACKAGE_MANAGER_YUM:
          /* TODO: determine number of new packages in Yum channels */
          break;
        case PACKAGE_MANAGER_APT:
          /* TODO: determine number of new packages in APT channels */
          break;
      }
    }
    fclose(file);
  } 
 
  msg = NULL;
 
  file = fdopen(smart->stderr, "r");
  if (file != NULL)
  {
    while ( (p = fgets(buf, sizeof(buf), file)) != NULL )
    {
      if (strstr(p, "error: ") != NULL)
      {
        p += strlen("error: ");
        msg = g_strdup(p);
        p = msg + strlen(msg) - 1;
        if (*p == '\n') *p = '\0';
      }
    }  
    fclose(file);
  }
  
  if (WIFEXITED(status) && WEXITSTATUS(status) == 0)
  {
    /* restore icon */
    smart_set_available(smart, smart->updates_available);

    /* NOTE: "packages" is the number of new packages in the channels */
    if (packages == 0)
      s = g_strdup(_("No new packages"));
    else if (packages > 0)
      s = g_strdup_printf(_("Channels have %d new packages"), packages);
    else /* unknown */
      s = g_strdup(_("Channels updated"));

    gtk_tooltips_set_tip(smart->tooltip, smart->button, s, NULL);
    g_free(s);
  }
  else
  {
    s = g_strdup_printf(_("Error updating%s%s"), msg ? ": " : "", msg ? msg : "");
    gtk_tooltips_set_tip(smart->tooltip, smart->button, s, NULL);
    if (smart->show_application)
      smart_set_icon(smart, smart->application_icon, smart->error_icon);
    else
      smart_set_icon(smart, smart->nopackage_icon, smart->error_icon);
    g_free(s);
  }

  if (msg)
  {
    if (strcmp(msg, "Configuration is in readonly mode.") == 0)
    {
        xfce_message_dialog(NULL, _("Smart Read-Only"), GTK_STOCK_DIALOG_WARNING,
                            _("Another process seems to be using Smart ?"), msg,
                            GTK_STOCK_OK, GTK_RESPONSE_OK, NULL);
    }
    g_free(msg);
  }
  
/*close(smart->stdout); closed above*/
/*close(smart->stderr); closed above*/
  g_spawn_close_pid(pid);

  smart->in_progress = FALSE;
}

static void
smart_update_channels_setup_child( gpointer   user_data)
{
  SmartPlugin *smart = user_data;
  /* nothing to do */
  UNUSED(smart);
}

static int
smart_update_channels( SmartPlugin    *smart)
{
  gchar *argv[] = { NULL, NULL, NULL, NULL };
  gchar *envp[] = { "PATH=" SMART_PATH, "LANG=" SMART_LANG, NULL };
  gchar *title, *command, *args;
  gboolean terminal = FALSE;
  GError *error = NULL;
  GPid pid;

  gtk_tooltips_set_tip(smart->tooltip, smart->button, _("Updating channels..."), NULL);
  if (smart->show_application)
    smart_set_icon(smart, smart->application_icon, smart->downloading_icon);
  else if (smart->updates_available <= 0)
    smart_set_icon(smart, smart->nopackage_icon, smart->downloading_icon);
  else
    smart_set_icon(smart, smart->newpackage_icon, smart->downloading_icon);
   
  switch (smart->package_manager)
  {
    case PACKAGE_MANAGER_SMART:
      argv[0] = SMART_UPDATE;
      /* TODO: channels */
      terminal = FALSE;
      break;
    case PACKAGE_MANAGER_YUM:
      argv[0] = PATH_SU;
      argv[1] = "-c";
      argv[2] = "\"" YUM_UPDATE " "\
                "check-update" "\""; /* yum doesn't have a method to just update */
      terminal = TRUE;
      break;
    case PACKAGE_MANAGER_APT:
      argv[0] = PATH_SUDO;
      argv[1] = APT_UPDATE;
      argv[2] = "update";
      terminal = TRUE;
      break;
    default:
      return 0;
  }

  smart->in_progress = TRUE;
  g_get_current_time(&smart->in_progress_since);

  if (terminal)
  {
    gchar *out, *err, *s;
  
    if (smart->package_manager == PACKAGE_MANAGER_YUM) title = YUM_UPDATE;
    if (smart->package_manager == PACKAGE_MANAGER_APT) title = APT_UPDATE;
  
    args = g_strdup_printf("%s %s %s",
                 argv[0], argv[1], argv[2]);
    command = g_strdup_printf("xterm -T \"%s (%s)\" -e /bin/sh -c \'%s\'",
                 _("Update Channels"), (title ? title : ""), args);

    if (!g_spawn_command_line_sync( command, &out, &err, NULL, &error))
    {
      xfce_message_dialog(NULL, title, GTK_STOCK_DIALOG_ERROR,
                          _("Failed to run update"), error->message,
                          GTK_STOCK_CLOSE, GTK_RESPONSE_ACCEPT, NULL);

      s = g_strdup_printf(_("Error updating%s%s"), err ? ": " : "", err ? err : "");
      smart_set_available(smart, -1);
      gtk_tooltips_set_tip(smart->tooltip, smart->button, s, NULL);
      if (smart->show_application)
        smart_set_icon(smart, smart->application_icon, smart->error_icon);
      else
        smart_set_icon(smart, smart->nopackage_icon, smart->error_icon);
      g_free(s);
    }
    else
    {
      smart_set_available(smart, -1);
      gtk_tooltips_set_tip(smart->tooltip, smart->button, _("Channels updated."), NULL);
    }

    g_free(args);
    g_free(command);

    g_error_free(error);
    smart->in_progress = FALSE;
    return 0;
  }

  if (!g_spawn_async_with_pipes( NULL /*cwd*/, argv, envp,
    G_SPAWN_SEARCH_PATH | G_SPAWN_DO_NOT_REAP_CHILD,
    &smart_update_channels_setup_child, smart,
    &pid, NULL /*stdin*/, &smart->stdout, &smart->stderr, &error))
  {
    if (smart->package_manager == PACKAGE_MANAGER_SMART && \
        g_error_matches(error, G_SPAWN_ERROR, G_SPAWN_ERROR_NOENT))
    {
      /* smart-update not available */
      smart_sudo_smart_gui(smart, "update");
    }
    else
    {
      xfce_message_dialog(NULL, _("Smart Unavailable"), GTK_STOCK_DIALOG_ERROR,
                          _("Failed to run update"), error->message,
                          GTK_STOCK_CLOSE, GTK_RESPONSE_ACCEPT, NULL);
    }

    g_error_free(error);
    smart->in_progress = FALSE;
    smart_set_available(smart, smart->updates_available);
    return 1;
  }
  else
  {
    g_child_watch_add(pid, smart_update_channels_watch_func, smart);
    return 0;
  }
 }

static gboolean
smart_timeout_function (gpointer user_data)
{
  SmartPlugin *smart = user_data;
  gint seconds;

  seconds = (gint) g_timer_elapsed(smart->timer, NULL);
  if (smart->in_progress || seconds < smart->check_interval * 60)
  {
    return TRUE;
  }
  
  smart_check_upgrades(smart);
  smart_set_interval(smart, smart->check_interval);
  
  return FALSE;
}

static void
smart_set_interval(SmartPlugin *smart, gint interval)
{
  if (smart->timeout)
    g_source_remove(smart->timeout);

  if (interval > 0)
  {
    g_timer_start(smart->timer);
    smart->timeout = g_timeout_add(TIMEOUT_TIME, smart_timeout_function, smart);
  }
  else
  {
    g_timer_stop(smart->timer);
    smart->timeout = 0;
  }

  smart->check_interval = interval;
}

static void
smart_check_upgrades_watch_func(GPid pid, gint status, gpointer user_data)
{
  SmartPlugin *smart = user_data;
  gboolean seen_header;
  char buf[1024], *p;
  int packages;
  FILE *file;
  gchar *msg;
  gchar *s;
  
  seen_header = FALSE;
  packages = 0;
  
  file = fdopen(smart->stdout, "r");
  if (file != NULL)
  {
    while ( (p = fgets(buf, sizeof(buf), file)) != NULL )
    {
      if (*p != '\n')
      {
        switch (smart->package_manager)
        {
          case PACKAGE_MANAGER_SMART:
            break;
          case PACKAGE_MANAGER_YUM:
            /* TODO: read package names */
            if (seen_header)
              packages++;
            break;
          case PACKAGE_MANAGER_APT:
            /* 0 upgraded, 0 newly installed, 0 to remove and 0 not upgraded. */
            if (strstr(p, " upgraded,") != NULL)
              packages = atoi(p);
            if (strstr(p, "Inst ") != NULL)
            {
              p += strlen("Inst ");
              /* TODO: read package names */
            }
            break;
        }
      }
      else
        seen_header = TRUE;
    }
    fclose(file);
  }

  msg = NULL;
  
  file = fdopen(smart->stderr, "r");
  if (file != NULL)
  {
    while ( (p = fgets(buf, sizeof(buf), file)) != NULL )
    {
      if (smart->package_manager == PACKAGE_MANAGER_APT &&
          strstr(p, "Failed to write temporary StateFile") != NULL)
      {
        /* skip this error, we don't want to write anything anyway */
        status = 0;
        continue;
      }
      if (smart->package_manager != PACKAGE_MANAGER_SMART ||
          strstr(p, "error: ") != NULL)
      {
        if (strstr(p, "error: ") == 0)
          p += strlen("error: ");
        if (strstr(p, "E: ") == 0)
          p += strlen("E: ");
        msg = g_strdup(p);
        p = msg + strlen(msg) - 1;
        if (*p == '\n') *p = '\0';
      }
      else /* PACKAGE_MANAGER_SMART */
      {
        /* TODO: read package names */
        if (*p != '\n') packages++;
      }
    }
    fclose(file);
  } 
 
  if (WIFEXITED(status) && (WEXITSTATUS(status) == 0 ||
      (smart->package_manager == PACKAGE_MANAGER_YUM &&
      WEXITSTATUS(status) == 100 /* packages available */)))
  {
    smart_set_available(smart, packages);
  }
  else
  {
    s = g_strdup_printf(_("Error checking%s%s"), msg ? ": " : "", msg ? msg : "");
    gtk_tooltips_set_tip(smart->tooltip, smart->button, s, NULL);
    if (smart->show_application)
      smart_set_icon(smart, smart->application_icon, smart->error_icon);
    else if (smart->updates_available <= 0)
      smart_set_icon(smart, smart->nopackage_icon, smart->error_icon);
    else
      smart_set_icon(smart, smart->newpackage_icon, smart->error_icon);
    g_free(s);
  }
 
  if (msg)
  {
    g_free(msg);
  }

/*close(smart->stdout); closed above */
/*close(smart->stderr); closed above */
  g_spawn_close_pid(pid);

  smart->in_progress = FALSE;
}

static void
smart_check_upgrades_setup_child( gpointer   user_data)
{
  SmartPlugin    *smart = user_data;
  /* nothing to do */
  UNUSED(smart);
}

static int
smart_check_upgrades( SmartPlugin    *smart)
{
  gchar *argv[] = { NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL };
  gchar *envp[] = { "PATH=" SMART_PATH, "LANG=" SMART_LANG, NULL };
  gchar *tempfile = NULL, *pkgcache = NULL, *srcpkgcache = NULL;
  GError *error = NULL;
  GPid pid;
  
  gtk_tooltips_set_tip(smart->tooltip, smart->button, _("Checking for upgrades..."), NULL);

  if (smart->show_application)
    smart_set_icon(smart, smart->application_icon, smart->downloading_icon);
  else if (smart->updates_available <= 0)
    smart_set_icon(smart, smart->nopackage_icon, smart->downloading_icon);
  else
    smart_set_icon(smart, smart->newpackage_icon, smart->downloading_icon);
  
  switch (smart->package_manager)
  {
    case PACKAGE_MANAGER_SMART:
      argv[0] = SMART_PROGRAM;
      argv[1] = "upgrade";
      argv[2] = "--dump";
      break;
    case PACKAGE_MANAGER_YUM:
      argv[0] = YUM_UPDATE;
      argv[1] = "-C"; /* don't update cache, we do that when running as root */
      argv[2] = "check-update";
      break;
    case PACKAGE_MANAGER_APT:
      tempfile = g_strdup_printf("%s%s%s-apt", g_get_tmp_dir(), G_DIR_SEPARATOR_S, g_get_user_name());
      argv[0] = APT_UPDATE;
      argv[1] = "--simulate";
      argv[2] = (smart->dist_upgrade) ? "dist-upgrade" : "upgrade";
      argv[3] = "--option"; /* regular cache files is only writable by root */
      argv[4] = pkgcache = g_strdup_printf("Dir::Cache::pkgcache=%spkg.cache", tempfile);
      argv[5] = "--option"; /* regular cache files is only writable by root */
      argv[6] = srcpkgcache = g_strdup_printf("Dir::Cache::srcpkgcache=%ssrc.cache", tempfile);
      argv[7] = "--option"; /* only root is able to get a lock on the cache */
      argv[8] = "Debug::nolocking=true";
      break;
    default:
      return 0;
  }

  smart->in_progress = TRUE;
  g_get_current_time(&smart->in_progress_since);

  if (!g_spawn_async_with_pipes( NULL /*cwd*/, argv, envp,
    G_SPAWN_SEARCH_PATH | G_SPAWN_DO_NOT_REAP_CHILD | G_SPAWN_LEAVE_DESCRIPTORS_OPEN,
    &smart_check_upgrades_setup_child, smart,
    &pid, NULL /*stdin*/, &smart->stdout, &smart->stderr, &error))
  {
        xfce_message_dialog(NULL, _("Smart Unavailable"), GTK_STOCK_DIALOG_ERROR,
                            _("Failed to run upgrade"), error->message,
                            GTK_STOCK_CLOSE, GTK_RESPONSE_ACCEPT, NULL);
        g_error_free(error);
        smart->in_progress = FALSE;
        smart_set_available(smart, smart->updates_available);
        if (tempfile) g_free(tempfile);
        if (pkgcache) g_free(pkgcache);
        if (srcpkgcache) g_free(srcpkgcache);
        return 1;
  }
  else
  {
    g_child_watch_add(pid, smart_check_upgrades_watch_func, smart);
    if (tempfile) g_free(tempfile);
    if (pkgcache) g_free(pkgcache);
    if (srcpkgcache) g_free(srcpkgcache);
    return 0;
  }
}

static void
smart_set_manager(SmartPlugin *smart, gint manager)
{
  if (manager != smart->package_manager)
   smart->updates_available = -1;

  smart->package_manager = manager;
}

static gboolean
smart_program_exists(gchar *program)
{
  gchar *path = "/usr/bin"; /* TODO: check PATH */
  gchar *bin;
  gboolean ok;
  
  bin = g_strdup_printf("%s/%s", path, program);
  ok = g_file_test(bin, G_FILE_TEST_EXISTS);
  g_free(bin);

  return ok;
}

static void
smart_sudo_smart_gui(SmartPlugin *smart, gchar *args)
{
  GError *error = NULL;
  gchar *program;
  gchar *command;
  
  switch (smart->package_manager)
  {
    case PACKAGE_MANAGER_SMART:
      if (smart_program_exists(SMART_SUIDBIN))
        program = g_strdup(SMART_SUIDBIN);
      else /* if (smart_program_exists(SMART_SUDOBIN)) */
        program = g_strdup_printf("%s -- %s", SMART_SUDOBIN, SMART_PROGRAM);
      /*else
        program = g_strdup(SMART_PROGRAM);*/
      break;
    case PACKAGE_MANAGER_YUM:
      if (smart_program_exists(SMART_SUDOBIN))
        program = g_strdup_printf("%s %s", SMART_SUDOBIN, YUM_PROGRAM);
      else
        program = g_strdup(YUM_PROGRAM); /* it is using consolehelper */
      break;
    case PACKAGE_MANAGER_APT:
      if (smart_program_exists(SMART_SUDOBIN))
        program = g_strdup_printf("%s %s", SMART_SUDOBIN, APT_PROGRAM);
      else
        program = g_strdup(APT_PROGRAM); /* it is using consolehelper */
      break;
    default:
      return;
  }

  command = g_strdup_printf("%s%s%s%s", program,
            (smart->package_manager == PACKAGE_MANAGER_SMART) ? " --gui" : "",
            args ? " " : "", args ? args : "");
  if (!xfce_exec(command, FALSE, FALSE, &error))
  {
    xfce_message_dialog(NULL, _("Smart Failed"), GTK_STOCK_DIALOG_ERROR,
                        _("Failed to run smart program"), error->message,
                        GTK_STOCK_CLOSE, GTK_RESPONSE_ACCEPT, NULL);
    g_error_free(error);
  }

  g_free(program);
  g_free(command);
}

void
smart_save (XfcePanelPlugin *plugin,
             SmartPlugin    *smart)
{
  XfceRc *rc;
  gchar  *file;

  /* get the config file location */
  file = xfce_panel_plugin_save_location (plugin, TRUE);

  if (G_UNLIKELY (file == NULL))
    {
       DBG ("Failed to open config file");
       return;
    }

  /* open the config file, read/write */
  rc = xfce_rc_simple_open (file, FALSE);
  g_free (file);

  if (G_LIKELY (rc != NULL))
    {
      /* save the settings */
      xfce_rc_write_int_entry  (rc, "interval", smart->check_interval);
      xfce_rc_write_bool_entry (rc, "upgrade", smart->upgrade_directly);
      xfce_rc_write_bool_entry (rc, "dist-upgrade", smart->dist_upgrade);
      xfce_rc_write_int_entry (rc, "manager", smart->package_manager);
      xfce_rc_write_bool_entry (rc, "appicon", smart->show_application);

      /* close the rc file */
      xfce_rc_close (rc);
    }
}



static void
smart_read (SmartPlugin *smart)
{
  XfceRc      *rc;
  gchar       *file;
  
  /* get the plugin config file location */
  file = xfce_panel_plugin_save_location (smart->plugin, TRUE);

  if (G_LIKELY (file != NULL))
    {
      /* open the config file, readonly */
      rc = xfce_rc_simple_open (file, TRUE);

      /* cleanup */
      g_free (file);

      if (G_LIKELY (rc != NULL))
        {
          /* read the settings */
          smart->check_interval = xfce_rc_read_int_entry (rc, "interval", 0);
          smart->upgrade_directly =  xfce_rc_read_bool_entry (rc, "upgrade", TRUE);
          smart->dist_upgrade =  xfce_rc_read_bool_entry (rc, "dist-upgrade", TRUE);
          smart->package_manager = xfce_rc_read_int_entry (rc, "manager", PACKAGE_MANAGER_SMART);
          smart->show_application =  xfce_rc_read_bool_entry (rc, "appicon", TRUE);

          /* cleanup */
          xfce_rc_close (rc);

          /* leave the function, everything went well */
          return;
        }
    }

  /* something went wrong, apply default values */
  DBG ("Applying default settings");

  smart->check_interval = 0;
  smart->upgrade_directly = TRUE;
  smart->dist_upgrade = TRUE;
  smart->package_manager = PACKAGE_MANAGER_SMART;
  smart->show_application = TRUE;
}



static gboolean
smart_button_press_cb(GtkWidget *w,
                          GdkEventButton *evt,
                          gpointer user_data)
{
    if(evt->button == 2)
        gtk_button_pressed(GTK_BUTTON(w));
    
    return FALSE;
}

static gboolean
smart_button_release_cb(GtkWidget *w, GdkEventButton *evt,
        gpointer user_data)
{
  SmartPlugin *smart = user_data;
    
  if(evt->x >= w->allocation.x
     && evt->x < w->allocation.x + w->allocation.width
     && evt->y >= w->allocation.y
     && evt->y < w->allocation.y + w->allocation.height)
  {
    switch(evt->button)
    {
      case 1:  /* left */
        if (smart->upgrade_directly && smart->updates_available > 0)
          smart_sudo_smart_gui(smart, "upgrade");
        else
          smart_sudo_smart_gui(smart, NULL);
        break;
      case 2:  /* middle */
        if (smart->in_progress)
        {
           GTimeVal now;

           g_get_current_time(&now);
           if (now.tv_sec - smart->in_progress_since.tv_sec > PROGRESS_TIME)
           {
             smart->in_progress = FALSE;
             smart_check_upgrades(smart);
           }
           else
           xfce_message_dialog(NULL, _("Upgrade In Progress"), GTK_STOCK_DIALOG_WARNING,
                               _("An upgrade is already in progress, please wait."), NULL,
                               GTK_STOCK_OK, GTK_RESPONSE_OK, NULL);
        }
        else
        {
           smart_check_upgrades(smart);
        }
        break;
    }
  }
  
  if(evt->button == 2)
    gtk_button_released(GTK_BUTTON(w));
    
  return FALSE;
}

static void
smart_menuitem_activate_cb(GtkMenuItem *item,
        gpointer user_data)
{
  SmartPlugin *smart = user_data;

  smart_update_channels(smart);
}

static void
smart_combobox_changed_cb(GtkComboBox *combo,
        gpointer user_data)
{
  SmartPlugin *smart = user_data;
  GtkWidget *button;
  gint manager;

  manager = gtk_combo_box_get_active(combo);

  smart_set_manager (smart, manager);

  button = g_object_get_data (G_OBJECT (smart->plugin), "dist-upgrade");

  if (button != NULL)
  gtk_widget_set_sensitive (button, smart->package_manager == PACKAGE_MANAGER_APT);

  if (smart->show_application) /* change the application icon too */
  smart_size_changed (smart->plugin, smart->current_icon_size, smart);
}

static void
smart_checkbox1_toggle_cb(GtkToggleButton *button,
        gpointer user_data)
{
  SmartPlugin *smart = user_data;

  smart->show_application = gtk_toggle_button_get_active(button);

  smart_size_changed (smart->plugin, smart->current_icon_size, smart);
}

static void
smart_spin_value_change_cb(GtkSpinButton *button,
        gpointer user_data)
{
  SmartPlugin *smart = user_data;
  gint interval;
    
  interval = gtk_spin_button_get_value_as_int(button);
    
  if (smart->check_interval == 0 && interval > 0)
  {
    /* check immediately when first enabled */
    smart_check_upgrades(smart);
  }

  smart_set_interval (smart, interval);
}

static void
smart_checkbox2_toggle_cb(GtkToggleButton *button,
        gpointer user_data)
{
  SmartPlugin *smart = user_data;

  smart->upgrade_directly = gtk_toggle_button_get_active(button);
}

static void
smart_checkbox3_toggle_cb(GtkToggleButton *button,
        gpointer user_data)
{
  SmartPlugin *smart = user_data;

  smart->dist_upgrade = gtk_toggle_button_get_active(button);
}

static SmartPlugin *
smart_new (XfcePanelPlugin *plugin)
{
  SmartPlugin   *smart;
  GtkOrientation  orientation;
  GtkWidget *dummy;
  GtkWidget *menuicon;
  
  /* allocate memory for the plugin structure */
  smart = panel_slice_new0 (SmartPlugin);

  /* pointer to plugin */
  smart->plugin = plugin;

  smart->in_progress = FALSE;
  smart->updates_available = -1;

  smart->timer = g_timer_new();
  smart->timeout = 0;
 
  /* read the user settings */
  smart_read (smart);

  /* get the current orientation */
  orientation = xfce_panel_plugin_get_orientation (plugin);

  dummy = gtk_invisible_new();
  gtk_widget_realize(dummy);
    
  smart->application_icon = xfce_themed_icon_load("xfce4-smart", 48);
  smart->nopackage_icon = xfce_themed_icon_load("xfce-nopackage", 48);
  smart->newpackage_icon = xfce_themed_icon_load("xfce-newpackage", 48);
  smart->current_icon_size = 48;

  smart->available_icon = xfce_themed_icon_load("xfce4-smart-available", 16);
  smart->downloading_icon = xfce_themed_icon_load("xfce4-smart-downloading", 16);
  smart->updated_icon = xfce_themed_icon_load("xfce4-smart-updated", 16);
  smart->error_icon = xfce_themed_icon_load("xfce4-smart-error", 16);

  /* create some panel widgets */
  smart->tooltip = gtk_tooltips_new();
    
  smart->button = gtk_button_new();
  gtk_button_set_relief(GTK_BUTTON (smart->button), GTK_RELIEF_NONE);
  gtk_widget_show(smart->button);
  gtk_container_add(GTK_CONTAINER (plugin), smart->button);
  g_signal_connect(smart->button, "button-press-event",
            G_CALLBACK(smart_button_press_cb), smart);
  g_signal_connect(smart->button, "button-release-event",
            G_CALLBACK(smart_button_release_cb), smart);

  smart->image = xfce_scaled_image_new();
  gtk_widget_show(smart->image);
  gtk_container_add(GTK_CONTAINER (smart->button), smart->image);

  smart->menuitem = gtk_image_menu_item_new_with_label(_("Update Channels"));
  gtk_widget_show (smart->menuitem);
  menuicon = gtk_image_new_from_stock(GTK_STOCK_REFRESH, GTK_ICON_SIZE_MENU);
  gtk_widget_show (menuicon);
  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM(smart->menuitem), menuicon);

  gtk_widget_destroy(dummy);

  /* set default icon and check */
  smart_size_changed (smart->plugin, smart->current_icon_size, smart);
  smart_set_available(smart, smart->updates_available);
  if (smart->check_interval > 0)
    smart_check_upgrades(smart);
  smart_set_interval(smart, smart->check_interval);

  return smart;
}

static void
smart_free (XfcePanelPlugin *plugin,
             SmartPlugin    *smart)
{
  GtkWidget *dialog;

  /* check if the dialog is still open. if so, destroy it */
  dialog = g_object_get_data (G_OBJECT (plugin), "dialog");
  if (G_UNLIKELY (dialog != NULL))
    gtk_widget_destroy (dialog);

  /* destroy the panel widgets */
  gtk_widget_destroy (smart->button);
  gtk_widget_destroy (smart->menuitem);

  gtk_object_sink(GTK_OBJECT (smart->tooltip));

  /* free the icons */
  g_object_unref(G_OBJECT(smart->application_icon));
  g_object_unref(G_OBJECT(smart->nopackage_icon));
  g_object_unref(G_OBJECT(smart->newpackage_icon));
  g_object_unref(G_OBJECT(smart->available_icon));
  g_object_unref(G_OBJECT(smart->downloading_icon));
  g_object_unref(G_OBJECT(smart->updated_icon));
  g_object_unref(G_OBJECT(smart->error_icon));

  if(smart->timeout)
    g_source_remove(smart->timeout);
  if (G_LIKELY (smart->timer != NULL))
    g_timer_destroy (smart->timer);

  /* free the plugin structure */
  panel_slice_free (SmartPlugin, smart);
}

static void
smart_orientation_changed (XfcePanelPlugin *plugin,
                            GtkOrientation   orientation,
                            SmartPlugin    *smart)
{
}

static gboolean
smart_size_changed (XfcePanelPlugin *plugin,
                     gint             size,
                     SmartPlugin    *smart)
{
  GtkOrientation orientation;
  gchar *iconname;
  gint iconsize;

  /* get the orientation of the plugin */
  orientation = xfce_panel_plugin_get_orientation (plugin);

  /* set the widget size */
  if (orientation == GTK_ORIENTATION_HORIZONTAL)
    gtk_widget_set_size_request (GTK_WIDGET (plugin), -1, size);
  else
    gtk_widget_set_size_request (GTK_WIDGET (plugin), size, -1);

  iconsize = size - MAX(GTK_WIDGET(smart->button)->style->xthickness, 
                        GTK_WIDGET(smart->button)->style->ythickness) - 1;

  switch (smart->package_manager)
  {
    case PACKAGE_MANAGER_SMART:
      iconname = "xfce4-" SMART_PROGRAM;
      break;
    case PACKAGE_MANAGER_YUM:
      iconname = "xfce4-" YUM_PROGRAM;
      break;
    case PACKAGE_MANAGER_APT:
      iconname = "xfce4-" APT_PROGRAM;
      break;
    default:
      iconname = "xfce-unknown";
      break;
  }

  if (smart->application_icon)
     g_object_unref(G_OBJECT(smart->application_icon));
  smart->application_icon = xfce_themed_icon_load(iconname, iconsize);

  if (smart->nopackage_icon)
     g_object_unref(G_OBJECT(smart->nopackage_icon));
  smart->nopackage_icon = xfce_themed_icon_load("xfce-nopackage", iconsize);
  if (smart->newpackage_icon)
     g_object_unref(G_OBJECT(smart->newpackage_icon));
  smart->newpackage_icon = xfce_themed_icon_load("xfce-newpackage", iconsize);

  smart->current_icon_size = size;

  gtk_widget_set_size_request(smart->button, size, size);
  smart_set_available(smart, smart->updates_available);

  /* we handled the orientation */
  return TRUE;
}


static void
smart_construct (XfcePanelPlugin *plugin)
{
  SmartPlugin *smart;

  /* setup transation domain */
  xfce_textdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");

  /* create the plugin */
  smart = smart_new (plugin);

  /* add the button to the panel */
  gtk_container_add (GTK_CONTAINER (plugin), smart->button);

  /* show the panel's right-click menu on this button */
  xfce_panel_plugin_add_action_widget (plugin, smart->button);

  /* connect plugin signals */
  g_signal_connect (G_OBJECT (plugin), "free-data",
                    G_CALLBACK (smart_free), smart);

  g_signal_connect (G_OBJECT (plugin), "save",
                    G_CALLBACK (smart_save), smart);

  g_signal_connect (G_OBJECT (plugin), "size-changed",
                    G_CALLBACK (smart_size_changed), smart);

  g_signal_connect (G_OBJECT (plugin), "orientation-changed",
                    G_CALLBACK (smart_orientation_changed), smart);

  /* show the configure menu item and connect signal */
  xfce_panel_plugin_menu_show_configure (plugin);
  g_signal_connect (G_OBJECT (plugin), "configure-plugin",
                    G_CALLBACK (smart_configure), smart);

  /* show the about menu item and connect signal */
  xfce_panel_plugin_menu_show_about (plugin);
  g_signal_connect (G_OBJECT (plugin), "about",
                    G_CALLBACK (smart_about), plugin);

  /* show the update menu item and connect signal */
  xfce_panel_plugin_menu_insert_item (plugin, GTK_MENU_ITEM(smart->menuitem));
  g_signal_connect (G_OBJECT (smart->menuitem), "activate",
                    G_CALLBACK (smart_menuitem_activate_cb), smart);
}


static void
smart_configure_response (GtkWidget    *dialog,
                           gint          response,
                           SmartPlugin *smart)
{
  gboolean result;
  gchar *website, *s;

  switch (smart->package_manager)
  {
    case PACKAGE_MANAGER_SMART:
      website = SMART_WEBSITE;
      break;
    case PACKAGE_MANAGER_YUM:
      website = YUM_WEBSITE;
      break;
    case PACKAGE_MANAGER_APT:
      website = APT_WEBSITE;
      break;
    default:
      website = "http://goodies.xfce.org/";
      break;
  }



  if (response == GTK_RESPONSE_HELP)
    {
      /* show help */
      s = g_strdup_printf ("exo-open --launch WebBrowser %s", website);
      result = g_spawn_command_line_async (s, NULL);
      g_free(s);

      if (G_UNLIKELY (result == FALSE))
        g_warning (_("Unable to open the following url: %s"), SMART_WEBSITE);
    }
  else if (response == GTK_RESPONSE_ACCEPT)
    {
      if (smart->in_progress)
      {
        xfce_message_dialog(NULL, _("Update In Progress"), GTK_STOCK_DIALOG_WARNING,
                            _("An update is already in progress, please wait."), NULL,
                            GTK_STOCK_OK, GTK_RESPONSE_OK, NULL);
      }
      else
      {
        smart_update_channels(smart);
      }
    }
  else if (response == GTK_RESPONSE_OK)
    {
      g_object_set_data (G_OBJECT (smart->plugin), "dist-upgrade", NULL);

      /* remove the dialog data from the plugin */
      g_object_set_data (G_OBJECT (smart->plugin), "dialog", NULL);
      
      /* unlock the panel menu */
      xfce_panel_plugin_unblock_menu (smart->plugin);

      /* save the plugin */
      smart_save (smart->plugin, smart);

      /* destroy the properties dialog */
      gtk_widget_destroy (dialog);
    }
}

static GtkWidget *
smart_custom_button_new(const gchar *text, const gchar *icon)
{
    GtkWidget *btn, *hbox, *img, *lbl;
    GdkPixbuf *pix;
    gint iw, ih;
    
    g_return_val_if_fail((text && *text) || icon, NULL);
    
    btn = gtk_button_new();
    
    hbox = gtk_hbox_new(FALSE, 4);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 0);
    gtk_widget_show(hbox);
    gtk_container_add(GTK_CONTAINER(btn), hbox);
    
    if(icon) {
        img = gtk_image_new_from_stock(icon, GTK_ICON_SIZE_BUTTON);
        if(!img || gtk_image_get_storage_type(GTK_IMAGE(img)) == GTK_IMAGE_EMPTY) {
            gtk_icon_size_lookup(GTK_ICON_SIZE_BUTTON, &iw, &ih);
            pix = xfce_themed_icon_load(icon, iw);
            if(pix) {
                if(img)
                    gtk_image_set_from_pixbuf(GTK_IMAGE(img), pix);
                else
                    img = gtk_image_new_from_pixbuf(pix);
                g_object_unref(G_OBJECT(pix));
            }
        }
        if(img) {
            gtk_widget_show(img);
            gtk_box_pack_start(GTK_BOX(hbox), img, FALSE, FALSE, 0);
        }
    }
    
    if(text) {
        lbl = gtk_label_new_with_mnemonic(text);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), btn);
    }
    
    return btn;
}

void
smart_configure (XfcePanelPlugin *plugin,
                  SmartPlugin    *smart)
{
  GtkWidget *dialog;
  GtkWidget *label;
  GtkWidget *button;
  GtkWidget *combo;
  GtkWidget *vbox, *hbox;
   
  /* block the plugin menu */
  xfce_panel_plugin_block_menu (plugin);

  /* create the dialog */
  dialog = xfce_titled_dialog_new_with_buttons (_("Smart Package Manager"),
                                                GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (plugin))),
                                                GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
                                              /*GTK_STOCK_HELP, GTK_RESPONSE_HELP,
                                                GTK_STOCK_CLOSE, GTK_RESPONSE_OK,*/
                                                NULL);

  button = gtk_button_new_from_stock(GTK_STOCK_HELP);
  gtk_widget_show(button);
  gtk_dialog_add_action_widget(GTK_DIALOG(dialog), button, GTK_RESPONSE_HELP);

  button = smart_custom_button_new(_("_Update..."), GTK_STOCK_REFRESH);
  gtk_widget_show(button);
  gtk_dialog_add_action_widget(GTK_DIALOG(dialog), button, GTK_RESPONSE_ACCEPT);

  button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
  gtk_widget_show(button);
  gtk_dialog_add_action_widget(GTK_DIALOG(dialog), button, GTK_RESPONSE_OK);

  vbox=gtk_vbox_new (FALSE,0);
  gtk_widget_show (vbox);
  gtk_container_add (GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), vbox);

  hbox = gtk_hbox_new (FALSE,0);
  gtk_widget_show (hbox);
  label = gtk_label_new(_("Package Manager: "));
  gtk_widget_show (label);
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  combo = gtk_combo_box_new_text ();
  gtk_combo_box_append_text (GTK_COMBO_BOX(combo), _("Smart"));
  gtk_combo_box_append_text (GTK_COMBO_BOX(combo), _("Yum+Extender"));
  gtk_combo_box_append_text (GTK_COMBO_BOX(combo), _("APT+Synaptic"));
  gtk_combo_box_set_active (GTK_COMBO_BOX(combo), smart->package_manager);
  gtk_widget_show (combo);
  g_signal_connect (G_OBJECT(combo),"changed", G_CALLBACK(smart_combobox_changed_cb), smart);
  gtk_box_pack_start(GTK_BOX(hbox),combo,FALSE,FALSE,0);
  gtk_box_pack_start (GTK_BOX(vbox),hbox,TRUE,TRUE,0);

  button = gtk_check_button_new_with_label(_("Show application icon for package manager"));
  gtk_widget_show (button);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), smart->show_application);
  g_signal_connect (G_OBJECT(button),"toggled", G_CALLBACK(smart_checkbox1_toggle_cb), smart);
  gtk_box_pack_start (GTK_BOX(vbox),button,TRUE,TRUE,0);

  hbox = gtk_hbox_new (FALSE,0);
  gtk_widget_show (hbox);
  label = gtk_label_new(_("Check for upgrades every "));
  gtk_widget_show (label);
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  button=gtk_spin_button_new_with_range(0,60,1);
  gtk_widget_show (button);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(button), smart->check_interval);
  g_signal_connect(G_OBJECT(button),"value-changed", G_CALLBACK(smart_spin_value_change_cb), smart);
  gtk_box_pack_start (GTK_BOX(hbox),button,FALSE,FALSE,0);
  label = gtk_label_new(_(" minutes"));
  gtk_widget_show (label);
  gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);
  gtk_box_pack_start(GTK_BOX(vbox),hbox,TRUE,TRUE,0);

  button = gtk_check_button_new_with_label(_("Upgrade on launch (if upgrades available)"));
  gtk_widget_show (button);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), smart->upgrade_directly);
  g_signal_connect (G_OBJECT(button),"toggled", G_CALLBACK(smart_checkbox2_toggle_cb), smart);
  gtk_box_pack_start (GTK_BOX(vbox),button,TRUE,TRUE,0);

  button = gtk_check_button_new_with_label(_("Smart upgrade to newer release (dist-upgrade)"));
  gtk_widget_show (button);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), smart->dist_upgrade);
  g_signal_connect (G_OBJECT(button),"toggled", G_CALLBACK(smart_checkbox3_toggle_cb), smart);
  gtk_box_pack_start (GTK_BOX(vbox),button,TRUE,TRUE,0);

  g_object_set_data (G_OBJECT (plugin), "dist-upgrade", button);
  gtk_widget_set_sensitive (button, smart->package_manager == PACKAGE_MANAGER_APT);

  /* center dialog on the screen */
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);

  /* set dialog icon */
  gtk_window_set_icon_name (GTK_WINDOW (dialog), "xfce4-settings");

  /* link the dialog to the plugin, so we can destroy it when the plugin
   * is closed, but the dialog is still open */
  g_object_set_data (G_OBJECT (plugin), "dialog", dialog);

  /* connect the reponse signal to the dialog */
  g_signal_connect (G_OBJECT (dialog), "response",
                    G_CALLBACK(smart_configure_response), smart);

  /* show the entire dialog */
  gtk_widget_show (dialog);
}

void
smart_about (XfcePanelPlugin *plugin)
{
  XfceAboutInfo *info;
  GtkWidget *dialog;
  GdkPixbuf *icon;

  info = xfce_about_info_new(
              _("Smart Package Manager - panel plugin"),
              VERSION,
              _("Check for updates using the Smart Package Manager"),
              XFCE_COPYRIGHT_TEXT("2007", "Anders F Bjorklund"),
              XFCE_LICENSE_GPL);
  xfce_about_info_set_homepage(info, SMART_WEBSITE);

  icon = xfce_themed_icon_load("xfce4-smart", 48);

  dialog = xfce_about_dialog_new(NULL, info, icon);
  gtk_window_set_default_size(GTK_WINDOW(dialog), 500, 400);

  xfce_about_info_free(info);
  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy (dialog);
  g_object_unref(G_OBJECT(icon));
}
