/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA 02110-1301, USA.                                          *
 ***************************************************************************/

//Own
#include "entry.h"

//Qt
#include <kmenu.h>

//KDE
#include <kmountpoint.h>
#include <kmessagebox.h>

//Ui
#include "password.h"
#include "map.h"

//KGRUBEditor
#include "core/path.h"

EntryAssistant::EntryAssistant( GRUB::ConfigFile::Entry *entry, QWidget *parent ) : KAssistantDialog( parent )
{
//SETUP UI
	setWindowIcon( KIcon( "document-properties" ) );
	setCaption( i18nc( "@title:window", "Entry Assistant" ) );
	setAttribute( Qt::WA_DeleteOnClose );
	showButton( KDialog::Help, false );
	setInitialSize( QSize( 600, 450 ) );

	QWidget *intro = new QWidget( this );
	ui_intro.setupUi( intro );
	addPage( intro, i18nc( "@title", "Introduction" ) );
	QWidget *step1 = new QWidget( this );
	ui_step1.setupUi( step1 );
	titlePage = addPage( step1, i18nc( "@title", "Step 1/10 - Title" ) );
	QWidget *step2 = new QWidget( this );
	ui_step2.setupUi( step2 );
	addPage( step2, i18nc( "@title", "Step 2/10 - Root" ) );
	QWidget *step3 = new QWidget( this );
	ui_step3.setupUi( step3 );
	addPage( step3, i18nc( "@title", "Step 3/10 - Kernel" ) );
	QWidget *step4 = new QWidget( this );
	ui_step4.setupUi( step4 );
	addPage( step4, i18nc( "@title", "Step 4/10 - Initrd" ) );
	QWidget *step5 = new QWidget( this );
	ui_step5.setupUi( step5 );
	addPage( step5, i18nc( "@title", "Step 5/10 - Lock" ) );
	QWidget *step6 = new QWidget( this );
	ui_step6.setupUi( step6 );
	addPage( step6, i18nc( "@title", "Step 6/10 - Password" ) );
	QWidget *step7 = new QWidget( this );
	ui_step7.setupUi( step7 );
	addPage( step7, i18nc( "@title", "Step 7/10 - MakeActive" ) );
	QWidget *step8 = new QWidget( this );
	ui_step8.setupUi( step8 );
	addPage( step8, i18nc( "@title", "Step 8/10 - SaveDefault" ) );
	QWidget *step9 = new QWidget( this );
	ui_step9.setupUi( step9 );
	addPage( step9, i18nc( "@title", "Step 9/10 - Chainloader" ) );
	QWidget *step10 = new QWidget( this );
	ui_step10.setupUi( step10 );
	addPage( step10, i18nc( "@title", "Step 10/10 - Maps" ) );
	QWidget *summary = new QWidget( this );
	ui_summary.setupUi( summary );
	summaryPage = addPage( summary, i18nc( "@title", "Summary" ) );
//PROPERTY SETUP
	tmp_entry = new GRUB::ConfigFile::Entry( *entry );
	m_entry = entry;
//SHOW ENTRY(IF ANY - ONLY USED IN EDIT MODE)
	ui_step1.klineedit_title->setText( entry->title() );
	ui_step2.grubdeviceeditor_root->lineEdit()->setText( entry->root() );
	ui_step3.gruburlrequester_kernel->setPath( entry->kernel().kernel() );
	ui_step3.klineedit_arguments->setText( entry->kernel().arguments() );
	ui_step4.gruburlrequester_initrd->setPath( entry->initrd() );
	ui_step5.checkBox_lock->setChecked( entry->lock() );
	showPassword();
	ui_step7.checkBox_makeactive->setChecked( entry->makeActive() );
	ui_step8.checkBox_savedefault->setChecked( entry->saveDefault() );
	ui_step9.klineedit_chainloader->setText( entry->chainLoader() );
	showMaps();
//MISC UI SETUP
	checkTitle( ui_step1.klineedit_title->text() );

	ui_step2.grubdeviceeditor_root->setMode( GRUBDeviceEditor::Partition );
	ui_step3.gruburlrequester_kernel->setAssociatedRootEditor( ui_step2.grubdeviceeditor_root->lineEdit() );
	ui_step4.gruburlrequester_initrd->setAssociatedRootEditor( ui_step2.grubdeviceeditor_root->lineEdit() );

	menu_kernelSuggestions = new KMenu( this );
	action_rootDevice = new QAction( i18nc( "@item:inmenu", "Root (Device)" ), menu_kernelSuggestions );
	menu_kernelSuggestions->addAction( action_rootDevice );
	action_rootUuid = new QAction( i18nc( "@item:inmenu", "Root (UUID)" ), menu_kernelSuggestions );
	menu_kernelSuggestions->addAction( action_rootUuid );
	menu_kernelSuggestions->addSeparator();
	action_ro = new QAction( i18nc( "@item:inmenu", "Read-Only" ), menu_kernelSuggestions );
	menu_kernelSuggestions->addAction( action_ro );
	action_quiet = new QAction( i18nc( "@item:inmenu", "Quiet" ), menu_kernelSuggestions );
	menu_kernelSuggestions->addAction( action_quiet );
	action_splash = new QAction( i18nc( "@item:inmenu", "Splash" ), menu_kernelSuggestions );
	menu_kernelSuggestions->addAction( action_splash );
	ui_step3.kpushbutton_suggestions->setMenu( menu_kernelSuggestions );
	ui_step3.kpushbutton_suggestions->setIcon( KIcon( "tools-wizard" ) );

	ui_step6.kpushbutton_create->setIcon( KIcon( "dialog-password" ) );
	ui_step6.kpushbutton_edit->setIcon( KIcon( "document-properties" ) );
	ui_step6.kpushbutton_delete->setIcon( KIcon( "edit-delete" ) );

	ui_step10.kpushbutton_add->setIcon( KIcon( "list-add" ) );
	ui_step10.kpushbutton_edit->setIcon( KIcon( "document-properties" ) );
	ui_step10.kpushbutton_remove->setIcon( KIcon( "list-remove" ) );
//SETUP CONNECTIONS
	setupConnections();
}
EntryAssistant::~EntryAssistant()
{
	delete tmp_entry;
	tmp_entry = 0;
	m_entry = 0;
}

void EntryAssistant::setupConnections()
{
	connect( ui_step1.klineedit_title, SIGNAL( textChanged( const QString & ) ), SLOT( checkTitle( const QString & ) ) );
	connect( action_rootDevice, SIGNAL( triggered() ), SLOT( appendRootDeviceArgument() ) );
	connect( action_rootUuid, SIGNAL( triggered() ), SLOT( appendRootUuidArgument() ) );
	connect( action_ro, SIGNAL( triggered() ), SLOT( appendReadOnlyArgument() ) );
	connect( action_quiet, SIGNAL( triggered() ), SLOT( appendQuietArgument() ) );
	connect( action_splash, SIGNAL( triggered() ), SLOT( appendSplashArgument() ) );
	connect( ui_step6.kpushbutton_create, SIGNAL( pressed() ), SLOT( createPassword() ) );
	connect( ui_step6.kpushbutton_edit, SIGNAL( pressed() ), SLOT( editPassword() ) );
	connect( ui_step6.kpushbutton_delete, SIGNAL( pressed() ), SLOT( deletePassword() ) );
	connect( ui_step10.kpushbutton_add, SIGNAL( pressed() ), SLOT( addMap() ) );
	connect( ui_step10.kpushbutton_edit, SIGNAL( pressed() ), SLOT( editMap() ) );
	connect( ui_step10.kpushbutton_remove, SIGNAL( pressed() ), SLOT( removeMap() ) );
	connect( this, SIGNAL( user2Clicked() ), SLOT( nextClicked() ) );
	connect( this, SIGNAL( user1Clicked() ), SLOT( finishClicked() ) );
}
void EntryAssistant::showMaps()
{
	ui_step10.klistwidget_maps->clear();

	if ( !tmp_entry->maps().isEmpty() )
	{
		foreach( const GRUB::ComplexCommand::Map map, tmp_entry->maps() )
		{
			ui_step10.klistwidget_maps->addItem( map.result() );
		}
	}
	ui_step10.kpushbutton_edit->setDisabled( tmp_entry->maps().isEmpty() );
	ui_step10.kpushbutton_remove->setDisabled( tmp_entry->maps().isEmpty() );
}
void EntryAssistant::showPassword()
{
	ui_step6.checkBox_md5->setChecked( tmp_entry->password().md5crypted() );
	ui_step6.klineedit_password->setText( tmp_entry->password().password() );
	ui_step6.klineedit_menu->setText( tmp_entry->password().configFile() );

	ui_step6.kpushbutton_edit->setDisabled( tmp_entry->password().isEmpty() );
	ui_step6.kpushbutton_delete->setDisabled( tmp_entry->password().isEmpty() );
}

void EntryAssistant::checkTitle( const QString &text )
{
	setValid( titlePage, !QString( text ).trimmed().isEmpty() );
}
void EntryAssistant::appendRootDeviceArgument()
{
	if ( ui_step3.gruburlrequester_kernel->url().isEmpty() )
		return;

	QString realKernelPath;
	if ( ui_step3.gruburlrequester_kernel->url().path().startsWith( "/" ) )
		realKernelPath = Core::Path::convertToGenericPath( ui_step2.grubdeviceeditor_root->lineEdit()->text() + ui_step3.gruburlrequester_kernel->url().path() );
	else
		realKernelPath = Core::Path::convertToGenericPath( ui_step3.gruburlrequester_kernel->url().path() );

	//Note that this check is necessary because if the user enters dummy text
	//such as 'jkh' mp would be null and ultimately crash the application.
	if ( KMountPoint::Ptr mp = KMountPoint::currentMountPoints().findByPath( realKernelPath ) )
	{
		foreach( const GRUB::Misc::Device device, Core::Devices::DeviceList )
		{
			if ( mp->mountPoint() == device.mountPoint() )
			{
				ui_step3.klineedit_arguments->setText( ui_step3.klineedit_arguments->text() + ( ui_step3.klineedit_arguments->text().isEmpty() ? QString() : QString( " " ) ) + "root=" + device.partition() );
				break;
			}
		}
	}
}
void EntryAssistant::appendRootUuidArgument()
{
	if ( ui_step3.gruburlrequester_kernel->url().isEmpty() )
		return;

	QString realKernelPath;
	if ( ui_step3.gruburlrequester_kernel->url().path().startsWith( "/" ) )
		realKernelPath = Core::Path::convertToGenericPath( ui_step2.grubdeviceeditor_root->lineEdit()->text() + ui_step3.gruburlrequester_kernel->url().path() );
	else
		realKernelPath = Core::Path::convertToGenericPath( ui_step3.gruburlrequester_kernel->url().path() );

	//Note that this check is necessary because if the user enters dummy text
	//such as 'jkh' mp would be null and ultimately crash the application.
	if ( KMountPoint::Ptr mp = KMountPoint::currentMountPoints().findByPath( realKernelPath ) )
	{
		foreach( const GRUB::Misc::Device device, Core::Devices::DeviceList )
		{
			if ( mp->mountPoint() == device.mountPoint() )
			{
				ui_step3.klineedit_arguments->setText( ui_step3.klineedit_arguments->text() + ( ui_step3.klineedit_arguments->text().isEmpty() ? QString() : QString( " " ) ) + "root=UUID=" + device.uuid() );
				break;
			}
		}
	}
}
void EntryAssistant::appendReadOnlyArgument()
{
	ui_step3.klineedit_arguments->setText( ui_step3.klineedit_arguments->text() + ( ui_step3.klineedit_arguments->text().isEmpty() ? QString() : QString( " " ) ) + "ro" );
}
void EntryAssistant::appendQuietArgument()
{
	ui_step3.klineedit_arguments->setText( ui_step3.klineedit_arguments->text() + ( ui_step3.klineedit_arguments->text().isEmpty() ? QString() : QString( " " ) ) + "quiet" );
}
void EntryAssistant::appendSplashArgument()
{
	ui_step3.klineedit_arguments->setText( ui_step3.klineedit_arguments->text() + ( ui_step3.klineedit_arguments->text().isEmpty() ? QString() : QString( " " ) ) + "splash" );
}
void EntryAssistant::createPassword()
{
	GRUB::ComplexCommand::Password tmp_password;
	PasswordAssistant *passwordAssistant = new PasswordAssistant( &tmp_password, this );
	if ( passwordAssistant->exec() )
	{
		tmp_entry->setPassword( tmp_password );
		showPassword();
	}
}
void EntryAssistant::editPassword()
{
	GRUB::ComplexCommand::Password tmp_password = tmp_entry->password();
	PasswordAssistant *passwordAssistant = new PasswordAssistant( &tmp_password, this );
	if ( passwordAssistant->exec() )
	{
		tmp_entry->setPassword( tmp_password );
		showPassword();
	}
}
void EntryAssistant::deletePassword()
{
	if ( KMessageBox::questionYesNo( this, i18nc( "@info", "Are you sure you want to delete the password?" ) ) == KMessageBox::Yes )
	{
		tmp_entry->clearPassword();
		showPassword();
	}
}
void EntryAssistant::addMap()
{
	GRUB::ComplexCommand::Map tmp_map;
	MapEditor *mapEditor = new MapEditor( &tmp_map, this );
	if ( mapEditor->exec() )
	{
		tmp_entry->addMap( tmp_map );
		showMaps();
	}
}
void EntryAssistant::editMap()
{
	if ( !ui_step10.klistwidget_maps->currentItem() )
		return;

	GRUB::ComplexCommand::Map tmp_map = tmp_entry->maps().at( ui_step10.klistwidget_maps->currentRow() );
	MapEditor *mapEditor = new MapEditor( &tmp_map, this );
	if ( mapEditor->exec() )
	{
		tmp_entry->replaceMap( ui_step10.klistwidget_maps->currentRow(), tmp_map );
		showMaps();
	}
}
void EntryAssistant::removeMap()
{
	if ( !ui_step10.klistwidget_maps->currentItem() )
		return;

	if ( KMessageBox::questionYesNo( this, i18nc( "@info", "Are you sure you want to remove this map?" ) ) == KMessageBox::Yes )
	{
		tmp_entry->removeMap( ui_step10.klistwidget_maps->currentRow() );
		showMaps();
	}
}

void EntryAssistant::nextClicked()
{
	if ( currentPage() == summaryPage )
	{
		tmp_entry->setTitle( ui_step1.klineedit_title->text() );
		tmp_entry->setRoot( ui_step2.grubdeviceeditor_root->lineEdit()->text() );
		tmp_entry->setKernel( GRUB::ComplexCommand::Kernel( ui_step3.gruburlrequester_kernel->url().path(), ui_step3.klineedit_arguments->text() ) );
		tmp_entry->setInitrd( ui_step4.gruburlrequester_initrd->url().path() );
		tmp_entry->setLock( ui_step5.checkBox_lock->isChecked() );
		//password is already set
		tmp_entry->setMakeActive( ui_step7.checkBox_makeactive->isChecked() );
		tmp_entry->setSaveDefault( ui_step8.checkBox_savedefault->isChecked() );
		tmp_entry->setChainLoader( ui_step9.klineedit_chainloader->text() );
		//maps are already set

		ui_summary.previewWidget->previewEntry( *tmp_entry );
	}
}
void EntryAssistant::finishClicked()
{
	m_entry->setTitle( ui_step1.klineedit_title->text() );
	m_entry->setRoot( ui_step2.grubdeviceeditor_root->lineEdit()->text() );
	m_entry->setKernel( GRUB::ComplexCommand::Kernel( ui_step3.gruburlrequester_kernel->url().path(), ui_step3.klineedit_arguments->text() ) );
	m_entry->setInitrd( ui_step4.gruburlrequester_initrd->url().path() );
	m_entry->setLock( ui_step5.checkBox_lock->isChecked() );
	m_entry->setPassword( tmp_entry->password() );
	m_entry->setMakeActive( ui_step7.checkBox_makeactive->isChecked() );
	m_entry->setSaveDefault( ui_step8.checkBox_savedefault->isChecked() );
	m_entry->setChainLoader( ui_step9.klineedit_chainloader->text() );
	m_entry->setMaps( tmp_entry->maps() );
}

#include "entry.moc"
