// -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 1999 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#else 
#error UNISTD_H
#endif

#include <gdk/gdk.h>

#if defined(USE_IMAGEMAGICK)
  #include <magick/magick.h>
  #include <gdk/gdkrgb.h>
  #include <gtk/gtk.h>
  extern GtkWidget* mainWindow;
  extern GdkGC* gc;
  #ifdef class
    #undef class
  #endif
#elif defined(USE_GDK_PIXBUF)
  #include <gdk-pixbuf/gdk-pixbuf.h>
#elif defined(USE_LIBGDK_IMLIB)
  #include <gdk_imlib.h>
#endif

#include "mmfile.h"
#include <utility>
#include <cstring>
#include <cstdlib>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

static char currentPalette[1024];
static int currentPaletteLength;

static bool isInitialized=false;

void mmfile_init(int* argc, char*** argv){

#if defined(USE_IMAGEMAGICK)
  gdk_rgb_init();
#elif defined(USE_LIBGDK_IMLIB)
  gdk_imlib_init();
#endif
  isInitialized=true;
}

#ifdef USE_INTERNAL_GRAPHICS
// ************************************************
#if defined(USE_IMAGEMAGICK)
  #if MagickLibVersion<0x0500
static GdkPixmap* MakePixmapFromFileSub(const char* fileName){
  GdkPixmap* pix=NULL;
  ImageInfo *ii=static_cast<ImageInfo *>(AllocateMemory(sizeof(ImageInfo)));
  GetImageInfo(ii);
  strcpy(ii->filename,fileName);
  Image* im=ReadImage(ii);
  if (im!=NULL) {
    int w=im->columns;
    int h=im->rows;
    pix=gdk_pixmap_new(mainWindow->window,w,h,-1);
    if (pix==NULL) return NULL;
    guchar* rgb=new guchar[w*h*3];

    guchar* rp=rgb;
    float* fr=new float[w*h];
    float* fg=new float[w*h];
    float* fb=new float[w*h];
    if (rgb!=NULL && fr!=NULL && fg!=NULL && fb!=NULL){
      GetPixels(im,fr,fg,fb,NULL);
      int count=0;
      for(int y=0;y<h;y++){
	for(int x=0;x<w;x++){
	  *(rp++)=static_cast<unsigned char>(255.0*(fr[count]));
	  *(rp++)=static_cast<unsigned char>(255.0*(fg[count]));
	  *(rp++)=static_cast<unsigned char>(255.0*(fb[count]));
	  count++;
	}
      }
      gdk_gc_set_function (gc,GDK_COPY);
      gdk_draw_rgb_image(pix,gc,0,0,w,h,GDK_RGB_DITHER_NORMAL,rgb,w*3);
      delete [] fb;
      delete [] fg;
      delete [] fr;
      delete [] rgb;
    }
    DestroyImage(im);
  }
  DestroyImageInfo(ii);
  return pix;
}

  #elif MagickLibVersion<0x0520
static GdkPixmap* MakePixmapFromFileSub(const char* fileName){
  Debug::DebugOut(Debug::MULTIMEDIA,"Make Image using ImageMagic 5.1.X\n");
  GdkPixmap* pix=NULL;
  ImageInfo *ii=static_cast<ImageInfo *>(AllocateMemory(sizeof(ImageInfo)));
  if (ii==NULL) return NULL;
  GetImageInfo(ii);
  strcpy(ii->filename,fileName);
  Image* im=ReadImage(ii);
  if (im!=NULL) {
    int w=im->columns;
    int h=im->rows;
    pix=gdk_pixmap_new(mainWindow->window,w,h,-1);
    guchar* rgb=new guchar[w*h*3];
    if (rgb!=NULL){
      GetPixels(im,0,0,w,h,"RGB",CharPixel,static_cast<void*>(rgb));
      gdk_gc_set_function (gc,GDK_COPY);
      gdk_draw_rgb_image(pix,gc,0,0,w,h,GDK_RGB_DITHER_NORMAL,rgb,w*3);
      delete [] rgb;
    }
    DestroyImage(im);
  }
  DestroyImageInfo(ii);
  return pix;
}
  #else // MagickLib 5.2.0 or later
static GdkPixmap* MakePixmapFromFileSub(const char* fileName){
  Debug::DebugOut(Debug::MULTIMEDIA,"Make Image using ImageMagic 5.2.X\n");
  GdkPixmap* pix=NULL;
  #if MagickLibVersion>0x0523
  ImageInfo *ii=static_cast<ImageInfo *>(AcquireMemory(sizeof(ImageInfo)));
  #else
  ImageInfo *ii=static_cast<ImageInfo *>(AllocateMemory(sizeof(ImageInfo)));
  #endif
  if (ii==NULL) return NULL;
  GetImageInfo(ii);
  ExceptionInfo ei;
  strcpy(ii->filename,fileName);
  Image* im=ReadImage(ii,&ei);
  if (im!=NULL) {
    int w=im->columns;
    int h=im->rows;
    pix=gdk_pixmap_new(mainWindow->window,w,h,-1);
    guchar* rgb=new guchar[w*h*3];
    if (rgb!=NULL){
      PixelPacket* pixels=GetImagePixels(im,0,0,w,h);
      if (pixels!=NULL){
	guchar* rgbp=rgb;
	PixelPacket* pixp=pixels;
	for(int y=0;y<h;y++){
	  for(int x=0;x<w;x++){
	    *(rgbp++)=DownScale(pixp->red);
	    *(rgbp++)=DownScale(pixp->green);
	    *(rgbp++)=DownScale(pixp->blue);
	    pixp++;
	  }
	}
	gdk_gc_set_function (gc,GDK_COPY);
	gdk_draw_rgb_image(pix,gc,0,0,w,h,GDK_RGB_DITHER_NORMAL,rgb,w*3);
      }
      delete [] rgb;
    }
    DestroyImage(im);
  }
  DestroyImageInfo(ii);
  return pix;
}
  #endif

// ********************************************
#elif defined(USE_GDK_PIXBUF)

inline GdkPixmap* MakePixmapFromFileSub(const char* fileName){
  GdkPixbuf* pb=gdk_pixbuf_new_from_file(fileName);
  if (pb==NULL) return NULL;
  GdkPixmap* p;
  gdk_pixbuf_render_pixmap_and_mask(pb,&p,NULL,0);
  gdk_pixbuf_unref(pb);
  return p;
}
// *********************************************
#elif defined(USE_LIBGDK_IMLIB)
inline GdkPixmap* MakePixmapFromFileSub(const char* fileName){
  GdkPixmap* p;
  char* fname=strdup(fileName);
  gdk_imlib_load_file_to_pixmap(fname, &p, NULL);
  free(fname);
  return p;
}
#endif
// *********************************************

GdkPixmap* MakePixmapFromFile(const char* fileName){
  if (!isInitialized) mmfile_init(NULL,NULL);
  if (!isInitialized) return NULL;
  char buf[16384];
  char *fileName2=NULL;
  const int BmpHeaderLength=0x36;
  unsigned char* ubuf=reinterpret_cast<unsigned char*>(buf);

  FILE* fp=fopen(fileName,"r");
  if (fp==NULL) return NULL;
  fread(buf,1,BmpHeaderLength,fp);
  // 8bit Windows BMP Check
  if (buf[0]=='B' && buf[1]=='M' && buf[0x1c]==8){
    // Keep Palette Information
    int length=65536*ubuf[12]+256*ubuf[11]+ubuf[10]-BmpHeaderLength;
    if (length>0) {
      if (length>1024) length=1024;
      fread(currentPalette,1,length,fp);
      currentPaletteLength=length;
    } else {
#ifdef HAVE_MKSTEMP
      fileName2=strdup("DCTXXXXXX");
      int fd=mkstemp(fileName2);
      if (fd<0){
	fclose(fp);
        free(fileName2);
        return NULL;
      }
      FILE* ofp=fdopen(fd,"w");
#else
      if ((fileName2=tempnam(NULL,"DCTXX"))==NULL) {
	fclose(fp);
	return NULL;
      }
      FILE* ofp=fopen(fileName2,"w");
#endif
      if (ofp==NULL) {
        free(fileName2);
	fclose(fp);
	return NULL;
      }
      int len=currentPaletteLength+BmpHeaderLength;
      ubuf[12]=(len>>16)&255;
      ubuf[11]=(len>>8)&255;
      ubuf[10]=len&255;
      fwrite(buf,1,BmpHeaderLength,ofp);
      fwrite(currentPalette,1,currentPaletteLength,ofp);

      int length=0x1000000*ubuf[0x5]+0x10000*ubuf[0x4]
	+0x100*ubuf[0x3]+ubuf[0x2];

      while(length>0){
	int l=16384;
	if (l>length) l=length;
	fread(buf,1,l,fp);
	fwrite(buf,1,l,ofp);
	length-=l;
      }
      fclose(ofp);
      fileName=fileName2;
    }
  }
  fclose(fp);

  GdkPixmap* pix=MakePixmapFromFileSub(fileName);

  if (fileName2!=NULL){
    unlink(fileName2);
    free(fileName2);
  }
  return pix;
}
#else // Without Graphics Libraries
GdkPixmap* MakePixmapFromFile(const char* fileName){
  return NULL; 
};
#endif

GdkPixmap* MakePixmapFromBlock(BlockIO* npf,const TAG& t){
  char buf[2048];
  byte* ubuf=reinterpret_cast<byte*>(buf);
  npf->Seek(t);
  Debug::DebugOut(Debug::WARNINGS,"Bitmap At %d:%d\n",t.Block(),t.Offset());
  if (npf->Read(ubuf,4));
  GdkPixmap* pix=NULL;
  if (strncmp(buf,"data",4)==0){
    int length=npf->GetRDWord();
    char* fileName=DupToTempFile(npf,TAG(-1,-1),length);
    if (fileName!=NULL){
      pix=MakePixmapFromFile(fileName);
      unlink(fileName);
      free(fileName);
    }
  }
  return pix;
}

char* DupToTempFile(BlockIO* npf,const TAG& start,int length){
  char* tmpfile;
  char buf[2048];
  byte* ubuf=reinterpret_cast<byte*>(buf);

#ifdef HAVE_MKSTEMP
  tmpfile=strdup("DCTXXXXXX");
  int fd=mkstemp(tmpfile);
  if (fd<0){
    free(tmpfile);
    return NULL;
  }
  FILE* fp=fdopen(fd,"w");
#else
  if ((tmpfile=tempnam(NULL,"DCTXX"))==NULL) return NULL;
  FILE* fp=fopen(tmpfile,"w");
#endif
  if (fp==NULL) {
    free(tmpfile);
    return NULL;
  }
  if (start!=TAG(-1,-1)) npf->Seek(start);
  while(length>0){
    int len=2048;
    if (len>length) len=length;
    npf->Read(ubuf,len);
    fwrite(buf,1,len,fp);
    length-=len;
  }
  fclose(fp);
  return tmpfile;
}

void DupToFile(BlockIO* npf,const TAG& start,int length,FILE* fp){
  char buf[2048];
  byte* ubuf=reinterpret_cast<byte*>(buf);
  if (start!=TAG(-1,-1)) npf->Seek(start);
  while(length>0){
    int len=2048;
    if (len>length) len=length;
    npf->Read(ubuf,len);
    fwrite(buf,1,len,fp);
    length-=len;
  }
  return;
}

