// -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 1999 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#include "gui.h"
#include "global.h"
#include "gaiji.h"
#include "honmon.h"
#include "mm.h"

#include "arrow.xpm"

#include <gdk/gdkrgb.h>

extern HISTORY hist;


static EUC_STRING DispChar(const DICCHAR& ch) {
  static byte table21[]=" \xa4\xa1,.\xa5:;?!\xde\xdfZZZ^~_ZZZZZZZZZ"
    "\xb0--/\\ZZ|ZZ`'\"\"()"
    "ZZ[]{}<>ZZ\xa2\xa3ZZZZ+-ZZZ=Z<>ZZZZZZZ'\"Z\\$ZZ%#&*@ZZZZZZZ";
  static byte table25[]="\xa7\xb1\xa8\xb2\xa9\xb3\xaa\xb4\xab\xb5" // $B%!(B-$B%*(B
    "\xb6\x76\xb7\x77\xb8\x78\xb9\x79\xba\x7a" // $B%+(B-$B%4(B
    "\xbb\x7b\xbc\x7c\xbd\x7d\xbe\x7e\xbf\x7f" // $B%5(B-$B%>(B
    "\xc0\x40\xc1\x41\xaf\xc2\x42\xc3\x43\xc4\x44" // $B%?(B-$B%I(B
    "\xc5\xc6\xc7\xc8\xc9" // $B%J(B-$B%N(B
    "\xca\x4a\x0a\xcb\x4b\x0b\xcc\x4c\x0c\xcd\x4d\x0d\xce\x4e\x0e" // $B%O(B-$B%](B
    "\xcf\xd0\xd1\xd2\xd3" "\xac\xd4\xad\xd5\xae\xd6" // $B%^(B-$B%h(B
    "\xd7\xd8\xd9\xda\xdb" " \xdc  \xa6\xdd   ";
  EUC_STRING result="";
  int high=255 & (ch.Code()>>8);
  int low=255 & ch.Code();
  ATTR a=ch.Attr();
  if (a.hankaku) {
    if (high==0x23) {
      result+=char(low);
    } else if (high==0x21 && table21[low-0x21]!='Z') {
      if (table21[low-0x21] & 0x80) {
        result+='\x8e';
        result+=char(table21[low-0x21]);
      } else {
        result+=char(table21[low-0x21]);
      }
    } else if (high==0x25) {
      int x=table25[low-0x21];
      if (x==' ') {
        result+=char(high | 0x80);
        result+=char(low | 0x80);
      } else if (x & 0x80) {
        result+=char(0x8e);
        result+=char(x);
      } else if (x & 0x40) {
	if (x>=0x60) x-=0x40;
        result+=char(0x8e);
        result+=char(x+0x80);
        result+=char(0x8e);
        result+=char(0xde);
      } else {
	if (x<0x20) x+=0x40;
        result+=char(0x8e);
        result+=char(x+0x80);
        result+=char(0x8e);
        result+=char(0xdf);
      }
    } else {
      result+=char(high | 0x80);
      result+=char(low | 0x80);
    }
  } else {
    result+=char(high | 0x80);
    result+=char(low | 0x80);
  }
  return result;
}

static GdkPixmap* NewWhitePixmap(int width,int height){
  GdkPixmap* pix=gdk_pixmap_new(mainWindow->window,width,height,-1);
  gdk_draw_rectangle(pix,gc_inv,true,0,0,width,height);
  return pix;
};

static GdkPixmap* NewColorPixmap(int width,int height,int color){
  GdkPixmap* pix=gdk_pixmap_new(mainWindow->window,width,height,-1);
  gdk_draw_rectangle(pix,gc_color[15-(color & 15)],true,0,0,width,height);
  return pix;
};

static GdkPixmap* NewPixmap(int width,int height,const ATTR& a){
  GdkPixmap* pix;
  if ((a.color & 0xf0)==0x10){
    pix=NewColorPixmap(width,height,a.color);
  } else {
    pix=NewWhitePixmap(width,height);
  }
  if (a.underline==5){
    const byte hatch[8]={0x22,0x22,0x00,0x00,0x88,0x88,0x00,0x00};
    static GdkImage* img=NULL;
    if (img==NULL) {
      img=gdk_image_new_bitmap(gtk_widget_get_visual(mainWindow),
			       (void*) hatch,16,4);
    }
    gdk_gc_set_function (gc_temp,GDK_AND);
    for(int x=0;x<width;x+=16){
      for(int y=0;y<height;y+=4){
	gdk_draw_image(pix,gc_temp,img,0,0,x,y,16,4);
      }
    }
    gdk_gc_set_function (gc_temp,GDK_COPY);
  }
  return pix;
};

static GdkPixmap* MakeGaijiPixmap(GAIJI *g,const DICCHAR& ch,int& width,int& height){
  int bwidth=-1;
  const byte* bitmap=NULL;
  int code=ch.Code();
  if (ch.Attr().hankaku) code|=0x10000;
  if (g!=NULL){
    GaijiChar* gaijiChar=g->Gaiji(code);
    //    if (gaijiChar==NULL) gaijiChar=g->Gaiji(code ^ 0x10000);
    if (gaijiChar!=NULL) {
      width=gaijiChar->Width();
      height=gaijiChar->Height();
      bwidth=((width-1)>>3)+1;
      bitmap=gaijiChar->Bitmap();
    } else {
      Debug::DebugOut(Debug::WARNINGS,"Gaiji Error. Code=%X\n",code);
    }
  } else {
    Debug::DebugOut(Debug::WARNINGS,"Gaiji Not Found. Code=%X\n",ch.Code());
  }
  if (bwidth<0 || bitmap==NULL){
    width=currentDict->charWidth;
    height=currentDict->charHeight+currentDict->charExtend;
    GdkPixmap* pix=NewPixmap(width,height,ch.Attr());
    gdk_draw_line(pix,gc_temp,0,0,width-1,0);
    gdk_draw_line(pix,gc_temp,0,0+height-1,width-1,height-1);
    gdk_draw_line(pix,gc_temp,0,0,0,0+height-1);
    gdk_draw_line(pix,gc_temp,width-1,0,0+width-1,0+height-1);
    gdk_draw_line(pix,gc_temp,0,0,width-1,height-1);
    gdk_draw_line(pix,gc_temp,0,height-1,width-1,0);
    return pix;
  } else {
    GdkPixmap* pix=NewPixmap(width+(ch.Attr().bold?1:0),
			     currentDict->charHeight+
			     currentDict->charExtend,ch.Attr());

    // Prepare for destroy_image
    char* buf=(char*)malloc(bwidth*height);
    memcpy(buf,bitmap,bwidth*height);

    GdkImage *img=
      gdk_image_new_bitmap(gtk_widget_get_visual(mainWindow),
    			   (void*) buf,width,height);


    gdk_gc_set_background(gc,&mainWindow->style->white);
    gdk_gc_set_function (gc,GDK_AND);
    gdk_draw_image(pix,gc,img,0,0,0,currentDict->gaijiOffset,width,height);

    if (ch.Attr().bold) {
      gdk_draw_image(pix,gc,img,0,0,1,
		     currentDict->gaijiOffset,width,height);
      width++;
    }
    height=currentDict->charHeight+currentDict->charExtend;
    gdk_gc_set_function (gc,GDK_COPY);
    gdk_image_destroy(img);
    if (ch.Attr().bold) width++;
    return pix;
  }
}

static GdkPixmap* Wider(GdkPixmap* psrc){

  GtkRequisition size;
  GtkWidget* pix=gtk_pixmap_new(psrc,NULL);
  gtk_widget_size_request(pix, &size);
  int w=size.width;
  int h=size.height;
  GdkPixmap* pdest=NewWhitePixmap(w*2,h);
  gdk_gc_set_function (gc_temp,GDK_COPY);
  psrc=GTK_PIXMAP(pix)->pixmap;
  for(int i=0;i<w;i++){
    gdk_draw_pixmap(pdest,gc_temp,psrc,i,0,i*2,0,1,h);
    gdk_draw_pixmap(pdest,gc_temp,psrc,i,0,i*2+1,0,1,h);
  }
  gtk_widget_destroy(pix);
  return pdest;
}

GdkPixmap* MakeCharPixmap(const DICCHAR& ch,int& width,int& height){
  GdkPixmap* pix=NULL;
  if (ch.Code()>0x8000){
    pix=MakeGaijiPixmap(currentGaiji,ch,width,height);
  } else {
    GdkFont* fnt;
    if (ch.Attr().subscript || ch.Attr().superscript){
      fnt=scriptFont;
    } else if (ch.Attr().bold){
      if (ch.Attr().italic){
	fnt=boldItalicFont;
      } else {
	fnt=boldFont;
      }
    } else {
      if (ch.Attr().italic){
	fnt=italicFont;
      } else {
	fnt=normalFont;
      }
    }

    NATIVE_STRING ich;
    if ((ch.Code() & 0xff00)==0x100) { // ISO-8859-1
      ich=char(ch.Code() & 0xff);
      //      setlocale(LC_ALL,"en_US.ISO8859-1");
      fnt=isoFont;
    } else {
      ich=_EC(DispChar(ch));
    }
    width=gdk_string_width(fnt,ich.c_str());
    height=currentDict->charHeight+currentDict->charExtend;
    pix=NewPixmap(width,height,ch.Attr());

    int h0=0;
    if (ch.Attr().superscript) h0=currentDict->charHeight/2;
    gdk_draw_string(pix,fnt,gc_temp,0,
		    currentDict->charHeight+currentDict->charOffset-h0,
		    ich.c_str());
    /*
    if ((ch.Code() & 0xff00)==0x100) { // ISO-8859-1
      setlocale(LC_ALL,"");
    }
    */
  }
  if (ch.Attr().underline!=0){
    int h=currentDict->charHeight+currentDict->charExtend-1;
    switch(ch.Attr().underline){
    case 1: // Normal Underline
      gdk_draw_line(pix,gc_temp,0,h,width-1,h);
      break;
    case 2: // Bold Underline
      gdk_draw_line(pix,gc_temp,0,h,width-1,h);
      gdk_draw_line(pix,gc_temp,0,h-1,width-1,h-1);
      break;
    case 3: // Wave Underline
      {
	for(int i=0;i<width;i+=4){
	  gdk_draw_line(pix,gc_temp,i,h,i+2,h-1);
	  gdk_draw_line(pix,gc_temp,i+2,h-2,i+3,h-1);
	}
      }
      break;
    case 4: // Dot 
      {
	gdk_draw_line(pix,gc_temp,width/2-1,1,width/2+1,1);
	gdk_draw_line(pix,gc_temp,width/2,0,width/2,2);
      }
      break;
    case 5: // Hatched 
    default: // No Underline
      break;
    }
  }
  if (ch.Attr().widechar){ 
    GdkPixmap* npix=Wider(pix);
    gdk_pixmap_unref(pix);
    pix=npix;
  }
  return pix;
}

extern gint TagJump(GtkWidget* widget,gpointer tag);

static DICCHAR jumpPointer[cMaxButton];
static int nButtons=0;

void ButtonInit(void){
  nButtons=0;
}

GtkWidget* MakePixmapFromDicText(const DICSTRING& str,bool em){
  GtkWidget* hbox=gtk_hbox_new(FALSE,0);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 0);

  DICCHAR ch;
  int width,height;

  int strlength=str.length();
  for(int i=0;i<strlength;i++){
    ch=str[i];
    if ((ch.Code()&0xff00)==0x1f00) continue;
    GdkPixmap* pix;
    if (ch.Code()==0xffff) {
      continue; // Ignore Multimedia Data in Entry
    } else {
      pix=MakeCharPixmap(ch,width,height);
    }
    if (pix!=NULL){
      if (em){
	gdk_draw_line(pix,gc_red,0,height-1,width-1,height-1);
      }
      GtkWidget* pixmap=gtk_pixmap_new(pix,NULL);
      
      gtk_box_pack_start (GTK_BOX (hbox), pixmap ,FALSE, TRUE, 0);
      gdk_pixmap_unref(pix);
    }
  }
  gtk_widget_show_all(hbox);
  return hbox;
}

int isClickableImage;
bool isInlineImage;

GtkWidget* clickableWidget;
std::vector<CLICKABLE_BOX> cBox;

gint ResultClick_callback(GtkWidget* widget,GdkEventButton* event){
  if (event->type!=GDK_BUTTON_PRESS) return FALSE;
  if (isClickableImage==-2 && widget==clickableWidget){
    int x=static_cast<guint>(event->x);
    int y=static_cast<guint>(event->y);
    int id=-1;
    DICCHAR ch;
    int cboxSize=cBox.size();
    for(int i=0;i<cboxSize;i++){
      if (cBox[i].x <= x && x<=cBox[i].x+cBox[i].w &&
	  cBox[i].y <= y && y<=cBox[i].y+cBox[i].h){
	id=cBox[i].ch.Attr().id;
	ch=cBox[i].ch;
	break;
      }
    }
    if (ch.Tag()==TAG(-1,-1)){
      if ((id & 0xf0)==0x80){
	CreateIndexSearchMenu(mainWindow,id);
      } else if ((id & 0xff)==0xff){
	CreateCombinedSearchMenu(mainWindow,id);
      } else {
	if (isIndexKey(id,currentDict->isEB)){
	  hist.Append(currentDict->IndexStart(id));
	  DisplayHonmon(currentDict->IndexStart(id));
	} else {
	  return FALSE;
	}
      }
    } else { // if (ch.Tag()!=TAG(0,0)) {
      if (ch.Attr().notText){
	DispatchMultiMedia(&ch);
      } else {
	hist.Append(ch.Tag());
	DisplayHonmon(ch.Tag());
      }
    } 
    return TRUE;
  }
  return FALSE;
}

gint EBMenuClick_callback(GtkWidget* widget,CLICKABLE_BOX* cbox){
  DICCHAR ch=cbox->ch;
  int id=ch.Attr().id;
  if (ch.Tag()==TAG(-1,-1)){
    if ((id & 0xf0)==0x80){
      CreateIndexSearchMenu(mainWindow,id);
    } else if ((id & 0xff)==0xff){
      CreateCombinedSearchMenu(mainWindow,id);
    } else {
      if (isIndexKey(id,currentDict->isEB)){
	hist.Append(currentDict->IndexStart(id));
	DisplayHonmon(currentDict->IndexStart(id));
      } else {
	return FALSE;
      }
    }
  } else if (ch.Tag()!=TAG(0,0)) {
    if (ch.Attr().notText){
      DispatchMultiMedia(&ch);
    } else {
      hist.Append(ch.Tag());
      DisplayHonmon(ch.Tag());
    }
  } else return FALSE;
  return TRUE;
}

GtkWidget* ArrowPixmap(){
  GdkBitmap *mask;
  GdkPixmap *pix;
  pix=gdk_pixmap_create_from_xpm_d(mainWindow->window,
				   &mask,
				   &mainWindow->style->bg[GTK_STATE_NORMAL],
				   arrow_xpm);
  GtkWidget* pixmap=gtk_pixmap_new(pix,mask);
  gdk_pixmap_unref(pix);
  gdk_bitmap_unref(mask);
  return pixmap;
}

GdkPixmap* DecodeControlCode(const DICSTRING& str,int& start,
			     int screenWidth,int cursorX,
			     int& w,int& h){
  const DICCHAR& ch=str[start];
  h=currentDict->charHeight+currentDict->charExtend;
  switch(ch.Code()){
  case 0x1f02:
    return NULL;
  case 0x1f03:
    start++;
    return NULL;
  case 0x1f10: // $BJ,3d6X;_(B
    {
      int chw,chh,x0=0;
      GdkPixmap* pi=NewPixmap(screenWidth,h,ch.Attr());
      for(start++;str[start].Code()!=0x1f11;start++){
	const DICCHAR& ch1=str[start];
	//	if ((ch1.Code()&0xff00)==0x1f00) continue;
	GdkPixmap* p=MakeCharPixmap(ch1,chw,chh);
	if (p!=NULL){
	  gdk_draw_pixmap(pi,gc,p,0,0,x0,0,chw,chh);
	  gdk_pixmap_unref(p);
	  x0+=chw;
	}
      }
      w=x0;
      GdkPixmap* pix=NewPixmap(w,h,ch.Attr());
      gdk_draw_pixmap(pix,gc,pi,0,0,0,0,w,h);
      gdk_pixmap_unref(pi);
      return pix;
    }
  case 0x1f11:  // Unreachable
    return NULL; 
  case 0x1f1A:  // Tab   or ignore
    {
      w=currentDict->charWidth*ch.Attr().id-cursorX;
      if (w<=8) w=8;
      return NewWhitePixmap(w,h);
    }
  case 0x1f1B:  //
    break;
  case 0x1f32:
  case 0x1f33:
  case 0x1f3f:
  case 0x1f42:
    if (isClickableImage==-1){
      CLICKABLE_BOX cb;
      cb.x=ch.Attr().id/10000;
      cb.y=ch.Attr().id%10000;
      cb.w=16;
      cb.h=16;
      if (cb.x>18) cb.x-=18; else cb.x=0;
      if (cb.y>8) cb.y-=8; else cb.y=0;
      Debug::DebugOut(Debug::MULTIMEDIA,
		      "Clickable Image (%d,%d) width=%d height=%d\n",
		      cb.x,cb.y,cb.w,cb.h);
      cb.ch=ch;
      cb.widget=ArrowPixmap();
      cBox.push_back(cb);
    }
    break;
  case 0x1f36:
    if (isClickableImage==-1){
      CLICKABLE_BOX cb;
      cb.x=ch.Tag().Block();
      cb.y=ch.Tag().Offset();
      cb.w=16;
      cb.h=16;
      if (cb.x>18) cb.x-=18; else cb.x=0;
      if (cb.y>8) cb.y-=8; else cb.y=0;
      cb.ch=ch;
      cb.ch.Tag(TAG(-1,-1));
      /*

      GdkBitmap *mask;
      GdkPixmap *pix;
      pix=gdk_pixmap_create_from_xpm_d(mainWindow->window,
			       &mask,
			       &mainWindow->style->bg[GTK_STATE_NORMAL],
			       arrow_xpm);
      cb.widget=gtk_pixmap_new(pix,mask);
      gdk_pixmap_unref(pix);
      gdk_bitmap_unref(mask);
      */
      cb.widget=ArrowPixmap();

      cBox.push_back(cb);
    }
    break;

  case 0x1f3c:
    return MakeInlinePixmap(ch,w,h);
  case 0x1f5c:
    return NULL;

  case 0x1f45:
    if (isClickableImage==0) isClickableImage=-1;
    isInlineImage=true;
    return NULL;
  case 0x1f65:
    isClickableImage=-2;  // Disable Next Clickable Image
    isInlineImage=false;
    start++;
    return NULL;

  case 0x1f4c:
    if (isClickableImage==0) isClickableImage=-1;
    isInlineImage=true;
    return NULL;
  case 0x1f6c:
    isClickableImage=-2;  // Disable Next Clickable Image
    isInlineImage=false;
    start++;
    return NULL;
  case 0x1f4B:
    //    DisplayHonmon(ch.Tag());
    break;
  case 0x1f4f:
    {
      BlockIO *pf=currentDict->BlockFile();
      TAG t=pf->Tell();
      pf->Seek(ch.Tag());
      if ((pf->GetWord() & 0xff00)==0x1100){
	CLICKABLE_BOX cb;
	pf->Read(NULL,4);
	cb.x=pf->GetBCD(4);
	cb.y=pf->GetBCD(4);
	cb.w=pf->GetBCD(4);
	cb.h=pf->GetBCD(4);
	pf->Read(NULL,12);
	cb.ch.Tag(pf->GetBCDTag());
	pf->Seek(cb.ch.Tag());
	if (pf->GetWord()==0x1f4b){
	  TAG t=pf->GetBCDTag();
	  if (pf->GetWord()==0x1f6b) {
	    cb.ch.Tag(t);
	  }
	}
	cBox.push_back(cb);
      }
      pf->Seek(t);
    }
    break;
  case 0x1fa2:
    gc_temp=gc_color[ch.Attr().color];
    break;
  case 0x1fa3:
    gc_temp=gc;
    break;
  default:
    break;
  }
  return NULL;
}

GtkWidget* MakeHonmonPixmap(int margin,const DICSTRING& str,int& start,
			    int screenWidth,int& stringWidth){

  GtkWidget* hbox=gtk_hbox_new(FALSE,0);
  GtkWidget* layout=NULL;
  gc_temp=gc;
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 0);
  int w,h;
  int cursorX=0;
  int kinsokuLevel=0,lastCode=-1;
  int leftMargin=0;
  

  cBox.clear();

  if (margin>1){
    w=currentDict->charWidth*(margin-1);
    GdkPixmap* pix=NewWhitePixmap(w,currentDict->charHeight);
    gtk_box_pack_start(GTK_BOX(hbox),gtk_pixmap_new(pix,NULL),FALSE,TRUE,0);
    gdk_pixmap_unref(pix);
    cursorX+=w;
    leftMargin=w;
  } 

  //  ATTR currentAttr=ATTR();
  TAG currentTag(0,0);
  GtkWidget* currentButton=NULL;
  GtkWidget* currentButtonBox=NULL;
  GdkPixmap* pix=NULL;
  GdkPixmap* basepix=NULL;

  int strLength=str.length(); 			    
  for(;start<strLength;start++){
    int current=start;
    const DICCHAR& ch=str[start];
    TAG chTag=ch.Tag();
    if ((ch.Code()&0xff00)==0x1f00) {
      pix=DecodeControlCode(str,start,screenWidth,cursorX,w,h);
      if (ch.Code()==0x1f3c) {
	basepix=pix;
	gdk_pixmap_ref(pix);
	chTag=TAG(0,0);
      }
      if (pix==NULL) continue;
    } else if (ch.Code()==0xffff && ch.Attr().notText){
      //      if (!currentDict->autoImageDisplay) continue;
      Debug::DebugOut(Debug::TEMP,"Code=%04X, Subcode=%04X\n",
		      ch.Code(),ch.Attr().id);
      if (!isInlineImage) continue;
      pix=MakeInlinePixmap(ch,w,h);
      basepix=pix;
      gdk_pixmap_ref(pix);
      chTag=TAG(0,0);
      if (pix==NULL) {
	continue;
      }
    } else {
      if (ch.Code()==0xffff) continue;
      pix=MakeCharPixmap(ch,w,h);
    };
    if (chTag!=currentTag && currentButton!=NULL){
      gtk_box_pack_start (GTK_BOX (hbox), currentButton ,FALSE, TRUE, 0);
      currentButton=NULL;
      currentTag=TAG(0,0);
    }

    if (chTag!=TAG(0,0)){
      gdk_draw_line(pix,gc_red,0,h-1,w-1,h-1);
    }
    if (screenWidth-cursorX<w && cursorX>leftMargin){ 
      // $B9TF,6XB'(B1$BJ8;zL\(B
      if (kinsokuLevel==0 && currentDict->IsKinsoku(0,ch.Code())) {
	kinsokuLevel++;
      // $B9TF,6XB'(B2$BJ8;zL\(B
      } else if (kinsokuLevel==1 && currentDict->IsKinsoku(1,ch.Code())) {
	kinsokuLevel++;
      // $B9TKv6XB'$*$^$1(B1$BJ8;zL\(B
      } else if (kinsokuLevel<2 && currentDict->IsKinsoku(2,lastCode)) {
	kinsokuLevel=2;
      } else {
	start=current;
	gdk_pixmap_unref(pix);
	break;
      }
    } 
    cursorX+=w;
    lastCode=ch.Code();
    GtkWidget* pixmap=gtk_pixmap_new(pix,NULL);
    if (chTag!=TAG(0,0)){
      if (chTag!=currentTag || currentButton==NULL) {
	currentButton=gtk_button_new();
	currentTag=chTag;
	currentButtonBox=gtk_hbox_new(FALSE,0);
	gtk_container_add(GTK_CONTAINER(currentButton), currentButtonBox);
	gtk_container_set_border_width (GTK_CONTAINER (currentButton), 0);
	jumpPointer[nButtons]=ch;
	gtk_signal_connect (GTK_OBJECT(currentButton),"clicked",
			    GTK_SIGNAL_FUNC(TagJump),jumpPointer+nButtons);
	nButtons++;
      }
      gtk_container_add(GTK_CONTAINER(currentButtonBox), pixmap);
    } else {
      if (isClickableImage==-1){
	layout=gtk_layout_new(NULL,NULL);
	GtkRequisition size;
	gtk_widget_size_request(pixmap, &size);

	gtk_widget_set_usize(layout,size.width,size.height);
	clickableWidget=layout;
	gtk_signal_connect(GTK_OBJECT(layout),"event",
			   GTK_SIGNAL_FUNC(ResultClick_callback),NULL);

	gtk_layout_put(GTK_LAYOUT(layout), pixmap, 0, 0);
	//	gtk_container_add(GTK_CONTAINER(layout),pixmap);
	// gtk_box_pack_start (GTK_BOX (hbox), layout,FALSE, TRUE, 0);
      } else {
	gtk_box_pack_start (GTK_BOX (hbox), pixmap ,FALSE, TRUE, 0);
      }
    }
    gdk_pixmap_unref(pix);
  }

  if (layout!=NULL){
    int cboxSize=cBox.size();
    for(int i=0;i<cboxSize;i++){
      GtkWidget* button=gtk_button_new();
      gtk_button_set_relief(GTK_BUTTON(button),GTK_RELIEF_HALF);
      gtk_container_set_border_width(GTK_CONTAINER(button), 0);
      /*
      GTK_WIDGET(button)->style->klass->xthickness=0;
      GTK_WIDGET(button)->style->klass->ythickness=0;
      */
      if (cBox[i].widget==NULL){
	GdkPixmap* subpix=gdk_pixmap_new(mainWindow->window,
				      cBox[i].w,cBox[i].h,-1);
	gdk_gc_set_function (gc_temp,GDK_COPY);
	if (basepix!=NULL){
	  gdk_draw_pixmap(subpix,gc,basepix,
			  cBox[i].x,cBox[i].y,0,0,cBox[i].w,cBox[i].h);
	}
	gtk_container_add(GTK_CONTAINER(button),gtk_pixmap_new(subpix,NULL));
	gdk_pixmap_unref(subpix);
      } else {
	gtk_container_add(GTK_CONTAINER(button),cBox[i].widget);
      }
      //      gdk_pixmap_unref(pix);
      gtk_signal_connect(GTK_OBJECT(button),"clicked",
			 GTK_SIGNAL_FUNC(EBMenuClick_callback),&cBox[i].ch);
      gtk_widget_show(button);
      gtk_layout_put(GTK_LAYOUT(layout),button,cBox[i].x,cBox[i].y);
    }
    if (basepix!=NULL) {
      gdk_pixmap_unref(basepix);
      basepix=NULL;
    }
    gtk_widget_show_all(layout);
    gtk_widget_destroy(hbox);
    return layout;
  }
  

  if (currentButton!=NULL){
    gtk_box_pack_start (GTK_BOX (hbox), currentButton ,FALSE, TRUE, 0);
  }
//  newline:
  stringWidth=cursorX; 
  gtk_widget_show_all(hbox);
  if (basepix!=NULL) {
    gdk_pixmap_unref(basepix);
    basepix=NULL;
  }
  return hbox;
}

void DisplayHonmon(const TAG& t){
  int width;
  gc_temp=gc;
  Debug::DebugOut(Debug::HONMON_CONTROL,"Start: %d:%d\n",t.Block(),t.Offset());

  isClickableImage=0;
  isInlineImage=false;

  static GtkWidget* resultList=NULL;
  if (resultList!=NULL){
    gtk_widget_destroy(resultList);
    resultList=NULL;
  } 
  if (t==TAG(0,0)) return;

  ButtonInit();
  resultList=gtk_vbox_new(FALSE,1);
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW(resultText),
					 resultList);
  gtk_widget_show_all(resultText);

  width=resultText->allocation.width-32;

  if (width<100) width=100;
  DICLINE* dl=new DICLINE(currentDict,t);

  int lineCount=0;

  while(dl->GetLine()){
    int start=0;
    int length=dl->txt().length();
    int textWidth;
    while(start<length){
      Debug::DebugOut(Debug::TEMP,"start=%d/length=%d\n",start,length);
      GtkWidget* pixmap=MakeHonmonPixmap(dl->Margin(),dl->txt(),start,
					 width,textWidth);
      if (pixmap==NULL) {
	Debug::DebugOut(Debug::WARNINGS,"Pixmap Error\n");
	break;
      }
      gtk_box_pack_start(GTK_BOX(resultList),pixmap,FALSE,FALSE,0);
      gtk_widget_show(pixmap);
      lineCount++;
      if (lineCount>500) goto ret;
    }
  }
  //  gtk_window_set_default_size(GTK_WINDOW(resultList),width,-2);

  //  gdk_window_resize(mainWindow->window,w,h);
 ret:
  delete dl;
}

