// -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 2001 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#include "def.h"
#include "filename.h"
#include <vector>
#include <string>
#include <cstdio>
#include <cstring>

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif
#ifdef HAVE_DIRENT_H
#include <dirent.h>
#endif

std::string FILENAME::FuzzyFind(std::string orgpath){
  std::string path=orgpath;
  std::vector<std::string> dirs;
  std::vector<std::string>::iterator dirp;
  std::string base;

  struct stat s_stat;

  try {
    if (stat(orgpath.c_str(),&s_stat)==0) throw(orgpath);
    if (path.length()==0) throw(path);

    int pt=path.find('/');
    if (pt<0) {
      if (path[0]=='~') {
	pt=path.length();
	path+='/';
      } else {
	path="./"+path;
	pt=1;
      }
    }
    base=path.substr(0,pt);
    path=path.substr(pt+1);
    if (base=="") {
      base="/";
    } else if (base[0]=='~') {
      if (base.length()>1) {
	std::string user=base.substr(1);
	struct passwd* pw=getpwnam(user.c_str());
	if (pw==NULL) throw(orgpath);
	base=pw->pw_dir;
      } else {
	base=getenv("HOME");
      }
    }
    while(path.length()>0){
      pt=path.find('/');
      std::string path1;
      if (pt<0) {
	path1=path;
	path="";
      } else {
	path1=path.substr(0,pt);
	path=path.substr(pt+1);
      }
      
      DIR* dir=opendir(base.c_str());
      if (dir==NULL) throw(orgpath);
      dirs.clear();
      if (base=="/") base="";
      struct dirent *ent;
      while((ent=readdir(dir))!=NULL){
	dirs.push_back(ent->d_name);
      }
      closedir(dir);
      for(dirp=dirs.begin();dirp!=dirs.end();++dirp){
	if ((*dirp)==path1){
	  base+="/"+path1;
	  break;
	}
      }
      if (dirp==dirs.end()){
	for(dirp=dirs.begin();dirp!=dirs.end();++dirp){
	  if (strcasecmp((*dirp).c_str(),path1.c_str())==0 ||
	      strcasecmp((*dirp).c_str(),(path1+".").c_str())==0){
	    base+="/"+(*dirp);
	    break;
	  }
	  // for ISO-9660 CD-ROM
	  int pt=(*dirp).find(';');
	  if (pt>0 && 
	      (strcasecmp((*dirp).substr(0,pt).c_str(),path1.c_str())==0 ||
	       strcasecmp((*dirp).substr(0,pt).c_str(),(path1+'.').c_str())==0
	       )) {
	    base+="/"+(*dirp);
	    break;
	  }
	}
	if (dirp==dirs.end()){
	  return base+"/"+path1;
	  //	  throw("X"+orgpath);
	}
      }
    }
Debug::DebugOut(Debug::LOW_LEVEL_FILE,
		"FileName='%s' from '%s'\n",base.c_str(),orgpath.c_str());
    return base;
  } catch (std::string opath) {
    Debug::DebugOut(Debug::LOW_LEVEL_FILE,
		    "Not Found: FileName='%s'\n",orgpath.c_str());
    return orgpath;
  }
}

static char buf[PATH_MAX+1];
const char* FuzzyFindFile(const char* path){
  strncpy(buf,FILENAME::FuzzyFind(path).c_str(),PATH_MAX);
  buf[PATH_MAX]=0;
  return buf;
}

const char* FileName(const char* path){
  std::string fname=FILENAME::FuzzyFind(path);
  if (fname.length()==0) return NULL;
  for(;;){
    int pt=fname.find('/');
    if (pt<0) break;
    fname=fname.substr(pt+1);
  }
  strncpy(buf,fname.c_str(),PATH_MAX);
  buf[PATH_MAX]=0;
  return buf;
}

