/**
 * @file logging.c Debug logging support
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include <libgalago/galago.h>
#include <stdio.h>
#include "logging.h"

static char *_filename = NULL;
static gboolean _verbose = FALSE;
static FILE *_fp = NULL;
static guint _domain_log_handler_id = 0;
static guint _glib_log_handler_id = 0;
static guint _galago_log_handler_id = 0;

static const gchar *
get_log_level_name(GLogLevelFlags log_level)
{
	if (log_level & G_LOG_LEVEL_ERROR)
		return "ERROR";
	else if (log_level & G_LOG_LEVEL_CRITICAL)
		return "CRITICAL";
	else if (log_level & G_LOG_LEVEL_WARNING)
		return "WARNING";
	else if (log_level & G_LOG_LEVEL_MESSAGE)
		return "MESSAGE";
	else if (log_level & G_LOG_LEVEL_INFO)
		return "INFO";
	else if (log_level & G_LOG_LEVEL_DEBUG)
		return "DEBUG";
	else
		return "UNKNOWN";
}

static void
output_log_info(FILE *fp, const gchar *log_domain, GLogLevelFlags log_level,
				const gchar *message)
{
	if (log_domain != NULL)
	{
		fprintf(fp, "%s: %s: %s\n",
				log_domain, get_log_level_name(log_level), message);
	}
	else
	{
		fprintf(fp, "%s: %s\n", get_log_level_name(log_level), message);
	}

	fflush(fp);
}

static void
log_handler(const gchar *log_domain, GLogLevelFlags log_level,
			const gchar *message, gpointer user_data)
{
	if (_fp != NULL)
		output_log_info(_fp, log_domain, log_level, message);

	if (_verbose)
		output_log_info(stderr, log_domain, log_level, message);
}

void
galagod_logging_set_filename(const char *filename)
{
	g_return_if_fail(*filename != '\0');

	if (_filename != NULL)
		g_free(_filename);

	_filename = (filename != NULL ? g_strdup(filename) : NULL);
}

void
galagod_logging_set_verbose(gboolean verbose)
{
	_verbose = verbose;
}

void
galagod_logging_start(void)
{
	if (_filename != NULL)
	{
		if ((_fp = fopen(_filename, "w")) == NULL)
			g_error("Can't open log file %s for writing", _filename);
	}

	_domain_log_handler_id =
		g_log_set_handler(NULL, G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL |
						  G_LOG_FLAG_RECURSION,
						  log_handler, NULL);

	_glib_log_handler_id =
		g_log_set_handler("GLib", G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL |
						  G_LOG_FLAG_RECURSION,
						  log_handler, NULL);

	_galago_log_handler_id =
		g_log_set_handler("Galago", G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL |
						  G_LOG_FLAG_RECURSION,
						  log_handler, NULL);
}

void
galagod_logging_stop(void)
{
	if (_fp != NULL)
	{
		fclose(_fp);
		_fp = NULL;
	}

	if (_filename != NULL)
	{
		g_free(_filename);
		_filename = NULL;
	}

	g_log_remove_handler(NULL, _domain_log_handler_id);
	g_log_remove_handler(NULL, _glib_log_handler_id);
	g_log_remove_handler(NULL, _galago_log_handler_id);
}
