/***************************************************************************
                           filterwidgetimpl.cpp
                           --------------------
    begin                : Sun Jul 4 2004
    copyright            : (C) 2004 by Dirk Ziegelmeier
    email                : dziegel@gmx.de
 ***************************************************************************/

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <klocale.h>
#include <kdebug.h>
#include <kconfig.h>
#include <kpushbutton.h>

#include <qcheckbox.h>
#include <qlistview.h>
#include <qlineedit.h>
#include <qcombobox.h>
#include <qstringlist.h>

#include "cfgdata.h"
#include "kdetv.h"
#include "pluginfactory.h"
#include "filterwidgetimpl.h"
#include "filtermanager.h"
#include "kdetvfilterplugin.h"
#include "pluginconfigwidget.h"


class FilterListItem : public QCheckListItem
{
public:
    FilterListItem(QListView* parent, PluginDesc* d)
        : QCheckListItem(parent, d->name, QCheckListItem::CheckBox)
    {
        setText(1, d->comment);
        setText(2, d->author);
        pd = d;
        setOn(pd->enabled);
    }

    virtual ~FilterListItem()
    {
    }

    PluginDesc *pd;
};

/***************************************************************************/

FilterWidgetImpl::FilterWidgetImpl(QWidget* parent, Kdetv* ktv, ConfigData* cfg)
    : FilterWidget(parent),
      SettingsDialogPage(i18n("Image Filters"),
                         i18n("Configure Image Filters"),
                         "filter"),
      _ktv(ktv),
      _cfg(cfg)
{
    connect(_DICombo, SIGNAL(highlighted(const QString&)),
            this, SLOT(diSelectionChanged()));
    connect(_DICombo, SIGNAL(activated(const QString&)),
            this, SLOT(diSelectionChanged()));
    connect(_DIConfigure, SIGNAL(clicked()),
            this, SLOT(configureDIPlugin()));

    connect(_PPListView, SIGNAL(selectionChanged()),
            this, SLOT(ppSelectionChanged()));
    connect(_PPListView, SIGNAL(doubleClicked(QListViewItem *)),
            this, SLOT(configurePPPlugin()));
    connect(_PPConfigure, SIGNAL(clicked()),
            this, SLOT(configurePPPlugin()));
}

FilterWidgetImpl::~FilterWidgetImpl()
{
}

void FilterWidgetImpl::setup()
{
    _cfg->pluginConfigHandle()->setGroup("Filter Plugins");

    _PPListView->clear();
    for (PluginDesc *plug = _ktv->pluginFactory()->postProcessPlugins().first();
         plug;
         plug = _ktv->pluginFactory()->postProcessPlugins().next()) {
        new FilterListItem(_PPListView, plug);
    }
    ppSelectionChanged();

    QStringList deinterlacers;
    for (PluginDesc *plug = _ktv->pluginFactory()->filterPlugins().first();
         plug;
         plug = _ktv->pluginFactory()->filterPlugins().next()) {
        deinterlacers.append(plug->name);
    }
    deinterlacers.sort();

    _DICombo->clear();
    _DICombo->insertStringList(deinterlacers);

    for (PluginDesc *plug = _ktv->pluginFactory()->filterPlugins().first();
         plug;
         plug = _ktv->pluginFactory()->filterPlugins().next()) {
        if(plug->enabled) {
            _DICombo->setCurrentText(plug->name);
            break;
        }
    }
    diSelectionChanged();
}

void FilterWidgetImpl::apply()
{
    bool rescanNecessary = false;

    _cfg->pluginConfigHandle()->setGroup("Filter Plugins");

    for (PluginDesc* p = _ktv->pluginFactory()->filterPlugins().first();
         p;
         p = _ktv->pluginFactory()->filterPlugins().next()) {
        bool e = p->enabled;

        if(p->name == _DICombo->currentText()) {
            p->enabled = true;
        } else {
            p->enabled = false;
        }

        if(e != p->enabled) {
            rescanNecessary = true;
        }

        _cfg->pluginConfigHandle()->writeEntry(p->name + "-" + p->author, p->enabled);
    }

    for (QListViewItemIterator it(_PPListView);
         it.current();
         it++) {
        FilterListItem* item = static_cast<FilterListItem*>(it.current());
        if (item->pd->enabled != item->isOn()) {
            rescanNecessary = true;
        }
        item->pd->enabled = item->isOn();
        _cfg->pluginConfigHandle()->writeEntry(item->pd->name + "-" + item->pd->author, item->pd->enabled);
    }

    if(rescanNecessary) {
        _ktv->stop();
        _ktv->filterManager()->scanPlugins();
        _ktv->start();
    }
}

void FilterWidgetImpl::defaults()
{
}

void FilterWidgetImpl::diSelectionChanged()
{
    PluginDesc* d = NULL;
    for (PluginDesc *plug = _ktv->pluginFactory()->filterPlugins().first();
         plug;
         plug = _ktv->pluginFactory()->filterPlugins().next()) {
        if(plug->name == _DICombo->currentText()) {
            d = plug;
            break;
        }
    }

    if(d) {
        _DIAuthor->setText(d->author);
        _DIAuthor->home(false);
        _DIDescription->setText(d->comment);
        _DIDescription->home(false);
        _DIConfigure->setEnabled(d->configurable);
    } else {
        _DIAuthor->setText("");
        _DIDescription->setText("");
        _DIConfigure->setEnabled(false);
    }
}

void FilterWidgetImpl::ppSelectionChanged()
{
    FilterListItem *item = static_cast<FilterListItem *>(_PPListView->selectedItem());
    if ( !item ) {
        _PPConfigure->setEnabled(false);
        return;
    }
    _PPConfigure->setEnabled(item->pd->configurable);
}

void FilterWidgetImpl::configurePPPlugin()
{
    FilterListItem* i = static_cast<FilterListItem*>(_PPListView->selectedItem());
    if (!i) return;
    PluginDesc* pd = i->pd;
    if (!pd->configurable) return;

    bool previousState = pd->enabled;
    pd->enabled        = true;
    KdetvPluginBase* p = _ktv->pluginFactory()->getPostProcessPlugin(pd);
    pd->enabled        = previousState;

    PluginConfigWidget w(0L, "Postprocess Plugin Config Dialog", true);
    w.setConfigWidget(pd->name, p->configWidget(&w, "Postprocess Configuration Widget"));
    if (w.exec() == QDialog::Accepted) {
        p->saveConfig();
    }

    p->destroy();
}

void FilterWidgetImpl::configureDIPlugin()
{
    PluginDesc* pd = NULL;
    for (PluginDesc *plug = _ktv->pluginFactory()->filterPlugins().first();
         plug;
         plug = _ktv->pluginFactory()->filterPlugins().next()) {
        if(plug->name == _DICombo->currentText()) {
            pd = plug;
            break;
        }
    }
    if(!pd) return;
    if(!pd->configurable) return;

    bool previousState = pd->enabled;
    pd->enabled        = true;
    KdetvPluginBase* p = _ktv->pluginFactory()->getFilterPlugin(pd);
    pd->enabled        = previousState;

    PluginConfigWidget w(0L, "Filter Plugin Config Dialog", true);
    w.setConfigWidget(pd->name, p->configWidget(&w, "Filter Configuration Widget"));
    if (w.exec() == QDialog::Accepted) {
        p->saveConfig();
    }

    p->destroy();
}

#include "filterwidgetimpl.moc"
