/* 
 * Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
 *
 * This file is part of Rast.
 * See the file COPYING for redistribution information.
 *
 */

#include <stdlib.h>

#include "rast/xmlrpc_client.h"

static int memory_error(int retcode);
static void print_error(rast_error_t *error);

int
main(int argc, char **argv)
{
    apr_pool_t *pool;
    char *server_url, *query;
    rast_db_t *db;
    rast_search_option_t *options;
    rast_result_t *result;
    rast_error_t *error;
    int i;
    const char *properties[] = {
        "filename",
        "title",
        "last_modified",
        "filesize"
    };

    if (argc < 3) {
        fprintf(stderr, "usage: %s <server_url> <query>\n", argv[0]);
        return 1;
    }
    server_url = argv[1];
    query = argv[2];

    apr_initialize();
    atexit(apr_terminate);
    error = rast_xmlrpc_client_initialize();
    if (error != RAST_OK) {
        return 1;
    }
    atexit(rast_xmlrpc_client_finalize);

    apr_pool_create_ex(&pool, NULL, memory_error, NULL);

    error = rast_xmlrpc_client_open(&db, server_url, RAST_DB_RDONLY, NULL,
                                    pool);
    if (error != RAST_OK) {
        print_error(error);
        return 1;
    }

    options = rast_search_option_create(pool);
    options->num_items = 10;
    options->need_summary = 1;
    options->summary_nchars = 200;
    options->properties = properties;
    options->num_properties = 4;

    error = rast_db_search(db, query, options, &result, pool);
    if (error != RAST_OK) {
        rast_db_close(db);
        print_error(error);
        return 1;
    }

    printf("hit_count: %d\n", result->hit_count);
    puts("");
    for (i = 0; i < result->num_items; i++) {
        printf("doc_id: %d\n", result->items[i]->doc_id);
        printf("score: %d\n", result->items[i]->score);
        printf("filename: %s\n",
               rast_value_string(result->items[i]->properties + 0));
        printf("title: %s\n",
               rast_value_string(result->items[i]->properties + 1));
        printf("last_modified: %s\n",
               rast_value_date(result->items[i]->properties + 2));
        printf("filesize: %u\n",
               rast_value_uint(result->items[i]->properties + 3));
        puts("summary:");
        fwrite(result->items[i]->summary, 1, result->items[i]->summary_nbytes,
               stdout);
        puts("");
        puts("");
    }

    rast_db_close(db);

    return 0;
}

static int
memory_error(int retcode)
{
    abort();
    return -1;  /* prevent compiler warnings */
}

static void
print_error(rast_error_t *error)
{
#ifdef RAST_DEBUG
    fprintf(stderr, "%s:%d: %s\n", error->file, error->line, error->message);
#else
    fprintf(stderr, "error: %s\n", error->message);
#endif
    rast_error_destroy(error);
}

/* vim: set filetype=c sw=4 expandtab : */
