# Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
#
# This file is part of Rast.
# See the file COPYING for redistribution information.
#

require "test/unit"
require "find"
require "yaml"

require "make-variable"
require "rast/database-generatable"

class CommandTest < Test::Unit::TestCase
  include Rast::DatabaseGeneratable

  CREATE_COMMAND = File.join(MakeVariable::TOP_BUILDDIR, "src", "rast-create")
  REGISTER_COMMAND = File.join(MakeVariable::TOP_BUILDDIR, "src",
                               "rast-register")
  SEARCH_COMMAND = File.join(MakeVariable::TOP_BUILDDIR, "scripts",
                             "rast-search")
  DELETE_COMMAND = File.join(MakeVariable::TOP_BUILDDIR, "src", "rast-delete")
  OPTIMIZE_COMMAND = File.join(MakeVariable::TOP_BUILDDIR, "src",
                               "rast-optimize")
  STATUS_COMMAND = File.join(MakeVariable::TOP_BUILDDIR, "src",
                             "rast-stat")

  def test_create
    db_name = generate_db_name
    result = system(CREATE_COMMAND, db_name)
    assert_equal(true, result)
    status = YAML.load(`#{STATUS_COMMAND} #{db_name}`)
    assert_equal("uri", status["properties"][0]["name"])
    assert_equal("RAST_TYPE_STRING", status["properties"][0]["type"])
    assert_equal(["RAST_PROPERTY_FLAG_SEARCH"],
                 status["properties"][0]["flags"])
    assert_equal("last_modified", status["properties"][1]["name"])
    assert_equal("RAST_TYPE_DATETIME", status["properties"][1]["type"])
    assert_equal(["RAST_PROPERTY_FLAG_SEARCH"],
                 status["properties"][1]["flags"])
    assert_equal(2, status["properties"].length)

    db_name = generate_db_name
    result = system(CREATE_COMMAND,
                    "--property=uri:string:search,full-text-search", db_name)
    assert_equal(true, result)
    status = YAML.load(`#{STATUS_COMMAND} #{db_name}`)
    assert_equal("last_modified", status["properties"][0]["name"])
    assert_equal("RAST_TYPE_DATETIME", status["properties"][0]["type"])
    assert_equal(["RAST_PROPERTY_FLAG_SEARCH"],
                 status["properties"][0]["flags"])
    assert_equal("uri", status["properties"][1]["name"])
    assert_equal("RAST_TYPE_STRING", status["properties"][1]["type"])
    assert_equal([
                   "RAST_PROPERTY_FLAG_SEARCH",
                   "RAST_PROPERTY_FLAG_FULL_TEXT_SEARCH",
                 ], status["properties"][1]["flags"])
    assert_equal(2, status["properties"].length)

    db_name = generate_db_name
    result = system(CREATE_COMMAND,
                    "--property=last_modified:uint:search",
                    "--property=user-defined:date:search",
                    "--property=uri:string:search,text-search", db_name)
    assert_equal(true, result)
    status = YAML.load(`#{STATUS_COMMAND} #{db_name}`)
    assert_equal("last_modified", status["properties"][0]["name"])
    assert_equal("RAST_TYPE_UINT", status["properties"][0]["type"])
    assert_equal(["RAST_PROPERTY_FLAG_SEARCH"],
                 status["properties"][0]["flags"])
    assert_equal("user-defined", status["properties"][1]["name"])
    assert_equal("RAST_TYPE_DATE", status["properties"][1]["type"])
    assert_equal(["RAST_PROPERTY_FLAG_SEARCH"],
                 status["properties"][1]["flags"])
    assert_equal("uri", status["properties"][2]["name"])
    assert_equal("RAST_TYPE_STRING", status["properties"][2]["type"])
    assert_equal([
                   "RAST_PROPERTY_FLAG_SEARCH",
                   "RAST_PROPERTY_FLAG_TEXT_SEARCH",
                 ], status["properties"][2]["flags"])
    assert_equal(3, status["properties"].length)

    db_name = generate_db_name
    result = system(CREATE_COMMAND,
                    "--property=uri:string", db_name)
    assert_equal(false, result)

    db_name = generate_db_name
    result = system(CREATE_COMMAND,
                    "--property=uri:uint:search", db_name)
    assert_equal(false, result)
  end

  def test_register
    db_name = generate_db_name
    filename_1 = File.join(MakeVariable::TOP_SRCDIR, "README.en")
    system(CREATE_COMMAND, "--preserve-text", db_name)
    s = `#{REGISTER_COMMAND} --verbose #{db_name} #{filename_1}`
    assert_match(/^doc_id : 1$/, s)
    assert_match(/^uri : file:\/\/#{filename_1}$/, s)
    s = `#{SEARCH_COMMAND} rast #{db_name}`
    assert_match(/^doc_id : 1$/, s)

    db_name = generate_db_name
    system(CREATE_COMMAND, db_name)
    result = system(REGISTER_COMMAND, db_name)
    assert_equal(false, result)
  end

  def test_search
    db_name = generate_db_name
    filename_1 = File.join(MakeVariable::TOP_SRCDIR, "README.en")
    filename_2 = File.join(MakeVariable::TOP_SRCDIR, "bootstrap")
    filename_3 = File.join(MakeVariable::TOP_SRCDIR, "Makefile.am")
    dirname_1 = File.join(MakeVariable::TOP_SRCDIR, "m4")

    system(CREATE_COMMAND, "--preserve-text", db_name)
    system(REGISTER_COMMAND, db_name, filename_1)
    assert_match(/^doc_id : 1$/, `#{SEARCH_COMMAND} rast #{db_name}`)

    system(REGISTER_COMMAND, "--mime-type=text/plain", db_name, filename_2,
           filename_3)
    s = `#{SEARCH_COMMAND} automake #{db_name}`
    assert_match(/^doc_id : 2$/, s)
    assert_no_match(/^doc_id : 1$/, s)
    assert_no_match(/^doc_id : 3$/, s)
    s = `#{SEARCH_COMMAND} subdir #{db_name}`
    assert_match(/^doc_id : 3$/, s)
    assert_no_match(/^doc_id : 1$/, s)
    assert_no_match(/^doc_id : 2$/, s)

    dirname_1 = generate_db_name
    Dir.mkdir(dirname_1)
    filenames_1 = []

    name = File.basename(filename_1)
    filename = File.join(dirname_1, name)
    FileUtils.cp(filename_1, filename, {:preserve => true})
    filenames_1.push(filename)

    name = File.basename(filename_2)
    filename = File.join(dirname_1, name)
    FileUtils.cp(filename_2, filename, {:preserve => true})
    filenames_1.push(filename)

    sub_dir = File.join(dirname_1, "sub_dir")
    Dir.mkdir(sub_dir)
    name = File.basename(filename_3)
    filename = File.join(sub_dir, name)
    FileUtils.cp(filename_3, filename, {:preserve => true})
    filenames_1.push(filename)

    system(REGISTER_COMMAND, "--mime-type=text/plain", db_name, dirname_1)
    Find.find(dirname_1) do |name|
      if File.file?(name)
        s = `#{SEARCH_COMMAND} 'uri = file://#{name}' #{db_name}`
        assert_equal(1, s.scan(/^doc_id : /).length,
                     "the check file is `#{name}'")
      end
    end

    name = File.basename(filename_2)
    filename = File.join(dirname_1, name)
    FileUtils.chmod(0644, filename)
    File.open(filename, "w") do |f|
      f.write("updated text")
    end

    system(REGISTER_COMMAND, "--mime-type=text/plain", db_name, filename)
    assert_match(/^doc_id : 7$/, `#{SEARCH_COMMAND} updated #{db_name}`)
    s = `#{SEARCH_COMMAND} automake #{db_name}`
    assert_match(/^doc_id : 2$/, s)
    assert_no_match(/^doc_id : 5$/, s)
  end

  def test_delete
    db_name = generate_db_name
    filename_1 = File.join(MakeVariable::TOP_SRCDIR, "README.en")

    system(CREATE_COMMAND, "--preserve-text", db_name)
    system(REGISTER_COMMAND, db_name, filename_1)
    assert_match(/^doc_id : 1$/, `#{SEARCH_COMMAND} full-text #{db_name}`)

    system(DELETE_COMMAND, db_name, filename_1)

    s = `#{SEARCH_COMMAND} full-text #{db_name}`
    assert_no_match(/^doc_id : 1$/, s)
  end

  def test_optimize
    db_name = generate_db_name
    filename_1 = File.join(MakeVariable::TOP_SRCDIR, "README.en")
    filename_2 = File.join(MakeVariable::TOP_SRCDIR, "README.ja")

    system(CREATE_COMMAND, "--preserve-text", "--property=title:string:search",
           db_name)
    system(REGISTER_COMMAND, db_name, filename_1)
    system(REGISTER_COMMAND, db_name, filename_2)

    s = `#{SEARCH_COMMAND} rast #{db_name}`
    assert_match(/^doc_id : 1$/, s)
    assert_match(/^doc_id : 2$/, s)

    system(OPTIMIZE_COMMAND, db_name)

    s = `#{SEARCH_COMMAND} rast #{db_name}`
    assert_match(/^doc_id : 1$/, s)
    assert_match(/^doc_id : 2$/, s)
  end
end
