# Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
#
# This file is part of Rast.
# See the file COPYING for redistribution information.
#

require "test/unit"

require "rast_test"
require "rast/filter"

module Rast
  module Filter
    load_mime_filter("powerpoint.rb")
    load_mime_filter("html.rb")

    class ApplicationPowerPointTest < Test::Unit::TestCase
      def test_invoke
        invoke_test_simple
        invoke_test_error
      end

      def invoke_test_simple
        ppt_filter = SpoofFilter.new("UTF-8")
        brigade = Brigade.new

        path = File.join(MakeVariable::TOP_SRCDIR, "tests", "data",
                         "filter", "ja", "powerpoint.ppt")
        file = File.open(path)
        bucket = FileBucket.new(file)
        brigade.insert_tail(bucket)
        brigade.insert_tail(EOSBucket.new)
        filter_module = ApplicationPowerPoint.new
        filter_module.invoke(ppt_filter, brigade,
                             "application/vnd.ms-powerpoint")

        assert_equal("text/html", ppt_filter.passed[0].mime_type)
        assert_equal(1, ppt_filter.passed.length)
        assert_equal("著者", ppt_filter.properties["author"])
        assert_equal("タイトル", ppt_filter.properties["title"])
        assert_equal(2, ppt_filter.properties.length)

        html_filter = SpoofFilter.new("UTF-8")
        filter_module = TextHtml.new
        filter_module.invoke(html_filter, ppt_filter.passed[0].brigade,
                             "text/html")

        buf = ""
        html_filter.passed[0].brigade.each do |bucket|
          buf.concat(bucket.read)
        end

        assert_match(/^\s*Microsoft PowerPoint テスト\s*$/m, buf)
        assert_match(/^\s*これは MS PowerPoint ファイルです\s*$/m, buf)
        assert_match(/^\s*1ページ目タイトル\s*$/m, buf)
        assert_match(/^\s*1ページ目本文\s*$/m, buf)
        assert_match(/^\s*2ページ目タイトル\s*$/m, buf)
        assert_match(/^\s*2ページ目本文\s*$/m, buf)
        assert_equal("text/plain", html_filter.passed[0].mime_type)
        assert_equal(1, html_filter.passed.length)
      end

      def invoke_test_error
        filter = SpoofFilter.new("UTF-8")
        brigade = Brigade.new
        brigade.insert_tail(TransientBucket.new("invalid type file"))
        brigade.insert_tail(EOSBucket.new)
        filter_module = ApplicationPowerPoint.new
        assert_raise(RastError) do
          filter_module.invoke(filter, brigade,
                               "application/vnd.ms-powerpoint")
        end
      end
    end
  end
end
