# Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
#
# This file is part of Rast.
# See the file COPYING for redistribution information.
#

require "test/unit"
require "fileutils"
require "test-utility"

require "rast_test"
require "rast/database-generatable"

module Rast
  class QueryTest < Test::Unit::TestCase
    include DatabaseGeneratable

    def test_parse
      assert_equal('(null)', Query.parse("").inspect)
      assert_equal('(null)', Query.parse(" ").inspect)
      assert_equal('"ruby"', Query.parse("ruby").inspect)
      assert_equal('"ruby"', Query.parse("ruby").inspect)
      assert_equal('"ruby"', Query.parse("ruby ").inspect)
      assert_equal('"ruby"', Query.parse(" ruby ").inspect)
      assert_equal('"ruby"', Query.parse(" ruby").inspect)
      assert_equal('"ruby"', Query.parse("ruby ").inspect)
      assert_equal('"ruby"', Query.parse("Ruby").inspect)
      assert_equal('"ruby"', Query.parse("ruby\0perl").inspect)
      assert_equal('"ruby 2.0"', Query.parse('"ruby 2.0"').inspect)
      assert_equal('"ruby 2.0"', Query.parse('"Ruby  2.0"').inspect)
      assert_equal('"ruby 2.0"', Query.parse(%Q'"rubY\t2.0"').inspect)
      assert_equal('"&|!():=<>"', Query.parse('"&|!():=<>"').inspect)
      assert_equal('(! "ruby" "perl")', Query.parse("ruby ! perl").inspect)
      assert_equal('(! "ruby" "perl")', Query.parse("ruby !perl").inspect)
      assert_equal('"ruby!perl"', Query.parse("ruby!perl").inspect)
      assert_equal('(& "ruby!" "perl")', Query.parse("ruby! perl").inspect)
      assert_equal('(& (! "ruby" "perl") "python")',
                   Query.parse("ruby ! perl python").inspect)
      assert_equal('(& "ruby" (! "perl" "python"))',
                   Query.parse("ruby perl ! python").inspect)
      assert_equal('(& "ruby" "perl")', Query.parse("ruby perl").inspect)
      assert_equal('(& "ruby" "perl")', Query.parse("ruby　perl").inspect)
      assert_equal('(& "ruby" "perl" "python")',
                   Query.parse("ruby perl python").inspect)
      assert_equal('(& "ruby" "perl")', Query.parse("ruby & perl").inspect)
      assert_equal('"ruby&perl"', Query.parse("ruby&perl").inspect)
      assert_equal('(& "ruby&" "perl")', Query.parse("ruby& perl").inspect)
      assert_equal('(& "ruby" "perl")', Query.parse("ruby &perl").inspect)
      assert_equal('(& "ruby" "perl" "python")',
                   Query.parse("ruby & perl & python").inspect)
      assert_equal('(| "ruby" "perl")', Query.parse("ruby | perl").inspect)
      assert_equal('(| "ruby" "perl" "python")',
                   Query.parse("ruby | perl | python").inspect)
      assert_equal('(| (& "ruby" "perl") "python")',
                   Query.parse("ruby & perl | python").inspect)
      assert_equal('(& (| "ruby" "perl") "python")',
                   Query.parse("ruby | perl & python").inspect)
      assert_equal('(& (| "ruby" "perl") "python" "number<" "10")',
                   Query.parse("ruby | perl & python & number< 10").inspect)
      assert_equal('(& (| "ruby" "perl") "python" (range "number" nil f "10" f))',
                   Query.parse("ruby | perl & python & number < 10").inspect)

      # paren
      assert_equal('(| "ruby" (& "perl" "python"))',
                   Query.parse("ruby | ( perl python )").inspect)
      assert_equal('(| "ruby" (& "perl" "python"))',
                   Query.parse("ruby | (perl python )").inspect)
      assert_equal('(| "ruby" (& "perl" "python"))',
                   Query.parse("ruby | ( perl python)").inspect)
      assert_equal('(| "ruby" (& "perl" "python"))',
                   Query.parse("ruby | (perl python)").inspect)

      assert_equal('(| "ruby" (| "perl" "python"))',
                   Query.parse("ruby | ( perl | python )").inspect)
      assert_equal('(| "ruby" (& "perl" "python"))',
                   Query.parse("ruby | ( perl & python )").inspect)

      # prop op
      assert_equal('(& "title:" "ruby")', Query.parse("title: ruby").inspect)
      assert_equal('(& "title:" "ruby")', Query.parse("title: Ruby").inspect)
      assert_equal('(title: "ruby")', Query.parse("title : ruby").inspect)
      assert_equal('(title: "ruby")', Query.parse("title : Ruby").inspect)
      assert_equal('(& "title:" "ruby 2.0")',
                   Query.parse('title: "Ruby  2.0"').inspect)
      assert_equal('(title: "ruby 2.0")',
                   Query.parse('title : "Ruby  2.0"').inspect)
      assert_equal('(& "title=" "ruby")',
                   Query.parse("title= ruby").inspect)
      assert_equal('(range "title" "ruby" t "ruby" t)', 
                   Query.parse("title = ruby").inspect)
      assert_equal('"title=ruby"', 
                   Query.parse("title=ruby").inspect)
      assert_equal('(& "date<" "ruby")',
                   Query.parse("date< ruby").inspect)
      assert_equal('(range "date" nil f "ruby" f)',
                   Query.parse("date <ruby").inspect)
      assert_equal('(range "date" nil f "ruby" f)', 
                   Query.parse("date < ruby").inspect)
      assert_equal('(range "date" "ruby" f nil f)',
                   Query.parse("date > ruby").inspect)
      assert_equal('(range "date" "ruby" t nil f)',
                   Query.parse("date >= ruby").inspect)
      assert_equal('(range "date" nil f "ruby" t)', 
                   Query.parse("date <= ruby").inspect)

      assert_equal('"1<size<10"',
                   Query.parse("1<size<10").inspect)
      assert_equal('(& "1<" (range "size" nil f "10" f))',
                   Query.parse("1< size < 10").inspect)
      assert_equal('(range "size" "1" f "10" f)',
                   Query.parse("1 <size < 10").inspect)
      assert_equal('(& (range "1" nil f "size<" f) "10")',
                   Query.parse("1 < size< 10").inspect)
      assert_equal('(range "size" "1" f "10" f)',
                   Query.parse("1 < size <10").inspect)
      assert_raise(RastError) do
        Query.parse("1<size < 10")
      end
      assert_equal('(& (range "1" nil f "size<" f) "10")',
                   Query.parse("1 <size< 10").inspect)
      assert_equal('(range "1" nil f "size<10" f)',
                   Query.parse("1 < size<10").inspect)
      assert_equal('(& "1<size<" "10")',
                   Query.parse("1<size< 10").inspect)
      assert_equal('(range "1" nil f "size<10" f)',
                   Query.parse("1 <size<10").inspect)
      assert_equal('(range "size" "1" f "10" f)',
                   Query.parse("1 < size < 10").inspect)
      assert_equal('(range "size" "1" t "10" f)',
                   Query.parse("1 <= size < 10").inspect)
      assert_equal('(range "size" "1" f "10" t)',
                   Query.parse("1 < size <= 10").inspect)
      assert_equal('(range "size" "1" t "10" t)',
                   Query.parse("1 <= size <= 10").inspect)
      assert_equal('(range "size" "1" f "10" f)',
                   Query.parse("10 > size > 1").inspect)
      assert_equal('(range "size" "1" f "10" t)',
                   Query.parse("10 >= size > 1").inspect)
      assert_equal('(range "size" "1" t "10" f)',
                   Query.parse("10 > size >= 1").inspect)
      assert_equal('(range "size" "1" t "10" t)',
                   Query.parse("10 >= size >= 1").inspect)
      assert_equal('"date<"',
                   Query.parse("date<").inspect)
      assert_raise(Rast::RastError) do
        Query.parse("< date")
      end
      assert_raise(Rast::RastError) do
        Query.parse("date <")
      end

      assert_equal('(range "filename" "abc>def" f nil f)',
                   Query.parse("filename > abc>def").inspect)
      assert_equal('(range "filename" "abc>def" f "ghi%jkl" t)',
                   Query.parse("abc>def < filename <= ghi%jkl").inspect)

      assert_equal('"a-b"', Query.parse("a-b").inspect)
      assert_equal('"a+b"', Query.parse("a+b").inspect)
      assert_equal('"a*b"', Query.parse("a*b").inspect)
      assert_equal('"a.b"', Query.parse("a.b").inspect)
      assert_equal('"a#b"', Query.parse("a#b").inspect)
      assert_equal('"a~b"', Query.parse("a~b").inspect)

      assert_equal('(! "ruby" "perl")', Query.parse("ruby -perl").inspect)
      assert_equal('(! "ruby" "perl")', Query.parse("ruby - perl").inspect)
      assert_equal('(& "ruby-" "perl")', Query.parse("ruby- perl").inspect)
      assert_equal('"ruby-perl"', Query.parse("ruby-perl").inspect)
      assert_equal('(& "ruby" (! "perl" "python"))',
                   Query.parse("ruby perl -python").inspect)

      assert_raise(Rast::RastError) do
        Query.parse("title/../../abc = foobar")
      end
    end

    def test_exec
      exec_test_null
      exec_test_term
      exec_test_property_pe
      exec_test_property_range
      exec_test_and
      exec_test_or
      exec_test_not
    end

    def exec_test_null
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        assert_raise(RastError) do
          Query.parse("").exec(db)
        end
      end
    end

    def exec_test_term
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("昨日は雨天なり", {})
        db.register("本日は晴天なり", {})
        db.register("明日は雪っす", {})
      end
      LocalDB.open(db_name) do |db|
        result = Query.parse("晴天").exec(db)
        assert_equal("晴天", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(1, result.candidates[0].terms[0].count)
        assert_equal(3, result.candidates[0].terms[0].pos)
        assert_equal(1, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        result = Query.parse("天なり").exec(db)
        assert_equal("天なり", result.terms[0].term)
        assert_equal(2, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(1, result.candidates[0].terms[0].count)
        assert_equal(4, result.candidates[0].terms[0].pos)
        assert_equal(1, result.candidates[0].terms.length)
        assert_equal(2, result.candidates[1].doc_id)
        assert_equal(1, result.candidates[1].terms[0].count)
        assert_equal(4, result.candidates[1].terms[0].pos)
        assert_equal(1, result.candidates[1].terms.length)
        assert_equal(2, result.candidates.length)
      end
    end

    def exec_test_property_pe
      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => true,
            "full_text_search" => false,
          },
        ]
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("", "title" => "abcd")
        db.register("", "title" => "bcde")
        db.register("", "title" => "cdef")
      end
      LocalDB.open(db_name) do |db|
        result = Query.parse("title : abc").exec(db)
        assert_equal(0, result.terms.length)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        result = Query.parse("title : cde").exec(db)
        assert_equal(0, result.terms.length)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(3, result.candidates[1].doc_id)
        assert_equal(0, result.candidates[1].terms.length)
        assert_equal(2, result.candidates.length)

        assert_raise(RastError) do
          Query.parse("tit : cde").exec(db)
        end
      end

      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
        ]
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("", "filename" => "abcd")
      end
      LocalDB.open(db_name) do |db|
        assert_raise(RastError) do
          Query.parse("filename : abcd").exec(db)
        end
      end
    end

    def exec_test_property_range
      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "string",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "datetime",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "number",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
        ]
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        properties = {
          "string" => "foo",
          "date" => "2005-02-21",
          "datetime" => "2005-06-16T18:16:29+0900",
          "number" => 100000,
        }
        db.register("", properties)
        properties = {
          "string" => "bar",
          "date" => "2005-01-01",
          "datetime" => "2005-06-16T18:16:50+0900",
          "number" => 1,
        }
        db.register("", properties)
        properties = {
          "string" => "bar",
          "date" => "2005-01-01",
          "datetime" => "2005-06-16T18:16:50",
          "number" => 1,
        }
        db.register("", properties)
      end
      LocalDB.open(db_name) do |db|
        result = Query.parse("string = foo").exec(db)
        assert_equal(0, result.terms.length)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        result = Query.parse('date = "2005-02-21"').exec(db)
        assert_equal(0, result.terms.length)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        result = Query.parse('datetime = "2005-06-16T18:16:29"').exec(db)
        assert_equal(0, result.terms.length)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        result = Query.parse("number = 100000").exec(db)
        assert_equal(0, result.terms.length)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        result = Query.parse("string = bar").exec(db)
        assert_equal(0, result.terms.length)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(3, result.candidates[1].doc_id)
        assert_equal(0, result.candidates[1].terms.length)
        assert_equal(2, result.candidates.length)

        result = Query.parse('date = "2005-01-01"').exec(db)
        assert_equal(0, result.terms.length)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(3, result.candidates[1].doc_id)
        assert_equal(0, result.candidates[1].terms.length)
        assert_equal(2, result.candidates.length)

        result = Query.parse('datetime = "2005-06-16T18:16:50"').exec(db)
        assert_equal(0, result.terms.length)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(3, result.candidates[1].doc_id)
        assert_equal(0, result.candidates[1].terms.length)
        assert_equal(2, result.candidates.length)

        result = Query.parse("number = 1").exec(db)
        assert_equal(0, result.terms.length)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(3, result.candidates[1].doc_id)
        assert_equal(0, result.candidates[1].terms.length)
        assert_equal(2, result.candidates.length)

        assert_raise(RastError) do
          Query.parse("numb = 1").exec(db)
        end

        result = Query.parse("number > 1").exec(db)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        result = Query.parse("number > 0").exec(db)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(2, result.candidates[1].doc_id)
        assert_equal(0, result.candidates[1].terms.length)
        assert_equal(3, result.candidates[2].doc_id)
        assert_equal(0, result.candidates[2].terms.length)
        assert_equal(3, result.candidates.length)

        result = Query.parse("number > 100").exec(db)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        result = Query.parse("number > 100000").exec(db)
        assert_equal(0, result.candidates.length)

        result = Query.parse("number > 100000000").exec(db)
        assert_equal(0, result.candidates.length)

        assert_raise(RastError) do
          Query.parse("numb > 1").exec(db)
        end

        result = Query.parse("number >= 100").exec(db)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        result = Query.parse("number >= 1").exec(db)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(2, result.candidates[1].doc_id)
        assert_equal(0, result.candidates[1].terms.length)
        assert_equal(3, result.candidates[2].doc_id)
        assert_equal(0, result.candidates[2].terms.length)
        assert_equal(3, result.candidates.length)

        result = Query.parse("number >= 100000").exec(db)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        result = Query.parse("number >= 10000000").exec(db)
        assert_equal(0, result.candidates.length)

        assert_raise(RastError) do
          Query.parse("numb >= 1").exec(db)
        end
        assert_raise(RastError) do
          Query.parse("number >= 1ab2").exec(db)
        end

        result = Query.parse("number < 0").exec(db)
        assert_equal(0, result.candidates.length)

        result = Query.parse("number < 1").exec(db)
        assert_equal(0, result.candidates.length)

        result = Query.parse("number < 100").exec(db)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(3, result.candidates[1].doc_id)
        assert_equal(0, result.candidates[1].terms.length)
        assert_equal(2, result.candidates.length)

        result = Query.parse("number < 100000").exec(db)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(3, result.candidates[1].doc_id)
        assert_equal(0, result.candidates[1].terms.length)
        assert_equal(2, result.candidates.length)

        result = Query.parse("number < 10000000").exec(db)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(2, result.candidates[1].doc_id)
        assert_equal(0, result.candidates[1].terms.length)
        assert_equal(3, result.candidates[2].doc_id)
        assert_equal(0, result.candidates[2].terms.length)
        assert_equal(3, result.candidates.length)

        result = Query.parse("number <= 0").exec(db)
        assert_equal(0, result.candidates.length)

        result = Query.parse("number <= 1").exec(db)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(3, result.candidates[1].doc_id)
        assert_equal(0, result.candidates[1].terms.length)
        assert_equal(2, result.candidates.length)

        result = Query.parse("number <= 100").exec(db)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(3, result.candidates[1].doc_id)
        assert_equal(0, result.candidates[1].terms.length)
        assert_equal(2, result.candidates.length)

        result = Query.parse("number <= 100000").exec(db)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(2, result.candidates[1].doc_id)
        assert_equal(0, result.candidates[1].terms.length)
        assert_equal(3, result.candidates[2].doc_id)
        assert_equal(0, result.candidates[2].terms.length)
        assert_equal(3, result.candidates.length)

        result = Query.parse("number <= 10000000").exec(db)
        assert_equal(1, result.candidates[0].doc_id)
        assert_equal(0, result.candidates[0].terms.length)
        assert_equal(2, result.candidates[1].doc_id)
        assert_equal(0, result.candidates[1].terms.length)
        assert_equal(3, result.candidates[2].doc_id)
        assert_equal(0, result.candidates[2].terms.length)
        assert_equal(3, result.candidates.length)

        options = {
          "encoding" => "utf8",
          "properties" => [
            {
              "name" => "type",
              "type" => Rast::PROPERTY_TYPE_STRING,
              "search" => true,
              "text_search" => true,
              "full_text_search" => false,
            },
            {
              "name" => "num",
              "type" => Rast::PROPERTY_TYPE_UINT,
              "search" => true,
              "text_search" => false,
              "full_text_search" => false,
            },
          ],
        }
        db_name = generate_db_name
        LocalDB.create(db_name, options)
        LocalDB.open(db_name) do |db|
          db.register("aiueo",
                      {"type" => "en", "num" => 5})
          db.register("qwertyasdfgh",
                      {"type" => "en", "num" => 12})
          db.register("明日は晴天なり",
                      {"type" => "ja", "num" => 7})
          db.register("foo",
                      {"type" => "en", "num" => 20})
        end
        LocalDB.open(db_name) do |db|
          result = Query.parse("type = en & num < 10").exec(db)
          assert_equal(1, result.candidates[0].doc_id)
          assert_equal(0, result.candidates[0].terms.length)
          assert_equal(1, result.candidates.length)
        end
      end

      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "type",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "num",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("aiueo", {"type" => "en", "num" => 5})
      end
      LocalDB.open(db_name) do |db|
        assert_raise(RastError) do
          Query.parse("type=en & num < 10 & num > 1").exec(db)
        end
      end

      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => true,
            "full_text_search" => false,
          },
        ]
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("", {"title" => "hello"})
      end
      LocalDB.open(db_name) do |db|
        assert_raise(RastError) do
          Query.parse("title > bye").exec(db)
        end
      end
    end

    def exec_test_and
      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
        ]
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("昨日は雨天なり", {"filename" => "ame.txt"})
        db.register("本日は晴天なり", {"filename" => "hare.txt"})
        db.register("明日は雪っす", {"filename" => "yuki.txt"})
      end
      LocalDB.open(db_name) do |db|
        result = Query.parse("天なり & 本日").exec(db)
        assert_equal("天なり", result.terms[0].term)
        assert_equal(2, result.terms[0].doc_count)
        assert_equal("本日", result.terms[1].term)
        assert_equal(1, result.terms[1].doc_count)
        assert_equal(2, result.terms.length)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(1, result.candidates[0].terms[0].count)
        assert_equal(4, result.candidates[0].terms[0].pos)
        assert_equal(1, result.candidates[0].terms[1].count)
        assert_equal(0, result.candidates[0].terms[1].pos)
        assert_equal(2, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        result = Query.parse("天なり & 本日 & は晴").exec(db)
        assert_equal("天なり", result.terms[0].term)
        assert_equal(2, result.terms[0].doc_count)
        assert_equal("本日", result.terms[1].term)
        assert_equal(1, result.terms[1].doc_count)
        assert_equal("は晴", result.terms[2].term)
        assert_equal(1, result.terms[2].doc_count)
        assert_equal(3, result.terms.length)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(1, result.candidates[0].terms[0].count)
        assert_equal(4, result.candidates[0].terms[0].pos)
        assert_equal(1, result.candidates[0].terms[1].count)
        assert_equal(0, result.candidates[0].terms[1].pos)
        assert_equal(1, result.candidates[0].terms[2].count)
        assert_equal(2, result.candidates[0].terms[2].pos)
        assert_equal(3, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        result = Query.parse("天なり & filename = \"hare.txt\"").exec(db)
        assert_equal("天なり", result.terms[0].term)
        assert_equal(2, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(1, result.candidates[0].terms[0].count)
        assert_equal(4, result.candidates[0].terms[0].pos)
        assert_equal(1, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)
      end
    end

    def exec_test_or
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("昨日は雨天なり", {})
        db.register("本日は晴天なり", {})
        db.register("明日は雪っす", {})
      end
      LocalDB.open(db_name) do |db|
        result = Query.parse("晴天 | 雨天").exec(db)
        assert_equal("晴天", result.terms[0].term)
        assert_equal(1,     result.terms[0].doc_count)
        assert_equal("雨天", result.terms[1].term)
        assert_equal(1,     result.terms[1].doc_count)
        assert_equal(2,     result.terms.length)
        assert_equal(1,     result.candidates[0].doc_id)
        assert_equal(1,     result.candidates[0].terms[0].count)
        assert_equal(3, result.candidates[0].terms[0].pos)
        assert_equal(2,     result.candidates[1].doc_id)
        assert_equal(1,     result.candidates[1].terms[0].count)
        assert_equal(3, result.candidates[1].terms[0].pos)
        assert_equal(2,     result.candidates.length)

        result = Query.parse("晴天 | 雨天 | 雪").exec(db)
        assert_equal("晴天", result.terms[0].term)
        assert_equal(1,     result.terms[0].doc_count)
        assert_equal("雨天", result.terms[1].term)
        assert_equal(1,     result.terms[1].doc_count)
        assert_equal("雪", result.terms[2].term)
        assert_equal(1,     result.terms[2].doc_count)
        assert_equal(3,     result.terms.length)
        assert_equal(1,     result.candidates[0].doc_id)
        assert_equal(1,     result.candidates[0].terms[0].count)
        assert_equal(3, result.candidates[0].terms[0].pos)
        assert_equal(2,     result.candidates[1].doc_id)
        assert_equal(1,     result.candidates[1].terms[0].count)
        assert_equal(3, result.candidates[1].terms[0].pos)
        assert_equal(3,     result.candidates[2].doc_id)
        assert_equal(1,     result.candidates[2].terms[0].count)
        assert_equal(3, result.candidates[2].terms[0].pos)
        assert_equal(3,     result.candidates.length)

        result = Query.parse("本日 | 晴天").exec(db)
        assert_equal("晴天", result.terms[1].term)
        assert_equal(1, result.terms[1].doc_count)
        assert_equal(2, result.terms.length)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(1, result.candidates[0].terms[0].count)
        assert_equal(0, result.candidates[0].terms[0].pos)
        assert_equal(1, result.candidates[0].terms[1].count)
        assert_equal(3, result.candidates[0].terms[1].pos)
        assert_equal(1, result.candidates.length)
      end
    end

    def exec_test_not
      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
        ]
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("昨日は雨天なり", {"filename" => "ame.txt"})
        db.register("本日は晴天なり", {"filename" => "hare.txt"})
        db.register("明日は雪っす", {"filename" => "yuki.txt"})
      end
      LocalDB.open(db_name) do |db|
        result = Query.parse("天なり ! 昨日").exec(db)
        assert_equal("天なり", result.terms[0].term)
        assert_equal(2, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(1, result.candidates[0].terms[0].count)
        assert_equal(4, result.candidates[0].terms[0].pos)
        assert_equal(1, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        result = Query.parse("日は ! 昨日 ! 明日").exec(db)
        assert_equal("日は", result.terms[0].term)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(1, result.candidates[0].terms[0].count)
        assert_equal(1, result.candidates[0].terms[0].pos)
        assert_equal(1, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        query = "天なり ! filename = \"ame.txt\""
        result = Query.parse(query).exec(db)
        assert_equal("天なり", result.terms[0].term)
        assert_equal(2, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(1, result.candidates[0].terms[0].count)
        assert_equal(4, result.candidates[0].terms[0].pos)
        assert_equal(1, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)

        query = "日は ! ( 昨日 ! 本日 | 明日 )"
        result = Query.parse(query).exec(db)
        assert_equal("日は", result.terms[0].term)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.candidates[0].doc_id)
        assert_equal(1, result.candidates[0].terms[0].count)
        assert_equal(1, result.candidates[0].terms[0].pos)
        assert_equal(1, result.candidates[0].terms.length)
        assert_equal(1, result.candidates.length)
      end
    end

    def test_optimize
      assert_equal('(null)', Query.parse("").optimize.inspect)
      assert_equal('"ruby"', Query.parse("ruby").optimize.inspect)
      assert_equal('(range "size" "10" f "100" f)',
                   Query.parse("size > 10 size < 100").optimize.inspect)
      assert_equal('(& "ruby" (range "size" "10" f "100" f))',
                   Query.parse("ruby size > 10 size < 100").optimize.inspect)
    end
  end
end
