/***************************************************************************
    smb4kcore  -  The main core class of Smb4K.
                             -------------------
    begin                : Do Apr 8 2004
    copyright            : (C) 2004-2008 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KCORE_H
#define SMB4KCORE_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <QObject>
#include <QList>

// KDE includes
#include <kdemacros.h>

// application specific includes
#include <smb4kscanner.h>
#include <smb4kmounter.h>
#include <smb4kbookmarkhandler.h>
#include <smb4kprint.h>
#include <smb4ksynchronizer.h>
#include <smb4kpreviewer.h>
#include <smb4ksearch.h>
#include <smb4ksudowriterinterface.h>

// forward declarations
class Smb4KWorkgroup;
class Smb4KHost;
class Smb4KShare;
class Smb4KCorePrivate;


/**
 * This is the top-level core class. It inherits most of the core classes.
 *
 * @author Alexander Reinholdt <dustpuppy@users.berlios.de>
 */

class KDE_EXPORT Smb4KCore : public QObject
{
  Q_OBJECT

  friend class Smb4KCorePrivate;

  public:
    /**
     * This enumeration determines with which application the mount point
     * of the current mounted share is to be opened.
     */
    enum OpenWith { FileManager,
                    Konsole };

    /**
     * Returns a static pointer to this class.
     */
    static Smb4KCore *self();

    /**
     * Returns the state the scanner is in.
     */
    static int scannerState() { return self()->m_scanner_state; }

    /**
     * Returns the state the mounter is in.
     */
    static int mounterState() { return self()->m_mounter_state; }

    /**
     * Returns the state the print object is in.
     */
    static int printState() { return self()->m_print_state; }

    /**
     * Returns the state the synchronizer is in.
     */
    static int synchronizerState() { return self()->m_syncer_state; }

    /**
     * Returns TRUE if one of the core classes is doing something.
     */
    static bool isRunning();

    /**
     * Returns the current state the core is in.
     */
    static int currentState() { return self()->m_current_state; }

    /**
     * Returns a pointer to the scanner object.
     */
    static Smb4KScanner *scanner() { return self()->m_scanner; }

    /**
     * Returns a pointer to the mounter object.
     */
    static Smb4KMounter *mounter() { return self()->m_mounter; }

    /**
     * Returns a pointer to the bookmark handler object.
     */
    static Smb4KBookmarkHandler *bookmarkHandler() { return self()->m_bookmarkHandler; }

    /**
     * Returns a pointer to the printer handler object.
     */
    static Smb4KPrint *print() { return self()->m_print; }

    /**
     * Returns a pointer to the synchronizer object.
     */
    static Smb4KSynchronizer *synchronizer() { return self()->m_synchronizer; }

    /**
     * Returns a pointer to the previewer object.
     */
    static Smb4KPreviewer *previewer() { return self()->m_previewer; }

    /**
     * Returns a pointer to the search object.
     */
    static Smb4KSearch *search() { return self()->m_search; }

    /**
     * Returns a pointer to the Smb4KSudoWriterInterface object.
     */
    static Smb4KSudoWriterInterface *sudoWriter() { return Smb4KSudoWriterInterface::self(); }

    /**
     * Aborts any action of the core.
     */
    static void abort();

    /**
     * Open the mount point of a share. Which application is used is determined by
     * the value of @p openWith and - maybe - by settings that were defined by the
     * user.
     *
     * @param share         The share that is to be opened.
     *
     * @param openWith      Integer of type Smb4KCore::OpenWith. Determines with which
     *                      application the share should be opened.
     */
    static void open( Smb4KShare *share,
                      OpenWith openWith = FileManager );

    /**
     * This function initializes the core classes.
     */
    void init();

  signals:
    /**
     * This signal is emitted, if one of the core objects
     * starts or stops running.
     */
    void runStateChanged();

  protected slots:
    /**
     * This slot sets the run state of the scanner.
     *
     * @param state         The run state the scanner is in. For definition of the
     *                      run states see smb4kdefs.h header file.
     */
    void slotSetScannerState( int state );

    /**
     * This slot sets the run state of the mounter.
     *
     * @param state         The run state the mounter is in. For definition of the
     *                      run states see smb4kdefs.h header file.
     */
    void slotSetMounterState( int state );

    /**
     * This slot sets the run state of the synchronizer.
     *
     * @param state         The run state the synchronizer is in. For definition of the
     *                      run states see smb4kdefs.h header file.
     */
    void slotSetSynchronizerState( int state );

    /**
     * This slot sets the run state of the print handler.
     *
     * @param state         The run state the print handler is in. For definition of the
     *                      run states see smb4kdefs.h header file.
     */
    void slotSetPrinterHandlerState( int state );

    /**
     * This slot sets the run state for the previewer.
     *
     * @param state         The state the previewer is in. For definition of the
     *                      run states see smb4kdefs.h header file.
     */
    void slotSetPreviewerState( int state );

    /**
     * This slot sets the run state for the search process.
     *
     * @param state         The state the search is in. For definition of the
     *                      tun states see smb4kdefs.h header file.
     */
    void slotSetSearchState( int state );

    /**
     * This slot is connected to the QApplication::aboutToQuit() signal. It is invoked
     * when the application is shut down by the KDE logout or by pressing CTRL+Q, etc.
     */
    void slotAboutToQuit();

  private:
    /**
     * The constructor.
     */
    Smb4KCore();

    /**
     * The destructor.
     */
    ~Smb4KCore();

    /**
     * The scanner object.
     */
    Smb4KScanner *m_scanner;

    /**
     * The mounter object.
     */
    Smb4KMounter *m_mounter;

    /**
     * The bookmark handler object.
     */
    Smb4KBookmarkHandler *m_bookmarkHandler;

    /**
     * The printer handler object.
     */
    Smb4KPrint *m_print;

    /**
     * The synchronizer object.
     */
    Smb4KSynchronizer *m_synchronizer;

    /**
     * The previewer object.
     */
    Smb4KPreviewer *m_previewer;

    /**
     * The search object
     */
    Smb4KSearch *m_search;

    /**
     * The state the sanner is in.
     */
    int m_scanner_state;

    /**
     * The state the mounter is in.
     */
    int m_mounter_state;

    /**
     * Holds the current state.
     */
    int m_current_state;

    /**
     * The state the printer handler is in.
     */
    int m_print_state;

    /**
     * The state the synchronizer is in.
     */
    int m_syncer_state;

    /**
     * The state the previewer is in.
     */
    int m_previewer_state;

    /**
     * The state the search is in.
     */
    int m_search_state;

    /**
     * This function is used to set the current state of the core.
     *
     * @param state         One of the states defined in smb4kdefs.h
     */
    void setCurrentState( int state );

    /**
     * Searches for the needed programs and emits an error
     * if mandatory ones are missing.
     */
    void searchPrograms();

    /**
     * Set default values for settings that depend on the system Smb4K is
     * running on and that have to be set dynamically.
     */
    void setDefaultSettings();
};

#endif
