/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <libxfcegui4/libxfcegui4.h>
#include <libintl.h>
#include <config.h>

#include "menubar.h"
#include "window.h"
#include "node.h"
#include "about_dialog.h"
#include "open_dialog.h"
#include "save_dialog.h"
#include "save.h"
#include "debug.h"
#include "entry_edit_dialog.h"
#include "treeview.h"
#include "string_edit_dialog.h"
#include "statusbar.h"

static void file_new_event_handler(struct xfbib *xfbib)
{
	_DEBUG(("file_new_event_handler"));
	free_list(xfbib->list);
	xfbib->filename = NULL;
	xfbib->list = NULL;
	update_treeview(xfbib);
	gtk_window_set_title(GTK_WINDOW(xfbib->window.window), create_title(xfbib));
	statusbar_push((GtkStatusbar *) xfbib->window.statusbar, 0, "New file.");
}

static void file_open_event_handler (struct xfbib *xfbib)
{
	_DEBUG(("file_open_event_handler"));
	show_open_dialog(xfbib);
}

static void file_save_event_handler (struct xfbib *xfbib)
{
	_DEBUG(("file_save_event_handler"));
	if (xfbib->filename == NULL) {
		/* Use save as*/
		_DEBUG(("No file open, opening save as dialog"));
		show_save_dialog(xfbib);
	} else {
		_DEBUG(("Saving data"));
		save(xfbib);
	}
}

static void file_saveas_event_handler (struct xfbib *xfbib)
{
	_DEBUG(("file_saveas_event_handler"));
	show_save_dialog(xfbib);
}

static void file_quit_event_handler (struct xfbib *xfbib)
{
	_DEBUG(("file_quit_event_handler"));
	free_list(xfbib->list);
	gtk_main_quit();
}

static void entry_add_event_handler (struct xfbib *xfbib)
{
	_DEBUG(("entry_add_event_handler"));
	show_entry_edit_dialog(xfbib, NULL);
}

static void entry_remove_event_handler (struct xfbib *xfbib)
{
	_DEBUG(("entry_remove_event_handler"));
	struct node *node;
	GtkTreeSelection *selection;
	GtkTreeModel     *model;
	GtkTreeIter       iter;

	/* This will only work in single or browse selection mode! */

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(xfbib->window.main_treeview));
	if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
		gtk_tree_model_get (model, &iter, COLUMN_ENTRY, &node, -1);
		remove_node(&(xfbib->list), node);
	} else {
		_DEBUG(("No row selected"));
	}
	update_treeview(xfbib);
}

static void variable_add_event_handler (struct xfbib *xfbib)
{
	_DEBUG(("variable_add_event_handler"));
	show_string_edit_dialog(xfbib, NULL);
}

static void variable_remove_event_handler (struct xfbib *xfbib)
{
	_DEBUG(("variable_remove_event_handler"));
	struct node *node;
	GtkTreeSelection *selection;
	GtkTreeModel     *model;
	GtkTreeIter       iter;

	/* This will only work in single or browse selection mode! */

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(xfbib->window.variable_treeview));
	if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
		gtk_tree_model_get (model, &iter, VAR_COLUMN_STRING, &node, -1);
		remove_node(&(xfbib->list), node);
	} else {
		_DEBUG(("No row selected"));
	}
	update_treeview(xfbib);
}

static void view_show_toolbar_event_handler (GtkCheckMenuItem *show_toolbar, struct xfbib *xfbib)
{
	_DEBUG(("view_show_toolbar_event_handler"));
	if (gtk_check_menu_item_get_active(show_toolbar))
		gtk_widget_show(xfbib->window.toolbar);
	else
		gtk_widget_hide(xfbib->window.toolbar);
}

static void view_show_var_list_event_handler (GtkCheckMenuItem *show_var_list, struct xfbib *xfbib)
{
	_DEBUG(("view_show_var_list_event_handler"));
	if (gtk_check_menu_item_get_active(show_var_list)) 
		gtk_widget_show(xfbib->window.scrolled_variable_window);
	else
		gtk_widget_hide(xfbib->window.scrolled_variable_window);
}

static void help_about_event_handler (GtkWidget *window)
{
	_DEBUG(("help_about_event_handler"));
	show_about_dialog();
}

static GtkWidget *create_file_menu(struct xfbib *xfbib, GtkAccelGroup *accel)
{
	_DEBUG(("create_file_menu"));
	GtkWidget *item, *menu, *new, *open, *save, *saveas, *quit;
	
	item = gtk_menu_item_new_with_label(_("File"));

	menu = gtk_menu_new();

	new = gtk_image_menu_item_new_from_stock(GTK_STOCK_NEW, accel);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), new);
	g_signal_connect_swapped (G_OBJECT (new), "activate",
			G_CALLBACK (file_new_event_handler), xfbib);

	open = gtk_image_menu_item_new_from_stock(GTK_STOCK_OPEN, accel);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), open);
	g_signal_connect_swapped (G_OBJECT (open), "activate",
			G_CALLBACK (file_open_event_handler), xfbib);

	save = gtk_image_menu_item_new_from_stock(GTK_STOCK_SAVE, accel);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), save);
	g_signal_connect_swapped (G_OBJECT (save), "activate",
			G_CALLBACK (file_save_event_handler), xfbib);
	/* Add this one later */
	//gtk_widget_set_sensitive (save, FALSE);

	saveas = gtk_image_menu_item_new_from_stock(GTK_STOCK_SAVE_AS, accel);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), saveas);
	g_signal_connect_swapped (G_OBJECT (saveas), "activate",
			G_CALLBACK (file_saveas_event_handler), xfbib);

	quit = gtk_image_menu_item_new_from_stock(GTK_STOCK_QUIT, accel);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), quit);
	g_signal_connect_swapped (G_OBJECT (quit), "activate",
			G_CALLBACK (file_quit_event_handler), xfbib);

	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), menu);

	return item;
}

static GtkWidget *create_entry_menu(struct xfbib *xfbib, GtkAccelGroup *accel)
{
	_DEBUG(("create_entry_menu"));
	
	GtkWidget *item, *menu, *add, *remove;
	
	item = gtk_menu_item_new_with_mnemonic(_("E_ntry"));
	menu = gtk_menu_new();
	add = gtk_image_menu_item_new_from_stock(GTK_STOCK_ADD, accel);
	gtk_menu_shell_append(GTK_MENU_SHELL (menu), add);
	remove = gtk_image_menu_item_new_from_stock(GTK_STOCK_REMOVE, accel);
	gtk_menu_shell_append(GTK_MENU_SHELL (menu), remove);
	
	g_signal_connect_swapped (G_OBJECT (GTK_IMAGE_MENU_ITEM (add)), "activate",
			G_CALLBACK (entry_add_event_handler), xfbib);
	g_signal_connect_swapped (G_OBJECT (GTK_IMAGE_MENU_ITEM (remove)), "activate",
			G_CALLBACK (entry_remove_event_handler), xfbib);
			
	gtk_menu_item_set_submenu(GTK_MENU_ITEM (item), menu);
	
	return item;
}

static GtkWidget *create_variable_menu(struct xfbib *xfbib, GtkAccelGroup *accel)
{
	_DEBUG(("create_variable_menu"));
	
	GtkWidget *item, *menu, *add, *remove;
	
	item = gtk_menu_item_new_with_mnemonic(_("Variable"));
	menu = gtk_menu_new();
	add = gtk_image_menu_item_new_from_stock(GTK_STOCK_ADD, accel);
	gtk_menu_shell_append(GTK_MENU_SHELL (menu), add);
	remove = gtk_image_menu_item_new_from_stock(GTK_STOCK_REMOVE, accel);
	gtk_menu_shell_append(GTK_MENU_SHELL (menu), remove);
	
	g_signal_connect_swapped (G_OBJECT (GTK_IMAGE_MENU_ITEM (add)), "activate",
			G_CALLBACK (variable_add_event_handler), xfbib);
	g_signal_connect_swapped (G_OBJECT (GTK_IMAGE_MENU_ITEM (remove)), "activate",
			G_CALLBACK (variable_remove_event_handler), xfbib);
			
	gtk_menu_item_set_submenu(GTK_MENU_ITEM (item), menu);
	
	return item;
}

static GtkWidget *create_view_menu(struct xfbib *xfbib)
{
	_DEBUG(("create_view_menu"));
	
	GtkWidget *item, *menu, *toolbar, *var_list;
	
	item = gtk_menu_item_new_with_mnemonic(_("_View"));
	menu = gtk_menu_new();
	toolbar = gtk_check_menu_item_new_with_mnemonic(_("Show _Toolbar"));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM (toolbar), TRUE);
	gtk_menu_shell_append(GTK_MENU_SHELL (menu), toolbar);

	var_list = gtk_check_menu_item_new_with_mnemonic(_("Show _Variable List"));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM (var_list), TRUE);
	gtk_menu_shell_append(GTK_MENU_SHELL (menu), var_list);


	g_signal_connect (G_OBJECT (GTK_CHECK_MENU_ITEM (toolbar)), "activate",
			G_CALLBACK (view_show_toolbar_event_handler), xfbib);
	g_signal_connect (G_OBJECT (GTK_CHECK_MENU_ITEM (var_list)), "activate",
			G_CALLBACK (view_show_var_list_event_handler), xfbib);
			
	gtk_menu_item_set_submenu(GTK_MENU_ITEM (item), menu);
	
	return item;
}

static GtkWidget *create_help_menu(GtkWidget *window, GtkAccelGroup *accel)
{
	_DEBUG(("create_help_menu"));

	GtkWidget *item, *menu, *about;
	/* Create Help menu */
	item = gtk_menu_item_new_with_label(_("Help"));
	
	menu = gtk_menu_new();

	about = gtk_image_menu_item_new_from_stock(GTK_STOCK_ABOUT, accel);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), about);
	g_signal_connect_swapped (G_OBJECT (about), "activate",
			G_CALLBACK (help_about_event_handler), window);

	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), menu);

	return item;
}


GtkWidget *create_menubar(struct xfbib *xfbib)
{
	_DEBUG(("create_menubar"));

	struct {
		GtkWidget *file;
		GtkWidget *entry;
		GtkWidget *variable;
		GtkWidget *view;
		GtkWidget *help;
	} items;
	GtkWidget *menubar;
	GtkAccelGroup *accel;

	accel = gtk_accel_group_new ();
	menubar = gtk_menu_bar_new();
	
	items.file = create_file_menu(xfbib, accel);
	gtk_menu_shell_append(GTK_MENU_SHELL(menubar), items.file);
	
	items.entry = create_entry_menu(xfbib, accel);
	gtk_menu_shell_append(GTK_MENU_SHELL(menubar), items.entry);
	
	items.variable = create_variable_menu(xfbib, accel);
	gtk_menu_shell_append(GTK_MENU_SHELL(menubar), items.variable);
	
	items.view = create_view_menu(xfbib);
	gtk_menu_shell_append(GTK_MENU_SHELL(menubar), items.view);

	items.help = create_help_menu(xfbib->window.window, accel);
	gtk_menu_shell_append(GTK_MENU_SHELL(menubar), items.help);
	
	gtk_window_add_accel_group (GTK_WINDOW (xfbib->window.window), accel);

	return menubar;
}
