/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <libxfcegui4/libxfcegui4.h>
#include <libintl.h>
#include <config.h>
#include <string.h>

#include "treeview.h"
#include "xfbib.h"
#include "entry_edit_dialog.h"
#include "string_edit_dialog.h"
#include "node.h"
#include "strings.h"
#include "debug.h"

static void main_row_activated_event (GtkTreeView *view, GtkTreePath *path,
		GtkTreeViewColumn *column, gpointer xfbib)
{
	_DEBUG(("main_row_activated_event"));
	struct node *node;
	GtkTreeIter iter;
	GtkTreeModel *model = gtk_tree_view_get_model(view);

	if (gtk_tree_model_get_iter(model, &iter, path)) {
		gtk_tree_model_get(model, &iter, COLUMN_ENTRY, &node, -1);
		if (node->node_type == ENTRY_NODE) {
			show_entry_edit_dialog(xfbib, node->node);
		} else
			_DEBUG(("Not an entry in main list"));
	}
}

static void var_row_activated_event (GtkTreeView *view, GtkTreePath *path,
		GtkTreeViewColumn *column, gpointer xfbib)
{
	_DEBUG(("var_row_activated_event"));
	struct node *node;
	GtkTreeIter iter;
	GtkTreeModel *model = gtk_tree_view_get_model(view);

	if (gtk_tree_model_get_iter(model, &iter, path)) {
		gtk_tree_model_get(model, &iter, VAR_COLUMN_STRING, &node, -1);
		if (node->node_type == STRING_NODE) {
			show_string_edit_dialog(xfbib, node->node);
		} else
			_DEBUG(("Not a string in variable list"));
	}
}

static void cell_edited (GtkCellRendererText *renderer, char *path, char *new_text, GtkTreeView *treeview)
{
	_DEBUG(("cell edited"));
	GtkTreeIter iter;
	GtkTreeModel *model;
	if (strcasecmp (new_text, "") != 0) {
  		model = gtk_tree_view_get_model (treeview);
  		if (gtk_tree_model_get_iter_from_string (model, &iter, path))
    		gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, new_text, -1);
  	}
}

GtkWidget *create_author_editor_treeview(struct xfbib *xfbib, char *title)
{
	_DEBUG(("create_author_editor_treeview"));
	GtkWidget *treeview;
	GtkListStore *list;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	treeview = gtk_tree_view_new();

    renderer = gtk_cell_renderer_text_new();
    g_object_set(renderer, "editable", TRUE, NULL);
    g_signal_connect(renderer, "edited", (GCallback) cell_edited, treeview);
    
    column = gtk_tree_view_column_new_with_attributes(title, renderer, "text", 0, NULL);
    gtk_tree_view_column_set_resizable(GTK_TREE_VIEW_COLUMN(column), TRUE);
    gtk_tree_view_column_set_min_width(GTK_TREE_VIEW_COLUMN(column), 100);
   	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);

	list = gtk_list_store_new (1, G_TYPE_STRING);

	gtk_tree_view_set_model(GTK_TREE_VIEW(treeview), GTK_TREE_MODEL(list));
	g_object_unref(list);
	return treeview; 
}

GtkWidget *create_variable_treeview(struct xfbib *xfbib)
{
	_DEBUG(("create_variable_treeview"));
	GtkWidget *treeview;
	GtkListStore *list;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	treeview = gtk_tree_view_new();

	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

	g_signal_connect(treeview, "row-activated", (GCallback) var_row_activated_event, xfbib);

    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes(_("Variable"), renderer, "text", 0, NULL);
    gtk_tree_view_column_set_resizable(GTK_TREE_VIEW_COLUMN(column), TRUE);
    gtk_tree_view_column_set_min_width(GTK_TREE_VIEW_COLUMN(column), 100);
   	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);

	list = gtk_list_store_new (VAR_N_COLUMNS, G_TYPE_STRING, G_TYPE_POINTER);

	gtk_tree_view_set_model(GTK_TREE_VIEW(treeview), GTK_TREE_MODEL(list));
	g_object_unref(list);
	return treeview; 
}

GtkWidget *create_main_treeview(struct xfbib *xfbib)
{
	_DEBUG(("create_main_treeview"));
	GtkWidget *treeview;
	GtkListStore *list;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	int n;

	treeview = gtk_tree_view_new();

	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

	g_signal_connect(treeview, "row-activated", (GCallback) main_row_activated_event, xfbib);

    const char *fields[]= {_("Key"),_("Type"),_("Author"),_("Title")};
    for (n = 0; n < 4; n++) {
        renderer = gtk_cell_renderer_text_new();
        column = gtk_tree_view_column_new_with_attributes(fields[n], renderer, "text", n, NULL);
        gtk_tree_view_column_set_resizable(GTK_TREE_VIEW_COLUMN(column), TRUE);
        gtk_tree_view_column_set_min_width(GTK_TREE_VIEW_COLUMN(column), 100);
    	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);
    }

	list = gtk_list_store_new (N_COLUMNS, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_POINTER);

	gtk_tree_view_set_model(GTK_TREE_VIEW(treeview), GTK_TREE_MODEL(list));
	g_object_unref(list);
	return treeview; 
}	

void update_treeview(struct xfbib *xfbib)
{
	_DEBUG(("update_treeview"));
	GtkTreeIter iter;
	struct node *n;
	struct entry *e;
	struct string *s;

	GtkListStore *main_list = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(xfbib->window.main_treeview)));
	GtkListStore *var_list = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(xfbib->window.variable_treeview)));

	gtk_list_store_clear (main_list);
	gtk_list_store_clear (var_list);

	for (n = xfbib->list; n != NULL; n = n->next) {
		switch(n->node_type) {
			case ENTRY_NODE:
				e = n->node;	
				gtk_list_store_append(main_list, &iter);	
				gtk_list_store_set (main_list, &iter, COLUMN_KEY, e->key, -1);
				gtk_list_store_set (main_list, &iter, COLUMN_TYPE, e->type, -1);
				gtk_list_store_set (main_list, &iter, COLUMN_AUTHOR, get_tag_value(e,"author"), -1);
				gtk_list_store_set (main_list, &iter, COLUMN_TITLE, get_tag_value(e, "title"), -1);
				gtk_list_store_set (main_list, &iter, COLUMN_ENTRY, n, -1);
				break;
			case STRING_NODE:
				s = n->node;
				gtk_list_store_append(var_list, &iter);	
				gtk_list_store_set (var_list, &iter, VAR_COLUMN_NAME, s->variable, -1);
				gtk_list_store_set (var_list, &iter, VAR_COLUMN_STRING, n, -1);
				break;
		}
	}
}
