 /*
  *  Copyright (C) 2008 Marián Kyral
  *  Copyright (C) 2008 Ezequiel R. Aguerre
  *  Copyright (C) 2008 Pierpaolo Vittorini
  *  Copyright (C) 2008 Ruan Strydom
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of the GNU General Public License as published by
  * the Free Software Foundation; either version 2 of the License, or
  * (at your option) any later version.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  * GNU General Public License for more details.
  *
  * You should have received a copy of the GNU General Public License
  * along with this program; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
  */

#include "yawp.h"
#include "yawpconfig.h"
#include "yawpwserv.h"

#include <QPainter>
#include <QFontMetrics>
#include <QSizeF>
#include <KConfigGroup>
#include <KConfigDialog>
#include <QTimer>
#include <QTimeLine>
#include <KGlobalSettings>
#include <QMessageBox>
#include <QFile>
#include <QLabel>
#include <QTextLayout>
#include <QPalette>
#include <QBitmap>
#include <QPainterPath>
#include <QGraphicsSceneResizeEvent>
#include <QDateTime>
#include <klocale.h>

#include <KAboutData>
#include <KAboutApplicationDialog>


#include <KColorScheme>
#include <Plasma/Theme>


#if KDE_IS_VERSION(4,1,90)
  #include <Plasma/ToolTipManager>
#endif

#include <Plasma/Dialog>
#include <QVBoxLayout>

#define YAWP_MINUTE 60000

#define YAWP_ORIG_SIZEX (273.0f)
#define YAWP_ORIG_SIZEY (255.0f)

//Day's title
#define YAWP_ORIG_DAYY      (102.0f * m_scale)
#define YAWP_ORIG_DAYH      (60.0f  * m_scale)

//Top large current degrees
#define YAWP_ORIG_DEGX      (160.0f * m_scale)
#define YAWP_ORIG_DEGY      (4.0f   * m_scale)
#define YAWP_ORIG_DEGH      (46.0f  * m_scale)

//Today' temp
#define YAWP_ORIG_CURHIGHY  (5.0f * m_scale)
#define YAWP_ORIG_CURHIGHH  (25.0f  * m_scale)
#define YAWP_ORIG_CURLOWY   (27.0f * m_scale)
#define YAWP_ORIG_CURLOWH   (25.0f  * m_scale)

//Today' wind and humidity
#define YAWP_ORIG_CURHUMY   (60.0f * m_scale)
#define YAWP_ORIG_CURHUMW   (100.0f * m_scale)
#define YAWP_ORIG_CURHUMH   (30.0f  * m_scale)

//Info and icons size
#define YAWP_ORIG_INFO          (25.0f * m_scale)
#define YAWP_ORIG_INFO_HEIGHT   (150.0f * m_scale)
#define YAWP_ORIG_INFO_ICON     (15.0f * m_scale)

//Day current
#define YAWP_ORIG_CURX      (10.0f  * m_scale)
#define YAWP_ORIG_CURY      (6.0f   * m_scale)
#define YAWP_ORIG_CURW      (90.0f  * m_scale)
#define YAWP_ORIG_CURH      (130.0f * m_scale)

//Padding
#define YAWP_ORIG_PADD      (10.0f  * m_scale)

//City label
#define YAWP_ORIG_CITYY     (75.0f * m_scale)

//Emblem sizes
#define YAWP_ORIG_EMBL      (90.0f  * m_scale)
#define YAWP_ORIG_EMBS      (50.0f  * m_scale)

//Day's emblems y
#define YAWP_ORIG_EMBY      (155.0f * m_scale)

//Day' temp
#define YAWP_ORIG_HIGHY     (206.0f  * m_scale)
#define YAWP_ORIG_HIGHH     (25.0f  * m_scale)
#define YAWP_ORIG_LOWY      (226.0f  * m_scale)
#define YAWP_ORIG_LOWH      (25.0f  * m_scale)

//Shadow
//#define YAWP_SHADOW         (2.0f * m_scale)
#define YAWP_SHADOW         (1.0f * m_scale)


#define DRAWTEXT(rect, align, text)({\
p->setPen(m_fontShadowColor);\
p->drawText(rect.translated(YAWP_SHADOW, YAWP_SHADOW), align, text);\
p->setPen(m_fontColor);\
p->drawText(rect, align, text);\
})

#define DRAWTEXTGREY(rect, align, text)({\
p->setPen(m_fontShadowColor);\
p->drawText(rect.translated(1.0f * m_scale, 1.0f * m_scale), align, text);\
p->setPen(m_fontLowerColor);\
p->drawText(rect, align, text);\
})

#define DRAWIMAGE(p, rect, name)({\
if(m_useCustomTheme)\
    m_customSvg.paint(p, rect, name);\
else\
  if (m_svg.hasElement(name))\
    m_svg.paint(p, rect, name);\
  else\
    m_svg.paint(p, rect, name.mid(0, name.lastIndexOf('-')));\
})

//-----------------------------------------------------------------------------
YaWP::YaWP(QObject *parent, const QVariantList &args) : Plasma::Applet(parent, args), m_svg(this) {

    //Setup services
    //Icon/emblem mappings
    Q_INIT_RESOURCE(yawpicons);
    m_googlewservice = new YaWPGoogleWeatherService();
    m_accuweatherwservice = new YaWPForecastfoxWeatherService();

    //NULLS
    m_cdiag = NULL;
    m_wservice = NULL;

    m_svg.setImagePath("widgets/yawp_theme13");
    m_svg.setContainsMultipleImages(true);

    setHasConfigurationInterface(true);
    setAspectRatioMode(Plasma::KeepAspectRatio);

    // Contextual actions - thanks to Ezequiel, ported from stdin plasmoid :-)
    QAction *refreshAction = new QAction ( i18n ( "&Refresh" ), this );
    refreshAction->setIcon ( KIcon ( "view-refresh" ) );
    connect ( refreshAction, SIGNAL(triggered()), this, SLOT(timeout()) );

    QAction *aboutAction = new QAction ( i18n ( "&About" ), this );
    aboutAction->setIcon ( KIcon ( "help-about" ) );
    connect ( aboutAction, SIGNAL(triggered()), this, SLOT(about()) );



    QAction *separator = new QAction ( this );
    separator->setSeparator ( true );

    m_actions.append ( refreshAction );
    m_actions.append ( aboutAction );
    m_actions.append ( separator );

  // About data
    m_aboutData = new KAboutData ( "plasma_yawp",
    QByteArray (), ki18n ( "yaWP" ), "0.2.3",
    ki18n ( "Yet Another Weather Applet" ),
    KAboutData::License_GPL,
    ki18n ( "Copyright (C) 2008-2009 Ruan Strydom\n"
            "Copyright (C) 2008-2009 Ezequiel R. Aguerre\n"
            "Copyright (C) 2008-2009 Pierpaolo Vittorini\n"
            "Copyright (C) 2008-2009 Marián Kyral"),
    ki18n ( "This plasmoid shows for the selected place "
            " current weather and forecast for 4 days." ),
    "http://www.kde-look.org/content/show.php?content=94106",
    "plasmafactory@jcell.co.za" );

    // Authors
    m_aboutData->addAuthor (  ki18n ( "Ruan Strydom" ),
    ki18n ( "Developer" ), "ruans@kr8.co.za" );
    m_aboutData->addAuthor ( ki18n ( "Ezequiel R. Aguerre" ),
    ki18n ( "Developer" ), "ezeaguerre@gmail.com" );
    m_aboutData->addAuthor ( ki18n ( "Pierpaolo Vittorini" ),
    ki18n ( "Developer" ), "pierpaolo.vittorini@gmail.com" );
    m_aboutData->addAuthor ( ki18n ( "Marián Kyral" ),
    ki18n ( "Developer" ), "mkyral@email.cz" );

    // Credits
    m_aboutData->addCredit ( ki18n ( "Ruan Strydom" ),
    ki18n ( "For the idea, especially great graphics" ), "ruans@kr8.co.za" );
    m_aboutData->addCredit ( ki18n ( "AccuWeather.com" ),
    ki18n ( "For the weather data" ), "", "http://www.accuweather.com"  );
    m_aboutData->addCredit ( ki18n ( "das Gnu" ),
    ki18n ( "German translation" ), "dasgnu@gmail.com" );
    m_aboutData->addCredit ( ki18n ( "Jesús Vidal Panalés" ),
    ki18n ( "Spanish translation" ));
    m_aboutData->addCredit ( ki18n ( "Bribanick Dominique" ),
    ki18n ( "French translation" ), "chepioq@gmail.com" );
    m_aboutData->addCredit ( ki18n ( "Mihail Pisanov" ),
    ki18n ( "Russian translation" ), "miha@52.ru" );
    m_aboutData->addCredit ( ki18n ( "Richard Fric" ),
    ki18n ( "Slovak translation" ), "Richard.Fric@kdemail.net" );
    m_aboutData->addCredit ( ki18n ( "Mihael Simonič" ),
    ki18n ( "Slovenian translation" ), "smihael@gmail.com" );
    m_aboutData->addCredit ( ki18n ( "Maciej Bulik" ),
    ki18n ( "Polish translation" ), "Maciej.Bulik@post.pl" );
    m_aboutData->addCredit ( ki18n ( "All people reporting bugs, send feedback and new feature requests" ) );

    m_aboutData->setProgramIconName ( "weather-clear" );

    // Translators
    m_aboutData->setTranslator ( ki18nc ( "NAME OF THE TRANSLATORS", "Your names" ),
    ki18nc ( "EMAIL OF THE TRANSLATORS", "Your emails" ) );
    // About data - end

    m_expandedArea = QSizeF(YAWP_ORIG_SIZEX, YAWP_ORIG_SIZEY);
    m_origExpandedArea = QSizeF(YAWP_ORIG_SIZEX, YAWP_ORIG_SIZEY);
    resize(m_expandedArea);
    m_panelLayoutSwitched = 0;

    m_font      = KGlobalSettings::generalFont();
    m_units     = YaWPDay::METRIC;
    m_city      = "Johannesburg";
    m_country   = "South Africa";
    m_theme     = "default";
    m_service   = "Google";
    m_interval  = 30;
    m_message   = tr2i18n("Connecting...", 0);
    m_location  = "AFR|ZA|SF004|JOHANNESBURG";
    m_useCustomTheme    = false;
    m_customTheme       = "";
    m_useCustomThemeBackground = false;
    m_panelActualTemp = false;
    m_panelActualIcon = false;
    m_panelActualBoth = true;
    m_panelForecastTemp = false;
    m_panelForecastIcon = false;
    m_panelForecastBoth = true;
    m_panelForecastDays = 0;
    m_inPanel = false;
    m_panelHorizontal = false;
    m_panelToolTipSatellite = true;
    m_panelItemCount = 2;
    m_dialog = 0;
    m_useCustomColor = false;
    setDefaultFontColors();

    m_info = YaWPWeatherInfo(5);

    m_wservice = m_accuweatherwservice;

    m_timer = new QTimer(this);
    m_timer->setInterval(5000);
    connect(m_timer, SIGNAL(timeout()), this, SLOT(timeout()));

    m_animation = new QTimeLine(500, this); // 0.5 second for every animation.
    m_animationActive = false;
    connect(m_animation, SIGNAL(frameChanged(int)), this, SLOT(animationTimeout(int)));
    connect(m_animation, SIGNAL(finished()), this, SLOT(animationFinished()));

    connect(Plasma::Theme::defaultTheme(), SIGNAL(themeChanged()), this, SLOT(updateFontColors()));

    connect(Solid::Networking::notifier(), SIGNAL(statusChanged(Solid::Networking::Status)),
      this, SLOT(solidStatusChanged(Solid::Networking::Status)));

    for (int i = 0; i < MAX_FORECAST_DAYS; i++) {
      m_emblem[i] = YaWP::EMBLEM_DAY;
      m_animationTickEmblem[i] = 0;
    }

    m_days = YaWP::DAYS_NAMES;
    m_animationTickDays = 0;

    m_infoIcon = YaWP::INFO_CLOSED;
    m_animationTickInfoIcon = 0;
    m_infoIconDay = 0;

    m_satellite = YaWP::SATELLITE_CLOSED;
    m_animationTickSatellite = 0;
}


//-----------------------------------------------------------------------------
YaWP::~YaWP() {
    if (hasFailedToLaunch()) {
      // Perform some cleanup
      delete m_googlewservice;
      delete m_accuweatherwservice;
    } else {
        saveConfig();
    }
    delete m_dialog;
    delete m_aboutData;
}


//-----------------------------------------------------------------------------
void YaWP::init() {
    m_dialog = new Plasma::Dialog;
    QVBoxLayout *layout = new QVBoxLayout(m_dialog);
    m_dialogLabel = new QLabel;
    m_dialogLabel->setAutoFillBackground ( true );

    QPalette palette = m_dialogLabel->palette ();
    palette.setColor(QPalette::Window, Qt::white);
    m_dialogLabel->setPalette ( palette );

    layout->addWidget(m_dialogLabel);

    m_timer->stop();
    loadConfig();
    Solid::Networking::Status status = Solid::Networking::status();
    if (status != Solid::Networking::Unknown) {
      if (status == Solid::Networking::Connected)
        m_timer->start();
      else
        updateMessage(tr2i18n("Network unavailable.", 0) );
    }
    else
      m_timer->start();
}


//-----------------------------------------------------------------------------
void YaWP::constraintsEvent(Plasma::Constraints constraints) {

  float w,h,w_orig,h_orig;
  bool formFactorChange = false;
  w = w_orig = contentsRect().width();
  h = h_orig = contentsRect().height();

//  kDebug() << "Before: contentsRect( " << contentsRect().width() << "," << contentsRect().height() << "):" << formFactor() << endl;

  m_expandedArea.scale(contentsRect().width(), contentsRect().height(), Qt::KeepAspectRatio);

  if (constraints & Plasma::FormFactorConstraint) {
    if (formFactor() == Plasma::Horizontal ||
      formFactor() == Plasma::Vertical) {
      if ( ! m_inPanel ) {
        // we are in panel for the first time
        formFactorChange = true;
        setBackgroundHints(NoBackground);

        if (formFactor() == Plasma::Horizontal)
          m_panelHorizontal = true;
        else
          m_panelHorizontal = false;

        updatePanelSize();
      }
      m_inPanel = true;
      if ( formFactorChange ) updateFontColors();
    } else {
      if (m_inPanel) {
        // we moved from panel to desktop, reset some variables.
        m_inPanel = false;
        m_panelHorizontal = false;
        w = m_origExpandedArea.width();
        h = m_origExpandedArea.height();
        m_expandedArea.scale(w ,h, Qt::IgnoreAspectRatio);
        setMinimumWidth(0);
        setMinimumHeight(0);
        formFactorChange = true;
        m_resized = true;
        setMinimumSize(QSizeF(0,0));
        m_panelLayoutSwitched = 0;
        updateFontColors();
        // We are going out of panel, disable tooltip
        #if KDE_IS_VERSION(4,1,90)
          Plasma::ToolTipManager::self()->clearContent(this);
        #endif
      }
      m_panelHorizontal = false;
      if (m_dialog->isVisible())
        m_dialog->hide();
//      setAspectRatioMode(Plasma::KeepAspectRatio);
    }
  }

  if ( ! m_inPanel ) {
    if (constraints & Plasma::FormFactorConstraint) {
        if(m_theme == "default" && !(m_useCustomTheme && m_useCustomThemeBackground))
            setBackgroundHints(DefaultBackground);
        else
            // no backround on panel
            setBackgroundHints(NoBackground);
    }
  }

  if (constraints & Plasma::SizeConstraint || formFactorChange) {
    // size changed
    formFactorChange = false;
    if (m_inPanel) {
      if ( h <= w ) {
        // Horizontal panel
        // Sometimes is width not enough
        if ( h * m_panelItemCount >= w ) {
          h = w / m_panelItemCount;
          // skip 3 resize events after this resize
//          kDebug() << "Horizontal resize" << endl;
          if (m_panelLayoutSwitched <= 0 ) m_panelLayoutSwitched = 3;
        }
        if ( w <= 0 ) w = 1;
        if ( h <= 0 ) h = 1;
        m_expandedArea.scale( h * m_panelItemCount, h, Qt::IgnoreAspectRatio);
        m_panelHorizontal = true;
        m_panelElementSize = h * 0.9f;
        m_panelElementBorderSize = h * 0.1f / 2.0f;
      } else {
        // Vertical panel
        // Sometimes is height not enough
        if ( w * m_panelItemCount > h ) {
          w = h / m_panelItemCount;
//          kDebug() << "Vertical resize" << endl;
          // skip 3 resize events after this resize
          if (m_panelLayoutSwitched <= 0 ) m_panelLayoutSwitched = 3;
        }
        if ( w <= 0 ) w = 1;
        if ( h <= 0 ) h = 1;
        m_expandedArea.scale( w, w * m_panelItemCount, Qt::IgnoreAspectRatio);
        m_panelHorizontal = false;
        m_panelElementSize = w * 0.9f;
        m_panelElementBorderSize = w * 0.1f / 2.0f;
      }
      setBackgroundHints(NoBackground);
//      setAspectRatioMode(Plasma::KeepAspectRatio);

      m_emblemSize = m_panelElementSize;
      if (m_animationTickEmblem[0] > 1)
        m_animationTickEmblem[0] = m_emblemSize;
    }
    else {
      // on desktop
      m_expandedArea.scale( w, h, Qt::KeepAspectRatio);
      m_origExpandedArea.setWidth(w);
      m_origExpandedArea.setHeight(h);
    }

//    kDebug() << "Sizes: " <<w << " : " << w_orig << " | " << h << " : " << h_orig << " Switch cnt. " << m_panelLayoutSwitched << endl;
    if ( (int) w != (int) w_orig || (int) h != (int) h_orig ) {
      if ( m_panelLayoutSwitched <= 0 || m_panelLayoutSwitched == 3 ) {
        // FIXME:
        //  When in panel and the applet layout is changed due to not enough space
        //  in one direction, plasma will generates these resize events:
        //  (103 , 104) - the original one
        //  (52 , 104)  - layout changed to horizontal
        //  (103 , 104) - plasma inform us, that we can use this size - we need to ignore this event
        //                otherwise we are back on the second event and it means a loop
        resize(m_expandedArea);
        if (m_panelLayoutSwitched > 0 ) m_panelLayoutSwitched-- ;
      } else {
        // skip this resize event
        m_panelLayoutSwitched-- ;
      }
    }
    m_resized = true;

    // set minimum sizes
    setMinimumWidth(0);
    setMinimumHeight(0);

    if (formFactor() == Plasma::Horizontal ) {
      if ( m_panelHorizontal ) setMinimumWidth( h * m_panelItemCount);
      else  setMinimumWidth(0);
      setMinimumHeight(0);
    }
    if (formFactor() == Plasma::Vertical ) {
      setMinimumWidth(0);
      if ( ! m_panelHorizontal ) setMinimumHeight( w * m_panelItemCount);
      else setMinimumHeight( 0 );
    }
  }

//  kDebug() << "After: contentsRect(" << contentsRect().width() << "," << contentsRect().height() << "):" << formFactor() << endl;
}

//-----------------------------------------------------------------------------

void YaWP::paintEmblem(QPainter * p, int i, const YaWPDay *day, float x0, float y0, float size , bool night) {
  QRectF rect;
  if (m_animationTickEmblem[i] < size / 2) {
    float y = y0 + m_animationTickEmblem[i] / 2;
    rect.setRect(x0, y, size, size - m_animationTickEmblem[i] * 2);
    // -1 is a magic constant - show current weather emblem :-)
    if ( i == -1 && m_inPanel ) {
      DRAWIMAGE(p, rect, day->emblemCurr());
    } else {
      if(!night) {
        p->setOpacity(0.5);
        DRAWIMAGE(p, rect, day->nightEmblem());
        p->setOpacity(1.0);
      }
      else {
        if (i == 0 && ! m_inPanel) {
          // Draw the current emblem instead of today Day forecast.
          DRAWIMAGE(p, rect, day->emblemCurr());
        } else {
          DRAWIMAGE(p, rect, day->emblem());
        }
      }
    }
  }
  else {
    float y = y0 + (size - m_animationTickEmblem[i]) / 2;
    rect.setRect(x0, y, size, (m_animationTickEmblem[i] - size / 2) * 2 );
    if (i == -1  && m_inPanel ) {
      DRAWIMAGE(p, rect, day->emblemCurr());
    } else {
      if(night) {
        p->setOpacity(0.5);
        DRAWIMAGE(p, rect, day->nightEmblem());
        p->setOpacity(1.0);
      }
      else {
        if (i == 0 && ! m_inPanel) {
          // Draw the current emblem instead of today Day forecast.
          DRAWIMAGE(p, rect, day->emblemCurr());
        } else {
          DRAWIMAGE(p, rect, day->emblem());
        }
      }
    }
  }
}

QString YaWP::veryShortDate()
{
  QLocale loc;
  QString df( loc.dateFormat( QLocale::ShortFormat ) );
  df.replace( QRegExp( "[y]+[^Mdy]+"), "" );
  df.replace( QRegExp( "[^Mdy]+[y]+"), "" );
  return df;
}

void YaWP::paintDayNames(QPainter * p, QFont font, QRectF rect, float colWidth ) {
  const YaWPDay *day = NULL;

  for (int i = 0; i < m_info.days().size() - 1; i++) {
    day = &(m_info.days().at(i + 1));

    font.setBold(true);
    font.setPixelSize(17.0f * m_scale);
    p->setFont(font);

    rect.setRect(m_animationTickDays + colWidth * i, YAWP_ORIG_DAYY, colWidth, YAWP_ORIG_DAYH);
    DRAWTEXT(rect, Qt::AlignHCenter | Qt::AlignVCenter, tr2i18n(day->date().toUtf8().data(), 0 ));

    rect.translate(-m_wrect.width(), 0);
    DRAWTEXT(rect, Qt::AlignHCenter | Qt::AlignVCenter, day->day().toString(veryShortDate()));
  }
}

void YaWP::paintDegrees(QPainter *p, int i, const YaWPDay *day, QFont font, float colWidth ) {
  float low_temp_scale, animation_scale;
  int low, high, display;

  animation_scale = 1.0;

  if (m_animationWhat == YaWP::EMBLEM && m_whichEmblem == i) {
    animation_scale = 2.0f * (m_animationTickEmblem[i] - m_emblemSize / 2.0f) / m_emblemSize;
    if (animation_scale < 0) {
      animation_scale = -animation_scale;
      display = m_emblem[i];
    }
    else {
      if (m_animationActive)
        if (m_emblem[i] == YaWP::EMBLEM_DAY) display = YaWP::EMBLEM_NIGHT;
        else display = YaWP::EMBLEM_DAY;
        else
          display = m_emblem[i];
    }
  }
  else
    display = m_emblem[i];

  if (animation_scale < 0.1) animation_scale = 0.1;

  if (display == YaWP::EMBLEM_DAY) {
    low = YaWPDay::convertDegrees(day->units(), m_units, day->low());
    high = YaWPDay::convertDegrees(day->units(), m_units, day->high());
  }
  else {
    low = YaWPDay::convertDegrees(day->units(), m_units, day->nightLow());
    high = YaWPDay::convertDegrees(day->units(), m_units, day->nightHigh());
  }

  if (high < -9 || low > 99) low_temp_scale = .8;
  else low_temp_scale = 1;

  QRect rect;

  if (i == 0) {
    font.setBold(true);
    font.setPixelSize(19.0f * m_scale * low_temp_scale);
    rect.setRect(YAWP_ORIG_PADD, YAWP_ORIG_CURHIGHY, colWidth / 1.5f, YAWP_ORIG_CURHIGHH);
  }
  else {
    font.setPixelSize(15.0f * m_scale);
    rect.setRect(colWidth * (i - 1), YAWP_ORIG_HIGHY, colWidth, YAWP_ORIG_HIGHH);
  }

  p->setFont(font);
  DRAWTEXT(rect, Qt::AlignHCenter | Qt::AlignVCenter, i18nc("High degrees", "H: %1", high) + QChar(0xB0));

  if (i == 0) {
    font.setPixelSize(17.0f * m_scale * low_temp_scale);
    rect.setRect(YAWP_ORIG_PADD, YAWP_ORIG_CURLOWY, colWidth / 1.5f, YAWP_ORIG_CURLOWH);
  }
  else {
    font.setPixelSize(14.0f * m_scale);
    rect.setRect(colWidth * (i - 1), YAWP_ORIG_LOWY, colWidth, YAWP_ORIG_LOWH);
  }

  p->setFont(font);
  DRAWTEXTGREY(rect, Qt::AlignHCenter | Qt::AlignVCenter, i18nc("Low degrees", "L: %1", low) + QChar(0xB0));
}

void YaWP::paintInformation(QPainter *p, QFont font) {
  if (m_animationTickInfoIcon > 0) {
    float animation_scale = m_animationTickInfoIcon / m_wrect.width();
    if (animation_scale < 0.1) animation_scale = 0.1;

    int y0 = YAWP_ORIG_CURHUMY + YAWP_ORIG_INFO / 2,
        h = YAWP_ORIG_INFO_HEIGHT * animation_scale;

    QRect rect(YAWP_ORIG_PADD, y0, m_wrect.width() - 2.0 * YAWP_ORIG_PADD, h);
    const YaWPDay *day = &(m_info.days().at(m_infoIconDay));

    QColor col (0, 0, 0, 191 );
    //QBrush brush(col);
    //p->fillRect(rect, brush);
    //p->drawRoundedRect(rect, 2.0f, 2.0f);
    DRAWIMAGE(p, rect, QString("panel"));

    font.setPixelSize(15.0f * m_scale * animation_scale);
    font.setBold(true);
    p->setFont(font);
    rect.setRect(YAWP_ORIG_PADD, y0 + 2.0f * YAWP_SHADOW, m_wrect.width() - 2.0 * YAWP_ORIG_PADD, 16.0f * m_scale * animation_scale);
    DRAWTEXT(rect, Qt::AlignHCenter | Qt::AlignVCenter, tr2i18n("Forecast details", 0 ));

    QString summary, dayname;
    if (m_days == DAYS_NAMES) {
      dayname = tr2i18n(day->dateLong().toUtf8().data());
      dayname[0].toUpper();
    }
    else
      dayname = day->day().toString(veryShortDate());

    if (m_emblem[m_infoIconDay] == YaWP::EMBLEM_NIGHT)
      summary = i18nc("Day in locality", "%1 night in %2", dayname, m_city);
    else
      summary = i18nc("Day in locality", "%1 in %2", dayname, m_city);
    summary += "\n\n";

    if (m_emblem[m_infoIconDay] == YaWP::EMBLEM_DAY) summary += tr2i18n(day->description().toUtf8().data());
    else summary += tr2i18n(day->nightDescription().toUtf8().data());

    // Format time according to KDE Local setting
    QString sunrise = KGlobal::locale()->formatTime(day->sunriseTime());
    QString sunset  = KGlobal::locale()->formatTime(day->sunsetTime());

    summary += "\n" + i18n("Sunrise at %1, sunset at %2", sunrise, sunset);

    int low = 0, high = 0;
    if (m_emblem[m_infoIconDay] == YaWP::EMBLEM_DAY) {
      high = YaWPDay::convertDegrees(day->units(), m_units, day->rfHigh());
      low = YaWPDay::convertDegrees(day->units(), m_units, day->rfLow());
    }
    else {
      high = YaWPDay::convertDegrees(day->units(), m_units, day->nightRfHigh());
      low = YaWPDay::convertDegrees(day->units(), m_units, day->nightRfLow());
    }
    summary += "\n" + i18n("Realfeel: %1", QString::number(high) + QChar(0xB0)
        + " / " + QString::number(low) + QChar(0xB0));

    font.setPixelSize(12.0f * m_scale * animation_scale);
    font.setBold(false);
    p->setFont(font);
    p->setPen(Qt::white);

    rect.setRect(YAWP_ORIG_PADD + 2 * YAWP_SHADOW, y0 + 4 * YAWP_SHADOW + 16.0f * m_scale * animation_scale, m_wrect.width() - 2.0 * YAWP_ORIG_PADD - 4 * YAWP_SHADOW, h - 16.0f * m_scale * animation_scale);

    p->setPen(QColor(220, 220, 220));
    p->drawText(rect, Qt::AlignLeft | Qt::AlignTop | Qt::TextWordWrap, summary);

    if (m_infoIconDay > 0) {
      rect.setRect(YAWP_ORIG_PADD + 2.0 * YAWP_SHADOW, y0 + 2.0 * YAWP_SHADOW, YAWP_ORIG_INFO_ICON * animation_scale, YAWP_ORIG_INFO_ICON * animation_scale);
      DRAWIMAGE(p, rect, QString("arrow-left"));
    }

    if (m_infoIconDay < (m_info.days().size() - 1)) {
      rect.setRect(m_wrect.width() - YAWP_ORIG_PADD - 2.0 * YAWP_SHADOW - YAWP_ORIG_INFO_ICON * animation_scale, y0 + 2.0 * YAWP_SHADOW, YAWP_ORIG_INFO_ICON * animation_scale, YAWP_ORIG_INFO_ICON * animation_scale);
      DRAWIMAGE(p, rect, QString("arrow-right"));
    }

    if (animation_scale > 0.9) {
      Qt::TransformationMode mode = m_animationActive ? Qt::FastTransformation : Qt::SmoothTransformation;
      QPixmap scaled = m_wservice->logo().scaled(m_wrect.width(), 10, Qt::KeepAspectRatio, mode);
      p->drawPixmap(YAWP_ORIG_PADD + 4 * YAWP_SHADOW, YAWP_ORIG_CURHUMY + YAWP_ORIG_INFO / 2 + YAWP_ORIG_INFO_HEIGHT - 10 - 4 * YAWP_SHADOW, scaled);
    }
  }
}

void YaWP::paintSatellite(QPainter *p ) {
  if (m_animationTickSatellite <= 0)
    return;

  float animation_scale = m_animationTickSatellite / m_wrect.width();
  if (animation_scale < 0.1) animation_scale = 0.1;

  int y0 = YAWP_ORIG_CURHUMY + YAWP_ORIG_INFO / 2;
  int h = (m_wrect.width() - 2.0 * YAWP_ORIG_PADD) * animation_scale * 0.75 - (YAWP_ORIG_PADD * 2.0);
  QRect rect(YAWP_ORIG_PADD, y0, m_wrect.width() - 2.0 * YAWP_ORIG_PADD, h);

  QPixmap image = m_wservice->image();
  if (!image.isNull()) {
    QPainterPath region;
    region.addRoundedRect(rect, YAWP_ORIG_PADD * 0.75, YAWP_ORIG_PADD * 0.75);
    p->save();
    p->setClipRegion(QRegion(region.toFillPolygon().toPolygon()));
    enum Qt::TransformationMode mode = m_animationActive ? Qt::FastTransformation : Qt::SmoothTransformation;
    QPixmap scaled = image.scaled(rect.width(), rect.height(), Qt::KeepAspectRatio, mode);
    p->drawPixmap(rect, scaled);
    p->restore();
  } else {
    DRAWIMAGE(p, rect, QString("panel"));
  }
}

//-----------------------------------------------------------------------------
void YaWP::paintPanel(QPainter *p, const QStyleOptionGraphicsItem *option, const QRect& contentsRect) {
    Q_UNUSED(option);
    float width, height, low_temp_scale, colWidth;
    const YaWPDay *day = NULL;
    QRectF rect;

    m_wrect = contentsRect;
    width = m_wrect.width();
    height = m_wrect.height();

    m_scale = width / YAWP_ORIG_SIZEX;
    colWidth = width / 3.0f;

    m_wrect = QRectF(QPointF(0, 0), m_expandedArea);
    day = &(m_info.days().at(0));
    m_scale = 0.25f;
    int panelCounter = 0;
//      p->translate (m_wrect.topLeft() );

    //Draw current degrees
    if (m_panelActualBoth || m_panelActualTemp) {
      if (YaWPDay::convertDegrees(day->units(), m_units, day->current()) < -9 ||
          YaWPDay::convertDegrees(day->units(), m_units, day->current()) > 99 )
        low_temp_scale = .7f;
      else
        low_temp_scale = 1.0f;

      QFont font = m_font;
      font.setBold(false);
      font.setPixelSize(m_panelElementSize * 0.6f * low_temp_scale);
      p->setFont(font);
      rect.setRect(m_panelElementBorderSize, m_panelElementBorderSize, m_emblemSize, m_emblemSize);
      DRAWTEXT(rect, Qt::AlignRight | Qt::AlignVCenter, QString("%1").arg(YaWPDay::convertDegrees(day->units(), m_units, day->current())) + QChar(0xB0));
      panelCounter += 1;
    }

    // paint actual weather emblem
    if (m_panelActualBoth || m_panelActualIcon) {
      if (m_panelHorizontal) {
        paintEmblem(p, -1, day, 0.0f + ((m_panelElementBorderSize  + m_panelElementSize) * panelCounter), 0.0f + m_panelElementBorderSize, m_emblemSize);
      } else {
        paintEmblem(p, -1, day, 0.0f + m_panelElementBorderSize, 0.0f + ((m_panelElementBorderSize + m_panelElementSize) * panelCounter), m_emblemSize);
      }
      panelCounter += 1;
    }

    // paint forecasts
    if (m_panelForecastDays > 0 ) {
      for (int i = 0; i < m_panelForecastDays; i++) {
        day = &(m_info.days().at(i));

        // paint forecast degrees
        if (m_panelForecastTemp || m_panelForecastBoth){
          QRectF rectHigh, rectLow;
          float low, high;

          QFont font = m_font;
          font.setBold(false);

          low=YaWPDay::convertDegrees(day->units(), m_units, day->low());
          high=YaWPDay::convertDegrees(day->units(), m_units, day->high());

          if (m_panelHorizontal) {
            rectHigh.setRect(0.0f + ((m_panelElementBorderSize  + m_panelElementSize) * panelCounter), m_panelElementBorderSize, m_emblemSize, m_emblemSize/2.0f);
            rectLow.setRect (0.0f + ((m_panelElementBorderSize  + m_panelElementSize) * panelCounter), m_panelElementBorderSize + m_emblemSize/2.0f, m_emblemSize, m_emblemSize/2.0f);
          } else {
            rectHigh.setRect(m_panelElementBorderSize, 0.0f + ((m_panelElementBorderSize  + m_panelElementSize) * panelCounter), m_emblemSize, m_emblemSize/2.0f);
            rectLow.setRect (m_panelElementBorderSize, 0.0f + ((m_panelElementBorderSize  + m_panelElementSize) * panelCounter)  + m_emblemSize/2.0f, m_emblemSize, m_emblemSize/2.0f);
          }
          font.setPixelSize(m_panelElementSize * 0.4f);
          p->setFont(font);
          DRAWTEXT(rectHigh, Qt::AlignHCenter | Qt::AlignVCenter, QString("%1").arg(high) + QChar(0xB0));

          font.setPixelSize(m_panelElementSize * 0.4f);
          p->setFont(font);
          DRAWTEXTGREY(rectLow, Qt::AlignHCenter | Qt::AlignVCenter, QString("%1").arg(low) + QChar(0xB0));

          panelCounter += 1;
        }

        bool dayTime;
        if (i==0) {
          dayTime = isDayTime(day->timeCurrent(), day->sunriseTime(), day->sunsetTime());
        }
        else
          dayTime = true;
        // paint forecast emblems
        if (m_panelForecastIcon || m_panelForecastBoth) {
          if (m_panelHorizontal) {
            paintEmblem(p, i, day, 0.0f + ((m_panelElementBorderSize  + m_panelElementSize) * panelCounter), 0.0f + m_panelElementBorderSize, m_emblemSize, dayTime);
          } else {
            paintEmblem(p, i, day, 0.0f + m_panelElementBorderSize, 0.0f + ((m_panelElementBorderSize + m_panelElementSize) * panelCounter), m_emblemSize, dayTime);
          }
          panelCounter++;
        }
      }
    }
    // panel tooltip
#if KDE_IS_VERSION(4,1,90)

    QColor fontColorBck, fontLowerColorBck, fontShadowColorBck;

    if ( m_theme != "default" && m_theme != "naked" && ! m_useCustomColor) {
      // set default colors when custom colors and not default or naked theme
      fontColorBck = m_fontColor;
      fontLowerColorBck = m_fontLowerColor;
      fontShadowColorBck = m_fontShadowColor;
      setDefaultFontColors();
    }

    Plasma::ToolTipContent toolTipData;
    QString subtext = QString();

    // yaWP image
    float temp = m_animationTickEmblem[0];

    QPixmap pix(YAWP_ORIG_SIZEX * 0.8f, YAWP_ORIG_SIZEY * 0.8f);
    pix.fill(Qt::transparent);
    QPainter pp(&pix);

    m_animationTickEmblem[0] = 0;
    m_inPanel = false;
    paintPanelTooltip(&pp, 0, pix.rect());
    m_inPanel = true;
    m_animationTickEmblem[0] = temp;

    pp.end();

    // yaWP image
    if (m_panelToolTipSatellite) subtext += "<table><tr><td>";

//    for (int i=0; i < m_info.days().size(); i++) { // only next two days for now
    for (int i=0; i < 3; i++) {
      // Weather description
      day = &(m_info.days().at(i));
      QString dayname = i18nc("Forecast for day",day->dateLong().toUtf8().data());
      dayname[0].toUpper();
      subtext += "<u>" + i18n("Forecast for %1",dayname) + " ("+ day->day().toString(veryShortDate()) +")</u><br />";
      subtext += i18n("Day: ") + i18n(day->description().toUtf8().constData()) + "<br />";
      subtext += i18n("Night: ") + i18n(day->nightDescription().toUtf8().constData()) + "<br />";

      // Sunset, sunrise and realfeal only for today, to limit tooltip size
      if ( i == 0 ) {
        // Format time according to KDE Local setting
        QString sunrise = KGlobal::locale()->formatTime(day->sunriseTime());
        QString sunset  = KGlobal::locale()->formatTime(day->sunsetTime());

        subtext += i18n("Sunrise at %1, sunset at %2", sunrise, sunset) + "<br />";

        int low = 0, high = 0;
          high = YaWPDay::convertDegrees(day->units(), m_units, day->rfHigh());
          low = YaWPDay::convertDegrees(day->units(), m_units, day->rfLow());

        subtext += i18n("Realfeel: %1", QString::number(high) + QChar(0xB0)
            + " / " + QString::number(low) + QChar(0xB0)) + "<br /><br />";
      }
    }

    // Add satellite image
    if (m_panelToolTipSatellite) {
      toolTipData.addResource(Plasma::ToolTipContent::ImageResource, QUrl("wicon://" + QString::number(2)),
                              QVariant(m_wservice->image().scaledToWidth(YAWP_ORIG_SIZEX * 0.8f, Qt::SmoothTransformation)));
      subtext += "</td><td valign=\"top\">";
      subtext += "<img src=\"wicon://" + QString::number(2) + "\"/>";
      subtext += "</td></tr></table>";
    }

    // Split Main text to two rows if no satellite image is shown
    if ( ! m_panelToolTipSatellite ) {
      toolTipData.setMainText( QString("%1</b><br /> %2<br /><b>").arg(m_city).arg(m_message) ); // City \n message
    } else {
      toolTipData.setMainText( QString("%1</b> - %2<b>").arg(m_city).arg(m_message) ); // City + message
    }
    toolTipData.setImage(pix); // yaWP tooltip image
    toolTipData.setSubText(subtext); // Some details
    toolTipData.setAutohide(false);
    Plasma::ToolTipManager::self()->setContent(this, toolTipData);

    if ( m_theme != "default" && m_theme != "naked"  && ! m_useCustomColor ) {
      // restore font colors
      m_fontColor = fontColorBck;
      m_fontLowerColor = fontLowerColorBck;
      m_fontShadowColor = fontShadowColorBck;
    }

#endif
//      update();

}

//-----------------------------------------------------------------------------
void YaWP::paintInterface(QPainter *p, const QStyleOptionGraphicsItem *option, const QRect &contentsRect) {
    Q_UNUSED(option);

    float width, height, colWidth;
    const YaWPDay *day = NULL;
    QRectF rect;

    m_wrect = contentsRect;
    width = m_wrect.width();
    height = m_wrect.height();

    m_scale = width / YAWP_ORIG_SIZEX;
    colWidth = width / 3.0f;

 /* **** PANEL **** */
    if (m_inPanel) {
      paintPanel(p, option, contentsRect);
      return;
    }

    if (m_animationActive) {
        p->setRenderHint(QPainter::SmoothPixmapTransform, false);
        p->setRenderHint(QPainter::Antialiasing, false);
        p->setRenderHint(QPainter::TextAntialiasing, false);
    } else {
        p->setRenderHint(QPainter::SmoothPixmapTransform);
        p->setRenderHint(QPainter::Antialiasing);
        p->setRenderHint(QPainter::TextAntialiasing);
    }

    if (m_resized) {
      if (m_infoIcon == YaWP::INFO_OPEN)
        m_animationTickInfoIcon = m_wrect.width ();
      if (m_animationTickSatellite > 0)
        m_animationTickSatellite = m_wrect.width ();
      if (m_animationTickEmblem[0] > 0)
        m_animationTickEmblem[0] = YAWP_ORIG_EMBL;
      m_resized = false;
    }

    //Draw back
    if(m_useCustomThemeBackground && m_useCustomTheme)
        DRAWIMAGE(p, m_wrect, QString("back-default"));
    else
        m_svg.paint(p, m_wrect, QString("back-%1").arg(m_theme));

    //Fix animations when rotated. (They leave the drawing area.)
    p->setClipRect(m_wrect.x() + (2.0f * m_scale), m_wrect.y() + (2.0f * m_scale), m_wrect.width() - (4.0f * m_scale), m_wrect.height() -(4.0f * m_scale));

    p->translate (m_wrect.topLeft() );

    QFont font = m_font;
    font.setBold(true);
    font.setPixelSize(19.0f * m_scale);
    p->setFont(font);

    //Draw head
    day = &(m_info.days().at(0));

    //Draw current degrees
    /*
    if (YaWPDay::convertDegrees(day->units(), m_units, day->current()) < -9 ||
        YaWPDay::convertDegrees(day->units(), m_units, day->current()) > 99 )
        low_temp_scale = .8;
    else
        low_temp_scale = 1;
    font.setPixelSize(52.0f * m_scale  * low_temp_scale);
    */
    font.setPixelSize(52.0f * m_scale);
    font.setBold(false);
    p->setFont(font);
    rect.setRect(YAWP_ORIG_DEGX, YAWP_ORIG_DEGY, width - YAWP_ORIG_DEGX - YAWP_ORIG_PADD / 1.5f, YAWP_ORIG_DEGH);
    DRAWTEXT(rect, Qt::AlignRight | Qt::AlignVCenter, QString("%1").arg(YaWPDay::convertDegrees(day->units(), m_units, day->current())) + QChar(0xB0));

    //Draw today high low degrees
    paintDegrees(p, 0, day, font, colWidth);

    //Draw icons
    rect.setRect(YAWP_ORIG_PADD / 2.0f, YAWP_ORIG_CURHUMY, YAWP_ORIG_INFO, YAWP_ORIG_INFO);
    DRAWIMAGE(p, rect, QString("info"));
    rect.translate(YAWP_ORIG_INFO + 2.0f * YAWP_SHADOW, 0);
    DRAWIMAGE(p, rect, QString("map"));

    //Draw today humidity and wind degrees
    font.setBold(false);
    font.setPixelSize(12.0f * m_scale);
    p->setFont(font);
    rect.setRect(width - YAWP_ORIG_PADD / 2.0f - YAWP_ORIG_CURHUMW, YAWP_ORIG_CURHUMY, YAWP_ORIG_CURHUMW, YAWP_ORIG_CURHUMH);
    if(m_units == YaWPDay::IMPERIAL)
      DRAWTEXT(rect, Qt::AlignHCenter | Qt::AlignVCenter, QString("%1 " + tr2i18n("mph") + " %2\n%3%").arg(YaWPDay::convertDistance(day->units(), m_units, day->windSpeedCurr())).arg(i18nc("Wind direction", day->windDirectionCurr().toUtf8().data())).arg(day->humidity()));
    else
      DRAWTEXT(rect, Qt::AlignHCenter | Qt::AlignVCenter, QString("%1 " + tr2i18n("kmph") + " %2\n%3%").arg(YaWPDay::convertDistance(day->units(), m_units, day->windSpeedCurr())).arg(i18nc("Wind direction", day->windDirectionCurr().toUtf8().data())).arg(day->humidity()));

    //Draw city
    font.setBold(false);
    font.setPixelSize(15.0f * m_scale);
    p->setFont(font);
    rect.setRect(YAWP_ORIG_PADD, YAWP_ORIG_CITYY, width - YAWP_ORIG_PADD, YAWP_ORIG_DAYH);
    DRAWTEXT(rect, Qt::AlignHCenter | Qt::AlignVCenter, m_message);

    //Draw today's emblem
    //Now is order: Current weather / Night forecast icon, no automatically switch for night emblem,
    //     covered by current weather icon (hope)
    paintEmblem(p, 0, day, (m_wrect.width() - YAWP_ORIG_EMBL) / 2.3f, YAWP_ORIG_PADD/1.8f, YAWP_ORIG_EMBL, true );
    //Draw days
    colWidth = m_wrect.width() / (m_info.days().size() - 1);
    paintDayNames(p, font, rect, colWidth);

    for (int i = 0; i < m_info.days().size() - 1; i++) {
        day = &(m_info.days().at(i + 1));

        paintEmblem(p, i + 1, day, colWidth * i + colWidth / 2.0f - YAWP_ORIG_EMBS / 2.0f, YAWP_ORIG_EMBY, YAWP_ORIG_EMBS);
        paintDegrees(p, i + 1, day, font, colWidth);
    }

    paintInformation(p, font);
    paintSatellite(p);
}

//-----------------------------------------------------------------------------
void YaWP::paintPanelTooltip(QPainter *p, const QStyleOptionGraphicsItem *option, const QRect &contentsRect) {
    Q_UNUSED(option);

    float width, height, colWidth;
    const YaWPDay *day = NULL;
    QRectF rect;

    m_wrect = contentsRect;
    width = m_wrect.width();
    height = m_wrect.height();

    m_scale = width / YAWP_ORIG_SIZEX;
    colWidth = width / 3.0f;

    p->setRenderHint(QPainter::SmoothPixmapTransform);
    p->setRenderHint(QPainter::Antialiasing);
    p->setRenderHint(QPainter::TextAntialiasing);

    //Draw back
    if(m_useCustomThemeBackground && m_useCustomTheme)
        DRAWIMAGE(p, m_wrect, QString("back-default"));
    else
        m_svg.paint(p, m_wrect, QString("back-%1").arg(m_theme));

    p->translate (m_wrect.topLeft() );

    QFont font = m_font;
    font.setBold(true);
    font.setPixelSize(19.0f * m_scale);
    p->setFont(font);

    //Draw head
    day = &(m_info.days().at(0));

    //Draw today humidity and wind degrees
    font.setBold(false);
    font.setPixelSize(12.0f * m_scale);
    p->setFont(font);
    rect.setRect(width - YAWP_ORIG_PADD / 2.0f - YAWP_ORIG_CURHUMW, YAWP_ORIG_CURHUMY, YAWP_ORIG_CURHUMW, YAWP_ORIG_CURHUMH);
    if(m_units == YaWPDay::IMPERIAL) {
      DRAWTEXT(rect, Qt::AlignHCenter | Qt::AlignVCenter, QString("%1 " + tr2i18n("mph") + " %2\n%3%").arg(YaWPDay::convertDistance(day->units(), m_units, day->windSpeedCurr())).arg(i18nc("Wind direction", day->windDirectionCurr().toUtf8().data())).arg(day->humidity()));
    } else {
      DRAWTEXT(rect, Qt::AlignHCenter | Qt::AlignVCenter, QString("%1 " + tr2i18n("kmph") + " %2\n%3%").arg(YaWPDay::convertDistance(day->units(), m_units, day->windSpeedCurr())).arg(i18nc("Wind direction", day->windDirectionCurr().toUtf8().data())).arg(day->humidity()));
}

    //Draw current degrees
    font.setPixelSize(52.0f * m_scale);
    font.setBold(false);
    p->setFont(font);
    rect.setRect(YAWP_ORIG_DEGX, YAWP_ORIG_DEGY, width - YAWP_ORIG_DEGX - YAWP_ORIG_PADD / 1.5f, YAWP_ORIG_DEGH);
    DRAWTEXT(rect, Qt::AlignRight | Qt::AlignVCenter, QString("%1").arg(YaWPDay::convertDegrees(day->units(), m_units, day->current())) + QChar(0xB0));

//    DRAWTEXT(rect, Qt::AlignHCenter | Qt::AlignVCenter, m_message);

    //Draw today's emblem
    rect.setRect((m_wrect.width() - YAWP_ORIG_EMBL) / 2.3f, YAWP_ORIG_PADD/1.8f,YAWP_ORIG_EMBL,YAWP_ORIG_EMBL);
    DRAWIMAGE(p, rect, day->emblemCurr());

    //Draw days
      //Todays night forecast emblem
    p->setOpacity(0.5);
    rect.setRect(YAWP_ORIG_PADD + YAWP_ORIG_EMBS/3.2f,YAWP_ORIG_PADD*2.0f + YAWP_ORIG_EMBS/2.0f, YAWP_ORIG_EMBS*1.2f,YAWP_ORIG_EMBS*1.2f);
    DRAWIMAGE(p, rect, day->nightEmblem());

      //Todays day forecast emblem
    p->setOpacity(1);
    rect.setRect(YAWP_ORIG_PADD,YAWP_ORIG_PADD*2.0f, YAWP_ORIG_EMBS*1.2f,YAWP_ORIG_EMBS*1.2f);
    DRAWIMAGE(p, rect, day->emblem());

      //Todays high forecast temperature
    font.setPixelSize(18.0f * m_scale);
    p->setFont(font);
    rect.setRect(YAWP_ORIG_PADD, YAWP_ORIG_PADD*0.5f, colWidth * 0.8f, YAWP_ORIG_HIGHH);
    DRAWTEXT(rect, Qt::AlignRight | Qt::AlignVCenter, i18nc("High degrees", "H: %1", QString("%1").arg(YaWPDay::convertDegrees(day->units(), m_units, day->high()))) + QChar(0xB0));

      //Todays Low forecast temperature
    rect.setRect(YAWP_ORIG_PADD, YAWP_ORIG_PADD*0.5f + YAWP_ORIG_EMBS*1.8f, colWidth * 0.8f, YAWP_ORIG_LOWH);
    DRAWTEXTGREY(rect, Qt::AlignLeft | Qt::AlignVCenter, i18nc("Low degrees", "L: %1", QString("%1").arg(YaWPDay::convertDegrees(day->units(), m_units, day->low()))) + QChar(0xB0));

    //Next days forecast
    colWidth = m_wrect.width() / (m_info.days().size() - 1);
    paintDayNames(p, font, rect, colWidth);

    for (int i = 0; i < m_info.days().size() - 1; i++) {
        day = &(m_info.days().at(i + 1));

//        paintEmblem(p, i + 1, day, colWidth * i + colWidth / 2.0f - YAWP_ORIG_EMBS / 2.0f, YAWP_ORIG_EMBY, YAWP_ORIG_EMBS);
//        paintDegrees(p, i + 1, day, font, colWidth);

        //Night forecast emblem
      p->setOpacity(0.5);
      rect.setRect(colWidth * i + colWidth / 2.0f - YAWP_ORIG_EMBS / 3.0f ,YAWP_ORIG_EMBY + YAWP_ORIG_HIGHH*0.5 + YAWP_ORIG_EMBS / 2.0f, YAWP_ORIG_EMBS,YAWP_ORIG_EMBS);
      DRAWIMAGE(p, rect, day->nightEmblem());

        //Day forecast emblem
      p->setOpacity(1);
      rect.setRect(colWidth * i + colWidth / 2.0f - YAWP_ORIG_EMBS/ 1.7f ,YAWP_ORIG_EMBY + YAWP_ORIG_HIGHH*0.5, YAWP_ORIG_EMBS,YAWP_ORIG_EMBS);
      DRAWIMAGE(p, rect, day->emblem());

        //High forecast temperature
      font.setPixelSize(13.0f * m_scale);
      p->setFont(font);
      rect.setRect(colWidth * i + colWidth / 2.0f - YAWP_ORIG_EMBS/ 1.7f, YAWP_ORIG_EMBY - YAWP_ORIG_HIGHH * 0.3f, colWidth * 0.8f, YAWP_ORIG_HIGHH);
      DRAWTEXT(rect, Qt::AlignRight | Qt::AlignVCenter, i18nc("High degrees", "H: %1", QString("%1").arg(YaWPDay::convertDegrees(day->units(), m_units, day->high()))) + QChar(0xB0));

        //Low forecast temperature
      rect.setRect(colWidth * i + colWidth / 2.0f - YAWP_ORIG_EMBS/ 1.7f, YAWP_ORIG_LOWY, colWidth * 0.8f, YAWP_ORIG_LOWH);
      DRAWTEXTGREY(rect, Qt::AlignLeft | Qt::AlignVCenter, i18nc("Low degrees", "L: %1", QString("%1").arg(YaWPDay::convertDegrees(day->units(), m_units, day->low()))) + QChar(0xB0));
    }
}

//-----------------------------------------------------------------------------
void YaWP::updateMessage(QString mess) {
    m_message = mess;
    update();
}


//-----------------------------------------------------------------------------
void YaWP::solidStatusChanged(Solid::Networking::Status status)
{
    if (status == Solid::Networking::Connected) {
      m_timer->start();
      timeout();
    }
    else {
      m_timer->stop();
      updateMessage(tr2i18n("Network unavailable.", 0) );
    }
}


//-----------------------------------------------------------------------------
void YaWP::timeout() {
    // Test again in case of refreshAction
    Solid::Networking::Status status = Solid::Networking::status();
    if (status != Solid::Networking::Unknown
        && status != Solid::Networking::Connected)
      return;

    updateMessage(tr2i18n("Connecting...", 0) );
#if KDE_VERSION_MINOR > 1
    setBusy(true);
#endif
    QMap<QString,QString> data;
    data["city"]=m_city;
    data["country"]=m_country;
    data["location"]=m_location;
    m_wservice = m_accuweatherwservice;
    connect(m_wservice, SIGNAL(completed(QObject *, int)), this, SLOT(updateInfo(QObject *, int)));
    m_wservice->update(data);
}


//-----------------------------------------------------------------------------
void YaWP::updateInfo(QObject * service, int exitCode) {

#if KDE_VERSION_MINOR > 1
    setBusy(false);
#endif
  if(exitCode == YaWPWeatherService::SUCCESS) {
    if(m_timer->interval() != YAWP_MINUTE * m_interval) {
      m_timer->setInterval(YAWP_MINUTE * m_interval);
    }
    m_info = ((YaWPWeatherService *)service)->info();
    updateMessage(m_info.message());
  }

  if(exitCode == YaWPWeatherService::CACHE) {
    m_timer->stop();
    m_timer->setInterval(YAWP_MINUTE);
    m_info = ((YaWPWeatherService *)service)->info();
    updateMessage(m_info.message() + " (cache)");
    m_timer->start();
  }

  if (exitCode == YaWPWeatherService::DATA_FAILED)
        updateMessage(tr2i18n("Bad city or country!", 0) );

  if (exitCode == YaWPWeatherService::CONNECTION_FAILED) {
    m_timer->stop();
    m_timer->setInterval(YAWP_MINUTE);
    updateMessage(tr2i18n("Connecting...", 0) );
    m_timer->start();
  }

  if (exitCode == YaWPWeatherService::ERROR)
    updateMessage(tr2i18n("ERROR!", 0) );
}


//-----------------------------------------------------------------------------
void YaWP::loadConfig() {
    KConfigGroup cfg = config();
    m_service = cfg.readEntry("service", m_service);
    m_location = cfg.readEntry("location", m_location);
    m_city = cfg.readEntry("city", m_city);
    m_country = cfg.readEntry("country", m_country);
    m_theme = cfg.readEntry("theme", m_theme);
    m_customTheme = cfg.readEntry("custom.theme", m_customTheme);
    m_useCustomTheme = cfg.readEntry("custom.theme.enabled", QString("%1").arg(m_useCustomTheme)).toInt();
    m_useCustomThemeBackground = cfg.readEntry("custom.theme.background.enabled", QString("%1").arg(m_useCustomThemeBackground)).toInt();
    //Ensure change
    constraintsEvent(Plasma::FormFactorConstraint);
    m_units = cfg.readEntry("units", QString("%1").arg((int)m_units)).toInt();
    m_interval = cfg.readEntry("interval", QString("%1").arg(m_interval)).toInt();
    if(m_interval < 15)
        m_interval = 30;
    if((int)m_units < 0)
        m_units = YaWPDay::METRIC;
    // Custom Colors
    m_useCustomColor = cfg.readEntry("use.custom.color", QString("%1").arg(m_useCustomColor)).toInt();
    m_fontColor      = cfg.readEntry("font.color", m_fontColor);
    m_fontLowerColor = cfg.readEntry("font.lower.color", m_fontLowerColor);

    if (m_useCustomColor) {
//      m_fontShadowColor = KColorScheme(QPalette::Active, KColorScheme::View, Plasma::Theme::defaultTheme()->colorScheme()).foreground(KColorScheme::InactiveText).color();
      m_fontShadowColor = QColor(0,0,0,100);
    } else {
        updateFontColors();
    }

    // panel support
    m_panelActualTemp   = cfg.readEntry("panel.actual.temp.enabled", QString("%1").arg(m_panelActualTemp)).toInt();
    m_panelActualIcon   = cfg.readEntry("panel.actual.icon.enabled", QString("%1").arg(m_panelActualIcon)).toInt();
    m_panelActualBoth   = cfg.readEntry("panel.actual.both.enabled", QString("%1").arg(m_panelActualBoth)).toInt();
    m_panelForecastTemp = cfg.readEntry("panel.forecast.temp.enabled", QString("%1").arg(m_panelForecastTemp)).toInt();
    m_panelForecastIcon = cfg.readEntry("panel.forecast.icon.enabled", QString("%1").arg(m_panelForecastIcon)).toInt();
    m_panelForecastBoth = cfg.readEntry("panel.forecast.both.enabled", QString("%1").arg(m_panelForecastBoth)).toInt();
    m_panelForecastDays = cfg.readEntry("panel.forecast.days", QString("%1").arg(m_panelForecastDays)).toInt();
    if ( m_panelForecastDays < 0 ) m_panelForecastDays = 0;
    if ( m_panelForecastDays > 5 ) m_panelForecastDays = 5;
    m_panelItemCount = m_panelActualBoth ? 2 : 1;
    m_panelItemCount += (m_panelForecastDays  * (m_panelForecastBoth ? 2 : 1));
    if (m_inPanel) updatePanelSize();

    m_panelToolTipSatellite = cfg.readEntry("panel.tooltip.satellite",QString("%1").arg(m_panelToolTipSatellite)).toInt();

    //Load custom theme if valid
    loadCustomTheme();
    kDebug() << m_location << endl;
}


//-----------------------------------------------------------------------------
void YaWP::saveConfig() {
    KConfigGroup cfg = config();
    cfg.writeEntry("units", QString("%1").arg((int)m_units));
    cfg.writeEntry("city", m_city);
    cfg.writeEntry("country", m_country);
    cfg.writeEntry("location", m_location);
    cfg.writeEntry("theme", m_theme);
    cfg.writeEntry("service", m_service);
    cfg.writeEntry("interval", QString("%1").arg(m_interval));
    cfg.writeEntry("custom.theme", m_customTheme);
    cfg.writeEntry("custom.theme.enabled", QString("%1").arg(m_useCustomTheme));
    cfg.writeEntry("custom.theme.background.enabled", QString("%1").arg(m_useCustomThemeBackground));
    cfg.writeEntry("use.custom.color", QString("%1").arg(m_useCustomColor));
    cfg.writeEntry("font.color", m_fontColor);
    cfg.writeEntry("font.lower.color", m_fontLowerColor);
    cfg.writeEntry("panel.actual.temp.enabled", QString("%1").arg(m_panelActualTemp));
    cfg.writeEntry("panel.actual.icon.enabled", QString("%1").arg(m_panelActualIcon));
    cfg.writeEntry("panel.actual.both.enabled", QString("%1").arg(m_panelActualBoth));
    cfg.writeEntry("panel.forecast.temp.enabled", QString("%1").arg(m_panelForecastTemp));
    cfg.writeEntry("panel.forecast.icon.enabled", QString("%1").arg(m_panelForecastIcon));
    cfg.writeEntry("panel.forecast.both.enabled", QString("%1").arg(m_panelForecastBoth));
    cfg.writeEntry("panel.forecast.days", QString("%1").arg(m_panelForecastDays));
    cfg.writeEntry("panel.tooltip.satellite", QString("%1").arg(m_panelToolTipSatellite));
}


//-----------------------------------------------------------------------------
void YaWP::createConfigurationInterface(KConfigDialog *parent) {
    if (m_cdiag != NULL)
      delete m_cdiag;
    m_cdiag = new YaWPConfigDialog(parent);
    connect(m_cdiag, SIGNAL(save()), this, SLOT(settingsChanged()));
    if (m_cdiag != NULL ) {
        //m_cdiag->setVisible(true);
        m_cdiag->setCity(m_city);
        m_cdiag->setCountry(m_country);
        m_cdiag->setService(m_service);
        m_cdiag->setLocation(m_location);
        m_cdiag->setInterval(m_interval);
        m_cdiag->setTheme(m_theme);
        m_cdiag->setUnits(m_units);
        m_cdiag->setUseCustomTheme(m_useCustomTheme);
        m_cdiag->setCustomTheme(m_customTheme);
        m_cdiag->setUseCustomThemeBackground(m_useCustomThemeBackground);
        m_cdiag->setUseCustomColor(m_useCustomColor);
        m_cdiag->setFontColor(m_fontColor);
        m_cdiag->setFontLowerColor(m_fontLowerColor);
        m_cdiag->setPanelActualTemp(m_panelActualTemp);
        m_cdiag->setPanelActualIcon(m_panelActualIcon);
        m_cdiag->setPanelActualBoth(m_panelActualBoth);
        m_cdiag->setPanelForecastTemp(m_panelForecastTemp);
        m_cdiag->setPanelForecastIcon(m_panelForecastIcon);
        m_cdiag->setPanelForecastBoth(m_panelForecastBoth);
        m_cdiag->setPanelForecastDays(m_panelForecastDays);
        m_cdiag->setPanelToolTipSatellite(m_panelToolTipSatellite);
    }
}


//-----------------------------------------------------------------------------
void YaWP::settingsChanged() {
    if (m_cdiag == NULL )
        return;
    QString location = m_location;
    m_theme = m_cdiag->theme();
    //Ensure change
    constraintsEvent(Plasma::FormFactorConstraint);
    m_units = m_cdiag->units();
    //m_cdiag->setVisible(false);
    m_city = m_cdiag->city();
    m_country = m_cdiag->country();
    m_interval = m_cdiag->interval();
    m_service = m_cdiag->service();
    m_location = m_cdiag->location();
    m_useCustomTheme = m_cdiag->useCustomTheme();
    m_customTheme = m_cdiag->customTheme();
    m_useCustomThemeBackground = m_cdiag->useCustomThemeBackground();
    if(m_interval < 15)
        m_interval = 30;
    if((int)m_units < 0)
        m_units = YaWPDay::METRIC;

    m_useCustomColor = m_cdiag->useCustomColor();
    m_fontColor = m_cdiag->fontColor();
    m_fontLowerColor = m_cdiag->fontLowerColor();

    if (m_useCustomColor) {
//      m_fontShadowColor = KColorScheme(QPalette::Active, KColorScheme::View, Plasma::Theme::defaultTheme()->colorScheme()).shade(KColorScheme::LightShade);
      m_fontShadowColor = QColor(0,0,0,100);
    } else {
        updateFontColors();
    }

    m_panelActualTemp=m_cdiag->panelActualTemp();
    m_panelActualIcon=m_cdiag->panelActualIcon();
    m_panelActualBoth=m_cdiag->panelActualBoth();
    m_panelForecastTemp=m_cdiag->panelForecastTemp();
    m_panelForecastIcon=m_cdiag->panelForecastIcon();
    m_panelForecastBoth=m_cdiag->panelForecastBoth();
    m_panelForecastDays=m_cdiag->panelForecastDays();

    if ( m_panelForecastDays < 0 ) m_panelForecastDays = 0;
    if ( m_panelForecastDays > 5 ) m_panelForecastDays = 5;
    m_panelItemCount = m_panelActualBoth ? 2 : 1;
    m_panelItemCount += (m_panelForecastDays  * (m_panelForecastBoth ? 2 : 1));

    m_panelToolTipSatellite = m_cdiag->panelToolTipSatellite();

    saveConfig();
    if (m_inPanel) updatePanelSize();

    //Load the custom theme if valid
    loadCustomTheme();


    if (location != m_location)
      m_info.reset();

    Solid::Networking::Status status = Solid::Networking::status();
    if (status != Solid::Networking::Unknown && location != m_location) {
      solidStatusChanged(status);
    }
    else
    {
      if (location != m_location)
        timeout();
      else
        update();
    }
}


void YaWP::loadCustomTheme() {
    if(!m_useCustomTheme)
        return;
    if(!QFile(m_customTheme).exists()) {
        m_useCustomTheme = false;
        kDebug() << "File does not exist: " << m_customTheme << endl;
        return;
    }
    /*TODO get the following paramaters from theme:
        rect The rectangle to draw in.
        font The font to use.
        color The font color to use.
    */
    m_customSvg.setImagePath(m_customTheme);
    m_customSvg.setContainsMultipleImages (true);
}

//-----------------------------------------------------------------------------
// Clickable areas
#define YAWP_CLICK_DAY_Y0   (120.0f * m_scale)
#define YAWP_CLICK_DAY_H    (28.0f * m_scale)

bool YaWP::clickOnDays(QPointF pos ) {
  QRectF rect(0, YAWP_CLICK_DAY_Y0, m_wrect.width(), YAWP_CLICK_DAY_H);
  return rect.contains(pos);
}

int YaWP::clickOnEmblem(QPointF pos ) {
  QRectF rect;

  rect.setRect((m_wrect.width() - YAWP_ORIG_EMBL) / 2.0f, YAWP_ORIG_PADD/1.8f, YAWP_ORIG_EMBL, YAWP_ORIG_EMBL);
  if (rect.contains(pos)) return 0;

  float colWidth = m_wrect.width() / (m_info.days().size() - 1);
  for (int i = 0; i < m_info.days().size() - 1; i++) {
    rect.setRect(colWidth * i + colWidth / 2.0f - YAWP_ORIG_EMBS / 2.0f, YAWP_ORIG_EMBY, YAWP_ORIG_EMBS, YAWP_ORIG_EMBS);
    if (rect.contains(pos)) return i + 1;
  }

  return -1;
}

int YaWP::clickOnPanelEmblem(QPointF pos ) {
  QRectF rect;
  int panelCounter = 0;
  int index = 0;

  if (m_panelActualBoth || m_panelActualIcon) {
    panelCounter = panelCounter + 1 * (m_panelActualBoth ? 2 : 1);
  }

  while (panelCounter <= m_panelItemCount) {
    if (m_panelForecastBoth || m_panelForecastIcon) {
      if (m_panelHorizontal) {
        rect.setRect( 0.0f + ((m_panelElementBorderSize  + m_panelElementSize) * panelCounter),
                      0.0f + m_panelElementBorderSize,
                      m_emblemSize * (m_panelActualBoth ? 2 : 1),
                      m_emblemSize);
      } else {
        rect.setRect( 0.0f + m_panelElementBorderSize,
                      0.0f + ((m_panelElementBorderSize  + m_panelElementSize) * panelCounter),
                      m_emblemSize,
                      m_emblemSize * (m_panelActualBoth ? 2 : 1));
      }
      if (rect.contains(pos)) return index;
      panelCounter = panelCounter + 1 * (m_panelActualBoth ? 2 : 1);
      index++;
    }
  }
//  float colWidth = m_wrect.width() / (m_info.days().size() - 1);
//  for (int i = 0; i < m_info.days().size() - 1; i++) {
//    rect.setRect(colWidth * i + colWidth / 2.0f - YAWP_ORIG_EMBS / 2.0f, YAWP_ORIG_EMBY, YAWP_ORIG_EMBS, YAWP_ORIG_EMBS);
//    if (rect.contains(pos)) return i + 1;
//  }

  return -1;
}

bool YaWP::clickOnInfoIcon(QPointF pos ) {
  QRectF rect(YAWP_ORIG_PADD / 2.0f, YAWP_ORIG_CURHUMY, YAWP_ORIG_INFO, YAWP_ORIG_INFO);
  return rect.contains(pos);
}

bool YaWP::clickOnInfoIconLeft(QPointF pos ) {
  QRectF rect(YAWP_ORIG_PADD + 2.0 * YAWP_SHADOW, YAWP_ORIG_CURHUMY + YAWP_ORIG_INFO / 2.0 + 2.0 * YAWP_SHADOW, YAWP_ORIG_INFO_ICON, YAWP_ORIG_INFO_ICON);
  return rect.contains(pos);
}

bool YaWP::clickOnInfoIconRight(QPointF pos ) {
  QRectF rect(m_wrect.width() - YAWP_ORIG_PADD - 2.0 * YAWP_SHADOW - YAWP_ORIG_INFO_ICON, YAWP_ORIG_CURHUMY + YAWP_ORIG_INFO / 2.0 + 2.0 * YAWP_SHADOW, YAWP_ORIG_INFO_ICON, YAWP_ORIG_INFO_ICON);
  return rect.contains(pos);
}

bool YaWP::clickOnSatelliteIcon(QPointF pos ) {
  QRectF rect(YAWP_ORIG_PADD / 2.0f + YAWP_ORIG_INFO + 2.0f * YAWP_SHADOW, YAWP_ORIG_CURHUMY, YAWP_ORIG_INFO, YAWP_ORIG_INFO);
  return rect.contains(pos);
}

#define ANIMATION_DAYS_DELTA            5
#define ANIMATION_EMBLEM_DELTA          (m_emblemSize / 10.0f)
#define ANIMATION_INFO_DELTA            (m_wrect.width() / 10.0f)
#define ANIMATION_SATELLITE_DELTA       (m_wrect.width() / 10.0f)

void YaWP::mousePressEvent(QGraphicsSceneMouseEvent* event) {
  if (event->button() == Qt::LeftButton) {
    if (m_animationActive) return;

    QPointF pos = event->pos() - m_wrect.topLeft();
    int emblem;

    if (m_inPanel)
      emblem = clickOnPanelEmblem(pos);
    else
      emblem = clickOnEmblem(pos);

    if (emblem >= 0) {
      m_animationWhat = YaWP::EMBLEM;
      m_whichEmblem = emblem;

      if (!m_inPanel) {
        if (emblem == 0) m_emblemSize = YAWP_ORIG_EMBL;
        else m_emblemSize = YAWP_ORIG_EMBS;
      }

      if (m_emblem[emblem] == YaWP::EMBLEM_DAY) m_animation->setDirection(QTimeLine::Forward);
      if (m_emblem[emblem] == YaWP::EMBLEM_NIGHT) m_animation->setDirection(QTimeLine::Backward);

      m_animation->setFrameRange(0, m_emblemSize);
      m_animation->setCurveShape(QTimeLine::EaseInOutCurve);
      m_animation->start();
      m_animationActive = true;
      return;
    }

    if (clickOnDays(pos)) {
      m_animationWhat = YaWP::DAYS;

      if (m_days == YaWP::DAYS_NAMES) m_animation->setDirection(QTimeLine::Forward);
      if (m_days == YaWP::DAYS_NUMBERS) m_animation->setDirection(QTimeLine::Backward);

      m_animation->setFrameRange(0, m_wrect.width());
      m_animation->setCurveShape(QTimeLine::EaseInOutCurve);
      m_animation->start();
      m_animationActive = true;
    }

    if (m_infoIcon == YaWP::INFO_OPEN) {
      if (clickOnInfoIconLeft(pos)) {
        if (m_infoIconDay > 0) {
          m_infoIconDay--;
          update(m_wrect.x() + YAWP_ORIG_PADD, m_wrect.y() + YAWP_ORIG_CURHUMY
              + YAWP_ORIG_INFO / 2, m_wrect.width() - 2.0f * YAWP_ORIG_PADD,
              YAWP_ORIG_INFO_HEIGHT);
          return;
        }
      }
      else if (clickOnInfoIconRight(pos)) {
        if (m_infoIconDay < (m_info.days().size() - 1)) {
          m_infoIconDay++;
          update(m_wrect.x() + YAWP_ORIG_PADD, m_wrect.y() + YAWP_ORIG_CURHUMY
              + YAWP_ORIG_INFO / 2, m_wrect.width() - 2.0f * YAWP_ORIG_PADD,
              YAWP_ORIG_INFO_HEIGHT);
          return;
        }
      }
    }
    if (clickOnInfoIcon(pos)) {
      if (m_satellite == YaWP::SATELLITE_CLOSED) {
        m_animationWhat = YaWP::INFO;

        if (m_infoIcon == YaWP::INFO_CLOSED) m_animation->setDirection(QTimeLine::Forward);
        if (m_infoIcon == YaWP::INFO_OPEN) m_animation->setDirection(QTimeLine::Backward);

        m_animation->setFrameRange(0, m_wrect.width());
        m_animation->setCurveShape(QTimeLine::EaseOutCurve);
        m_animation->start();
        m_animationActive = true;
      }
    }

    if (clickOnSatelliteIcon(pos)) {
      if (m_infoIcon == YaWP::INFO_CLOSED) {
        m_animationWhat = YaWP::SATELLITE;

        if (m_satellite == YaWP::SATELLITE_CLOSED) m_animation->setDirection(QTimeLine::Forward);
        if (m_satellite == YaWP::SATELLITE_OPEN) m_animation->setDirection(QTimeLine::Backward);

        m_animation->setFrameRange(0, m_wrect.width());
        m_animation->setCurveShape(QTimeLine::EaseOutCurve);
        m_animation->start();
        m_animationActive = true;
      }
    }
  }
}

void YaWP::mouseDoubleClickEvent(QGraphicsSceneMouseEvent *) {
  if (!m_inPanel)
    return;
  if (m_dialog->isVisible())
    m_dialog->hide();
  else {
#if 1
    YaWPDay day = m_info.days().at(0);
    QString info =
        i18n("City: %1",m_city) + "\n"
        + i18n("Temperature: %1", YaWPDay::convertDegrees(day.units(), m_units, day.current()))
        + QChar(0xB0) + "\n" + i18n("Humidity: %1%", day.humidity())
        + "\n" + i18n("Sunrise: %1", day.sunrise())
        + "\n" + i18n("Sunset: %1", day.sunset());

    info += "\n\n" + i18n("Day:") + "\n"
        + i18n("Temperature: %1 / %2", QString::number(YaWPDay::convertDegrees(day.units(), m_units, day.high()))
            + QChar(0xB0),
         QString::number(YaWPDay::convertDegrees(day.units(), m_units, day.low())) + QChar(0xB0))
        + "\n" + i18n("Real feel: %1 / %2", QString::number(YaWPDay::convertDegrees(day.units(), m_units, day.rfHigh()))
            + QChar(0xB0),
         QString::number(YaWPDay::convertDegrees(day.units(), m_units, day.rfLow())) + QChar(0xB0))
        + "\n";
    info += i18n("Wind: %1 %2 %3", YaWPDay::convertDistance(day.units(), m_units, day.windSpeed()), (m_units == YaWPDay::IMPERIAL ?
            i18n("mph") : i18n("kmph")), i18n(day.windDirection().toUtf8().constData()));
    info += "\n" + i18n(day.description().toUtf8().constData());

    if (day.hasNightValues()) {
      info += "\n\n" + i18n("Night:") + "\n"
        + i18n("Temperature: %1 / %2", QString::number(YaWPDay::convertDegrees(day.units(), m_units, day.nightHigh()))
            + QChar(0xB0),
         QString::number(YaWPDay::convertDegrees(day.units(), m_units, day.nightLow())) + QChar(0xB0))
        + "\n" + i18n("Real feel: %1 / %2", QString::number(YaWPDay::convertDegrees(day.units(), m_units, day.nightRfHigh()))
            + QChar(0xB0),
         QString::number(YaWPDay::convertDegrees(day.units(), m_units, day.nightRfLow())) + QChar(0xB0))
        + "\n";
      info += i18n("Wind: %1 %2 %3", YaWPDay::convertDistance(day.units(), m_units, day.nightWindSpeed()), (m_units == YaWPDay::IMPERIAL ?
        i18n("mph") : i18n("kmph")), i18n(day.nightWindDirection().toUtf8().constData()));
      info += "\n" + i18n(day.nightDescription().toUtf8().constData());
    }

    m_dialogLabel->setText(info);
#else
    float temp = m_animationTickEmblem[0];

    QPixmap pix(YAWP_ORIG_SIZEX, YAWP_ORIG_SIZEY);
    pix.fill(Qt::black);
    QPainter p(&pix);

    m_animationTickEmblem[0] = 0;
    m_inPanel = false;
    paintInterface(&p, 0, pix.rect());
    m_inPanel = true;
    m_animationTickEmblem[0] = temp;

    p.end();
    m_dialogLabel->setPixmap(pix);
#endif
    m_dialog->move(popupPosition(m_dialog->sizeHint()));
    m_dialog->show();
  }
}

//-----------------------------------------------------------------------------
// NOTE: All those "+2" down there are meant to correct truncation.
//       Let's suppose we have a coordinate x=2.9 and a width=4.7,
//       these should be mapped to the coordinates 3 and 8, however,
//       when casting to int we truncate the number and we get x=2
//       and width=4 (coordinates 2 and 6), so width+=2 gives us x=2
//       and width=6, wich translates to coordinates 2 and 8.
void YaWP::animationTimeout(int frame) {
  int x = m_wrect.x(), y = m_wrect.y();
  if (m_inPanel) {
    // Cut&Paste emblem code.
    m_animationTickEmblem[m_whichEmblem] = frame;
    if (frame >= (int)m_emblemSize)
      m_emblem[m_whichEmblem] = YaWP::EMBLEM_NIGHT;
    else if (frame <= 0)
      m_emblem[m_whichEmblem] = YaWP::EMBLEM_DAY;
    update();
    return;
  }
  switch (m_animationWhat) {

    // ANIMATION REGARDS AN EMBLEM
    case YaWP::EMBLEM :
      m_animationTickEmblem[m_whichEmblem] = frame;
      if (frame >= (int)m_emblemSize)
        m_emblem[m_whichEmblem] = YaWP::EMBLEM_NIGHT;
      else if (frame <= 0)
        m_emblem[m_whichEmblem] = YaWP::EMBLEM_DAY;
      {
        QRect rect;
        if (m_whichEmblem == 0) {
          rect.setRect(x + (m_wrect.width() - YAWP_ORIG_EMBL)/2.4f, y +
              YAWP_ORIG_PADD/1.8f, YAWP_ORIG_EMBL + 2, YAWP_ORIG_EMBL + 2);
        }
        else {
          float colWidth = m_wrect.width() / (m_info.days().size() - 1);
          float s = YAWP_ORIG_EMBS + 2;
          rect.setRect(x + colWidth * (m_whichEmblem - 1) + colWidth / 2.0f -
              YAWP_ORIG_EMBS / 2.0f, y + YAWP_ORIG_EMBY, s, s);
        }
        update(rect);
        // Qt should merge these 2 updates using a QRegion
        // This update is necessary to actualize the
        // information in the info box, but is only needed
        // once after the animation has finished.
        if (m_infoIcon == YaWP::INFO_OPEN &&
            m_whichEmblem == m_infoIconDay &&
            (frame == 0 || frame >= (int)m_emblemSize))
          update(x + YAWP_ORIG_PADD, y + YAWP_ORIG_CURHUMY +
              YAWP_ORIG_INFO / 2, m_wrect.width() - 2.0f * YAWP_ORIG_PADD,
              YAWP_ORIG_INFO_HEIGHT);
      }
      break;

      // ANIMATION REGARDS THE DAYS
    case YaWP::DAYS :
      m_animationTickDays = frame;
      if (frame >= (int)m_wrect.width())
        m_days = YaWP::DAYS_NUMBERS;
      else if (frame <= 0)
        m_days = YaWP::DAYS_NAMES;
      update(x, y + YAWP_ORIG_DAYY, m_wrect.width() + 2, YAWP_ORIG_DAYH + 2);
      if (m_infoIcon == YaWP::INFO_OPEN &&
          (frame == 0 || frame >= (int)m_wrect.width()))
        update(x + YAWP_ORIG_PADD, y + YAWP_ORIG_CURHUMY +
            YAWP_ORIG_INFO / 2, m_wrect.width() - 2.0f * YAWP_ORIG_PADD,
            YAWP_ORIG_INFO_HEIGHT);
      break;

      // ANIMATION REGARDS THE INFO
    case YaWP::INFO :
      {
        // Here we add 4 instead of 2, because we subtract
        // 1 from the x and y, so, to be coherent with what
        // I explained at the top of this function we need to
        // add 1 more to the width and height, giving us 3, but
        // I don't like 3 because it's odd... so I use 4 :-)
        // I think we need to subtract 1 to x and y because of
        // the border's pen width...
        if (frame >= (int)m_wrect.width())
          m_infoIcon = YaWP::INFO_OPEN;
        else if (frame <= 0)
          m_infoIcon = YaWP::INFO_CLOSED;
        if (frame > m_animationTickInfoIcon)
          m_animationTickInfoIcon = frame;
        float animation_scale = m_animationTickInfoIcon / m_wrect.width();
        if (animation_scale < 0.1f) animation_scale = 0.1f;
        int y0 = YAWP_ORIG_CURHUMY + YAWP_ORIG_INFO / 2;
        float h = YAWP_ORIG_INFO_HEIGHT * animation_scale + 4 + 2.0f * YAWP_SHADOW;
        QRect rect(YAWP_ORIG_PADD + x - 1, y0 + y - 1, m_wrect.width() - 2.0
            * YAWP_ORIG_PADD + 4, h);
        update(rect);
        m_animationTickInfoIcon = frame;
      }
      break;

      // ANIMATION REGARDS THE INFO
    case YaWP::SATELLITE :
      {
        if (frame >= (int)m_wrect.width())
          m_satellite = YaWP::SATELLITE_OPEN;
        else if (frame <= 0)
          m_satellite = YaWP::SATELLITE_CLOSED;
        if (frame > m_animationTickSatellite)
          m_animationTickSatellite = frame;
        float animation_scale = m_animationTickSatellite / m_wrect.width();
        if (animation_scale < 0.1f) animation_scale = 0.1f;
        int y0 = YAWP_ORIG_CURHUMY + YAWP_ORIG_INFO / 2;
        float h = (m_wrect.width() - 2.0f * YAWP_ORIG_PADD) * animation_scale * 0.75f + 2;
        QRect rect(x + YAWP_ORIG_PADD, y + y0, m_wrect.width() - 2.0f
            * YAWP_ORIG_PADD + 2, h);
        update(rect);
        m_animationTickSatellite = frame;
      }
      break;
  }
}

void YaWP::animationFinished() {
  m_animationActive = false;
}

void YaWP::updatePanelSize() {
//  FIXME: with panel spacer plasmoid it doesn't work correctly
//         when returning back to smaller size (e.g. without forecast)
  if (m_panelHorizontal) {
    resize( contentsRect().height() * m_panelItemCount, contentsRect().height());
//    resize( 10 * m_panelItemCount, 10);
  } else {
    resize( contentsRect().width(), contentsRect().width() * m_panelItemCount);
//    resize( 10, 10 * m_panelItemCount);
  }
  setMinimumSize(contentsRect().width(), contentsRect().height());
}

bool YaWP::isDayTime(QTime currentTime, QTime sunriseTime, QTime sunsetTime) {
  if(currentTime > sunsetTime || currentTime < sunriseTime)
    return ( false );
  else
    return( true );
};

void YaWP::setDefaultFontColors() {
  m_fontColor = QColor(Qt::white);
  m_fontShadowColor = QColor(0,0,0,100);
  m_fontLowerColor = QColor(Qt::gray);
}

void YaWP::updateFontColors() {
  if (m_useCustomColor) {
    KConfigGroup cfg  = config();
    m_fontColor       = cfg.readEntry("font.color", m_fontColor);
    m_fontLowerColor  = cfg.readEntry("font.lower.color", m_fontLowerColor);
    m_fontShadowColor = QColor(0,0,0,100);
  } else {
    if ( m_theme != "default" && m_theme != "naked" and ! m_inPanel) {
      setDefaultFontColors();
    } else {
      m_fontColor = KColorScheme(QPalette::Active, KColorScheme::View, Plasma::Theme::defaultTheme()->colorScheme()).foreground().color();
      m_fontLowerColor = KColorScheme(QPalette::Active, KColorScheme::View,
                                      Plasma::Theme::defaultTheme()->colorScheme()).foreground(KColorScheme::InactiveText).color();
//      m_fontShadowColor = KColorScheme(QPalette::Active, KColorScheme::View,
//                                      Plasma::Theme::defaultTheme()->colorScheme()).foreground(KColorScheme::InactiveText).color();
      if (m_fontColor.red() < 25 &&  m_fontColor.green() < 25 && m_fontColor.blue() < 25) {
//        m_fontShadowColor = QColor(Qt::white);
        m_fontShadowColor = QColor(0,0,0,100);
        m_fontLowerColor = m_fontColor.lighter();
      } else {
        m_fontShadowColor = QColor(0,0,0,100);
        m_fontLowerColor = m_fontColor.darker(125);
      }
    }
  }
}

void YaWP::about () const
{
  KAboutApplicationDialog *aboutDialog = new KAboutApplicationDialog ( m_aboutData );
  connect ( aboutDialog, SIGNAL ( finished () ), aboutDialog, SLOT ( deleteLater () ) );
  aboutDialog->show ();
}

K_EXPORT_PLASMA_APPLET(yawp, YaWP)

#include "yawp.moc"
