 /*
  *  Copyright (C) 2008 Marián Kyral
  *  Copyright (C) 2008 Ezequiel R. Aguerre
  *  Copyright (C) 2008 Pierpaolo Vittorini
  *  Copyright (C) 2008 Ruan Strydom
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of the GNU General Public License as published by
  * the Free Software Foundation; either version 2 of the License, or
  * (at your option) any later version.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  * GNU General Public License for more details.
  *
  * You should have received a copy of the GNU General Public License
  * along with this program; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
  */

#include "yawpconfig.h"
#include "yawp.h"
#include "yawpwday.h"
#include <QDomDocument>
#include <QRegExp>
#include <QFile>
#include <QDir>
#include <QMetaType>
#include <QResource>
#include <QProgressBar>
#include <kio/netaccess.h>
#include <kio/job.h>
#include <KProgressDialog>
#include <QDesktopServices>
#include <KFileDialog>
#include <KConfigDialog>
#include <KColorScheme>
#include <Plasma/Theme>


//-----------------------------------------------------------------------------
YaWPConfigDialog::YaWPConfigDialog(KConfigDialog *parent) {
  QWidget *unitsPage = new QWidget;
  QWidget *themePage = new QWidget;
  QWidget *locationPage = new QWidget;
  QWidget *panelPage = new QWidget;

  Ui::UnitsPage::setupUi(unitsPage);
  Ui::ThemePage::setupUi(themePage);
  Ui::LocationPage::setupUi(locationPage);
  Ui::PanelPage::setupUi(panelPage);

  parent->setButtons(KDialog::Ok | KDialog::Cancel | KDialog::Apply);
  parent->addPage(locationPage, i18n("Location"), "preferences-desktop-locale");
  parent->addPage(unitsPage, i18n("Settings"), "preferences-system");
  parent->addPage(panelPage, i18n("Panel"), "preferences-system-windows");
  parent->addPage(themePage, i18n("Theme"), "plasma");

  connect(parent, SIGNAL(applyClicked()), this, SIGNAL(save()));
  connect(parent, SIGNAL(okClicked()), this, SIGNAL(save()));

  //labelVersion->setText(QString("yaWP %1: %2").arg(tr2i18n( "Version", 0 )).arg(YAWP_VERSION));

  m_location = "AFR|ZA|SF004|JOHANNESBURG";
  m_parent = parent;
  m_diag = NULL;

  m_theme_map.insert(0, "default");
  m_theme_map.insert(1, "purple");
  m_theme_map.insert(2, "green");
  m_theme_map.insert(3, "black");
  m_theme_map.insert(4, "blue");
  m_theme_map.insert(5, "red");
  m_theme_map.insert(6, "yellow");
  m_theme_map.insert(7, "funky");
  m_theme_map.insert(8, "naked");

  connect(checkFileTheme, SIGNAL(stateChanged(int)), this, SLOT(fileSelected(int)) );
  connect(buttonFile, SIGNAL(clicked()), this, SLOT(showFileDialog()));
  connect(buttonFind, SIGNAL(clicked()), this, SLOT(find()));
  connect(buttonLogo, SIGNAL(clicked()), this, SLOT(showAccuweatherPage()));
  connect(listCity, SIGNAL(itemClicked(QListWidgetItem *)), this, SLOT(itemClicked(QListWidgetItem *)));
  connect(checkBoxUseCustomColor, SIGNAL(stateChanged(int)), this, SLOT(customColorSelected(int)) );
}


//-----------------------------------------------------------------------------
YaWPConfigDialog::~YaWPConfigDialog() {
}

//-----------------------------------------------------------------------------
void YaWPConfigDialog::setCustomThemeEnabled(bool value) {
    buttonFile->setEnabled(value);
    editFile->setEnabled(value);
    checkBackground->setEnabled(value);
}

//-----------------------------------------------------------------------------
void YaWPConfigDialog::setUseCustomTheme(bool enabled) {
    if(enabled)
        checkFileTheme->setCheckState(Qt::Checked);
    else
        checkFileTheme->setCheckState(Qt::Unchecked);
}

//-----------------------------------------------------------------------------
bool YaWPConfigDialog::useCustomTheme() const {
    return checkFileTheme->checkState() == Qt::Checked;
}

//-----------------------------------------------------------------------------
void YaWPConfigDialog::setCustomTheme(const QString &file) {
    editFile->setText(file);
}

//-----------------------------------------------------------------------------
QString YaWPConfigDialog::customTheme() const {
    return editFile->text();
}

//-----------------------------------------------------------------------------
void YaWPConfigDialog::setUseCustomThemeBackground(bool enabled) {
    if(enabled)
        checkBackground->setCheckState(Qt::Checked);
    else
        checkBackground->setCheckState(Qt::Unchecked);
}

//-----------------------------------------------------------------------------
bool YaWPConfigDialog::useCustomThemeBackground() const {
    return checkBackground->checkState() == Qt::Checked;
}

//-----------------------------------------------------------------------------
void YaWPConfigDialog::fileSelected(int state) {
    setCustomThemeEnabled(state == Qt::Checked);
}

//-----------------------------------------------------------------------------
void YaWPConfigDialog::showFileDialog() {
    QString filename = KFileDialog::getOpenFileName(KUrl(), "*.svg *.svgz|Scalable Vector Graphics", m_parent);
    if (!filename.isEmpty())
        editFile->setText(filename);
}

//-----------------------------------------------------------------------------
void YaWPConfigDialog::setInterval(int inter){
    QString text = QString("%1").arg(inter);
    for(int i = 0; i < comboInterval->count(); i++)
    {
        if(comboInterval->itemText(i) == text)
            comboInterval->setCurrentIndex(i);
    }
}


//-----------------------------------------------------------------------------
int YaWPConfigDialog::interval() const {
    return comboInterval->currentText().toInt();
}


//-----------------------------------------------------------------------------
void YaWPConfigDialog::setTheme(const QString &theme) {
//    for(int i = 0; i < comboTheme->count(); i++)
//        if(comboTheme->itemText(i).toLower() == theme.toLower())
//            comboTheme->setCurrentIndex(i);

    comboTheme->setCurrentIndex(m_theme_map.key(theme));
}


//-----------------------------------------------------------------------------
QString YaWPConfigDialog::theme() const {
    //return comboTheme->currentText();
    return m_theme_map[comboTheme->currentIndex()];
}


//-----------------------------------------------------------------------------
void YaWPConfigDialog::setService(const QString &service) {
    Q_UNUSED(service);
}


//-----------------------------------------------------------------------------
QString YaWPConfigDialog::service() const {
    //TODO Implement more services
    return "accuweather";
}


//-----------------------------------------------------------------------------
void YaWPConfigDialog::setCity(const QString &city) {
    m_city = city;
    editCity->setText(m_city);
}


//-----------------------------------------------------------------------------
QString YaWPConfigDialog::city() const {
    return m_city;
}


//-----------------------------------------------------------------------------
void YaWPConfigDialog::setCountry(const QString &country) {
    m_country = country;
}


//-----------------------------------------------------------------------------
QString YaWPConfigDialog::country() const {
    return m_country;
}


//-----------------------------------------------------------------------------
void YaWPConfigDialog::setLocation(const QString &code) {
    m_location = code;
}


//-----------------------------------------------------------------------------
QString YaWPConfigDialog::location() {
    return m_location;
}


//-----------------------------------------------------------------------------
void YaWPConfigDialog::setUnits(int units) {
    QString text;
    switch(units){
        case YaWPDay::IMPERIAL:
            text = tr2i18n("Imperial");
            break;
        case YaWPDay::METRIC:
            text = tr2i18n("Metric");
            break;
        default:
            text = tr2i18n("Metric");
    }
    for(int i = 0; i < comboUnits->count(); i++)
        if(comboUnits->itemText(i) == text)
            comboUnits->setCurrentIndex(i);
}
//-----------------------------------------------------------------------------
int YaWPConfigDialog::units() const {
    if(comboUnits->currentText() == tr2i18n("Imperial"))
        return YaWPDay::IMPERIAL;
    if(comboUnits->currentText() == tr2i18n("Metric"))
        return YaWPDay::METRIC;
    return YaWPDay::METRIC;
}

//-----------------------------------------------------------------------------
void YaWPConfigDialog::showAccuweatherPage() {
    //TODO need to use the kde stuff here
    QDesktopServices::openUrl(QUrl("http://www.accuweather.com"));
}

//-----------------------------------------------------------------------------
void YaWPConfigDialog::slotData( KIO::Job *job, const QByteArray &data) {
    Q_UNUSED(job);
    m_xmldata += data;
}

//-----------------------------------------------------------------------------
void YaWPConfigDialog::find() {
    m_diag = new KProgressDialog(m_parent, i18nc("Progress dialog caption",
          "Please wait..."), i18nc("Progress dialog description","YaWP is "
          "retrieving the list of cities from the internet, please wait."));
    m_diag->setAllowCancel(false);
    m_diag->setModal(false);
    m_diag->setAutoClose(false);

    if(editCity->text().length() < 1)
        return;
    listCity->clear();

    m_diag->progressBar()->setMinimum(0);
    m_diag->progressBar()->setMaximum(0);
    m_diag->setVisible(true);

    QString urlFormat;
    QString fileName;
    QMap<QString, QString> metaData;

    m_xmldata.clear();

    urlFormat = QString("http://ruan.accu-weather.com/widget/ruan/city-find.asp?location=%1").
      arg(YaWPWeatherService::toPercentEncoding(editCity->text()));
    KUrl url(urlFormat);

    /*NOTE This seams like a bug ? Doing a workarround*/
    qRegisterMetaType<KIO::filesize_t>("KIO::filesize_t");
    qRegisterMetaType<KIO::MetaData>("KIO::MetaData");

    KIO::Job *job = KIO::get( url, KIO::NoReload, KIO::HideProgressInfo );
    metaData.insert( "PropagateHttpHeader", "true" );
    connect(job, SIGNAL(data( KIO::Job*, const QByteArray&)), this, SLOT( slotData( KIO::Job *, const QByteArray & ) ) );
    connect(job, SIGNAL(result(KJob *)), this, SLOT( slotResult(KJob*)));

    job->start();
}

//-----------------------------------------------------------------------------
void YaWPConfigDialog::slotResult( KJob *job) {
    if(job->error() == 0)
        parseInfo(m_xmldata);
    if (m_diag != NULL) {
      m_diag->setVisible(false);
      delete m_diag;
      m_diag = 0;
    }
}

//-----------------------------------------------------------------------------
bool YaWPConfigDialog::parseInfo(const QByteArray &data) {
    bool success = false;

    QDomDocument doc("weatherdoc");
    doc.setContent(data);

    QDomNodeList nodeList = doc.elementsByTagName("location");
    for(int i = 0; i < nodeList.size(); i++) {
        QDomNode node = nodeList.at(i);
        QString city = "";
        QString state = "";
        QString code = "";
        if(node.nodeName() == "location") {
            QDomElement element = node.toElement();
            if(!element.isNull()) {
                QDomAttr attribute = element.attributeNode("city");
                if(!attribute.isNull()) {
                    city = attribute.value();
                }
                attribute = element.attributeNode("state");
                if(!attribute.isNull()) {
                    state = attribute.value();
                }
                attribute = element.attributeNode("location");
                if(!attribute.isNull()) {
                    code = attribute.value();
                }
            }
        }
        success = true;
        QListWidgetItem * item = new QListWidgetItem;

        QString thecountry;
        QRegExp regex("([\\w\\s]*)(.*)");
        if(regex.indexIn(state) > -1)
          thecountry = i18nc("Country or state", regex.cap(1).toUtf8().constData())
            + regex.cap(2);
        else
          thecountry = state;
        item->setText(i18nc("City Country/State in the search dialog",
              "%1 %2", city, thecountry));

        item->setData(LOCATION_DATA, code);
        item->setData(CITY_DATA, city);
        QRegExp regx("(\\w*)(.*)");
        if(regx.indexIn(state) > -1) {
            state = regx.cap(1);
        }
        item->setData(COUNTRY_DATA, state);
        listCity->insertItem(0, item);
    }

    return success;
}

void YaWPConfigDialog::itemClicked(QListWidgetItem *item) {
    Q_UNUSED(item);
    if (item == NULL) {
        editCity->setText("");
        m_location = m_city = m_country = QString ();
    } else {
        m_location = item->data(LOCATION_DATA).toString();
        m_city = item->data(CITY_DATA).toString();
        m_country = item->data(COUNTRY_DATA).toString();
        editCity->setText(m_city);
    }
}

//-----------------------------------------------------------------------------
// Panel mode option
//-----------------------------------------------------------------------------
  void YaWPConfigDialog::setPanelActualTemp(bool enabled) {
    radioActualTemp->setChecked(enabled);
};

//-----------------------------------------------------------------------------
  bool YaWPConfigDialog::panelActualTemp() const {
    return radioActualTemp->isChecked();
};

//-----------------------------------------------------------------------------
  void YaWPConfigDialog::setPanelActualIcon(bool enabled) {
    radioActualIcon->setChecked(enabled);
};

//-----------------------------------------------------------------------------
  bool YaWPConfigDialog::panelActualIcon() const {
    return radioActualIcon->isChecked();
};

//-----------------------------------------------------------------------------
  void YaWPConfigDialog::setPanelActualBoth(bool enabled) {
    radioActualBoth->setChecked(enabled);
};

//-----------------------------------------------------------------------------
  bool YaWPConfigDialog::panelActualBoth() const {
    return radioActualBoth->isChecked();
};

//-----------------------------------------------------------------------------
  void YaWPConfigDialog::setPanelForecastTemp(bool enabled) {
    radioForecastTemp->setChecked(enabled);
};

//-----------------------------------------------------------------------------
  bool YaWPConfigDialog::panelForecastTemp() const {
    return radioForecastTemp->isChecked();
};

//-----------------------------------------------------------------------------
  void YaWPConfigDialog::setPanelForecastIcon(bool enabled) {
    radioForecastIcon->setChecked(enabled);
};

//-----------------------------------------------------------------------------
  bool YaWPConfigDialog::panelForecastIcon() const {
    return radioForecastIcon->isChecked();
};

//-----------------------------------------------------------------------------
  void YaWPConfigDialog::setPanelForecastBoth(bool enabled) {
    radioForecastBoth->setChecked(enabled);
};

//-----------------------------------------------------------------------------
  bool YaWPConfigDialog::panelForecastBoth() const {
    return radioForecastBoth->isChecked();
};

//-----------------------------------------------------------------------------
  void YaWPConfigDialog::setPanelForecastDays(int days) {
    QString text = QString("%1").arg(days);
    for(int i = 0; i < comboForecastDays->count(); i++)
      if(comboForecastDays->itemText(i) == text)
         comboForecastDays->setCurrentIndex(i);

};

//-----------------------------------------------------------------------------
  int YaWPConfigDialog::panelForecastDays() const {
    return comboForecastDays->currentText().toInt();
};

//-----------------------------------------------------------------------------
// Custom colors options
//-----------------------------------------------------------------------------

void YaWPConfigDialog::setCustomColorsEnabled(bool value) {

    label_comboColor->setEnabled(value);
    label_comboLowerColor->setEnabled(value);

    comboColor->setEnabled(value);
    comboLowerColor->setEnabled(value);
}

//-----------------------------------------------------------------------------
void YaWPConfigDialog::customColorSelected(int state) {
    setCustomColorsEnabled(state == Qt::Checked);
}

//-----------------------------------------------------------------------------

  void YaWPConfigDialog::setUseCustomColor(bool enabled) {
    checkBoxUseCustomColor->setChecked(enabled);
};

//-----------------------------------------------------------------------------
  bool YaWPConfigDialog::useCustomColor() const {
    return checkBoxUseCustomColor->isChecked();
};

//-----------------------------------------------------------------------------
  void YaWPConfigDialog::setFontColor(QColor color){
  comboColor->setColor(color);
};

//-----------------------------------------------------------------------------
  QColor YaWPConfigDialog::fontColor() const {
    if (useCustomColor()) {
            return comboColor->color();
    } else {
        return KColorScheme(QPalette::Active, KColorScheme::View, Plasma::Theme::defaultTheme()->colorScheme()).foreground().color();
    }
};

//-----------------------------------------------------------------------------
  void YaWPConfigDialog::setFontLowerColor(QColor color){
  comboLowerColor->setColor(color);
};

//-----------------------------------------------------------------------------
  QColor YaWPConfigDialog::fontLowerColor() const {
    if (useCustomColor()) {
            return comboLowerColor->color();
    } else {
        return KColorScheme(QPalette::Active, KColorScheme::View, Plasma::Theme::defaultTheme()->colorScheme()).foreground(KColorScheme::InactiveText).color();
    }
};

//-----------------------------------------------------------------------------
  void YaWPConfigDialog::setPanelToolTipSatellite(bool enabled){
    checkBoxToolTipSatellite->setChecked(enabled);

};

//-----------------------------------------------------------------------------
  bool YaWPConfigDialog::panelToolTipSatellite() const {
    return checkBoxToolTipSatellite->isChecked();
};

//checkBoxToolTipSatellite


//-----------------------------------------------------------------------------
