
/*
 *  devicelisting.cpp
 *
 *  Copyright (C) 2009 David Hubner <hubnerd@ntlworld.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

#include "devicelisting.h"

DeviceListing::DeviceListing(QWidget *parent, InfoPanel *info) : 
  QTreeWidget(parent), iPanel(info) {
    
    // Check if clicked
    connect(this,SIGNAL(itemClicked(QTreeWidgetItem *,int)),this,SLOT(deviceItemClicked(QTreeWidgetItem *,int)));
    
    // Check if item is added
    connect(Solid::DeviceNotifier::instance(),SIGNAL(deviceAdded(QString)),this,SLOT(deviceAddedSlot(QString)));
  
    // Check if item is removed
    connect(Solid::DeviceNotifier::instance(),SIGNAL(deviceRemoved(QString)),this,SLOT(deviceRemovedSlot(QString)));
    
    createMenuActions();
    setHeaderLabels(QStringList(i18n("Devices")));
    populateListing();   
}

DeviceListing::~DeviceListing() {
  
  clear();
}

void DeviceListing::createMenuActions() {
 
  colAct = new QAction(i18n("Collapse All"), this);
  connect(colAct, SIGNAL(triggered()), this, SLOT(collapseAllDevices()));
  
  expAct = new QAction(i18n("Expand All"), this);
  connect(expAct, SIGNAL(triggered()), this, SLOT(expandAllDevices()));
  
  allAct = new QAction(i18n("Show All Devices"), this);
  connect(allAct, SIGNAL(triggered()), this, SLOT(showAllDevices()));
  
  relAct = new QAction(i18n("Show Relevent Devices"), this);
  connect(relAct, SIGNAL(triggered()), this, SLOT(showReleventDevices()));
}

void DeviceListing::contextMenuEvent(QContextMenuEvent *event) {
  
  QMenu menu(this);
  
  menu.addAction(colAct);
  menu.addAction(expAct);
  menu.addAction(allAct);
  menu.addAction(relAct);
  menu.exec(event->globalPos());
}

QTreeWidgetItem *DeviceListing::createListItems(Solid::DeviceInterface::Type type) {

      switch(type) {
	case Solid::DeviceInterface::Processor: 
	  return new SolProcessorDevice(type);
	case Solid::DeviceInterface::StorageDrive:
	  return new SolStorageDevice(type);
	case Solid::DeviceInterface::NetworkInterface:
	  return new SolNetworkDevice(type);
	case Solid::DeviceInterface::AudioInterface:
	  return new SolAudioDevice(type);
	case Solid::DeviceInterface::Camera:
	  return new SolCameraDevice(type);
	case Solid::DeviceInterface::PortableMediaPlayer:
	  return new SolMediaPlayerDevice(type);
	case Solid::DeviceInterface::Button:
	  return new SolButtonDevice(type);
	case Solid::DeviceInterface::Battery:
	  return new SolBatteryDevice(type);
	case Solid::DeviceInterface::AcAdapter:
	  return new SolAcAdapterDevice(type);
	case Solid::DeviceInterface::DvbInterface:
	  return new SolDvbDevice(type);
	case Solid::DeviceInterface::SerialInterface:
	  return new SolSerialDevice(type);
	case Solid::DeviceInterface::SmartCardReader:
	  return new SolSmartCardDevice(type);
	case Solid::DeviceInterface::Video:
	  return new SolVideoDevice(type);
	default:
	  return new SolDevice(type,i18n("Unknown"));
      }
}

void DeviceListing::populateListing(show showStatus) {
 
  Solid::DeviceInterface::Type needHardware[] = { 
    Solid::DeviceInterface::Processor,
    Solid::DeviceInterface::StorageDrive,
    Solid::DeviceInterface::NetworkInterface,
    Solid::DeviceInterface::AudioInterface,
    Solid::DeviceInterface::Video,
    Solid::DeviceInterface::SerialInterface,
    Solid::DeviceInterface::SmartCardReader,
    Solid::DeviceInterface::DvbInterface,
    Solid::DeviceInterface::Button,
    Solid::DeviceInterface::Battery, 
    Solid::DeviceInterface::AcAdapter,
    Solid::DeviceInterface::PortableMediaPlayer,
    Solid::DeviceInterface::Camera
  };
  
  clear();
 
  for(unsigned int i=0;i<(sizeof(needHardware)/sizeof(Solid::DeviceInterface::Type));i++) { 
    QTreeWidgetItem *tmpDevice = createListItems(needHardware[i]);
    deviceMap[needHardware[i]] = static_cast<SolDevice *>(tmpDevice); 
    if((tmpDevice->childCount() > 0) || (showStatus==ALL)) {
     addTopLevelItem(tmpDevice);
    }
  }
}

void DeviceListing::deviceItemClicked(QTreeWidgetItem *listItemIn ,int columnIn) {

  Q_UNUSED(columnIn);
  
  SolDevice *listItem = static_cast<SolDevice *>(listItemIn);
  if(listItem->isDeviceSet()) {
    
    iPanel->setTopInfo(listItem->getDeviceIcon(),listItem->getDevice());
    
    QVListLayout *bottomLay = listItem->getInfoPanelLayout();
    if(!bottomLay) return;
    
    iPanel->setBottomInfo(bottomLay);
  }
}

void DeviceListing::deviceAddedSlot(QString udi) {
  
  SolidHelper *solhelp = new SolidHelper();
  QList<Solid::Device> list = Solid::Device::allDevices();
  
  foreach(Solid::Device dev, list) {
    
    if(dev.udi() == udi) {
      Solid::DeviceInterface::Type deviceType = solhelp->deviceType(&dev);
      QTreeWidgetItem *parent = getTreeWidgetItem(this,dev.parentUdi());       

      // Incase of bad index
      if(deviceMap[deviceType] == NULL) {
	QTreeWidgetItem *topItem = topLevelItem(0);
	if(topItem == 0) { delete solhelp; return; }
	deviceMap[deviceType] = static_cast<SolDevice *>(topItem);
      }
      
      switch(deviceType) {
	case Solid::DeviceInterface::Processor: 
	  new SolProcessorDevice(deviceMap[deviceType],dev); break;
	case Solid::DeviceInterface::NetworkInterface:
	  new SolNetworkDevice(deviceMap[deviceType],dev); break;
	case Solid::DeviceInterface::AudioInterface:
	  deviceMap[deviceType]->addItem(dev); break;
	case Solid::DeviceInterface::Camera:
	  new SolCameraDevice(deviceMap[deviceType],dev); break;
	case Solid::DeviceInterface::PortableMediaPlayer:
	  new SolMediaPlayerDevice(deviceMap[deviceType],dev); break;
	case Solid::DeviceInterface::Button:
	  new SolButtonDevice(deviceMap[deviceType],dev); break;
	case Solid::DeviceInterface::Battery:
	  new SolBatteryDevice(deviceMap[deviceType],dev); break;
	case Solid::DeviceInterface::AcAdapter:
	 new SolAcAdapterDevice(deviceMap[deviceType],dev); break;
	case Solid::DeviceInterface::DvbInterface:
	  new SolDvbDevice(deviceMap[deviceType],dev); break;
	case Solid::DeviceInterface::SerialInterface:
	  new SolSerialDevice(deviceMap[deviceType],dev); break;
	case Solid::DeviceInterface::SmartCardReader:
	  new SolSmartCardDevice(deviceMap[deviceType],dev); break;
	case Solid::DeviceInterface::Video:
	  new SolVideoDevice(deviceMap[deviceType],dev); break;
	case Solid::DeviceInterface::StorageDrive:
	  new SolStorageDevice(deviceMap[deviceType],dev,SolStorageDevice::NOCHILDREN); break;
	case Solid::DeviceInterface::StorageVolume:
	  if(parent == NULL) break;
	  new SolVolumeDevice(parent,dev); break;  
	default:
	break;
      }
    }
  }
  delete solhelp;
}

void DeviceListing::deviceRemovedSlot(QString udi) {

  QTreeWidgetItem *item = getTreeWidgetItem(this,udi);
  if(item == NULL) return;
  
  delete item;
}

void DeviceListing::collapseAllDevices() {
  
  collapseAll();
}

void DeviceListing::expandAllDevices() {
 
  expandAll();
}

void DeviceListing::showAllDevices() {
 
  populateListing(ALL);
}

void DeviceListing::showReleventDevices() {
 
  populateListing(RELEVENT);
}