/*
   FALCON - The Falcon Programming Language.
   FILE: sequence.h

   Definition of abstract sequence class.
   -------------------------------------------------------------------
   Author: Giancarlo Niccolai
   Begin: 2007-12-01

   -------------------------------------------------------------------
   (C) Copyright 2004: the FALCON developers (see list in AUTHORS file)

   See LICENSE file for licensing details.
*/

/** \file
   Definition of abstract sequence class.
*/

#ifndef flc_sequence_H
#define flc_sequence_H

#include <falcon/setup.h>
#include <falcon/falcondata.h>
#include <falcon/citerator.h>

namespace Falcon {

/** Abstract sequence class.
   A sequence is a special user data wich is used as internal mean
   by sequence oriented falcon classes. It may be also used by
   extension code to create special lists or generator objects.

   The sequence must be able to create a CoreIterator object for
   itself. The CoreIterator will be used internaly by the VM or
   eventually wrapped in a Falcon iterator object and then
   given to the script.
*/

class FALCON_DYN_CLASS Sequence: public FalconData
{
public:
   /** Mark this class as a sequence. */
   virtual bool isSequence() const { return true; }

   /** Returns the first element of the sequence.
      If the sequence has not an underlying storage, it may
      generate a temporary item, as the item is immediately
      copied into some destination by the caller.

      Guarantees are taken so that this method is never called
      when v_empty() returns false.
      \return a valid reference to the first item of the sequence
   */
   virtual const Item &front() const = 0;

   /** Returns the first element of the sequence.
      If the sequence has not an underlying storage, it may
      generate a temporary item, as the item is immediately
      copied into some destination by the caller.

      This method is never used by the engine to modify the
      underlying item.

      Guarantees are taken so that this method is never called
      when v_empty() returns false.
      \return a valid reference to the first item of the sequence
   */
   virtual const Item &back() const = 0;

   /** Gets a CoreIterator valid for this sequence.
      \param tail if false, get an iterator to the first element,
         else get an iterator to the last element.
      \return a subclass of a CoreIterator that can iterate on this sequence.
   */
   virtual CoreIterator *getIterator( bool tail = false ) = 0;

   /** Inserts an element in a position indicated by the iterator.
      The implementation must check that the iterator is a valid
      iterator created by this object and pointing to a valid position.

      Insertion happens at given position, shifting all the remaining
      elements forward; after a succesful insert, the iterator must
      point to the newly inserted element, and the previously current
      element is found safely in the next() position of the iterator.

      Valid iterators (generated by this owner) pointing to invalid
      positions must be treated as pointing to last-past-one element;
      insertion causes append on tail, and at return they must be
      valid and point to the last valid element (the one just inserted).

      If the iterator cannot be used, in example because their owner is
      not this item, the function must return immediately false; in the
      right context, an appropriate error will be raised by the caller.

      \param iterator an iterator (possibily invalid or not generated by
         this class).
      \param data the item to be inserted
      \return true if the iterator was valid for this object.
   */
   virtual bool insert( CoreIterator *iter, const Item &data ) = 0;

   /** Deletes the element at position indicated by the iterator.
      The implementation must check that the iterator is a valid
      iterator created by this object and pointing to a valid position.

      Deletion happens at given position, shifting all the remaining
      elements backward; after a succesful erase, the iterator must
      point to the element that was previously next in the series,
      or must be invalidated if the removed element was the last.

      If the sequence is empty or if the iterator cannot be used, in
      example because their owner is
      not this item, the function must return immediately false; in the
      right context, an appropriate error will be raised by the caller.

      \param iterator an iterator (possibily invalid or not generated by
         this class).
      \param data the item to be inserted
      \return true if the iterator was valid for this object.
   */
   virtual bool erase( CoreIterator *iter ) = 0;

   /** Removes all the items in the sequence. */
   virtual void clear() = 0;

   /** Tells if the series is empty.
      \return false if there is at least one valid item in the series,
         false otherwise.
   */
   virtual bool empty() const =0;
};

}

#endif

/* end of sequence.h */
