/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

// C++ Headers
#include <iostream>
#include <fstream>
#include <cstdlib>

// GTLCore Headers
#include <GTLCore/AbstractImage.h>
#include <GTLCore/Debug.h>
#include <GTLCore/Image.h>
#include <GTLCore/PixelDescription.h>
#include <GTLCore/Region.h>

// OpenShiva Headers
#include <OpenShiva/Debug.h>
#include <OpenShiva/Kernel.h>
#include <OpenShiva/Version.h>

// GTLImageIO Headers
#include <GTLImageIO/ImageDC.h>
#include <GTLImageIO/ImageDCRegistry.h>
#include <OpenShiva/LibrariesManager.h>
#include <GTLCore/CompilationMessages.h>

void printVersion()
{
  std::cout << OpenShiva::LibraryShortName() << " - " << OpenShiva::LibraryName() << " - " << OpenShiva::LibraryVersionString() << std::endl;
  std::cout << OpenShiva::LibraryCopyright() << std::endl;
  std::cout << "Shiva Version : " << OpenShiva::LanguageVersion() << std::endl;
}
void printHelp()
{
  std::cout << "Usage : shivatester [option] fileName.shiva" << std::endl;
  std::cout << std::endl;
  std::cout << "Options : " << std::endl;
  std::cout << "  -r --run-test           run the function 'run-test'" << std::endl;
  std::cout << "  -c --compare            compare the result of the program with a file stored on disk" << std::endl;
//   std::cout << "  -L --module-dir   add a location where to find modules" << std::endl;
  std::cout << std::endl;
  std::cout << "  -h --help               print this message" << std::endl;
  std::cout << "  -v --version            print the version information" << std::endl;
}
#define ARG_IS(a,b) argv[ai] == GTLCore::String(a) or argv[ai] == GTLCore::String(b)

enum ShivaTesterMode {
  STM_DEFAULT,
  STM_RUN_TEST,
  STM_COMPARE_RESULT
};

int main(int argc, char** argv)
{
  GTLCore::String fileName = "";
  std::list< GTLCore::String > fileNames;
  ShivaTesterMode mode = STM_DEFAULT;
  for(int ai = 1; ai < argc; ai++)
  {
    if(ARG_IS("-h","--help"))
    {
      printHelp();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-c","--compare") ) {
      mode = STM_COMPARE_RESULT;
    } else if(ARG_IS("-v","--version"))
    {
      printVersion();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-L", "--module-dir")) {
      if( ai == argc )
      {
        std::cerr << "Expected directory after -L --module-dir." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        OpenShiva::LibrariesManager::instance()->addDirectory(argv[ai]);
      }
    } else {
      fileNames.push_back( argv[ai] );
    }
  }
  if( fileNames.size() > 0 )
  {
    fileName = *( fileNames.begin() );
    fileNames.pop_front();
  }
  if( fileName == "" or (fileNames.size() < 1 and mode == STM_COMPARE_RESULT) )
  {
    std::cerr << "Invalid command line parameters." << std::endl;
    printHelp();
  } else {
    GTLCore::String source;
    std::ifstream in;
    in.open(fileName.c_str() );
    if(not in)
    {
      std::cerr << "Impossible to open file " << fileName << std::endl;
      return EXIT_FAILURE;
    }
    std::string str;
    std::getline(in,str);
    while ( in ) {
      source += str;
      source += "\n";
      std::getline(in,str);
    }
    
    GTLCore::String errMsg;
    GTLCore::RegionI region;
    GTLCore::AbstractImage* resultImage = 0;
    int channels = 4;
    if(mode == STM_COMPARE_RESULT)
    {
      GTLCore::String resultFileName = *( --fileNames.end() );
      fileNames.pop_back();
      const GTLImageIO::ImageDC* decoder = GTLImageIO::ImageDCRegistry::instance()->decoder( resultFileName );
      if( not decoder )
      {
        std::cerr << "Can't find decoder for " << resultFileName << std::endl;
        return EXIT_FAILURE;
      }
      resultImage = decoder->decode( resultFileName, &region, &errMsg );
      if(not resultImage)
      {
        std::cerr << "Can't open result file: " << resultFileName << std::endl;
        return EXIT_FAILURE;
      }
      channels = resultImage->pixelDescription().channels();
    }
    OpenShiva::Kernel p(channels);;
    p.setSource( source );
    p.compile();
    if(not p.isCompiled())
    {
      std::cout << "Error: " << std::endl << p.compilationMessages().toString() << std::endl;
      return EXIT_FAILURE;
    }
    switch( mode )
    {
      case STM_COMPARE_RESULT:
      {
        GTL_ASSERT(resultImage);
        GTLCore::Image image(region.columns(), region.rows(), resultImage->pixelDescription() );
        std::list<const GTLCore::AbstractImage*> images;
        std::list<GTLCore::RegionI> inputDOD;
        for( std::list< GTLCore::String >::iterator it = fileNames.begin();
            it != fileNames.end(); ++it )
        {
          const GTLImageIO::ImageDC* decoder = GTLImageIO::ImageDCRegistry::instance()->decoder( *it );
          if( not decoder )
          {
            std::cerr << "Can't find decoder for " << *it << std::endl;
            return EXIT_FAILURE;
          }
          GTLCore::RegionI region;
          GTLCore::AbstractImage* image = decoder->decode( *it, &region, &errMsg );
          if( not image )
          {
            std::cerr << "Can't read image " << *it << " : " << errMsg << std::endl;
            return EXIT_FAILURE;
          }
          images.push_back( image );
          inputDOD.push_back( region );
        }
        if( p.hasGeneratedFunction() )
        {
          GTLCore::RegionF gregion = p.generated();
          if( gregion.toRegionI() != region )
          {
            std::cerr << "Generated region (" << gregion << ") and result images are different (" << region << ")" << std::endl;
            return EXIT_FAILURE;
          }
        }
        if( p.hasChangedFunction() )
        {
          GTLCore::RegionF cregion;
          int i = 0;
          for( std::list<GTLCore::RegionI>::iterator it = inputDOD.begin(); it != inputDOD.end(); ++it, ++i)
          {
            cregion += p.changed( *it, i, inputDOD );
          }
          if( cregion.toRegionI() != region )
          {
            std::cerr << "Changed region (" << cregion << ") and result images are different (" << region << ")" << std::endl;
            return EXIT_FAILURE;
          }
        }
        std::cout << "Start evaluation..." << std::endl;
        p.evaluatePixels( region, images, &image );
        std::cout << "Start comparison..." << std::endl;
        int errorCount = image.compare( resultImage, region );
        if( errorCount != 0 )
        {
          std::cout << errorCount << " different pixels" << std::endl;
          return EXIT_FAILURE;
        }
        break;
      }
      case STM_RUN_TEST:
      case STM_DEFAULT:
        if( p.hasTestFunction() )
        {
          return p.runTest();
        } else {
          std::cout << "No test function" << std::endl;
          return EXIT_FAILURE;
        }
    }
  }
  return EXIT_SUCCESS;
}
