/************************************
libcommie2 - an interface to the democratic peoples republic of oauth
Copyright (C) 2010 Chris Fuenty <zimmy@zimmy.co.uk>

This class handles the construction of HMAC-SHA1 OAuth Requests

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
************************************/

#include "oauth.h"

CommieLib::CommieLib(QByteArray consumerKey, QByteArray consumerSecret, QObject *parent)
    :QObject(parent),
    m_consumerKey(consumerKey),
    m_consumerSecret(consumerSecret),
    m_busy(false),
    m_action(ACTION_NONE)
{
    m_http = new QNetworkAccessManager(this);
    connect(m_http, SIGNAL(finished(QNetworkReply*)), this, SLOT(finished(QNetworkReply*)));
}

CommieLib::~CommieLib()
{

}

void CommieLib::obtainRequestToken(QUrl requestUrl, int httpMethod, ParameterMap params)
{
    token_t token;
    if(httpMethod == HTTP_POST)
        doPost(requestUrl, token, params);
    else
        doGet(requestUrl, token, params);

    m_action = ACTION_REQUEST;
}

void CommieLib::obtainAccessToken(QUrl requestUrl, int httpMethod, token_t token, ParameterMap params)
{
    if(httpMethod == HTTP_POST)
        doPost(requestUrl, token, params);
    else
        doGet(requestUrl, token, params);

    m_action = ACTION_ACCESS;
}

void CommieLib::doGet(QUrl requestUrl, token_t token, ParameterMap extras)
{
    /* get a parameter map with all the nice little extras built in */
    ParameterMap params = constructSignatureAndBuildParams(requestUrl, HTTP_GET, token, extras);

    /* construct this into a nice little url encoded form post */
    QStringList baList;
    foreach(QByteArray key, params.keys())
    {
        QString temp;
        temp.append(key.toPercentEncoding());
        temp.append("=\"");
        temp.append(params.value(key).toPercentEncoding());
        temp.append("\"");
        baList.append(temp);
    }
    QString sdata = baList.join(", ");

    /* OAuth says unicode aware */
    QByteArray data = sdata.toUtf8();
    data.prepend("OAuth ");

    QNetworkRequest request(requestUrl);
    request.setRawHeader("Authorization", data);
    /* don't know if we should send this */
    //request.setRawHeader("User-Agent", "Cardinal/0.4 (libcommie/0.1)");
    m_http->get(request);
    m_busy = true;
}

QByteArray CommieLib::obtainAuthorizationHeader(QUrl requestUrl, int httpMethod, token_t token, ParameterMap extras)
{
    /* get a parameter map with all the nice little extras built in */
    ParameterMap params = constructSignatureAndBuildParams(requestUrl, httpMethod, token, extras);

    if(httpMethod == HTTP_POST)
    {
        foreach(QByteArray key, extras.keys())
        {
            params.remove(key);
        }
    }

    /* construct this into a nice little url encoded form post */
    QStringList baList;
    QString sdata;
    foreach(QByteArray key, params.keys())
    {
        QString temp;
        temp.append(key.toPercentEncoding());
        temp.append("=\"");
        temp.append(params.value(key).toPercentEncoding());
        temp.append("\"");
        baList.append(temp);
        sdata = baList.join(", ");
    }

    /* OAuth says unicode aware */

    QByteArray data = sdata.toUtf8();
    data.prepend("OAuth ");

    return data;
}

ParameterMap CommieLib::obtainAuthorizationHeaderAsParams(QUrl requestUrl, int httpMethod, token_t token, ParameterMap extras)
{
    /* get a parameter map with all the nice little extras built in */
    ParameterMap params = constructSignatureAndBuildParams(requestUrl, httpMethod, token, extras);

    return params;
}

void CommieLib::doPost(QUrl requestUrl, token_t token, ParameterMap extras)
{
    /* get a parameter map with all the nice little extras built in */
    ParameterMap params = constructSignatureAndBuildParams(requestUrl, HTTP_POST, token, extras);

    /* construct this into a nice little url encoded form post */
    QStringList baList;
    foreach(QByteArray key, params.keys())
    {
        QString temp;
        temp.append(key);
        temp.append("=");
        temp.append(params.value(key));
        baList.append(temp);
    }
    QString sdata = baList.join("&");

    /* OAuth says unicode aware */
    QByteArray data = sdata.toUtf8().toPercentEncoding();

    QNetworkRequest request(requestUrl);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    /* don't know if we should send this */
    //request.setRawHeader("User-Agent", "Cardinal/0.4 (libcommie/0.1)");
    m_http->post(request, data);

    m_busy = true;
}


ParameterMap CommieLib::constructSignatureAndBuildParams(QUrl requestUrl, int httpMethod, token_t token, ParameterMap extra)
{    
    /* we need to construct a base sig here */
    /* append the HTTP method, a standard ampersand, URL encoded host,
       sans query string, then a normal ampersand */
    QByteArray signature;
    if(httpMethod == HTTP_POST)
        signature.append("POST");
    else
        signature.append("GET");
    
    signature.append("&");
    requestUrl.queryItems().clear();
    signature.append(requestUrl.toString().toUtf8().toPercentEncoding());
    signature.append("&");

    /* now we need to append our sorted paramaters, url encoded keys,
       url encoded values, seperated by url encoded equals and URL ENCODED ampersands */
    /* add in some basic params we WILL need */

    uint ts = QDateTime::currentDateTime().toTime_t();
    QByteArray nonce = createNonce(ts);
    ParameterMap params;
    params.insert("oauth_consumer_key", m_consumerKey);
    params.insert("oauth_nonce", nonce);
    params.insert("oauth_signature_method", "HMAC-SHA1");
    params.insert("oauth_timestamp", QByteArray::number(ts));
    params.insert("oauth_version", OAUTH_VERSION);

    /* if we have a token, insert this */
    if(!token.token.isNull())
        params.insert("oauth_token", token.token);

    /* insert our extra parameters */
    params.unite(extra);

    QStringList baList;
    foreach(QByteArray key, params.keys())
    {
        QString temp;
        temp.append(key.toPercentEncoding());
        temp.append("=");
        temp.append(params.value(key).toPercentEncoding());
        baList.append(temp);
    }
    QString data = baList.join("&");

    QByteArray base = data.toUtf8().toPercentEncoding();

    /* if token secret is empty, this should just be consumer secret plus a standard ampersand */
    QByteArray key(m_consumerSecret + "&" + token.secret);

    signature.append(base);
    signature.trimmed();
    /* let's do some nasty casting here */
    unsigned char md_val[EVP_MAX_MD_SIZE];
    unsigned int md_len;

    /* get an HMAC signature, signing our base with our consumer secret */
    QByteArray digest;
    HMAC_CTX ctx;
    HMAC_CTX_init(&ctx);
    HMAC_Init_ex(&ctx, key.data(), key.size(), EVP_sha1(), 0);
    HMAC_Update(&ctx, (unsigned char *)signature.data(), signature.size());
    HMAC_Final(&ctx, md_val, &md_len);
    HMAC_CTX_cleanup(&ctx);

    
    /* append md_len bytes of md_val, casted as a const char* (no data loss should occur) */
    digest.append((const char*)md_val, md_len);

    /*append the signature to our parameters */
    params.insert("oauth_signature", digest.toBase64());

    /* return our completed params */
    return params;
}

QByteArray CommieLib::createNonce(time_t ts)
{
    /* a rather shitty nonce constructor */
    QString nonceStr;
    QByteArray nonce;
    for(int i = 0; i < 5; i++)
    {
        int diff = 'Z'-'A';
        nonceStr.append(QChar('A' + (rand() % diff)));
        nonceStr.append(QChar('A' + (rand() % diff)));
        nonceStr.append(QChar('A' + (rand() % diff)));
        nonceStr.append(QChar('A' + (rand() % diff)));
        nonceStr.append(rand());
    }

    nonceStr.append(QString::number(ts));

    nonce = nonceStr.toUtf8().toBase64();
    return nonce;
}

void CommieLib::finished(QNetworkReply *reply)
{
    QStringList oauthItems = QString(reply->readAll()).split("&");
    int result = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
    m_busy = false;
    m_error = result;
    token_t token;
    switch(m_error)
    {
    case HTTP_OK:
        switch(m_action)
        {
        case ACTION_REQUEST:
            foreach(QString item, oauthItems)
            {
                QStringList data = item.split("=");
                if(data.at(0) == "oauth_token")
                    token.token = data.at(1).toUtf8();
                if(data.at(0) == "oauth_token_secret")
                    token.secret = data.at(1).toUtf8();
            }

            emit hasRequestToken(token);
            break;
        case ACTION_ACCESS:
            foreach(QString item, oauthItems)
            {
                QStringList data = item.split("=");
                if(data.at(0) == "oauth_token")
                    token.token = data.at(1).toUtf8();
                if(data.at(0) == "oauth_token_secret")
                    token.secret = data.at(1).toUtf8();
            }

            emit hasAccessToken(token);
            break;
        case ACTION_COMMAND:
        case ACTION_NONE:
        default:
            break;
        };
    case HTTP_NOT_FOUND:
    case HTTP_UNAUTHORIZED:
    case HTTP_FORBIDDEN:
    default:
        emit error(result, reply->readAll());
        break;
    };
}
