/************************************

cardinal - a proper cross platform twitter client
Copyright (C) 2009-2010 Chris Fuenty <zimmy@zimmy.co.uk>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

************************************/

#include "StatusTableView.h"

StatusTableView::StatusTableView(QWidget *parent, QString Username, QString Password, bool search)
    :QTableWidget(parent),
    m_user(Username),
    m_pass(Password),
    m_isOnline(false),
    m_search(search)
{
    insertColumn(0);

    /* lets get our style on */
    setGridStyle(Qt::SolidLine);
    setShowGrid(true);
    verticalHeader()->hide();
    horizontalHeader()->hide();
    setAlternatingRowColors(true);
    horizontalHeader()->setStretchLastSection(true);
    setSelectionMode(QAbstractItemView::SingleSelection);

    bool useId = (m_search == true ? false : true);
    m_profilePics = new ProfilePictureManager(this, useId);
    if(search == false)
    {
        connect(m_profilePics, SIGNAL(pictureRecieved(unsigned long long int, QByteArray)),
            this, SLOT(pictureRecieved(unsigned long long int, QByteArray)));
    }
    else
    {
        connect(m_profilePics, SIGNAL(pictureRecievedAlt(QString, QByteArray)),
            this, SLOT(pictureRecievedAlt(QString, QByteArray)));

    }
    connect(this, SIGNAL(itemSelectionChanged()), this, SLOT(tableItemActivated()));
    /* let's prepare our identity */
    ident = new IdentityManager(m_search);
    connect(ident, SIGNAL(addNewTweet(tweet_t, int)), this, SLOT(addNewTweet(tweet_t, int)));
    connect(ident, SIGNAL(hideEditor()), this, SIGNAL(hideEditor()));
    connect(ident, SIGNAL(isOnline()), this, SLOT(isOnline()));
    connect(ident, SIGNAL(showInfoDialog(user_t)), this, SLOT(showInfoDialog(user_t)));
    connect(ident, SIGNAL(updateAPIData(apilimit_t)), this, SLOT(updateAPIData(apilimit_t)));
    connect(ident, SIGNAL(working(bool)), this, SIGNAL(working(bool)));
    connect(ident, SIGNAL(showNotification()), this, SLOT(showNotification()));
    connect(ident, SIGNAL(retweetedByMe(unsigned long long int)), this, SLOT(retweetedByMe(unsigned long long int)));
    connect(ident, SIGNAL(idsForRetweet(unsigned long long int, QList<user_t>)), this, SLOT(idsForRetweet(unsigned long long int, QList<user_t>)));
    connect(ident, SIGNAL(statusRemoved(unsigned long long int)), this, SLOT(updateRemoved(unsigned long long int)));

    if(m_search == false)
        SqlArchive::InitializeSqlDatabase(Username);

    action_Reply = new QAction(this);
    action_GetInfoH = new QAction(this);

    action_GetInfoH->setEnabled(false);
    action_Reply->setEnabled(false);
    action_Reply->setText(tr("Reply..."));
    action_Reply->setShortcut(QApplication::translate("Cardinal", "Ctrl+R", 0, QApplication::UnicodeUTF8));
    action_Reply->setIcon(QIcon(QString(":/Icons/16px/Status-Reply")));
    action_GetInfoH->setText(tr("Get Info On..."));
    action_GetInfoH->setIcon(QIcon(QString(":/Icons/16px/Get-Info")));

    connect(action_GetInfoH, SIGNAL(triggered()), this, SLOT(loadUserInfo()));
    connect(action_Reply, SIGNAL(triggered()), this, SLOT(signalReply()));

    row = 0;

    ident->initConnection(Username);
    createViewMenu();

    if(m_search == false)
    {
        QList<tweet_t> tList = SqlArchive::RetrieveStatusUpdates(Username);
        foreach(tweet_t mtw, tList)
        {
            addNewTweet(mtw, TIMELINE_TWEET);
        }
    }
}

StatusTableView::~StatusTableView()
{
    delete ident;
    delete m_profilePics;
}

void StatusTableView::loadUserInfo()
{
    int row = currentRow();
    twInfo = m_tweets.at(row);

    getUserInfo(twInfo.uid);
}

int StatusTableView::findSlotByReply(unsigned long long int reply)
{
    if(m_tweets.count() == 0)
        return 0;

    /* if we don't have the source, bail */
    if(!m_id.contains(reply))
        return -1;

    for(int i = 0; i < m_tweets.count(); i++)
    {
        if(reply == m_tweets.at(i).id)
            return i;
    }

    ident->getTweetById(reply);
    return -1;
}

int StatusTableView::findSlotByOrignal(unsigned long long int id)
{
    if(m_tweets.count() == 0)
        return 0;

    for(int i = 0; i < m_tweets.count(); i++) {
        if(id == m_tweets.at(i).reply)
            return i;
    }

    return 0;
}

int StatusTableView::findSlot(time_t time)
{
    int count = m_tweets.count();
    if(m_tweets.count() == 0)
        return 0;

    for(int i = 0; i < m_tweets.count(); i++) {
        if(time > m_tweets.at(i).time)
            return i;
    }

    return count;
}

tweet_t StatusTableView::categorizeTweet(tweet_t twInfo)
{
    twInfo.categories.append(addCategoryByUser(twInfo.screenName));
    if(QDateTime::fromTime_t(twInfo.time).time().hour() >= 20 || QDateTime::fromTime_t(twInfo.time).time().hour() <= 4)
    {
        twInfo.categories.append(tr("Updates at Night"));
    }
    else
    {
        twInfo.categories.append(tr("Updates at Day"));
    }

    if(twInfo.tweet.contains("RT"))
        twInfo.categories.append(tr("Retweets"));

    if(twInfo.screenName == m_user)
        twInfo.categories.append(tr("My Updates"));

    if(twInfo.favorite == true)
        twInfo.categories.append(tr("Favorites"));

    return twInfo;
}

void StatusTableView::addNewTweet(tweet_t twInfo, int type = TIMELINE_TWEET)
{
    if(twInfo.time == 0)
        return;

    if(twContains(twInfo.id) && type != NEW_STATUS)
        return;

    twInfo = categorizeTweet(twInfo);
    tweet *t = new tweet(this, twInfo, m_user);

    int indent = 0;
    if(twInfo.screenName != ident->identSelf().screenName ||
        type == NEW_STATUS || type == THREAD || ident->isFirstRun() == true)
    {
        /* were assuming twitter and the databases sort this shit properly by time */
        int row;

        /* okay, if we're a thread, let's see if we have our original */
        if(type == THREAD)
        {
            row = findSlotByOrignal(twInfo.id);
            row++;
        }
        else if(twInfo.reply != 0 && type != THREAD)
        {
            /* we're not a thread, hwoever, we have a reply ID, find our reply */
            int trow = findSlotByReply(twInfo.reply);
            /* did we get a row? */
            if(trow == -1)
                row = findSlot(twInfo.time);
            else
                row = trow;

            if(row+1 <= rowCount())
                row++;
        }
        else
        {
            row = findSlot(twInfo.time);
        }
        if(twInfo.categories.contains(tr("Direct Messages")))
            t->prefixDirect();

        if(type == MENTION)
        {
            t->tw.categories.append(tr("Mentions"));
            t->prefixMention();
        }

        if(twInfo.tweet.contains(QString("@%1").arg(m_user)))
        {
            t->tw.categories.append(tr("Mentions"));
            t->prefixMention();
        }

        insertRow(row);
        QModelIndex idx = model()->index(row, 0);
        QSettings settings("Cardinal", "Cardinal");

        m_tweets.insert(row, twInfo);
        m_id.insert(row, twInfo.id);

        if(m_search == false)
            m_profilePics->retrievePicture(twInfo.uid, twInfo.picUri);
        else
            m_profilePics->retrievePictureByName(twInfo.screenName, twInfo.picUri);

        setIndexWidget(idx, t);

        /* indent as needed */
        indent = indentCount(twInfo);

        t->show();

        /* indent our status update */
        if(indent != 0)
        {
            for(int i = 0; i < indent; i++)
                t->indent();
        }

        resizeRowToContents(row);

        /* thanks to people like @williamtm whoring around shitty products
           like the apple iPad, cardinal now has an ignore feature --stitch */

        QStringList ignores = settings.value("IgnoreList").toStringList();
        if(ignores.count() != 0)
        {
            for(int i = 0; i < ignores.count(); i++)
            {
                QRegExp rx(ignores.at(i), Qt::CaseInsensitive);
                rx.setPatternSyntax(QRegExp::RegExp2);
                if(rx.indexIn(twInfo.tweet) != -1)
                {
                    t->ignore();
                    hideRow(row);
                    break;
                }
                rx.setPatternSyntax(QRegExp::Wildcard);
                if(rx.indexIn(twInfo.tweet) != -1)
                {
                    t->ignore();
                    hideRow(row);
                    break;
                }
            }
        }


        if(ident->isFirstRun() == false && twInfo.id != ident->identSelf().id)
            emit addNotification(twInfo);
        connect(t, SIGNAL(requestReply(tweet_t)), this, SIGNAL(reply(tweet_t)));
        connect(t, SIGNAL(requestUserInfo(unsigned long long int)), this, SLOT(getUserInfo(unsigned long long int)));
        connect(t, SIGNAL(retweetUpdate(unsigned long long int)), this, SLOT(retweetUpdate(unsigned long long int)));
        connect(t, SIGNAL(deleteUpdate(unsigned long long int)), this, SLOT(deleteUpdate(unsigned long long int)));
        connect(t, SIGNAL(markFavorite(unsigned long long int, bool)), this, SLOT(markFavorite(unsigned long long int, bool)));
        connect(t, SIGNAL(requestDM(unsigned long long int)), this, SIGNAL(directMsg(unsigned long long int)));
        if(m_search == false)
        {
            SqlArchive::InsertStatusUpdate(twInfo, type, m_user);
            SqlArchive::MarkFavorite(twInfo.id, twInfo.favorite, m_user);
        }
        if(type == NEW_STATUS)
            m_sentTweets.append(twInfo.time);
    }
}

void StatusTableView::submitTweet(QString txt, unsigned long long int id = 0)
{
    lastActiveItem++;
    ident->submitTweet(txt, id);
}

void StatusTableView::submitDM(QString txt, unsigned long long int tid)
{
    ident->submitDM(txt, tid);
}

void StatusTableView::showSortView(QString category)
{
    int count = rowCount();

    if(category == "ALL")
    {
        for(int i = 0; i < count; ++i)
        {
            tweet *t = qobject_cast<tweet*>(cellWidget(i, 0));
            tweet_t tw = t->returnTweet();
            showRow(i);
        }
    }
    else
    {
        for(int i = 0; i < count; ++i)
        {
            tweet *t = qobject_cast<tweet*>(cellWidget(i, 0));
            tweet_t tw = t->returnTweet();
            if(tw.categories.contains(category))
            {
                showRow(i);
            }
            else
            {
                hideRow(i);
            }
        }
    }
}

void StatusTableView::tableItemActivated()
{
    if(lastActiveItem >= 0)
    {
        QModelIndex idx = model()->index(lastActiveItem, 0);
        tweet *t = qobject_cast<tweet*>(indexWidget(idx));
        if(t)
            t->itemDeactivated();
    }

    int row = currentRow();
    twInfo = m_tweets.at(row);
    action_Reply->setEnabled(true);
    action_GetInfoH->setEnabled(true);
    action_GetInfoH->setText(tr("Get Info On ") + twInfo.screenName);

    QModelIndex idx = model()->index(row, 0);
    tweet *t = qobject_cast<tweet*>(indexWidget(idx));
    t->itemActivated();

    lastActiveItem = currentRow();

    m_selectedUser = twInfo.screenName;
    m_replyId = twInfo.id;

    emit selectedChanged();
}

bool StatusTableView::twContains(unsigned long long int id)
{
    if(m_tweets.count() == 0)
        return false;

    if(m_id.indexOf(id) != -1)
        return true;

    return false;
}

void StatusTableView::getUserInfo(unsigned long long int id)
{
    ident->GetUserDetails(id);
}

void StatusTableView::block(unsigned long long int uid, QString user)
{
    Q_UNUSED(user);
    ident->BlockUser(QString::number(uid));
}

void StatusTableView::follow(unsigned long long int uid, QString user)
{
    ident->AddFriendship(uid);
    QMessageBox msg(QMessageBox::Information, tr("Following User"), tr("Congrats, you are now friends with ") + user + tr(". Their updates will be available on your canvas."), QMessageBox::Ok, this);
    msg.exec();
}

void StatusTableView::unfollow(unsigned long long int uid, QString user)
{
    ident->RemoveFriendship(uid);
    QMessageBox msg(QMessageBox::Information, tr("Unfollowing User"), tr("You are no longer friends with ") + user + tr(" and you will not be notified of their updates"), QMessageBox::Ok, this);
    msg.exec();
}

void StatusTableView::showInfoDialog(user_t user)
{
    InfoWnd *wnd = new InfoWnd(user, m_user, this);
    wnd->setModal(false);
    connect(wnd, SIGNAL(finished(int)), this, SLOT(infoFinished(int)));
    connect(wnd, SIGNAL(blockUser(unsigned long long int, QString)), this, SLOT(block(unsigned long long int, QString)));
    connect(wnd, SIGNAL(follow(unsigned long long int, QString)), this, SLOT(follow(unsigned long long int, QString)));
    connect(wnd, SIGNAL(unfollow(unsigned long long int, QString)), this, SLOT(unfollow(unsigned long long int, QString)));
    connect(wnd, SIGNAL(getUserInfo(unsigned long long int)), this, SLOT(getUserInfo(unsigned long long int)));
    wnd->exec();
}

void StatusTableView::resizeRow(QSize size, int row)
{
    setRowHeight(row, size.height());
}

QString StatusTableView::resetString(unsigned int seconds)
{
    if(seconds <= 60)
        return QString(QString::number(seconds) + " Seconds");

    int minutes = seconds / 60;

    if(minutes <= 60)
        return QString(QString::number(minutes) + " Minutes");

    double hours = minutes / 60;

    //hours = round(hours);

    if(hours != 1) {
        return QString(QString::number(hours) + " Hours");
    } else {
        return QString("1 hour");
    }
}

QStringList StatusTableView::addCategoryByUser(QString username)
{
    QStringList categories;
    QSettings settings("Cardinal", "Cardinal");
    settings.beginGroup("Accounts");
    settings.beginGroup(m_user);
    settings.beginGroup("Categories");
    QStringList availableCats = settings.childGroups();
    for (int i = 0; i < availableCats.count(); ++i)
    {
        settings.beginGroup(availableCats[i]);
        QStringList cats = settings.childKeys();
        for (int j = 0; j < cats.count(); ++j)
        {
            if(cats[j].toLower() == username.toLower() && settings.value(cats[j]) == "Username")
            {
                categories.append(availableCats[i]);
                break;
            }
        }
        settings.endGroup();
    }
    settings.endGroup();
    settings.endGroup();
    settings.endGroup();
    return categories;
}

void StatusTableView::signalReply()
{
    emit reply(twInfo);
}

void StatusTableView::isOnline()
{
    emit setOnline();
    m_isOnline = true;
    action_Reply->setEnabled(true);
}

void StatusTableView::reload()
{
    if(ident)
        ident->reload();
}

void StatusTableView::resizeEvent(QResizeEvent *event)
{
    for(int i = 0; i < rowCount(); i++)
    {
        int newWidth = event->size().width();
        tweet *t = qobject_cast<tweet*>(indexWidget(model()->index(i, 0)));
        if(t)
        {
            t->resize(3, newWidth);
            int heightForWidth = t->heightForWidth(newWidth);
            setRowHeight(i, heightForWidth);
        }
    }
    QTableWidget::resizeEvent(event);
}

void StatusTableView::updateAPIData(apilimit_t apiData)
{
    QString text(QString::number(apiData.remainingHits) + tr(" remaining API calls"));
    m_statusBarText = text;
    emit updateStatusBar(text);
}

void StatusTableView::showIgnoredUpdates(bool checked)
{
    if(checked == true)
    {
        for(int i = 0; i < rowCount() - 1; i++)
        {
            QModelIndex idx = model()->index(i, 0);
            tweet *t = qobject_cast<tweet*>(indexWidget(idx));
            t->show();
            t->unignore();
            setRowHidden(i, false);
        }
    }
    else
    {
        for(int i = 0; i < rowCount() - 1; i++)
        {
            QModelIndex idx = model()->index(i, 0);
            tweet *t = qobject_cast<tweet*>(indexWidget(idx));
            QSettings settings("Cardinal", "Cardinal");
            QStringList ignores = settings.value("IgnoreList").toStringList();
            if(ignores.count() != 0)
            {
                for(int j = 0; j < ignores.count(); j++)
                {
                    QRegExp rx(ignores.at(j), Qt::CaseInsensitive);
                    rx.setPatternSyntax(QRegExp::RegExp2);
                    if(rx.indexIn(t->tw.tweet) != -1)
                    {
                        t->ignore();
                        hideRow(i);
                        break;
                    }
                    rx.setPatternSyntax(QRegExp::Wildcard);
                    if(rx.indexIn(t->tw.tweet) != -1)
                    {
                        t->ignore();
                        hideRow(i);
                        break;
                    }
                }
            }
        }
    }
}

void StatusTableView::createViewMenu()
{
    menu_View = new QMenu(this);
    showIgnored = new QAction(this);
    showIgnored->setText(tr("Ignored Status Updates"));
    showIgnored->setObjectName("showIgnored");
    showIgnored->setCheckable(true);
    connect(showIgnored, SIGNAL(triggered(bool)), this, SLOT(showIgnoredUpdates(bool)));

    menu_View->setObjectName("ViewMenu");
    menu_View->menuAction()->setObjectName("ViewMenu");
    menu_View->setTitle(tr("View"));
    menu_View->addAction(showIgnored);
    showIgnored->setChecked(false);
    menu_View->addSeparator();
    menu_View->addAction(tr("All Status Updates"))->setCheckable(true);
    menu_View->actions()[2]->setChecked(true);
    menu_View->addSeparator();
    menu_View->addAction(tr("My Updates"))->setCheckable(true);
    menu_View->addAction(tr("Direct Messages"))->setCheckable(true);
    menu_View->addAction(tr("Mentions"))->setCheckable(true);
    menu_View->addAction(tr("Favorites"))->setCheckable(true);
    menu_View->addAction(tr("Updates at Night"))->setCheckable(true);
    menu_View->addAction(tr("Updates at Day"))->setCheckable(true);
    menu_View->addAction(tr("Retweets"))->setCheckable(true);
    menu_View->addSeparator();
    connect(menu_View, SIGNAL(triggered(QAction*)), this, SLOT(changeView(QAction*)));

    /* View Items */
    QSettings settings("Cardinal", "Cardinal");
    settings.beginGroup("Accounts");
    settings.beginGroup(m_user);
    settings.beginGroup("Categories");
    QStringList cats = settings.childGroups();
    for (int i = 0; i < cats.count(); ++i)
    {
        menu_View->addAction(cats[i])->setCheckable(true);
    }
    settings.endGroup();
    settings.endGroup();
    settings.endGroup();
}

void StatusTableView::changeView(QAction *action)
{
    if(action->objectName() == QString("showIgnored"))
        return;

    if(action->text() == tr("All Status Updates"))
    {
        showSortView("ALL");
        setAllUnchecked();
        action->setChecked(true);
    }
    else
    {
        showSortView(action->text());
        setAllUnchecked();
        action->setChecked(true);
    }
}

void StatusTableView::setAllUnchecked()
{
    QList<QAction*> lst = menu_View->actions();
    for(int i = 0; i < lst.count(); ++i)
    {
        if(lst[i]->objectName() != "showIgnored")
            lst[i]->setChecked(false);
    }
}

/* void StatusTableView::deleteUpdate()
{

} */

void StatusTableView::retweetUpdate(unsigned long long int id)
{
    ident->Retweet(id);
}

void StatusTableView::retweetedByMe(unsigned long long int id)
{
    for(int i = 0; i < m_tweets.count(); i++)
    {
        tweet_t twInfo = m_tweets.at(i);
        if(twInfo.id == id)
        {
            if(!twInfo.categories.contains("Retweets"))
                twInfo.categories.append("Retweets");
            QModelIndex idx = model()->index(i, 0);
            tweet *t = qobject_cast<tweet*>(indexWidget(idx));
            t->setRetweetedByMe(true);
            break;
        }
    }
}

void StatusTableView::idsForRetweet(unsigned long long int id, QList<user_t> users)
{
    for(int i = 0; i < m_tweets.count(); i++)
    {
        tweet_t twInfo = m_tweets.at(i);
        if(twInfo.id == id)
        {
            QModelIndex idx = model()->index(i, 0);
            tweet *t = qobject_cast<tweet*>(indexWidget(idx));
            t->setRetweeters(users);
            if(!t->tw.categories.contains("Retweets"))
                t->tw.categories.append("Retweets");
            break;
        }
    }
}

void StatusTableView::pictureRecieved(unsigned long long int id, QByteArray picture)
{
    for(int i = 0; i < m_tweets.count(); i++)
    {
        tweet_t twInfo = m_tweets.at(i);
        if(twInfo.uid == id)
        {
            QModelIndex idx = model()->index(i, 0);
            tweet *t = qobject_cast<tweet*>(indexWidget(idx));
            if(t)
                t->setProfileImage(picture);
        }
    }
}

void StatusTableView::pictureRecievedAlt(QString name, QByteArray picture)
{
    if(m_search == false)
        return;

    for(int i = 0; i < m_tweets.count(); i++)
    {
        tweet_t twInfo = m_tweets.at(i);
        if(twInfo.screenName == name)
        {
            QModelIndex idx = model()->index(i, 0);
            tweet *t = qobject_cast<tweet*>(indexWidget(idx));
            if(t)
                t->setProfileImage(picture);
        }
    }
}

int StatusTableView::indentCount(tweet_t twInfo)
{
    /* we initially have a 0 indent count */
    int indent = 0;

    /* If we are not in reply to anything, don't indent */
    if(twInfo.reply == 0)
        return 0;

    /* do we have a reply? */
    if(twContains(twInfo.reply))
    {
        indent = 1;
        indent += indentCount(updateForId(twInfo.reply));
    }

    /* we max out at 5 */
    if(indent > 5)
        indent = 5;

    return indent;
}

tweet_t StatusTableView::updateForId(unsigned long long id)
{
    foreach(tweet_t tw, m_tweets)
    {
        if(tw.id == id)
            return tw;
    }

    tweet_t twInfo;
    return twInfo;
}

void StatusTableView::deleteUpdate(unsigned long long id)
{
    ident->DeleteStatus(id);
}

void StatusTableView::updateRemoved(unsigned long long int id)
{
    for(int i = 0; i < m_tweets.count(); i++)
    {
        tweet_t twInfo = m_tweets.at(i);
        if(twInfo.id == id)
        {
            QModelIndex idx = model()->index(i, 0);
            tweet *t = qobject_cast<tweet*>(indexWidget(idx));
            t->setParent(0);
            delete t;
            t = NULL;
            removeRow(i);
            m_tweets.removeAt(i);
            break;
        }
    }

    SqlArchive::DeleteUpdate(id, m_user);
}

void StatusTableView::markFavorite(unsigned long long int id, bool favorite)
{
    ident->MarkAsFavorite(id, favorite);
}

void StatusTableView::search(QString search)
{
    ident->search(search);
}
