/*
 * Compiz cube model plugin
 *
 * cubemodel.c
 *
 * This plugin displays wavefront (.obj) 3D mesh models inside of
 * the transparent cube.
 *
 * Copyright : (C) 2008 by David Mikos
 * E-mail    : infiniteloopcounter@gmail.com
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

/*
 * Model loader code based on cubemodel/cubefx plugin "cubemodelModel.c.in"
 * code - originally written by Joel Bosvel (b0le).
 */

/*
 * Note - The textures in animations are only
 *        displayed from the first frame.
 */

#define _GNU_SOURCE /* for strndup */

#include <errno.h>
#include <pthread.h>

#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

#include "cubemodel.h"


/**************************
* Gets path from object   *
* and file name from file *
* and returns full path   *
* to the file             *
**************************/

static char *
findPath (char *object,
	  char *file)
{
    char *filePath; /* string containing /full/path/to/file */
    int  i;

    if (!file || !object)
	return NULL;

    if (file[0] == '/')
	return strdup (file);

    filePath = strdup (object);
    if (!filePath)
	return NULL;

    for (i = strlen (filePath) - 1; i >= 0; i--)
    {
	if (filePath[i] == '/')
	{
	    filePath[i + 1]='\0'; /* end string at last /
				     (gives path to object) */
	    break;
	}
    }

    filePath = (char* ) realloc (filePath,
  				 sizeof (char) *
				 (strlen (filePath) + strlen (file) + 1));
    if (!filePath)
	return NULL;

    strcat (filePath, file);

    return filePath;
}

static int
addNumToString (char         **sp,
                unsigned int size,
                int          offset,
                char         *post,
                unsigned int x,
                unsigned int maxNumZeros)
{
    unsigned int  c = 0;
    int  numZeros = 0;
    char *s = *sp;
    unsigned int  i = x, j;

    while (i != 0)
    {
	c++;
	i /= 10;
    }

    if (maxNumZeros > c)
	numZeros = maxNumZeros - c;

    j = offset + c + numZeros + strlen (post) + 4 + 1;
    if (j > size)
    {
	size = j;
	s = *sp = (char *) realloc (*sp, size * sizeof (char));
    }

    snprintf (s + offset, size - offset, "%0*d%s.obj", maxNumZeros, x, post);

    return size;
}

static int
addVertex (unsigned int **indices,
	   int		nUniqueIndices,
	   int		iTexture,
	   int		iNormal)
{
    unsigned int i, len;

    if (*indices == NULL)
    {
	*indices = (unsigned int *) malloc (4 * sizeof (unsigned int));
	(*indices)[0] = 1; /* store size of array as 1st element */
	(*indices)[1] = iTexture;
	(*indices)[2] = iNormal;
	(*indices)[3] = nUniqueIndices; /* ptr to new unique vertex index */

	return -1; /* new vertex/texture/normal */
    }

    len = (*indices)[0];
    for (i = 0; i < len; i++)
    {
	if ((*indices)[1 + 3 * i] == (unsigned int ) iTexture &&
	    (*indices)[2 + 3 * i] == (unsigned int ) iNormal)
	{
	    return (*indices)[3 + 3 * i]; /* found same vertex/texture/normal
					     before */
	}
    }

    *indices = (unsigned int *) realloc (*indices, (1 + 3 * ((*indices)[0] + 1)) *
                        	  sizeof (unsigned int));
    (*indices)[1 + 3 * (*indices)[0]] = iTexture;
    (*indices)[2 + 3 * (*indices)[0]] = iNormal;
    (*indices)[3 + 3 * (*indices)[0]] = nUniqueIndices;
    (*indices)[0]++;

    return -1;
}

bool
CubemodelScreen::compileDList (CubemodelObject *data)
{
    if (!data->animation && data->finishedLoading && !data->compiledDList)
    {
	data->dList = glGenLists (1);
	glNewList (data->dList, GL_COMPILE);

	glDisable (GL_CULL_FACE);
	glEnable  (GL_NORMALIZE);
	glEnable  (GL_DEPTH_TEST);

	glDisable (GL_COLOR_MATERIAL);

	drawVBOModel (data,
	              (float *) data->reorderedVertex[0],
	              (float *) data->reorderedNormal[0]);
	glEndList ();

	data->compiledDList = true;

	return true;
    }
    return false;
}

void
CubemodelScreen::loadMaterials (CubemodelObject *data,
				char            *approxPath,
				char            *filename,
				mtlStruct       **material,
				int             *nMat)
{
    int i;

    /* getLine stuff */
    char *strline    = NULL;
    int tempBufferSize = 2048; /* get character data from files
				  in these size chunks */
    fileParser *fParser = NULL;


    mtlStruct *currentMaterial = NULL;

    char *mtlFilename;
    FILE *mtlfp;

    int nMaterial = *nMat;

    if (nMaterial == 0)
	*material = NULL;

    mtlFilename = findPath (approxPath, filename);
    if (!mtlFilename)
	return;

    mtlfp = fopen (mtlFilename, "r");

    if (mtlFilename)
	free (mtlFilename);

    if (!mtlfp)
    {
	compLogMessage ("cubemodel", CompLogLevelWarn,
	                "Failed to open material file : %s", mtlFilename);
	return;
    }

    fParser = initFileParser (mtlfp, tempBufferSize);

    /* now read all the materials in the mtllib referenced file */

    while ((strline = getLineToken2 (fParser, false)))
    {
	char *tmpPtr[3] = { NULL, NULL, NULL }; /* used to check numerical
						   parameters*/
	float tmpNum[3] = {0, 0, 0 };
	float tmpIllum = 100;

	if (strline[0] == '\0')
	    continue;

	if (!strcmp (strline, "newmtl"))
	{
	    strline = getLineToken2 (fParser, true);
	    if (!strline)
		continue;

	    *material = (mtlStruct *) realloc (*material, sizeof (mtlStruct) *
	                                       (nMaterial + 1));
	    currentMaterial = &((*material)[nMaterial]);

	    nMaterial++;

	    currentMaterial->name = strdup (strline);

	    /* set defaults */
	    currentMaterial->Ns[0] = 100;
	    currentMaterial->Ni[0] = 1;
	    currentMaterial->illum = 2;

	    for (i = 0; i < 3; i++)
	    {
		currentMaterial->Ka[i] = 0.2;
		currentMaterial->Kd[i] = 0.8;
		currentMaterial->Ks[i] = 1;
	    }
	    currentMaterial->Ka[3] = 1;
	    currentMaterial->Kd[3] = 1;
	    currentMaterial->Ks[3] = 1;

	    currentMaterial->map_Ka = -1;
	    currentMaterial->map_Kd = -1;
	    currentMaterial->map_Ks = -1;
	    currentMaterial->map_d  = -1;

	    currentMaterial->map_params = -1;
	}

	if (!currentMaterial)
	    continue;

	for (i = 0; i < 3; i++)
	{
	    tmpPtr[i] = getLineToken2 (fParser, true);
	    if (!tmpPtr[i])
		break;

	    tmpNum[i] = atof (tmpPtr[i]);

	    if (i == 0)
		tmpIllum = atoi (tmpPtr[i]);
	}

	if (!strcmp (strline, "Ns"))
	{
	    currentMaterial->Ns[0] = tmpNum[0];
	}
	else if (!strcmp (strline, "Ka"))
	{
	    for (i = 0; i < 3; i++)
		currentMaterial->Ka[i] = tmpNum[i];
	}
	else if (!strcmp (strline, "Kd"))
	{
	    for (i = 0; i < 3; i++)
		currentMaterial->Kd[i] = tmpNum[i];
	}
	else if (!strcmp (strline, "Ks"))
	{
	    for (i = 0; i < 3; i++)
		currentMaterial->Ks[i] = tmpNum[i];
	}
	else if (!strcmp (strline, "Ni"))
	{
	    currentMaterial->Ni[0] = tmpNum[0];
	}
	else if (!strcmp (strline, "d") || !strcmp(strline,"Tr"))
	{
	    currentMaterial->Ka[3] = tmpNum[0];
	    currentMaterial->Kd[3] = tmpNum[0];
	    currentMaterial->Ks[3] = tmpNum[0];
	}
	else if (!strcmp (strline, "illum"))
	{
	    currentMaterial->illum = tmpIllum;
	}
	else if (!strcmp (strline, "map_Ka") || !strcmp (strline, "map_Kd") ||
		 !strcmp (strline, "map_Ks") || !strcmp (strline, "map_d" ) )
	{
	    char *tmpName = NULL;

	    if (!data->tex)
	    {
		data->tex = new GLTexture::List[1];
		if (!data->tex)
		{
		    compLogMessage ("cubemodel", CompLogLevelWarn,
		                    "Error allocating texture memory");
		    break;
		}
		data->texName = NULL;

		data->texWidth  = (unsigned int *) malloc (sizeof (unsigned int));
		if (!data->texWidth)
		{
		    delete[] data->tex;
		    data->tex = NULL;
		    break;
		}
		data->texHeight = (unsigned int *) malloc (sizeof (unsigned int));
		if (!data->texHeight)
		{
		    delete[] data->tex;
		    free (data->texWidth);
		    data->tex = NULL;
		    break;
		}

		data->nTex = 0;
	    }
	    else
	    {
		bool match = false;

		if (!data->texName && data->nTex > 0)
		    continue;

		for (i = 0; i < data->nTex; i++)
		{
		    if (!data->texName[i])
			break;

		    if (!strcmp (tmpPtr[0], data->texName[i]))
		    {
			if (!strcmp (strline, "map_Ka"))
			    currentMaterial->map_Ka = i;
			else if (!strcmp (strline, "map_Kd"))
			    currentMaterial->map_Kd = i;
			else if (!strcmp (strline, "map_Ks"))
			    currentMaterial->map_Ks = i;
			else if (!strcmp (strline, "map_d"))
			    currentMaterial->map_d = i;

			currentMaterial->width  = data->texWidth[i];
			currentMaterial->height = data->texHeight[i];

			currentMaterial->map_params = i;

			match = true;
			break;
		    }
		}

		if (match)
		    continue;

		GLTexture::List *tmpTexList = NULL;
		int newTexSize = data->nTex + 1;

		memcpy (tmpTexList, data->tex, sizeof (GLTexture::List) *
						data->nTex);
		
		delete[] data->tex;
		
		data->tex = new GLTexture::List[newTexSize];
		
		memcpy (data->tex, tmpTexList, data->nTex);
		
		data->texWidth = (unsigned int *) realloc (data->texWidth,
		                          sizeof (unsigned int) *
		                          (data->nTex + 1));
		data->texHeight= (unsigned int *) realloc (data->texHeight,
		                          sizeof (unsigned int) *
		                          (data->nTex + 1));
	    }

	    if (!data->tex)
	    {
		compLogMessage ("cubemodel", CompLogLevelWarn,
		                "CompTexture is not malloced properly");
	    }
	    else
	    {
		tmpName = findPath (approxPath, tmpPtr[0]);
		CompString texTmpName = CompString (tmpName);
		CompSize currentMaterialSize = 
					       CompSize (currentMaterial->width,
						       currentMaterial->height);
		data->tex[data->nTex] =
		    GLTexture::readImageToTexture (texTmpName,
						   currentMaterialSize);
		if (!data->tex[data->nTex].size ())
		{
		    compLogMessage ("cubemodel", CompLogLevelWarn,
		                   "Failed to load image: %s", tmpName);

		    data->tex[data->nTex].clear ();
		}
		else
		{
		    data->texName  = (char **) realloc (data->texName,
					      	        sizeof (char *) *
					      	        (data->nTex + 1));

		    data->texName[data->nTex] = strdup (tmpPtr[0]);

		    if (!strcmp (strline, "map_Ka"))
			currentMaterial->map_Ka = data->nTex;
		    else if (!strcmp (strline, "map_Kd"))
			currentMaterial->map_Kd = data->nTex;
		    else if (!strcmp (strline, "map_Ks"))
			currentMaterial->map_Ks = data->nTex;
		    else if (!strcmp (strline, "map_d"))
			currentMaterial->map_d = data->nTex;

		    data->texWidth[data->nTex]  = currentMaterial->width;
		    data->texHeight[data->nTex] = currentMaterial->height;

		    currentMaterial->map_params = data->nTex;

		    data->nTex++;
		}
	    }

	    if (tmpName)
		free (tmpName);
	    tmpName = NULL;
	}

	if (!fParser->lastTokenOnLine)
	    skipLine (fParser);
    }

    freeFileParser(fParser);

    fclose (mtlfp);

    *nMat = nMaterial;
}

bool
CubemodelScreen::initLoadModelObject (CubemodelObject *modelData)
{
    char *filename = modelData->filename;
    char *post     = modelData->post;

    int size            = modelData->size;
    int lenBaseFilename = modelData->lenBaseFilename;
    int startFileNum    = modelData->startFileNum;
    int maxNumZeros     = modelData->maxNumZeros;

    /* getLine stuff */
    char *strline    = NULL;
    int tempBufferSize = 4 * 1024; /* get character data from files
				      in these size chunks */
    fileParser *fParser = NULL;

    int nVertex = 0;
    int nNormal = 0;
    int nTexture = 0;
    int nIndices = 0;

    FILE *fp;

    modelData->nMaterial[0] = 0;
    modelData->material[0] = NULL;

    /* First pass - count how much data we need to store and
     *		  - load the materials from any mtllib references.
     */

    if (modelData->animation)
	size = addNumToString (&filename, size, lenBaseFilename, post,
	                       startFileNum, maxNumZeros);

    fp = fopen (filename, "r");
    if (!fp)
    {
	compLogMessage ("cubemodel", CompLogLevelWarn,
	                "Failed to open model file - %s", filename);
	return false;
    }

    fParser = initFileParser (fp, tempBufferSize);

    while ((strline = getLineToken2 (fParser, false)))
    {
	if (strline[0] == '\0')
	    continue;

	if (!strcmp (strline, "v"))
	    nVertex++;
	else if (!strcmp (strline, "vn"))
	    nNormal++;
	else if (!strcmp (strline, "vt"))
	    nTexture++;
	else if (!strcmp (strline, "f") || !strcmp (strline, "fo") ||
		 !strcmp (strline, "p") || !strcmp (strline, "l") )
	{
	    while (getLineToken2 (fParser, true))
		nIndices++;
	}
	else if (!strcmp (strline, "mtllib"))
	{
	    while ((strline = getLineToken2 (fParser, true)))
	    {
		loadMaterials (modelData, filename, strline,
		               &(modelData->material[0]),
		               &(modelData->nMaterial[0]));
	    }
	}

	if (!fParser->lastTokenOnLine)
	    skipLine (fParser);
    }

    modelData->reorderedVertex[0]  = (vect3d *) malloc (sizeof (vect3d) * nIndices);
    modelData->reorderedTexture[0] = (vect2d *) malloc (sizeof (vect2d) * nIndices);
    modelData->reorderedNormal[0]  = (vect3d *) malloc (sizeof (vect3d) * nIndices);

    modelData->indices = (unsigned int *) malloc (sizeof (unsigned int) * nIndices);
    modelData->reorderedVertexBuffer  = (vect3d *) malloc (sizeof (vect3d) * nIndices);
    modelData->reorderedTextureBuffer = (vect2d *) malloc (sizeof (vect2d) * nIndices);
    modelData->reorderedNormalBuffer  = (vect3d *) malloc (sizeof (vect3d) * nIndices);

    modelData->nVertex  = nVertex;
    modelData->nNormal  = nNormal;
    modelData->nTexture = nTexture;
    modelData->nIndices = nIndices;

    freeFileParser (fParser);

    return true;
}

bool
CubemodelScreen::loadModelObject (CubemodelObject *modelData)
{
    int i, j;

    char *filename = modelData->filename;
    char *post     = modelData->post;

    int size            = modelData->size;
    int lenBaseFilename = modelData->lenBaseFilename;
    int startFileNum    = modelData->startFileNum;
    int maxNumZeros     = modelData->maxNumZeros;

    /* getLine stuff */
    char *strline    = NULL;
    int tempBufferSize = 4 * 1024; /* get character data from files
				      in these size chunks */
    fileParser *fParser = NULL;

    int fileCounter = modelData->fileCounter;

    int nVertex=0;
    int nNormal=0;
    int nTexture=0;
    int nIndices=0;
    int nUniqueIndices = 0;

    unsigned int **tmpIndices = NULL; /* reorder indices per vertex
					 (store alternating corresponding
					 textures/normals) */
    vect3d *vertex  = NULL;
    vect3d *normal  = NULL;
    vect2d *texture = NULL;

    FILE *fp;

    int nGroups  = 0;

    int oldPolyCount = 0;
    bool oldUsingNormal = false;
    bool oldUsingTexture = false;

    /* store size of each array */
    int sVertex = 0;
    int sTexture = 0;
    int sNormal = 0;
    int sIndices = 0;

    int fc;

    fParser = initFileParser (NULL, tempBufferSize);

    for (fc = 0; fc < fileCounter; fc++)
    {
	int lastLoadedMaterial = -1;
	int prevLoadedMaterial = -1;

	if (modelData->animation)
	    size = addNumToString (&filename, size, lenBaseFilename,
	                           post, startFileNum+fc, maxNumZeros);

	fp = fopen(filename, "r");
	if (!fp)
	{
	    compLogMessage ("cubemodel", CompLogLevelWarn,
	                    "Failed to open model file - %s", filename);
	    free (normal);
	    free (tmpIndices);
	    free (texture);
	    free (vertex);
	    freeFileParser (fParser);
	    return false;
	}

	updateFileParser (fParser, fp);

	if (fc == 0)
	{
	    nVertex  = modelData->nVertex;
	    nTexture = modelData->nTexture;
	    nNormal  = modelData->nNormal;
	    nIndices = modelData->nIndices;

	    sVertex  = nVertex;
	    sTexture = nTexture;
	    sNormal  = nNormal;
	    sIndices = nIndices;
	}
	else
	{
	    sIndices = modelData->nIndices;
	}

	modelData->reorderedVertex[fc]  = (vect3d *) malloc (sizeof (vect3d) * sIndices);
	modelData->reorderedTexture[fc] = (vect2d *) malloc (sizeof (vect2d) * sIndices);
	modelData->reorderedNormal[fc]  = (vect3d *) malloc (sizeof (vect3d) * sIndices);

	if (fc == 0)
	{
	    vertex  = (vect3d *) malloc (sizeof (vect3d) * nVertex);
	    texture = (vect2d *) malloc (sizeof (vect2d) * nTexture);
	    normal  = (vect3d *) malloc (sizeof (vect3d) * nNormal);

	    modelData->indices = (unsigned int *) malloc (sizeof (unsigned int) * nIndices);
	    modelData->reorderedVertexBuffer  = (vect3d *) malloc (sizeof (vect3d) *
	                                                           nIndices);
	    modelData->reorderedTextureBuffer = (vect2d *) malloc (sizeof (vect2d) *
	                                                           nIndices);
	    modelData->reorderedNormalBuffer  = (vect3d *) malloc (sizeof (vect3d) *
	                                                           nIndices);

	    modelData->nVertex  = nVertex;
	    modelData->nNormal  = nNormal;
	    modelData->nTexture = nTexture;
	    modelData->nIndices = nIndices;

	    tmpIndices = (unsigned int **) malloc (sizeof (unsigned int **) * sVertex);
	    for (i = 0; i < sVertex; i++)
		tmpIndices[i] = NULL;
	}
	else
	{
	    for (i = 0; i < sVertex; i++)
	    {
		if (tmpIndices[i])
		    tmpIndices[i][0] = 0; /* set length to 0 */
	    }
	}

	nVertex  = 0;
	nNormal  = 0;
	nTexture = 0;
	nIndices = 0;
	nUniqueIndices = 0;

	/* Second pass - fill arrays
	 * 	       - reorder data and store into vertex/normal/texture
	 * 		 buffers
	 */

	while ((strline = getLineToken2 (fParser, false)))
	{
	    int complexity = 0;
	    int polyCount  = 0;
	    bool updateGroup  = false;
	    bool usingNormal  = false;
	    bool usingTexture = false;

	    if (strline[0] == '\0')
		continue;

	    if (!strcmp (strline, "v"))
	    {
		if (sVertex <= nVertex)
		{
		    sVertex++;
		    vertex     = (vect3d *) realloc (vertex, sizeof (vect3d) * sVertex);
		    tmpIndices = (unsigned int **) realloc (tmpIndices,
					  sizeof (unsigned int *) * sVertex);
		    tmpIndices[sVertex - 1] = NULL;
		}

		for (i = 0; i < 3; i++)
		{
		    strline = getLineToken2 (fParser, true);

		    if (!strline)
		    {
			vertex[nVertex].r[0] = 0;
			vertex[nVertex].r[1] = 0;
			vertex[nVertex].r[2] = 0;
			break;
		    }
		    vertex[nVertex].r[i] = atof (strline);
		}
		nVertex++;
	    }
	    else if (!strcmp (strline, "vn"))
	    {
		if (sNormal <= nNormal)
		{
		    sNormal++;
		    normal = (vect3d *) realloc (normal, sizeof (vect3d) * sNormal);
		}

		for (i = 0; i < 3; i++)
		{
		    strline = getLineToken2 (fParser, true);

		    if (!strline)
		    {
			normal[nNormal].r[0] = 0;
			normal[nNormal].r[1] = 0;
			normal[nNormal].r[2] = 1;
			break;
		    }
		    normal[nNormal].r[i] = atof(strline);
		}
		nNormal++;
	    }
	    else if (!strcmp (strline, "vt"))
	    {
		if (sTexture <= nTexture)
		{
		    sVertex++;
		    texture = (vect2d *) realloc (texture, sizeof (vect2d) * sTexture);
		}

		/* load the 1D/2D coordinates for textures */
		for (i = 0; i < 2; i++)
		{
		    strline = getLineToken2 (fParser, true);
		    if (!strline)
		    {
			if (i == 0)
			    texture[nTexture].r[0] = 0;

			texture[nTexture].r[1] = 0;
			break;
		    }
		    texture[nTexture].r[i] = atof (strline);
		}
		nTexture++;
	    }
	    else if (!strcmp (strline, "usemtl") && fc == 0)
	    {
		/* parse mtl file(s) and load specified material */
		strline = getLineToken2 (fParser, true);
		if (!strline)
		    continue;

		for (j = 0; j < modelData->nMaterial[fc]; j++)
		{
		    if (!strcmp (strline, modelData->material[fc][j].name))
		    {
			lastLoadedMaterial = j;
			updateGroup = true;
			break;
		    }
		}
	    }
	    else if (!strcmp (strline, "f") || !strcmp (strline, "fo") ||
		     !strcmp (strline, "p") || !strcmp (strline, "l"))
	    {
		char *tmpPtr; /* used to check value of vertex/texture/normal */

		if (!strcmp (strline, "l"))
		    complexity = 1;
		else if (!strcmp (strline, "f") || !strcmp (strline, "fo"))
		    complexity = 2;

		while ((strline = getLineToken2 (fParser, true)))
		{
		    int vertexIndex  = -1;
		    int textureIndex = -1;
		    int normalIndex  = -1;
		    int tmpInd;


		    tmpPtr = strsep (&strline, "/");
		    if (tmpPtr)
		    {
			vertexIndex = atoi (tmpPtr);
			if (vertexIndex > 0)
			{
			    /* skip vertex index past last in obj file */
			    if (vertexIndex > modelData->nVertex)
				break;
			    vertexIndex--;
			}
			else if (vertexIndex < 0)
			{
			    vertexIndex += nVertex;

			    /* skip vertex index < 0 in obj file */
			    if (vertexIndex < 0)
				break;
			}
			else /* skip vertex index of 0 in obj file */
			    break;
		    }
		    else
			break;

		    tmpPtr = strsep (&strline, "/");
		    if (tmpPtr && complexity != 0)
		    {
			/* texture */

			if (tmpPtr[0] != '\0')
			{
			    textureIndex = atoi (tmpPtr);

			    if (textureIndex > 0)
			    {
				 /* skip normal index past last in obj file */
				if (textureIndex > modelData->nTexture)
				    break;
				textureIndex--;
			    }
			    else if (textureIndex < 0)
			    {
				textureIndex += nTexture;

				/* skip texture index < 0 in obj file */
				if (textureIndex < 0)
				    break;
			    }
			    else /* skip texture index of 0 in obj file */
				break;

			    usingTexture = true;
			}

			tmpPtr = strsep (&strline, "/");
			if (tmpPtr)
			{
			    if (tmpPtr[0]!='\0' && complexity == 2)
			    {
				/* normal */

				normalIndex = atoi (tmpPtr);

				if (normalIndex > 0)
				{
				    /* skip normal index past last in obj file */
				    if (normalIndex > modelData->nNormal)
					break;
				    normalIndex--;
				}
				else if (normalIndex < 0)
				{
				    normalIndex += nNormal;

				    /* skip normal index < 0 in obj file */
				    if (normalIndex < 0)
					break;
				}
				else /* skip normal index of 0 in obj file */
				    break;

				usingNormal = true;
			    }
			}
		    }

		    /* reorder vertices/textures/normals */

		    tmpInd = addVertex (&tmpIndices[vertexIndex],
					nUniqueIndices, textureIndex,
					normalIndex);
		    if (tmpInd < 0)
		    {
			if (nUniqueIndices >= sIndices)
			{
			    sIndices++;
			    modelData->reorderedVertex[fc]  =
				(vect3d *) realloc (modelData->reorderedVertex[fc],
				         sizeof (vect3d) * sIndices);
			    modelData->reorderedTexture[fc] =
				(vect2d *) realloc (modelData->reorderedTexture[fc],
				         sizeof (vect2d) * sIndices);
			    modelData->reorderedNormal[fc]  =
				(vect3d *) realloc (modelData->reorderedNormal[fc],
				         sizeof (vect3d) * sIndices);
			}

			if (vertexIndex < nVertex)
			{
			    memcpy (modelData->reorderedVertex[fc]
				    [nUniqueIndices].r,
				    vertex[vertexIndex].r,
				    3 * sizeof (float));
			}
			else
			{
			    for (i = 0; i < 3; i++)
				modelData->reorderedVertex[fc]
				    [nUniqueIndices].r[i] = 0;
			}

			if (textureIndex >= 0 && textureIndex < nTexture)
			{
			    memcpy (modelData->reorderedTexture[fc]
				    [nUniqueIndices].r,
				    texture[textureIndex].r,
				    2 * sizeof (float));


			    /* scale as per 1st loaded texture for
			     * that material (1st frame)*/

			    if (lastLoadedMaterial >=0 && modelData->tex)
			    {
				mtlStruct *currentMaterial =
				    &(modelData->material[fc]
				      [lastLoadedMaterial]);

				if (currentMaterial->map_params >= 0)
				{

				    /* FIXME: This looks horrible */

				    GLTexture::Matrix ct =
					((&(modelData->tex
					[currentMaterial->map_params]))->at (0)->
					matrix ());

				    modelData->reorderedTexture[fc]
				    [nUniqueIndices].r[0] =
					COMP_TEX_COORD_X (ct,
					(currentMaterial->width - 1) *
					(texture[textureIndex].r[0]));
				    modelData->reorderedTexture[fc]
				    [nUniqueIndices].r[1] =
					COMP_TEX_COORD_Y (ct,
					(currentMaterial->height - 1) *
					(1 - texture[textureIndex].r[1]));
				}
			    }
			}
			else
			{
			    modelData->reorderedTexture[fc]
			    [nUniqueIndices].r[0] = 0;
			    modelData->reorderedTexture[fc]
			    [nUniqueIndices].r[1] = 0;
			}

			if (normalIndex >= 0 && normalIndex < nNormal)
			    memcpy (modelData->reorderedNormal[fc]
				    [nUniqueIndices].r,
				    normal[normalIndex].r,
				    3 * sizeof (float));
			else
			{
			    modelData->reorderedNormal[fc]
				[nUniqueIndices].r[0] = 0;
			    modelData->reorderedNormal[fc]
				[nUniqueIndices].r[1] = 0;
			    modelData->reorderedNormal[fc]
				[nUniqueIndices].r[2] = 1;
			}

			modelData->indices[nIndices] = nUniqueIndices;
			nUniqueIndices++;
		    }
		    else
			modelData->indices[nIndices] = tmpInd;

		    nIndices++;
		    polyCount++;
		}

		updateGroup = true;
	    }

	    if (!fParser->lastTokenOnLine)
		skipLine (fParser);

	    if (updateGroup && fc == 0)
	    {
		if (polyCount !=0 &&
		    (polyCount != oldPolyCount       ||
		     usingNormal != oldUsingNormal   ||
		     usingTexture != oldUsingTexture ||
		     lastLoadedMaterial != prevLoadedMaterial))
		{
		    oldPolyCount = polyCount;
		    oldUsingTexture = usingTexture;
		    oldUsingNormal  = usingNormal;
		    prevLoadedMaterial = lastLoadedMaterial;

		    modelData->group = (groupIndices *) realloc (modelData->group,
								  (nGroups + 1) *
								  sizeof (groupIndices));

		    modelData->group[nGroups].polyCount = polyCount;
		    modelData->group[nGroups].complexity = complexity;
		    modelData->group[nGroups].startV = nIndices - polyCount;

		    modelData->group[nGroups].materialIndex =
			lastLoadedMaterial;

		    if (nGroups > 0)
			modelData->group[nGroups - 1].numV = nIndices -
			    polyCount - modelData->group[nGroups - 1].startV;

		    modelData->group[nGroups].texture = usingTexture;
		    modelData->group[nGroups].normal  = usingNormal;

		    nGroups++;
		}
	    }
	}

	if (nGroups != 0 && fc == 0)
	    modelData->group[nGroups - 1].numV = nIndices -
		modelData->group[nGroups - 1].startV;

	if (fc == 0)
	    modelData->nUniqueIndices = nUniqueIndices;

	fclose (fp);

	if (fc == 0 && modelData->animation)
	{ /* set up 1st frame for display */
	    vect3d *reorderedVertex  = modelData->reorderedVertex[0];
	    vect3d *reorderedNormal  = modelData->reorderedNormal[0];

	    for (i = 0; i < modelData->nUniqueIndices; i++)
	    {
		for (j = 0; j < 3; j++)
		{
		    modelData->reorderedVertexBuffer[i].r[j] =
			reorderedVertex[i].r[j];
		    modelData->reorderedNormalBuffer[i].r[j] =
			reorderedNormal[i].r[j];
		}
	    }
	}

    }

    modelData->nGroups = nGroups;

    if (vertex)
	free (vertex);
    if (normal)
	free (normal);
    if (texture)
	free (texture);


    if (tmpIndices)
    {
	for (i = 0; i < sVertex; i++)
	    if (tmpIndices[i])
		free (tmpIndices[i]);
	free (tmpIndices);
    }

    freeFileParser (fParser);

    modelData->finishedLoading = true;

    return true;
}

static void *
loadModelObjectThread (void *ptr)
{
    CubemodelObject *modelData = (CubemodelObject *) ptr;
    modelData->threadRunning = true;

    CubemodelScreen::get (screen)->loadModelObject (modelData);

    modelData->updateAttributes = true;
    modelData->threadRunning = false;

    pthread_exit (NULL);
}

bool
CubemodelScreen::addModelObject (CubemodelObject *modelData,
				 CompString      sFile,
				 float           *translate,
				 float           *rotate,
				 float           rotateSpeed,
				 float           *scale,
				 float           *color,
				 bool            animation,
				 float           fps)
{
    int  i, size;
    int  fileCounter = 0; /* value checked in cubemodelDeleteModelObject */
    int  lenFilename;
    int  startFileNum = 0;
    int  maxNumZeros = 6;
    int  lenBaseFilename;
    const char *file = sFile.c_str ();
    FILE *fp;
    bool flag;

    modelData->fileCounter = 0;
    modelData->updateAttributes = false;

    if (!file)
	return false;
    if (!strlen (file))
	return false;

    modelData->rotate[0] = rotate[0]; /* rotation angle */
    modelData->rotate[1] = rotate[1]; /* rotateX */
    modelData->rotate[2] = rotate[2]; /* rotateY */
    modelData->rotate[3] = rotate[3]; /* rotateZ */

    modelData->translate[0] = translate[0]; /* translateX */
    modelData->translate[1] = translate[1]; /* translateY */
    modelData->translate[2] = translate[2]; /* translateZ */

    modelData->scaleGlobal = scale[0];
    modelData->scale[0] = scale[1]; /* scaleX */
    modelData->scale[1] = scale[2]; /* scaleY */
    modelData->scale[2] = scale[3]; /* scaleZ */

    modelData->rotateSpeed = rotateSpeed;
    modelData->animation   = animation;
    modelData->fps         = fps;
    modelData->time        = 0;

    if (!color)
    {
	modelData->color[0] = 1;
	modelData->color[1] = 1;
	modelData->color[2] = 1;
	modelData->color[3] = 1;
    }
    else
    {
	modelData->color[0] = color[0]; /* R */
	modelData->color[1] = color[1]; /* G */
	modelData->color[2] = color[2]; /* B */
	modelData->color[3] = color[3]; /* alpha */
    }

    /* set to NULL so delete will not crash for threads */
    modelData->reorderedVertex        = NULL;
    modelData->reorderedTexture	      = NULL;
    modelData->reorderedNormal        = NULL;
    modelData->nMaterial 	      = NULL;
    modelData->material 	      = NULL;
    modelData->tex      	      = NULL;
    modelData->texName   	      = NULL;
    modelData->texWidth   	      = NULL;
    modelData->texHeight   	      = NULL;
    modelData->reorderedVertexBuffer  = NULL;
    modelData->reorderedTextureBuffer = NULL;
    modelData->reorderedNormalBuffer  = NULL;
    modelData->indices 		      = NULL;
    modelData->group                  = NULL;


    modelData->compiledDList = false;
    modelData->finishedLoading = false;

    modelData->threadRunning = false;

    modelData->post = NULL;
    modelData->filename = NULL;

    lenFilename = strlen (file);
    size = lenFilename + 1 + 4;

    if (lenFilename > 3)
    {
	if (strstr (file + lenFilename - 4, ".obj"))
	{
	    lenFilename -= 4;
	    size -= 4;
	}
    }

    modelData->filename = (char *) calloc (size, sizeof (char));
    if (!modelData->filename)
        return false;

    strncpy (modelData->filename, file, lenFilename);
    if (!modelData->animation)
	strcat (modelData->filename, ".obj");

    lenBaseFilename = lenFilename;

    if (modelData->animation)
    {
	char *start, *numbers = NULL;
	char *post = modelData->filename + lenFilename;

	start = strrchr (modelData->filename, '/');
	if (!start)
	    start = modelData->filename;

	start++;

	bool lastCharANumber = false;

	while (*start)
	{
	    if (*start >= '0' && *start <= '9')
	    {
		if (!numbers || !lastCharANumber)
		    numbers = start;
		post = start + 1;

		lastCharANumber = true;
	    }
	    else
		lastCharANumber = false;

	    start++;
	}

	if (numbers)
	{
	    lenBaseFilename = numbers - modelData->filename;
	    maxNumZeros     = post - numbers;

	    modelData->post = strdup (post);
	    if (!modelData->post)
		return false;

	    strncpy (modelData->filename, file, lenBaseFilename);
	    startFileNum = strtol (numbers, NULL, 10);
	}
	else
	{
	    modelData->animation = false;
	    strcat (modelData->filename, ".obj");
	}
    }

    /* verify existence of files and/or check for animation frames */

    do
    {
	if (modelData->animation)
	    size = addNumToString (&modelData->filename, size,
				   lenBaseFilename, modelData->post,
				   startFileNum + fileCounter, maxNumZeros);

	fp = fopen (modelData->filename, "r");
	if (fp)
	{
	    printf ("opened %s\n", modelData->filename);

	    fclose (fp);
	    fileCounter++;
	}
    }
    while (modelData->animation && fp);

    modelData->fileCounter = fileCounter;
    if (!fileCounter)
    {
	compLogMessage ("cubemodel", CompLogLevelWarn,
			"Failed to open model file : %s", modelData->filename);

	if (modelData->filename)
	    free (modelData->filename);
	if (modelData->post)
	    free (modelData->post);

	return false;
    }

    modelData->reorderedVertex  = (vect3d **) malloc (sizeof (vect3d *) * fileCounter);
    modelData->reorderedTexture = (vect2d **) malloc (sizeof (vect2d *) * fileCounter);
    modelData->reorderedNormal  = (vect3d **) malloc (sizeof (vect3d *) * fileCounter);

    modelData->reorderedVertexBuffer  = NULL;
    modelData->reorderedTextureBuffer = NULL;
    modelData->reorderedNormalBuffer  = NULL;

    modelData->material  = (mtlStruct **) malloc (sizeof (mtlStruct *) * fileCounter);
    modelData->nMaterial = (int *) malloc (sizeof (int) * fileCounter);

    for (i = 0; i < fileCounter; i++)
    {
	modelData->material[i]  = 0;
	modelData->nMaterial[i] = 0;
    }

    modelData->tex = NULL;
    modelData->texName = NULL;
    modelData->nTex = 0;
    modelData->texWidth = NULL;
    modelData->texHeight = NULL;

    modelData->indices = NULL;
    modelData->group = NULL;

    modelData->size = size;
    modelData->lenBaseFilename = lenBaseFilename;
    modelData->startFileNum = startFileNum;
    modelData->maxNumZeros = maxNumZeros;

    flag = initLoadModelObject (modelData);

    if (flag)
    {
	if  (optionGetConcurrentLoad ())
	{
	    int iret;

	    modelData->threadRunning = true;

	    iret = pthread_create (&modelData->thread, NULL,
				   loadModelObjectThread,
				   (void*) modelData);
	    if (!iret)
		return true;

	    compLogMessage ("cubemodel", CompLogLevelWarn,
			    "Error creating thread: %s\n"
			    "Trying single threaded approach", file);
	    modelData->threadRunning = false;
	}

	flag = loadModelObject (modelData);
    }

    return flag;
}

bool
CubemodelScreen::deleteModelObject (CubemodelObject *data)
{
    int i, fc;

    if (!data)
	return false;

    if (data->fileCounter == 0)
	return false;

    if (data->threadRunning)
    {
	int ret;

	ret = pthread_join (data->thread, NULL); /* not best in all
						    circumstances */
	if (ret)
	{
	    compLogMessage ("cubemodel", CompLogLevelWarn,
			    "Could not synchronize with thread.\n"
			    "Possible memory leak)");
	    return false;
	}
    }

    if (data->filename)
	free (data->filename);

    if (data->post)
	free (data->post);

    if (!data->animation && data->compiledDList)
	glDeleteLists (data->dList, 1);

    for (fc = 0; fc < data->fileCounter; fc++)
    {
	if (data->reorderedVertex && data->reorderedVertex[fc])
	    free (data->reorderedVertex[fc]);
	if (data->reorderedTexture && data->reorderedTexture[fc])
	    free (data->reorderedTexture[fc]);
	if (data->reorderedNormal && data->reorderedNormal[fc])
	    free (data->reorderedNormal[fc]);

	if (data->nMaterial)
	{
	    for (i = 0; i< data->nMaterial[fc]; i++)
	    {
		if (data->material[fc][i].name)
		    free (data->material[fc][i].name);
	    }
	}

	if (data->material && data->material[fc])
	    free(data->material[fc]);
    }

    if (data->tex)
    {
	delete[] data->tex;
    }

    if (data->texName)
    {
	for (i = 0; i < data->nTex; i++)
	{
	    if (data->texName[i])
		free (data->texName[i]);
	}
    }

    if (data->texWidth)
	free (data->texWidth);
    if (data->texHeight)
	free (data->texHeight);

    if (data->reorderedVertex)
	free (data->reorderedVertex);
    if (data->reorderedTexture)
	free (data->reorderedTexture);
    if (data->reorderedNormal)
	free (data->reorderedNormal);
    if (data->material)
	free (data->material);

    if (data->reorderedVertexBuffer)
	free (data->reorderedVertexBuffer);
    if (data->reorderedTextureBuffer)
	free (data->reorderedTextureBuffer);
    if (data->reorderedNormalBuffer)
	free (data->reorderedNormalBuffer);

    if (data->indices)
	free (data->indices);
    if (data->group)
	free (data->group);

    return true;
}

bool
CubemodelScreen::drawModelObject (CubemodelObject *data,
			  	  float           scale)
{
    if (!data->fileCounter || !data->finishedLoading)
	return false;

    if (!data->animation && !data->compiledDList)
	compileDList (data);

    /* Rotate, translate and scale  */

    glTranslatef (data->translate[0], data->translate[2], data->translate[1]);

    glScalef (data->scaleGlobal * data->scale[0],
	      data->scaleGlobal * data->scale[1],
	      data->scaleGlobal * data->scale[2]);

    glScalef (scale, scale, scale);

    glRotatef (data->rotate[0], data->rotate[1],
	       data->rotate[2], data->rotate[3]);

    glDisable (GL_CULL_FACE);
    glEnable (GL_NORMALIZE);
    glEnable (GL_DEPTH_TEST);

    glEnable (GL_COLOR_MATERIAL);
    glColor4fv (data->color);

    if (data->animation)
    {
	drawVBOModel (data,
	              (float *) data->reorderedVertexBuffer,
	              (float *) data->reorderedNormalBuffer);
    }
    else
    {
	glCallList (data->dList);
    }

    return true;
}

bool
CubemodelScreen::updateModelObject (CubemodelObject *data,
				    float	    time)
{
    int i, j;

    if (!data->fileCounter || !data->finishedLoading)
	return false;

    if (!data->animation && !data->compiledDList)
	compileDList (data);

    data->rotate[0] += 360 * time * data->rotateSpeed;
    data->rotate[0] = fmodf (data->rotate[0], 360.0f);

    if (data->animation && data->fps)
    {
	float  t, dt, dt2;
	int    ti, ti2;
	vect3d *reorderedVertex, *reorderedVertex2;
	vect3d *reorderedNormal, *reorderedNormal2;

	data->time += time * data->fps;
	data->time = fmodf (data->time, (float) data->fileCounter);

	t = data->time;
	if (t < 0)
	    t += (float) data->fileCounter;

	ti  = (int) t;
	ti2 = (ti + 1) % data->fileCounter;
	dt  = t - ti;
	dt2 = 1 - dt;

	reorderedVertex  = data->reorderedVertex[ti];
	reorderedVertex2 = data->reorderedVertex[ti2];
	reorderedNormal  = data->reorderedNormal[ti];
	reorderedNormal2 = data->reorderedNormal[ti2];

	for (i = 0; i < data->nUniqueIndices; i++)
	{
	    for (j = 0; j < 3; j++)
	    {
		data->reorderedVertexBuffer[i].r[j] =
		    dt2 * (reorderedVertex[i].r[j]) +
		    dt  * (reorderedVertex2[i].r[j]);
		data->reorderedNormalBuffer[i].r[j] =
		    dt2 * (reorderedNormal[i].r[j]) +
		    dt  * (reorderedNormal2[i].r[j]);
	    }
	}
    }

    return true;
}

static void
setMaterial (const float *shininess,
	     const float *ambient,
	     const float *diffuse,
	     const float *specular)
{
    glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

    glMaterialfv (GL_FRONT_AND_BACK, GL_SHININESS, shininess);
    glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT, ambient);
    glMaterialfv (GL_FRONT_AND_BACK, GL_DIFFUSE, diffuse);
    glMaterialfv (GL_FRONT_AND_BACK, GL_SPECULAR, specular);
}

bool
CubemodelScreen::drawVBOModel (CubemodelObject *data,
			       float           *vertex,
		       	       float           *normal)
{
    groupIndices *group;
    int          i, j;

    static const float white[4] = { 1.0, 1.0, 1.0, 1.0 };
    static const float black[4] = { 0.0, 0.0, 0.0, 0.0 };
    static const float defaultShininess[1] = { 100 };

    float *v = vertex;
    float *n = normal;
    float *t = (float *) data->reorderedTexture[0];

    GLTexture::List *currentTexture = NULL;
    int             currentTextureIndex = -1; /* last loaded texture -
					     to prevent multiple loadings */

    bool prevNormal = true, prevTexture = false;

    int prevMaterialIndex = -1;

    int ambientTextureIndex     = -1;
    int diffuseTextureIndex     = -1;
    int specularTextureIndex    = -1;
    int transparentTextureIndex = -1;

    const float *ambient   = white;
    const float *diffuse   = white;
    const float *specular  = white;
    const float *shininess = defaultShininess;

    glVertexPointer (3, GL_FLOAT, 0, v);
    glNormalPointer (GL_FLOAT, 0, n);
    glTexCoordPointer (2, GL_FLOAT, 0, t);

    glEnableClientState (GL_VERTEX_ARRAY);
    glEnableClientState (GL_NORMAL_ARRAY);
    glDisableClientState (GL_TEXTURE_COORD_ARRAY);
    glDisable (GL_TEXTURE_2D);

    for (i = 0; i < data->nGroups; i++)
    {
	GLenum cap = GL_QUADS;

	group = &(data->group[i]);
	if (group->polyCount < 1)
	    continue;

	if (group->polyCount == 3)
	    cap = GL_TRIANGLES;
	if (group->polyCount == 2 || group->complexity == 1)
	    cap = GL_LINE_LOOP;
	if (group->polyCount == 1 || group->complexity == 0)
	    cap = GL_POINTS;

	if (group->normal && !prevNormal)
	{
	    glEnableClientState (GL_NORMAL_ARRAY);
	    prevNormal = true;
	}
	else if (!group->normal && prevNormal)
	{
	    glDisableClientState (GL_NORMAL_ARRAY);
	    prevNormal = false;
	}

	if (group->materialIndex >= 0)
	{
	    if (group->materialIndex != prevMaterialIndex)
	    {
		glDisable (GL_COLOR_MATERIAL);

		ambientTextureIndex     =
		    data->material[0][group->materialIndex].map_Ka;
		diffuseTextureIndex     =
		    data->material[0][group->materialIndex].map_Kd;
		specularTextureIndex    =
		    data->material[0][group->materialIndex].map_Ks;
		transparentTextureIndex =
		    data->material[0][group->materialIndex].map_d;

		ambient   = data->material[0][group->materialIndex].Ka;
		diffuse   = data->material[0][group->materialIndex].Kd;
		specular  = data->material[0][group->materialIndex].Ks;
		shininess = data->material[0][group->materialIndex].Ns;

		setMaterial (shininess, ambient, diffuse, specular);

		switch (data->material[0][group->materialIndex].illum) {
		case 0:
		    glDisable (GL_LIGHTING);
		    break;
		case 1:
		    specular = black;
		default:
		    glEnable (GL_LIGHTING);
		}
	    }
	    prevMaterialIndex = group->materialIndex;
	}

	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	if (group->texture && transparentTextureIndex >= 0)
	{
	    if (!prevTexture)
	    {
		glEnableClientState (GL_TEXTURE_COORD_ARRAY);
		glEnable (GL_TEXTURE_2D);
		prevTexture = true;
	    }

	    if (transparentTextureIndex >= 0)
	    {
		if (!currentTexture ||
		    transparentTextureIndex != currentTextureIndex)
		{
		    currentTextureIndex = transparentTextureIndex;
		    if (currentTexture)
			currentTexture[0].at (0)->disable ();

		    currentTexture =  &(data->tex[transparentTextureIndex]);
		    if (currentTexture)
		    {
			glEnable (currentTexture[0].at (0)->target ());
			currentTexture[0].at (0)->enable (GLTexture::Good);
		    }
		}

		glBlendFunc (GL_SRC_ALPHA, GL_ONE);
		setMaterial (shininess, white, white, white);

		if (data->group[i].polyCount < 5)
		    glDrawElements (cap, group->numV, GL_UNSIGNED_INT,
				    data->indices + group->startV);
		else
		{
		    for (j = 0; j < group->numV / group->polyCount; j++)
		    {
			glDrawElements (GL_POLYGON,
					group->polyCount,
					GL_UNSIGNED_INT,
					data->indices + group->startV +
					j * group->polyCount);
		    }
		}

		glBlendFunc (GL_ONE_MINUS_DST_ALPHA, GL_SRC_COLOR);
		setMaterial (shininess, ambient, diffuse, specular);
	    }
	}

	if (group->texture && diffuseTextureIndex >= 0)
	{
	    if (!prevTexture)
	    {
		glEnableClientState (GL_TEXTURE_COORD_ARRAY);
		glEnable (GL_TEXTURE_2D);
		prevTexture = true;
	    }

	    glMaterialfv (GL_FRONT_AND_BACK, GL_DIFFUSE, white);

	    if (!currentTexture || diffuseTextureIndex != currentTextureIndex)
	    {
		currentTextureIndex = diffuseTextureIndex;
		if (currentTexture)
		    currentTexture[0].at (0)->disable ();

		currentTexture =  &(data->tex[diffuseTextureIndex]);
		if (currentTexture)
		{
		    glEnable (currentTexture[0].at (0)->target ());
		    currentTexture[0].at (0)->enable (GLTexture::Good);
		}
	    }
	}
	else
	{
	    if (prevTexture)
	    {
		glDisable (GL_TEXTURE_2D);
		glDisableClientState (GL_TEXTURE_COORD_ARRAY);
		prevTexture = false;
	    }

	    glMaterialfv (GL_FRONT_AND_BACK, GL_DIFFUSE, diffuse);
	}

	if (data->group[i].polyCount < 5)
	    glDrawElements (cap, group->numV, GL_UNSIGNED_INT,
			    data->indices + group->startV);
	else
	{
	    for (j = 0; j < group->numV/group->polyCount; j++)
	    {
		glDrawElements (GL_POLYGON, group->polyCount, GL_UNSIGNED_INT,
				data->indices + group->startV +
				j * group->polyCount);
	    }
	}
    }

    if (currentTexture)
	currentTexture[0].at (0)->disable ();

    glDisable (GL_TEXTURE_2D);
    glDisableClientState (GL_NORMAL_ARRAY);
    glEnableClientState (GL_TEXTURE_COORD_ARRAY);

    return true;
}
