/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <Decibel/Types>

#include <QtCore/QStringList>
#include <QtCore/QDateTime>

#include <QtTapioca/ConnectionManagerFactory>
#include <QtTapioca/Connection>
#include <QtTapioca/Channel>
#include <QtTapioca/Handle>

// ChannelInfo:

Decibel::ChannelInfo::ChannelInfo() :
    channel_type(0),
    target_handle_type(0),
    target_handle(0)
{ }

Decibel::ChannelInfo::ChannelInfo(const QtTapioca::Connection * connection,
                                  const QtTapioca::Channel * channel,
                                  const bool isIncoming)
{
    Q_ASSERT(0 != connection);
    Q_ASSERT(0 != channel);
    connection_service = connection->serviceName();
    connection_path = QDBusObjectPath(connection->objectPath());
    channel_service = channel->serviceName();
    channel_path = QDBusObjectPath(channel->objectPath());
    channel_type = channel->type();
    incoming = isIncoming;
    Q_ASSERT(0 != channel->target());
    Q_ASSERT(0 != channel->target()->handle());
    target_handle_type = channel->target()->handle()->type();
    target_handle = channel->target()->handle()->id();
}

Decibel::ChannelInfo::ChannelInfo(const Decibel::ChannelInfo & channel_info) :
    connection_service(channel_info.connection_service),
    connection_path(channel_info.connection_path),
    channel_service(channel_info.channel_service),
    channel_path(channel_info.channel_path),
    channel_type(channel_info.channel_type),
    target_handle_type(channel_info.target_handle_type),
    target_handle(channel_info.target_handle),
    incoming(channel_info.incoming)
{ }

QtTapioca::Connection * Decibel::ChannelInfo::connection(QObject * parent) const
{
    if (isNull()) { return 0; }
    return new QtTapioca::Connection(connection_service,
                                     connection_path.path(),
                                     parent);
}

QtTapioca::Channel * Decibel::ChannelInfo::channel() const
{
    if (isNull()) { return 0; }
    QList<QtTapioca::Channel *> channels = connection()->openChannels();
    QtTapioca::Channel * channel;
    foreach(channel, channels)
    {
        if (channel->serviceName() == channel_service &&
            channel->objectPath() == channel_path.path() &&
            // channel->target()->handle()->type() == target_handle_type &&
            // channel->target()->handle()->id() == target_handle &&
            channel->type() == channel_type)
        { return channel; }
    }
    return 0;
}

bool Decibel::ChannelInfo::isNull() const
{ return connection_service.isEmpty(); }

Decibel::ChannelInfo &
Decibel::ChannelInfo::operator = (const Decibel::ChannelInfo & channel_info)
{
    connection_service = channel_info.connection_service;
    connection_path = channel_info.connection_path;
    channel_service = channel_info.channel_service;
    channel_path = channel_info.channel_path;
    channel_type = channel_info.channel_type;
    target_handle_type = channel_info.target_handle_type;
    target_handle = channel_info.target_handle;
    incoming = channel_info.incoming;
    return *this;
}

const QDBusArgument & operator << (QDBusArgument & arg,
                                   const Decibel::ChannelInfo & channel_info)
{
    arg.beginStructure();
    arg << channel_info.connection_service
        << channel_info.connection_path
        << channel_info.channel_service
        << channel_info.channel_path
        << channel_info.channel_type
        << channel_info.target_handle_type
        << channel_info.target_handle
        << channel_info.incoming;
    arg.endStructure();
    return arg;
}

const QDBusArgument & operator >> (const QDBusArgument & arg,
                                   Decibel::ChannelInfo & channel_info)
{
    arg.beginStructure();
    arg >> channel_info.connection_service
        >> channel_info.connection_path
        >> channel_info.channel_service
        >> channel_info.channel_path
        >> channel_info.channel_type
        >> channel_info.target_handle_type
        >> channel_info.target_handle
        >> channel_info.incoming;
    arg.endStructure();
    return arg;

}

// ComponentInfo:

Decibel::ComponentInfo::ComponentInfo() { }

Decibel::ComponentInfo::ComponentInfo(const QString & name,
                                      const QStringList & protocols,
                                      const QList<int> & types,
                                      const QList<int> & targets,
                                      const QString & service,
                                      const QDBusObjectPath & path) :
    display_name(name),
    possible_protocol_list(protocols),
    possible_type_list(types),
    possible_target_type_list(targets),
    service_name(service),
    object_path(path)
{ }

Decibel::ComponentInfo::ComponentInfo(const Decibel::ComponentInfo & info) :
    display_name(info.display_name),
    possible_protocol_list(info.possible_protocol_list),
    possible_type_list(info.possible_type_list),
    possible_target_type_list(info.possible_target_type_list),
    service_name(info.service_name),
    object_path(info.object_path)
{ }

bool Decibel::ComponentInfo::sameObject(const Decibel::ComponentInfo & info) const
{
    if (info.service_name == service_name &&
        info.object_path.path() == object_path.path())
    { return true; }
    else
    { return false; }
}

// LogEntry:

Decibel::LogEntry::LogEntry() { }

Decibel::LogEntry::LogEntry(const LogEntry & in) :
    startTime(in.startTime),
    connectTime(in.connectTime),
    endTime(in.endTime),
    protocol(in.protocol),
    incoming(in.incoming),
    contacts(in.contacts)
{ }

Decibel::LogEntry::LogEntry(const uint st, const uint ct, const uint et,
                            const QString & proto, const bool ic,
                            const QStringList cont) :
    startTime(st), connectTime(ct), endTime(et),
    protocol(proto), incoming(ic),
    contacts(cont)
{ }

Decibel::LogEntry::~LogEntry()
{ }

// Operater >> and <<:

const QDBusArgument & operator << (QDBusArgument & arg,
                                   const Decibel::ComponentInfo & info)
{
    arg.beginStructure();
    arg << info.display_name
        << info.possible_protocol_list
        << info.possible_type_list
        << info.possible_target_type_list
        << info.service_name
        << info.object_path;
    arg.endStructure();
    return arg;
}

const QDBusArgument & operator >> (const QDBusArgument & arg,
                                   Decibel::ComponentInfo & info)
{
    arg.beginStructure();
    arg >> info.display_name
        >> info.possible_protocol_list
        >> info.possible_type_list
        >> info.possible_target_type_list
        >> info.service_name
        >> info.object_path;
    arg.endStructure();
    return arg;
}

// ComponentSetup:

Decibel::Component::Component() { }

Decibel::Component::Component(const uint handle) :
     component_handle(handle)
{ }

Decibel::Component::Component(const QString & name,
                              const QStringList & protocols,
                              const QList<int> & types,
                              const QList<int> & targets,
                              const uint handle) :
    display_name(name),
    protocol_list(protocols),
    type_list(types),
    target_type_list(targets),
    component_handle(handle)
{ }

Decibel::Component::Component(const Decibel::Component & component) :
    display_name(component.display_name),
    protocol_list(component.protocol_list),
    type_list(component.type_list),
    target_type_list(component.target_type_list),
    component_handle(component.component_handle)
{ }

const QDBusArgument & operator << (QDBusArgument & arg,
                                   const Decibel::Component & component)
{
    arg.beginStructure();
    arg << component.display_name
        << component.protocol_list
        << component.type_list
        << component.target_type_list
        << component.component_handle;
    arg.endStructure();
    return arg;
}

const QDBusArgument & operator >> (const QDBusArgument & arg,
                                   Decibel::Component & component)
{
    arg.beginStructure();
    arg >> component.display_name
        >> component.protocol_list
        >> component.type_list
        >> component.target_type_list
        >> component.component_handle;
    arg.endStructure();
    return arg;
}

const QDBusArgument & operator << (QDBusArgument & arg,
                                   const Decibel::LogEntry & entry)
{
    arg.beginStructure();
    arg << entry.startTime
        << entry.connectTime
        << entry.endTime
        << entry.protocol
        << entry.incoming
        << entry.contacts;
    arg.endStructure();
    return arg;
}

const QDBusArgument & operator >> (const QDBusArgument & arg,
                                   Decibel::LogEntry & entry)
{
    arg.beginStructure();
    arg >> entry.startTime
        >> entry.connectTime
        >> entry.endTime
        >> entry.protocol
        >> entry.incoming
        >> entry.contacts;
    arg.endStructure();
    return arg;
}

// Helper functions:

void Decibel::registerTypes()
{
    static bool registered = false;
    if (!registered)
    {
        qDBusRegisterMetaType<QVariantMap>();
        qDBusRegisterMetaType<Decibel::ChannelInfo>();
        qDBusRegisterMetaType<Decibel::ComponentInfo>();
        qDBusRegisterMetaType<Decibel::Component>();
        qDBusRegisterMetaType<Decibel::LogEntry>();
        qDBusRegisterMetaType<QList<Decibel::Component> >();

        // types from QtTapioca:
        QtTapioca::ConnectionManagerFactory::registerTypes();

        registered = true;
    }
}
