/*
Copyright (c) 2010 by Drake Justice <djustice.kde@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.
*/

#include <QMessageBox>
#include <QDebug>

#include <KLocale>

#include "formatthread.h"

FormatThread::FormatThread(QObject* parent, const QString& device_path, int disc_size): QThread(parent)
{
    m_device = device_path;
    m_hadError = false;
    m_discSize = disc_size;
}

void FormatThread::run()
{
  qDebug() << "FormatThread::run()";

    connect(&m_process, SIGNAL(finished(int)), this, SLOT(processFinished(int)));
    connect(&m_process, SIGNAL(error(QProcess::ProcessError)), this, SLOT(processError(QProcess::ProcessError)));
    connect(&m_timer, SIGNAL(timeout()), this, SLOT(processUpdate()));

    m_process.setProcessChannelMode(QProcess::MergedChannels);

    if (m_discSize < 815) {            // cd-rw
        m_process.start("wodim -vvv -dev=" + m_device + " blank=fast");
    } else if (m_discSize < 4600) {     // dvd-rw
        m_process.start("wodim -vvv -dev=" + m_device + " -format");
    } else if (m_discSize < 9200) {     // dvd-dl-rw

    } else if (m_discSize < 87000) {     // blu-ray-rw

    }

    m_timer.start(500);

    exec();
}

void FormatThread::processUpdate()
{
    QString output = m_process.readAll();

  qDebug() << output;

    if (output.contains("exclusively (Device or resource busy)...")) {
        m_hadError = true;
        cancelProcess();
        QMessageBox::information(0, i18n("Error"), i18n("The disc is either mounted or in use. discburner will not format it."));
    }
}

void FormatThread::processFinished(int exit_code)
{
  qDebug() << "FormatThread::processFinished() with exit_code: " << exit_code;

    m_timer.stop();

    emit finished();

    if (m_wasCanceled)
        return;

    if (m_hadError)
        return;

    if (m_discSize < 815)
    { } else if (m_discSize < 4600) {
        emit jobStatus("wiped.dvd+rw");
        QMessageBox::information(0, "DiscBurner", "DVDRW formatted, ready to be overwritten");
        return;
    } else if (m_discSize < 9200) {
    } else if (m_discSize < 87000) {
    }

    QMessageBox::information(0, "DiscBurner", i18n("Formatting complete"));
}

void FormatThread::processError(QProcess::ProcessError error)
{
    m_hadError = true;

    if(m_wasCanceled)
        return;

    switch(error)
    {
    case 0:
        QMessageBox::information(0, "DiscBurner", i18n("Formatting failed, check that you have sufficient permissions, and that wodim is installed."));
        break;

    case 1:
        QMessageBox::information(0, "DiscBurner", i18n("Formatting crashed.  Has valid input been given?  Does the wodim binary work?"));
        break;

    case 2:
        QMessageBox::information(0, "DiscBurner", i18n("Formatting failed: timed out."));
        break;

    case 3:
        QMessageBox::information(0, "DiscBurner", i18n("Formatting failed: unable to communicate with wodim."));
        break;

    case 4:
        QMessageBox::information(0, "DiscBurner", i18n("Formatting failed due to a write error. Check disk space and permissions."));
        break;

    case 5:
        QMessageBox::information(0, "DiscBurner", i18n("Formatting failed with an unknown error."));
        break;
    };
}

void FormatThread::cancelProcess()
{
    m_process.kill();
    m_wasCanceled = true;
}
