// -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 1999 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#include "def.h"
#include "dict.h"
#include "tag.h"
#include <cstdio>
const char *GetIndexName(int id,bool isEB) {
  static bool initialized=false;
  static std::map<int,std::string> indexName;
  if (id<0 || id>0xff) return NULL;
  if (!initialized){
    struct iName {
      int id;
      char* name;
    };
    static iName iname[]= {
      {0x00,N_("Honmon Data")},
      {0x01,N_("MENU")},
      {0x02,N_("Copyright")},
      {0x03,N_("Index Search - Headword")},
      {0x04,N_("Word Search (Kana) - Headword")},
      {0x05,N_("Word Search - Headword")},
      {0x06,N_("Endword Search (Kana) - Headword")},
      {0x07,N_("Endword Search - Headword")},
      {0x08,N_("Word Search (Latin) - Headword")},
      {0x09,N_("Endword Search (Latin) - Headword")},
      {0x0a,N_("Index Search (Cross) - Headword")},
      {0x0b,N_("Group Number Search - Headword")},
      {0x0d,N_("Multi-parameter Search - Headword")},
      {0x0e,N_("Auto Demo ?")},
      {0x0f,N_("Figure Index")},
      {0x10,N_("Menu Search")},
      {0x11,N_("Menu Search (Color)")},
      {0x12,N_("forbidden characters")},
      {0x16,N_("Multi-parameter Search - big headline")},
      {0x20,N_("Menu Data")},
      {0x21,N_("Copyright Data")},
      {0x22,N_("Menu 3?")},
      {0x23,N_("Copyright (one line)")},
      {0x24,N_("Sound")},
      {0x30,N_("Kana Index")},
      {0x40,N_("Latin Index")},
      {0x50,N_("Kanji Index")},
      {0x60,N_("Word Index")},
      {0x61,N_("English Word Jump Index")},
      {0x70,N_("Endword Kana Search Index")},
      {0x71,N_("Endword Word Search Index")},
      {0x72,N_("Endword Latin Search Index")},
      {0x73,N_("Endword Misc Search Index")},
      {0x80,N_("Multi-word Search Index")},
      {0x81,N_("Multi-parameter Search Index")},
      {0x90,N_("Word Kana Search Index")},
      {0x91,N_("Word Search Index")},
      {0x92,N_("Word Search Latin Index")},
      {0x93,N_("Word Search Misc Index")},
      {0xa0,N_("Group Number Index")},
      {0xa1,N_("Multi-parameter Search Index Item")},
      {0xb0,N_("Page Index")},
      {0xb1,N_("Kanji Search Index")},
      {0xd0,N_("Voice/Figure Data")},
      {0xd1,N_("Picture Data")},
      {0xd2,N_("JPEG Bitmap Data")},
      {0xd8,N_("PCM? Sound Data")},
      {0xe0,N_("Color Data")},
      {0xee,N_("Auto Demo (Color)")},
      {0xef,N_("Figure Index (Color)")},
      {0xf0,N_("ANSHO?")},
      {0xf1,N_("16x16 dot extended characters")},
      {0xf2,N_("8x16 dot extended characters")},
      {0xf3,N_("24x24 dot extended characters")},
      {0xf4,N_("12x24 dot extended characters")},
      {0xf5,N_("30x30 dot extended characters")},
      {0xf6,N_("15x30 dot extended characters")},
      {0xf7,N_("48x48 dot extended characters")},
      {0xf8,N_("24x48 dot extended characters")},
      {0xfb,N_("Gaiji Conversion Table?")},
      {0xff,N_("Multi-parameter Search Control Data")},
      {0x108,N_("Menu Search")},
    };
    for(int i=0;i<int(sizeof(iname)/sizeof(iName));i++){
      indexName[iname[i].id]=_(iname[i].name);
    }
    initialized=true;
  }
  if (isEB && id==8) return indexName[0x108].c_str(); 
  if (indexName.find(id)!=indexName.end()) return indexName[id].c_str();
  return NULL;
}

bool isMenuKey(int type,bool isEB){
  static bool tblIndexKey[256];
  static bool initialized=false;
  if (!initialized){
    for(int i=0;i<256;i++) tblIndexKey[i]=false;
    for(int i=0x80;i<0x8f;i++) tblIndexKey[i]=true;  // ֤ Index

    struct TblIndexKey {
      int id;
      bool isMenu;
    };
    static TblIndexKey tbl[]={
      {0x01,true},    // MENU
      {0x02,true},    // ɽ
    //{0x0e,true},    // Auto Demo?
    //{0x0f,true},    // * ǰ
      {0x10,true},    // MENU?
      {0x11,true},    // Color MENU
      {0x20,true},    // MENU 2
      {0x21,true},    // ɽ 2
      {0x22,true},    // MENU 3
      {0x23,true},    // ɽ 3
    //{0xee,true},    // Auto Demo(Color)
    //{0xef,true},    // Figure Index(Color)
      {0xff,true},    // ʣ縡
    };
    for(int i=0;i<int(sizeof(tbl)/sizeof(TblIndexKey));i++){
      tblIndexKey[tbl[i].id]=tbl[i].isMenu;
    }
    initialized=true;
  }
  if (type>0xff) return true; // ʣ縡
  if (type<0) return false;  // Error?
  //  if (isEB && type==8) return true;
  return tblIndexKey[type];
}

bool isIndexKey(int type,bool isEB){
  static bool tblIndexKey[256];
  static bool initialized=false;
  if (!initialized){
    for(int i=0;i<256;i++) tblIndexKey[i]=false;
    for(int i=0x80;i<0x8f;i++) tblIndexKey[i]=true;  // ֤ Index
    struct TblIndexKey {
      int id;
      bool isKey;
    };
    static TblIndexKey tbl[]={
      {0x01,true},     // MENU
      {0x02,true},     // ɽ
      {0x0e,true},     // Auto Demo?
      {0x0f,true},     // * ǰ
      {0x10,true},     // MENU?
      {0x11,true},     // Color MENU
      {0x20,true},     // MENU 2
      {0x21,true},     // ɽ 2
      {0x22,true},     // MENU 3
      {0x23,true},     // ɽ 3
      {0xee,true},     // Auto Demo(Color)
      {0xef,true},     // Figure Index(Color)
      {0xff,true},     // ʣ縡
    };
    for(int i=0;i<int(sizeof(tbl)/sizeof(TblIndexKey));i++){
      tblIndexKey[tbl[i].id]=tbl[i].isKey;
    }
    initialized=true;
  }

  if (isMenuKey(type,isEB)) return true;
  if (type>0xff) return true; // ʣ縡
  if (type<0) return false;  // Error?
  if (isEB && type==0x08) return true;
  return tblIndexKey[type];
}

INDEX_HEAD::INDEX_HEAD(byte* &data,int indexDecodeMethod) {
  type=*data;
  start=GetDWord(data+2);
  length=GetDWord(data+6);
  typeName=std::string("");
  indexDecodeMethodAvailability=data[10];
  if (indexDecodeMethod==2 || 
      (indexDecodeMethod==0 && indexDecodeMethodAvailability==2)){
    //    sel->Setup(5,15);
    codeConvMode.Setup(data+10);
  } else {
    codeConvMode.Setup(NULL);
  }
  data+=16;
};

SUBINDEX::SUBINDEX(byte* &data,int indexDecodeMethod){
  int nSubSub=*data;
  Debug::DebugOut(Debug::INDEX_DECODE|Debug::VERBOSE,"SUBINDEX %d\n",nSubSub);
  memcpy(typeName,data+2,30);
  typeName[30]=0;
  data+=32;
  for(int i=0;i<nSubSub;i++){
    int n=*data;
    subSubIndex[n]=INDEX_HEAD(data,indexDecodeMethod);
  }
}

MINDEX_HEAD::MINDEX_HEAD(byte* &data,int indexDecodeMethod){
  int nSub=GetWord(data);
  Debug::DebugOut(Debug::INDEX_DECODE|Debug::VERBOSE,"MINDEX_HEAD %d\n",nSub);
  data+=16;
  for(int i=0;i<nSub;i++){
    subIndex.push_back(SUBINDEX(data,indexDecodeMethod));
  }
  typeName=std::string("");
}

void INDEX_HEAD::DebugOut(int level,bool isEB){
  if (GetIndexName(type,isEB)!=NULL) {
    Debug::DebugOut(level,"%02X %7ld %7ld %s\n",type,start,length,
		    GetIndexName(type,isEB));
  }
}  

void MINDEX_HEAD::DebugOut(int level){
  std::vector<SUBINDEX>::iterator i;
  for(i=subIndex.begin();i!=subIndex.end();++i){
    i->DebugOut(level);
  }
}

void SUBINDEX::DebugOut(int level){
  std::map<int,INDEX_HEAD>::iterator i;
  Debug::DebugOut(level,">  %s\n",
		  CODECONV::Jis2Euc(std::string((char*)typeName)).c_str());
  for(i=subSubIndex.begin();i!=subSubIndex.end();++i){
    Debug::DebugOut(level,">> ");
    i->second.DebugOut(level);
  }
}

std::string MINDEX_HEAD::SubIndexName(int n) const { 
  if (n>=0 && n<nSubIndex()){
    return std::string(reinterpret_cast<const char*>(subIndex[n].typeName)); 
  } else {
    return std::string("");
  }
};

SUBINDEX* MINDEX_HEAD::SubIndex(int n) {
  if (n>=0 && n<nSubIndex()){
    return &subIndex[n];
  } else {
    return NULL;
  }
};

INDEX_HEAD* SUBINDEX::SubSubIndex(int n) {
  if (subSubIndex.find(n)!=subSubIndex.end()){
    return &(subSubIndex[n]);
  } else {
    return NULL;
  }
};


INDEXITEM::INDEXITEM() : keyword("") {};

INDEXITEM::INDEXITEM(const std::string& key,const TAG& a,
		     const std::string& dkey,const TAG& l){
  keyword=key;
  dkeyword="";
  adrs=a;
  keywordTag=l;
  dkeyword=dkey;
  Debug::DebugOut(Debug::INDEX_DECODE|Debug::VERBOSE,
	     " 2: len=%d : (%6d:%4d), (%6d:%4d) : %s\n",key.length(),
	     adrs.Block(),adrs.Offset(),keywordTag.Block(),keywordTag.Offset(),
	     CODECONV::Jis2Euc(key).c_str());
};

void INDEXITEM::DebugOut(int level) const {
  Debug::DebugOut(level,"INDEXITEM:a(%6d:%4d):l(%6d:%4d):key=(%d)'%s',(%d)'%s':\n",
	     adrs.Block(),adrs.Offset(),
	     keywordTag.Block(),keywordTag.Offset(),keyword.length(),
		  CODECONV::Jis2Euc(keyword).c_str(),dkeyword.length(),
		  CODECONV::Jis2Euc(dkeyword).c_str());
}

/*
bool INDEXITEM::operator==(const INDEXITEM& itm) const {
  return ((itm.adrs==adrs &&
	   CODECONV::CompareKana(itm.keyword,keyword)==0) && 
	  (itm.keywordTag==keywordTag || itm.dkeyword==dkeyword));   
};

bool INDEXITEM::operator<(const INDEXITEM& itm) const {
  return ( 
	  CODECONV::CompareKana(keyword,itm.keyword)<0 || 
	  adrs<itm.adrs || 
	   ((itm.adrs==adrs && itm.keyword==keyword) && 
	    (keywordTag < itm.keywordTag ||
	     CODECONV::CompareKana(dkeyword,itm.dkeyword)<0)));
};
*/

bool INDEXITEM::operator==(const INDEXITEM& itm) const {
  return ((itm.adrs==adrs &&
	   itm.keyword==keyword) && 
	  (itm.keywordTag==keywordTag || itm.dkeyword==dkeyword));   
};

bool INDEXITEM::operator<(const INDEXITEM& itm) const {
  return ( 
	  keyword<itm.keyword || 
	  adrs<itm.adrs || 
	   ((itm.adrs==adrs && itm.keyword==keyword) && 
	    (keywordTag < itm.keywordTag ||
	     dkeyword<itm.dkeyword)));
};

int INDEXITEM::FuzzyCompare(const INDEXITEM& itm) const {
  if (adrs==itm.adrs && 
      (keywordTag == itm.keywordTag || dkeyword==itm.dkeyword)) return 0;
  if (adrs<itm.adrs || 
      (adrs==itm.adrs && keywordTag < itm.keywordTag)) return -1;
  return 1;
}

INDEXITEM::INDEXITEM(const INDEXITEM& i){
  copy(i);
};
    
const INDEXITEM& INDEXITEM::operator = (const INDEXITEM& i){
  copy(i);
  return *this;
};

void INDEXITEM::copy(const INDEXITEM& i){
  keyword=i.keyword;
  dkeyword=i.dkeyword;
  adrs=i.adrs;
  keywordTag=i.keywordTag;
};


bool INDEX::operator==(const INDEX& idx) const { 
  return (block==idx.block && currentPoint==idx.currentPoint); 
};
bool INDEX::operator<(const INDEX& idx) const { 
  return (block<idx.block || 
	  (block==idx.block && currentPoint<idx.currentPoint)); 
};


bool INDEX::DecodeUpperFixedIndex(byte* data){
  int len=data[1];
  int nItem=GetWord(data+2);
  data+=4;
  char* buf=new char[len+1];
  Debug::DebugOut(Debug::INDEX_DECODE|Debug::VERBOSE,
	     "̥ǥå(Ĺ %d) : ȥ %d :\n",
	   len,nItem);
  for(int i=0;i<nItem;i++){
    memcpy(buf,reinterpret_cast<char*>(data),len);
    buf[len]=0;
    std::string key=std::string(buf);
    indexItem.push_back(INDEXITEM(key,TAG(GetDWord(data+len),0),"",TAG(0,0)));
    data+=len+4;
  }
  delete[] buf;
  return true;
}

bool INDEX::DecodeUpperVariableIndex(byte* data){
  int nItem=GetWord(data+2);
  data+=4;
  Debug::DebugOut(Debug::INDEX_DECODE|Debug::VERBOSE,
	     "̥ǥå(Ĺ) : ȥ = %d :\n",nItem);
  for(int i=0;i<nItem;i++){
    int len=*data++;
    if (!isGroupIndex()) { // ̷ȥ꤫鹽
      indexItem.push_back(
	       INDEXITEM(std::string(reinterpret_cast<char*>(data),len),
			    TAG(data+len),"",TAG(0,0)));
      data+=len+6;
    } 
  }
  return true;
}

bool INDEX::DecodeUpperIndex(byte* data){
  if (keyLength>0) return DecodeUpperFixedIndex(data);
  else return DecodeUpperVariableIndex(data);
}

bool INDEX::DecodeLowerFixedIndex(BlockIO* p,int blk){
  Debug::ErrorOut("ǲ̸ĹǥåϤޤݡ̵");
  exit(1);
}

bool INDEX::isShortIndex() { 
  //  fprintf(stderr,"%02x %d\n",type,
  //	  ((type & 0xf0)!=0x90 && (type & 0xf0)!=0x70 && (type & 0xf0)!=0xa0));
  return ((type & 0xf0)!=0x90 && 
	  (type & 0xf0)!=0x70 &&
	  (type & 0xf0)!=0xa0); 
};

bool INDEX::isEB() { return isEb; };

bool INDEX::isUpperIndex() { return (blockType & 0x80)==0; };
bool INDEX::isIndexHead()   { return (blockType&0x40)!=0 && currentPoint==0;};
bool INDEX::isIndexTail()   { return (blockType & 0x20)!=0 && 
				currentPoint>=
                                  static_cast<int>(indexItem.size()-1); };
bool INDEX::isGroupIndex()  { return (blockType & 0x10)!=0; };
int INDEX::unknownType()    { return (blockType & 0x0f); };

bool INDEX::isVariableIndex() { return (keyLength==0); };

bool INDEX::DecodeLowerVariableIndex(BlockIO* p,int blk){
  int n=0,blockCount=0;
  std::string key0="";
  TAG keywordTag0=TAG(0,0);
  byte data[cBlockSize];
  byte* dp;
  do {
    p->ReadBlock(blk++,data);
    blockCount++;
    int nItem=GetWord(data+2);
    Debug::DebugOut(Debug::INDEX_DECODE,
		    "ǲ̥ǥå %d : ȥ %d : type = %02X\n",
		    blk-1,nItem,blockType);
    dp=data+4;
    if (isGroupIndex()){
      for(int i=0;i<nItem;i++){
	//	Debug::ErrorOut("at %d\n",dp-data);
	int tp=*dp++;
	//	fprintf(stderr,"at %d %x\n",dp-1-data,tp);
	switch(tp & 0xf0) {
	case 0x00: // ܥȥ
	  {
	    int len=*dp++;

	    TAG t=p->Tell();
	    char k[cTmpLength];
	    p->Seek(TAG(dp+len+6));
	    p->GetLine(k,cTmpLength);
	    p->Seek(t);

	    indexItem.push_back(INDEXITEM(std::string((char*)dp,len),
					  TAG(dp+len),k,TAG(dp+len+6)));
	    dp+=len+12;
	  }
	  break;
	case 0x80: // ĥȥ
	  {
	    int len=*dp++;
	    switch(type & 0xf0) {
	    case 0x70:
	    case 0x90:
	      n=GetWord(dp);
	      dp+=2;
	      key0=std::string((char*)dp,len);
	      dp+=len;
	      break;
	    case 0xa0:
	      n=GetDWord(dp);
	      dp+=4;
	      key0=std::string((char*)dp,len);
	      dp+=len;
	      break;
	    case 0x80:
	      n=GetDWord(dp);
	      dp+=4;
	      key0=std::string((char*)dp,len);
	      dp+=len;
	      keywordTag0=TAG(dp);
	      dp+=6;
	      break;
	    default:
	      n=GetWord(dp);
	      dp+=2;
	      key0=std::string((char*)dp,len);
	      dp+=len;
	      keywordTag0=TAG(dp);
	      dp+=6;
	      break;
	    } 
	  }
	  break;
	case 0xc0: //ĥȥо
	  {
	    switch(type & 0xf0){
	    case 0xa0:
	      {
		TAG t=p->Tell();
		char k[cTmpLength];
		p->Seek(TAG(dp+6));
		p->GetLine(k,cTmpLength);
		p->Seek(t);
		indexItem.push_back(INDEXITEM(key0,TAG(dp),k,TAG(dp+6)));
		dp+=12;
	      }
	      break;
	    case 0x70:
	    case 0x90:
	      {
		int len=*dp++;

		TAG t=p->Tell();
		char k[cTmpLength];
		p->Seek(TAG(dp+len+6));
		p->GetLine(k,cTmpLength);
		p->Seek(t);

		indexItem.push_back(INDEXITEM(std::string((char*)dp,len),
					      TAG(dp+len),
					      k,TAG(dp+len+6)));
		dp+=len+12;
	      }
	      break;
	    case 0x80:
	    default:
	      TAG t=p->Tell();
	      char k[cTmpLength];
	      p->Seek(keywordTag0);
	      p->GetLine(k,cTmpLength);
	      p->Seek(t);
	      indexItem.push_back(INDEXITEM(key0,TAG(dp),k,keywordTag0));
	      dp+=6;
	      break;
	    }
	  }
	  --n;
	  if (n==0 && blockCount>1) return true;
	  break;

	default:
	  Debug::ErrorOut("Unknown Type 0x%02x\n",tp);
	  Debug::ErrorOut("at %d\n",dp-data);
	  exit(1);
	}
      }
    } else {
      for(int i=0;i<nItem;i++){
	int len=*dp++;

	TAG t=p->Tell();
	char k[cTmpLength];
	p->Seek(TAG(dp+len+6));
	p->GetLine(k,cTmpLength);
	p->Seek(t);

	indexItem.push_back(INDEXITEM(std::string((char*)dp,len),TAG(dp+len),
				      k,TAG(dp+len+6)));
	dp+=len+12;
      }
    }
  } while(n>0);
  return true;
}

bool INDEX::DecodeLowerIndex(BlockIO* p,int blk){
  if (keyLength>0) return DecodeLowerFixedIndex(p,blk);
  return DecodeLowerVariableIndex(p,blk);
}

INDEX::INDEX(BlockIO* p,int pp,int tp,bool iseb) {
  isEb=iseb;
  ReadIndex(p,pp,tp);
}

bool INDEX::ReadIndex(BlockIO* p,int pp,int tp) {
  pf=p;
  currentPoint=0;
  block=pp;
  byte data[cBlockSize];
  if (!p->ReadBlock(block,data)) return false;

  type=tp;
  blockType=data[0];
  keyLength=data[1];
  indexItem.clear();

  Debug::DebugOut(Debug::INDEX_DECODE,
		  "Block=%d : INDEX TYPE %02x\n",block,blockType);

  if (isUpperIndex()){
    DecodeUpperIndex(data);
  } else {
    DecodeLowerIndex(pf,block);
  }
  return true;
}

bool INDEX::Prev(){
  if (currentPoint>0){
    currentPoint--;
    return true;
  }
  if (!isIndexHead()){
    ReadIndex(pf,block-1,type);
    currentPoint=indexItem.size()-1;
    return true;
  }
  return false;
}

bool INDEX::Next(){
  if (currentPoint<static_cast<int>(indexItem.size()-1)){
    currentPoint++;
    return true;
  }
  if (!isIndexTail()){
    ReadIndex(pf,block+1,type);
    return true;
  }
  return false;
}

  
INDEX::INDEX(INDEX* idx) {
  Dup(*idx);
}

INDEX::~INDEX() {
  pf=NULL;
}

void INDEX::DebugOut(int level){
  Debug::DebugOut(Debug::INDEX_DECODE,"INDEX Block=%d : type=%02x\n",
		  block,blockType);
  Debug::DebugOut(level,
	  "INDEX: Type=%02x : nItem=%d \n",
	  type,indexItem.size());
  std::vector<INDEXITEM>::iterator i;
  for(i=indexItem.begin();i!=indexItem.end();++i){
    i->DebugOut(level);
  }
}

void INDEX::Dup(const INDEX& idx){
  if (idx.pf==NULL){
    Debug::DebugOut(Debug::WARNINGS,"INVALID INDEX\n");
    return;
  }
  pf=idx.pf;
  
  type=idx.type;
  blockType=idx.blockType;
  keyLength=idx.keyLength;
  block=idx.block;
  currentPoint=idx.currentPoint;
  indexItem=idx.indexItem;
  isEb=idx.isEb;
}

INDEX::INDEX(const INDEX& idx){
  Dup(idx);
}

const INDEX& INDEX::operator=(const INDEX& idx){
  Dup(idx);
  return *this;
}

