/***************************************************************************
 *   Copyright (C) 2003 by Gav Wood                                        *
 *   gav@cs.york.ac.uk                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 ***************************************************************************/
#define __GEDDEI_BUILD

#include "processor.h"
#include "domprocessor.h"
#include "processorgroup.h"
using namespace Geddei;

#define MESSAGES 0

namespace Geddei
{

ProcessorGroup::ProcessorGroup(const bool adopt) : theAdopt(adopt)
{
	theProcessors.clear();
}

ProcessorGroup::~ProcessorGroup()
{
	if(theAdopt) deleteAll();
}

const bool ProcessorGroup::exists(const QString &name)
{
	return theProcessors.count(name);
}

Processor &ProcessorGroup::get(const QString &name)
{
	if(theProcessors.count(name))
		return *(theProcessors[name]);
	else
	{	qFatal("*** FATAL: Attempting to attain a Processor object that does not exist.\n"
			   "           You used the non-existant name %s. Bailing.", name.latin1());
		return *((Processor *)(NULL));
	}
}

DomProcessor &ProcessorGroup::dom(const QString &name)
{
	if(theProcessors.count(name))
		return *dynamic_cast<DomProcessor *>(theProcessors[name]);
	else
	{	qFatal("*** FATAL: Attempting to attain a Processor object that does not exist.\n"
			   "           You used the non-existant name %s. Bailing.", name.latin1());
		return *((DomProcessor *)(NULL));
	}
}

void ProcessorGroup::add(Processor *o)
{
	for(QMap<QString, Processor *>::Iterator i = theProcessors.begin(); i != theProcessors.end(); i++)
		if(i.data() == o) return;
		else if(i.key() == o->name())
			qDebug("*** ERROR: You are using the same name for multiple Processor objects.\n"
				   "           You have to use a unique name for each object or grouping will not\n"
				   "           work properly.");
	theProcessors[o->name()] = o;
	o->setGroup(*this);
}

void ProcessorGroup::remove(Processor *o)
{
	for(QMap<QString, Processor *>::Iterator i = theProcessors.begin(); i != theProcessors.end(); i++)
		if(i.data() == o)
		{	theProcessors.erase(i);
			o->setNoGroup();
			return;
		}
}

const bool ProcessorGroup::confirmTypes() const
{
	if(MESSAGES) qDebug("ProcessorGroup::confirmTypes()");
	for(QMap<QString, Processor *>::ConstIterator i = theProcessors.begin(); i != theProcessors.end(); i++)
	{	if(MESSAGES) qDebug("ProcessorGroup::confirmTypes(): Confirming %p...", i.data());
		if(!i.data()->confirmTypes()) return false;
	}
	return true;
}

void ProcessorGroup::deleteAll()
{
	QMap<QString, Processor *> theProcessorsBU = theProcessors;
	for(QMap<QString, Processor *>::ConstIterator i = theProcessorsBU.begin(); i != theProcessorsBU.end(); i++)
		delete i.data();
	theProcessors.clear();
}

void ProcessorGroup::disconnectAll()
{
	for(QMap<QString, Processor *>::ConstIterator i = theProcessors.begin(); i != theProcessors.end(); i++)
		i.data()->disconnectAll();
}

const bool ProcessorGroup::go(const bool waitUntilGoing) const
{
	if(!confirmTypes()) return false;
	for(QMap<QString, Processor *>::ConstIterator i = theProcessors.begin(); i != theProcessors.end(); i++)
		if(!i.data()->go()) return false;
	if(waitUntilGoing) return !ProcessorGroup::waitUntilGoing();
	return true;
}

const Processor::ErrorType ProcessorGroup::waitUntilGoing(Processor **errorProc, int *errorData) const
{
	Processor::ErrorType ret;
	for(QMap<QString, Processor *>::ConstIterator i = theProcessors.begin(); i != theProcessors.end(); i++)
		if((ret = i.data()->waitUntilGoing(errorData)) != Processor::NoError)
		{	if(errorProc) *errorProc = i.data();
			for(QMap<QString, Processor *>::ConstIterator j = theProcessors.begin(); j != i; j++)
				j.data()->stop();
			return ret;
		}
	return Processor::NoError;
}

void ProcessorGroup::stop(const bool resetToo) const
{
	for(QMap<QString, Processor *>::ConstIterator i = theProcessors.begin(); i != theProcessors.end(); i++)
		i.data()->stop();
	if(resetToo) reset();
}

void ProcessorGroup::reset() const
{
	for(QMap<QString, Processor *>::ConstIterator i = theProcessors.begin(); i != theProcessors.end(); i++)
		i.data()->reset();
}

}
