/*
 * Dialog handling code
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#define G_LOG_DOMAIN "dialog"

#include <gnubversion.h>
#include <gnome.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <assert.h>
#include <unistd.h>
#include <svn_client.h>

#include "gvn-checkout.h"

/* Shorthand macros for important GUI widgets */
#define MAIN_WINDOW gnubversion_get_widget ( "mainWindow" )
#define REVISIONCHOOSER GNUBVERSION_REVISIONCHOOSER(gnubversion_get_widget("revisionChooser"))
#define OPTIONS GNUBVERSION_COMMONOPTIONS(gnubversion_get_widget("options"))

/* Flags for keeping track of *why* we disabled the OK button. */
#define OKFLAGS_BAD_URL  ( 1 << 0 )
#define OKFLAGS_BAD_DEST ( 1 << 1 )
#define OKFLAGS_BAD_WCNAME ( 1 << 2 )

static int okbutton_flags = OKFLAGS_BAD_URL;

static worker_comms_t worker_comms;    // FIXME: Booh hiss. Global variable!

static void
set_status ( const char *status )
{
    GnomeAppBar *appbar = GNOME_APPBAR ( gnubversion_get_widget ( "mainAppBar" ) );
    gnome_appbar_clear_stack ( appbar );
    if ( status != NULL )
        gnome_appbar_push ( appbar, status );
}

/*
 * Enables/disables the OK button in the main window.
 */
static void 
update_okbutton ( int mask, gboolean flagset, const char *message )
{
    if ( flagset )
        okbutton_flags |= mask;
    else
        okbutton_flags &= ~mask;

    gtk_widget_set_sensitive ( gnubversion_get_widget ( "checkoutOkButton"), 
                               ( okbutton_flags == 0 ) );
    if ( flagset )
        set_status ( message );
    else
        set_status ( NULL );
}

static void 
check_destination()
{
    GnomeFileEntry *gef = GNOME_FILE_ENTRY ( gnubversion_get_widget ( "checkoutDestination" ) );
    char *path = gnome_file_entry_get_full_path ( gef, FALSE );
    update_okbutton ( OKFLAGS_BAD_DEST, 
                      !g_file_test ( path, G_FILE_TEST_IS_DIR ), 
                      _( "Directory not found" ) );
}

GLADE_CALLBACK void
checkoutDestination_changed_cb  ( GtkWidget *widget, gpointer userdata )
{
    check_destination ();
}

static void 
check_sandbox_name ()
{
    gboolean isbad = FALSE;
    const char *wcname;

    if ( get_toggle_button_value ( "workingCopyNameDefaultRadio" ) )
    {
        GtkWidget *widget = gnubversion_get_widget ( "workingCopyNameDefault" );
        wcname = gtk_label_get_text ( GTK_LABEL ( widget ) );
    }
    else
    {
        GtkWidget *widget = gnubversion_get_widget ( "workingCopyNameCustom" );
        wcname = gtk_entry_get_text ( GTK_ENTRY ( widget ) );
    }
    
    if ( ( wcname == NULL ) || ( *wcname == 0 ) )
        isbad = TRUE;
    else if ( strchr ( wcname, '/' ) != NULL)
        isbad = TRUE;

    update_okbutton( OKFLAGS_BAD_WCNAME, isbad, _( "Invalid working copy name" ) );
}

static void
set_default_sandbox( const char *name )
{
    GtkLabel *def = GTK_LABEL ( gnubversion_get_widget ( "workingCopyNameDefault" ) );
    gtk_label_set_text (def, name );
    check_sandbox_name ();
}

static int
count_slashes(const char *str)
{
    int kount = 0;
    
    for (; ( str = strchr ( str, '/') ) != NULL; kount++)
        str++;
    
    return kount;
}

static void
check_repository_url()
{
    GtkComboBox *combo = GTK_COMBO_BOX ( gnubversion_get_widget ( "checkoutURL" ) );
    char *c = gtk_combo_box_get_active_text ( combo );
    gboolean badURL;

    badURL = ( ( c == NULL ) || ( *c == 0 ) );

    if ( badURL )
        set_default_sandbox ( "" );
    else if ( count_slashes ( c ) < 3)
        badURL = TRUE;
    else
    {
        /* The part after the last '/' becomes the default sandbox name */
        char *lastslash = strrchr ( c, '/' );

        if ( lastslash == NULL )
            badURL = TRUE;
        else if ( *(lastslash+1) == 0 )
            badURL = TRUE;
        else
            set_default_sandbox ( lastslash+1 );
    }

    update_okbutton ( OKFLAGS_BAD_URL, badURL, _( "Invalid URL" ) );
}

GLADE_CALLBACK void
checkoutURL_changed_cb  ( GtkWidget *widget, gpointer userdata )
{
    check_repository_url ();
}

/* Read the values from the GUI and populate
 * the worker_comms_t structure
 */
static void 
fillin_worker_comms ( worker_comms_t *wc )
{
    gnubversion_revisionchooser_get_revision (REVISIONCHOOSER, &wc->revision);
 
    GtkComboBox *urlbox = GTK_COMBO_BOX ( gnubversion_get_widget ( "checkoutURL" ) );
    wc->repository_url = gtk_combo_box_get_active_text ( urlbox );
    gvn_debug ( "url: %s", wc->repository_url );

    wc->recursive = gnubversion_commonoptions_get_recursive ( OPTIONS );
    wc->ignore_externals = gnubversion_commonoptions_get_ignore_externals ( OPTIONS );

    GnomeFileEntry *destEntry = GNOME_FILE_ENTRY ( gnubversion_get_widget ( "checkoutDestination" ) );
    wc->destination_path = gnome_file_entry_get_full_path ( destEntry, FALSE );

    if ( get_toggle_button_value ( "workingCopyNameDefaultRadio" ) )
    {
        GtkWidget * wcname;
        wcname = gnubversion_get_widget ( "workingCopyNameDefault" );
        wc->working_copy_name = gtk_label_get_text ( GTK_LABEL ( wcname ) );
    }
    else
    {
        GtkWidget * wcname;
        wcname = gnubversion_get_widget ( "workingCopyNameCustom" );
        wc->working_copy_name = gtk_entry_get_text ( GTK_ENTRY ( wcname ) );
    }
}

static gint ask(GtkWindow *window, const char *format, ...)
{
    va_list ap;

    va_start(ap, format);
    GString *str = g_string_new("");

    g_string_vprintf(str, format, ap);

    GtkWidget *dialog = gnome_question_dialog_parented(str->str, NULL, NULL, window);
    assert(dialog != NULL);

    int retval;
    retval = gnome_dialog_run(GNOME_DIALOG(dialog));

    g_string_free(str, TRUE);
    va_end(ap);
    return retval;
}

static gboolean is_sandbox_directory_bad(GtkWidget *widget)
{
    GtkWindow *window = GTK_WINDOW( gtk_widget_get_ancestor(widget, GTK_TYPE_WINDOW));
    gboolean retval = FALSE;

    GString *str = g_string_new(worker_comms.destination_path);
    g_string_append(str,"/");
    g_string_append(str,worker_comms.working_copy_name);

    if (g_file_test(str->str, G_FILE_TEST_EXISTS))
    {
        if (g_file_test(str->str, G_FILE_TEST_IS_DIR))
        {
            retval = !ask (window, 
                           _("The directory %s already exists. This may obstruct "
                             "files in the the new working copy.\n\n"
                             "Do you want to continue the checkout anyway?"), 
                           str->str);
            gvn_debug("retval : %d", retval);
        }
        else
        {
            gnubversion_error(window, _("%s already exists, but is not a directory"), str->str);
            retval = TRUE;
        }
    }
    g_string_free(str, TRUE);
    return retval;
}

static void progressWindow_closed_cb(GnubVersionProgressWindow *progress, gpointer data)
{
    gvn_debug("start of progressWindow_closed_cb");

    if (gnubversion_progresswindow_was_task_successfull(progress))
    {
        gtk_main_quit();
    }
    else
    {
        gtk_widget_show_all (MAIN_WINDOW);
        set_status ( NULL );
    }
    gvn_debug("end of progressWindow_closed_cb");
}

GLADE_CALLBACK void
start_checkout_cb  ( GtkWidget *widget, gpointer userdata )
{
    fillin_worker_comms ( &worker_comms );

    if (is_sandbox_directory_bad(widget))
        return;

    gtk_widget_hide ( MAIN_WINDOW);

    if (worker_comms.progressWindow != NULL)
        g_object_unref(worker_comms.progressWindow);

    worker_comms.progressWindow = gnubversion_progresswindow_new( _("Preparing Checkout"));

    assert(worker_comms.progressWindow != NULL);
    gnubversion_auth_baton->progressWindow = worker_comms.progressWindow;
    
    gtk_window_set_title ( GTK_WINDOW (worker_comms.progressWindow), _("Checkout Progress"));
    gtk_widget_show ( GTK_WIDGET (worker_comms.progressWindow));

    gnubversion_progresswindow_connect__window_closed(worker_comms.progressWindow, progressWindow_closed_cb, &worker_comms);

    GThread *checkoutThread = NULL;
    checkoutThread = g_thread_create ( checkout_thread_main, &worker_comms, TRUE, NULL );
    assert ( checkoutThread != NULL );

    gvn_debug ("Main thread");
}

GLADE_CALLBACK void
workingCopyNameDefaultRadio_toggled_cb ( GtkWidget *widget, gpointer userdata )
{
    check_sandbox_name();
}

GLADE_CALLBACK void
workingCopyNameCustomRadio_toggled_cb ( GtkWidget *widget, gpointer userdata )
{
    GtkWidget *customName = gnubversion_get_widget ( "workingCopyNameCustom" );

    gtk_widget_set_sensitive(customName, get_toggle_button_value ("workingCopyNameCustomRadio"));
    check_sandbox_name();
}

GLADE_CALLBACK void
workingCopyNameCustom_changed_cb ( GtkWidget *widget, gpointer userdata )
{
    check_sandbox_name();
}

void 
dialog_init (
    const char *repository, 
    const char *revision,
    gboolean ignore_externals,
    gboolean recursive,
    const char *destination_path,
    const char *sandbox
)
{
    if ( (repository != NULL) && (*repository != 0))
        gtk_entry_set_text( GTK_ENTRY ( gnubversion_get_widget("checkoutURLEntry")), 
                            repository);

    if ( revision != NULL )
    {
        svn_opt_revision_t start_rev, end_rev;
        start_rev.kind = svn_opt_revision_unspecified;
        end_rev.kind = svn_opt_revision_unspecified;

        if ( svn_opt_parse_revision ( &start_rev, &end_rev, revision, gnubversion_apr_pool ) )
            g_error( _("Revision %s is not a valid revision number"), revision);
        else if ( end_rev.kind != svn_opt_revision_unspecified)
            g_error( _("Revision %s is not valid : ranges are not allowed"), revision);
        else
        {
            switch (start_rev.kind)
            {
                svn_revnum_t rev;
                case svn_opt_revision_head:
                    gnubversion_revisionchooser_set_head(REVISIONCHOOSER);
                    break;
                case svn_opt_revision_number:
                    rev = SVN_STR_TO_REV(revision);
                    if ( ! SVN_IS_VALID_REVNUM (rev) )
                    {
                        g_error( _("Revision %s is not a valid revision number"), revision );
                    }
                    gnubversion_revisionchooser_set_number(REVISIONCHOOSER, rev);
                    break;
                case svn_opt_revision_date:
                    gnubversion_revisionchooser_set_date_apr(REVISIONCHOOSER, start_rev.value.date);
                    break;
                case svn_opt_revision_unspecified:
                case svn_opt_revision_committed:
                case svn_opt_revision_base:
                case svn_opt_revision_working:
                case svn_opt_revision_previous:
                    g_error( _("Revision %s is not a valid revision number format"), revision);
                    break;
                default:
                    g_error( _("Internal error: cannot handle revision number %s of kind (%d)"), 
                            revision, start_rev.kind);
                    break;
            }
        }
    }

    gnubversion_commonoptions_set_ignore_externals (OPTIONS, ignore_externals);
    gnubversion_commonoptions_set_recursive (OPTIONS, recursive);

    // Fill in the destination
    if (destination_path != NULL)
    {
        gnome_file_entry_set_filename ( GNOME_FILE_ENTRY (gnubversion_get_widget ("checkoutDestination")),
                                        destination_path);
    }
    if ( (sandbox != NULL) && (strlen(sandbox) > 0) )
    {
        gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( gnubversion_get_widget ( "workingCopyNameDefaultRadio" ) ), 
                                        FALSE);
        gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( gnubversion_get_widget ( "workingCopyNameCustomRadio" ) ), 
                                        TRUE);
        gtk_entry_set_text( GTK_ENTRY ( gnubversion_get_widget ("workingCopyNameCustom") ), sandbox);
    }
    else
    {
        gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( gnubversion_get_widget ( "workingCopyNameDefaultRadio" ) ), 
                                        TRUE);
        gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( gnubversion_get_widget ( "workingCopyNameCustomRadio" ) ), 
                                        FALSE);
    }

    // Finally: show the GUI to the user and let him/her get on with doing stuff
    gtk_widget_show_all (MAIN_WINDOW);
    check_sandbox_name ();
    check_destination ();
    check_repository_url ();
    worker_comms.progressWindow = NULL;
}

static void dialog_noninteractive_finished(
    GnubVersionProgressWindow *progress, 
    gboolean successfull, 
    gpointer userdata)
{
    gvn_debug("Task finished");
    progressWindow_closed_cb(progress, NULL);
}

int dialog_noninteractive_start(void)
{
    start_checkout_cb( gnubversion_get_widget("checkoutOkButton"), NULL); 
    gnubversion_progresswindow_connect__task_finished(worker_comms.progressWindow, 
                                                    dialog_noninteractive_finished, NULL);
    gtk_main ();
    return 0;
}
