/*
 * Worker thread for commit
 *
 * This file contains all of the routines executed directly
 * by the worker thread
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#define G_LOG_DOMAIN "worker-thread"

#include <gnubversion.h>
#include <assert.h>
#include <svn_client.h>

#include "gvn-commit.h"

svn_error_t *
get_commit_log2 (
    const char **log_msg,
    const char **tmp_file,
    const apr_array_header_t *commit_items,
    void *svn_baton,
    apr_pool_t *pool)
{
    struct worker_thread_baton *baton = svn_baton;

    *tmp_file = NULL;
    *log_msg = apr_pstrdup(pool, baton->commitMessage);

    return SVN_NO_ERROR;
}

static void 
cancel_requested(GnubVersionProgressWindow *progress, gpointer user_data)
{
    gvn_debug("Cancel_requested signal received");
    struct worker_thread_baton *baton = user_data;

    baton->cancel_requested = TRUE;

    GnubVersionThreadMessage *msg;
    msg = gnubversion_threadmessage_new_info(_("Cancelling Commit..."));
    gnubversion_progresswindow_send_message(baton->progressWindow, msg);
}


static svn_error_t *
cancel_func( void *svn_baton )
{
    struct worker_thread_baton *baton = svn_baton;

    if ( baton->cancel_requested )
    {
        gvn_debug("Flagging up cancellation");
        return svn_error_create(SVN_ERR_CANCELLED, NULL,
                                "Operation cancelled by user");
    }
    else
        return SVN_NO_ERROR;
}

gpointer 
commit_thread_main( gpointer data )
{
    struct worker_thread_baton *baton = data;
    gvn_debug("thread started");

    apr_pool_t *pool;
    apr_status_t apr_err = apr_pool_create(&pool, gnubversion_apr_pool);
    if (apr_err)
    {
        GError *gerr = g_error_from_apr(apr_err /* , "Cannot create memory pool" */);
        GnubVersionThreadMessage *msg = gnubversion_threadmessage_new_bye_error(gerr);
        gnubversion_progresswindow_send_message(baton->progressWindow, msg);
        return NULL;
    }

    baton->cancel_requested = FALSE;
    gnubversion_progresswindow_connect__cancel_requested(baton->progressWindow, cancel_requested, baton);

    gvn_debug("getting filename list");
    apr_array_header_t *targets;
    targets = gnubversion_fileselectorstore_get_ticked_fullnames_apr(baton->store);
    assert(targets != NULL);

    svn_error_t * err = NULL;
    svn_commit_info_t *commit_info = NULL;

    gnubversion_svn_client_ctx->log_msg_func2 = get_commit_log2;
    gnubversion_svn_client_ctx->log_msg_baton2 = baton;

    gnubversion_svn_client_ctx->cancel_func = cancel_func;
    gnubversion_svn_client_ctx->cancel_baton = baton;

    gnubversion_svn_client_ctx->notify_func2 = gnubversion_progresswindow_svn_notify2;
    gnubversion_svn_client_ctx->notify_baton2 = baton->progressWindow;

    gvn_debug("committing");
    err = svn_client_commit3(&commit_info,
                             targets,
                             FALSE, /* recurse */
                             FALSE, /* keep_locks */
                             gnubversion_svn_client_ctx,
                             pool);

    gnubversion_svn_client_ctx->log_msg_func2 = NULL;
    gnubversion_svn_client_ctx->cancel_func = NULL;
    gnubversion_svn_client_ctx->notify_func2 = NULL;

    GnubVersionThreadMessage *msg;
    if (err) 
    {
        msg = gnubversion_threadmessage_new_bye_error(g_error_from_svn(err));
    }
    else
    {
        msg = gnubversion_threadmessage_new_bye_success(_("Commit Complete"));
    }
    gnubversion_progresswindow_send_message(baton->progressWindow, msg);

    apr_pool_destroy(pool);
    gvn_debug("bye bye");
    g_free(baton);
    return NULL;
}
