#define APP_NAME "gnubversion-auth-gui"

#include <glib/gi18n.h>
#include <libgnomeui/libgnomeui.h>
#include <gnubversion-threadmessage.h>
#include <assert.h>

#define RESPONSE_ACCEPT_TEMP 1
#define RESPONSE_ACCEPT_PERM 2

void
gnubversion_gui_auth_simple_prompt ( GnubVersionThreadMessage *msg)
{
    assert(msg != NULL);
    assert(GNUBVERSION_IS_THREADMESSAGE(msg));
    assert( gnubversion_threadmessage_get_message_type(msg) == MESSAGE_TYPE_ASK_USERNAME_PASSWORD );

    GnubVersionThreadMessage *response;
    GString *realm = gnubversion_threadmessage_get_realm(msg);
    GString *username = gnubversion_threadmessage_get_username(msg);
    GString *password = gnubversion_threadmessage_get_password(msg);
    gboolean may_save = gnubversion_threadmessage_get_may_save(msg);

    g_debug("realm = %s", realm ? realm->str : NULL);
    g_debug("username = %s", username ? username->str : NULL);
    g_debug("password = %s", password ? password->str : NULL);
    g_debug("may_save = %d", may_save);

    GnomePasswordDialog *pd = GNOME_PASSWORD_DIALOG(gnome_password_dialog_new( "GnubVersion Authentication",
                                                        "Subversion requires authentication for this repository",
                                                        (username != NULL) ? username->str : "",
                                                        (password != NULL) ? password->str : "",
                                                        FALSE
                                                        ));

    gnome_password_dialog_set_domain(pd, realm->str);
    gnome_password_dialog_set_readonly_domain(pd, TRUE);
    gnome_password_dialog_set_show_domain(pd, TRUE);

    gnome_password_dialog_set_show_username(pd, TRUE);
    gnome_password_dialog_set_readonly_username(pd, FALSE);
    gnome_password_dialog_set_show_new_password(pd, FALSE);
    gnome_password_dialog_set_show_userpass_buttons(pd, FALSE);

    if (may_save)
    {
        gnome_password_dialog_set_show_remember(pd, TRUE);
        gnome_password_dialog_set_remember(pd, GNOME_PASSWORD_DIALOG_REMEMBER_SESSION);
    }
    else
    {
        gnome_password_dialog_set_show_remember(pd, FALSE);
        gnome_password_dialog_set_remember(pd, GNOME_PASSWORD_DIALOG_REMEMBER_NOTHING);
    }

    if ( gnome_password_dialog_run_and_block(pd) )
    {
        gboolean remember = FALSE;

        switch (gnome_password_dialog_get_remember(pd))
        {
            case GNOME_PASSWORD_DIALOG_REMEMBER_FOREVER:
            case GNOME_PASSWORD_DIALOG_REMEMBER_SESSION:
                remember = TRUE;
                break;
            case GNOME_PASSWORD_DIALOG_REMEMBER_NOTHING:
                remember = FALSE;
                break;
        }

        response = gnubversion_threadmessage_new_username_password_response(
                                    gnome_password_dialog_get_username(pd),
                                    gnome_password_dialog_get_password(pd),
                                    remember);
    }
    else
    {
        /* User pressed Cancel */
        response = gnubversion_threadmessage_new_cancel_response();
    }
    g_async_queue_push(gnubversion_threadmessage_get_response_queue(msg), response);
}

void
gnubversion_gui_auth_username_prompt ( GnubVersionThreadMessage *msg)
{
    assert(msg != NULL);
    assert(GNUBVERSION_IS_THREADMESSAGE(msg));
    assert( gnubversion_threadmessage_get_message_type(msg) == MESSAGE_TYPE_ASK_USERNAME);

    GnubVersionThreadMessage *response;
    GString *realm = gnubversion_threadmessage_get_realm(msg);
    GString *username = gnubversion_threadmessage_get_username(msg);
    gboolean may_save = gnubversion_threadmessage_get_may_save(msg);

    g_debug("realm = %p", realm);
    g_debug("username = %p", username);
    g_debug("may_save = %d", may_save);

    GnomePasswordDialog *pd = GNOME_PASSWORD_DIALOG(gnome_password_dialog_new( "GnubVersion Authentication",
                                                        "Subversion requires authentication for this repository",
                                                        (username != NULL) ? username->str : "",
                                                        "",
                                                        may_save
                                                        ));

    gnome_password_dialog_set_domain(pd, realm->str);
    gnome_password_dialog_set_readonly_domain(pd, TRUE);
    gnome_password_dialog_set_show_domain(pd, TRUE);

    gnome_password_dialog_set_show_username(pd, TRUE);
    gnome_password_dialog_set_show_new_password(pd, FALSE);
    gnome_password_dialog_set_show_userpass_buttons(pd, FALSE);

    gnome_password_dialog_set_show_password(pd, FALSE);

    if (may_save)
    {
        gnome_password_dialog_set_show_remember(pd, TRUE);
        gnome_password_dialog_set_remember(pd, GNOME_PASSWORD_DIALOG_REMEMBER_SESSION);
    }
    else
    {
        gnome_password_dialog_set_show_remember(pd, FALSE);
        gnome_password_dialog_set_remember(pd, GNOME_PASSWORD_DIALOG_REMEMBER_NOTHING);
    }

    if ( gnome_password_dialog_run_and_block(pd) )
    {
        gboolean remember = FALSE;

        switch (gnome_password_dialog_get_remember(pd))
        {
            case GNOME_PASSWORD_DIALOG_REMEMBER_FOREVER:
            case GNOME_PASSWORD_DIALOG_REMEMBER_SESSION:
                remember = TRUE;
                break;
            case GNOME_PASSWORD_DIALOG_REMEMBER_NOTHING:
                remember = FALSE;
                break;
        }

        response = gnubversion_threadmessage_new_username_response(
                                    gnome_password_dialog_get_username(pd),
                                    remember);
    }
    else
    {
        /* User pressed Cancel */
        response = gnubversion_threadmessage_new_cancel_response();
    }
    g_async_queue_push(gnubversion_threadmessage_get_response_queue(msg), response);
}

void
gnubversion_gui_ask_cert ( GnubVersionThreadMessage *msg)
{
    assert(msg != NULL);
    assert(GNUBVERSION_IS_THREADMESSAGE(msg));
    assert( gnubversion_threadmessage_get_message_type(msg) == MESSAGE_TYPE_ASK_CERT);

    gboolean may_save = gnubversion_threadmessage_get_may_save(msg);
    const char *question = gnubversion_threadmessage_get_question(msg);

    g_debug("question = %s", question);
    g_debug("may_save = %d", may_save);

    GtkWidget *dialog, *label;

    dialog = gtk_dialog_new_with_buttons( _("GnubVersion: SSL Certificate Error"),
                                         NULL,
                                         GTK_DIALOG_DESTROY_WITH_PARENT,
                                         GTK_STOCK_CANCEL,
                                         GTK_RESPONSE_CANCEL,
                                         NULL);

    gtk_dialog_add_button( GTK_DIALOG(dialog), _("Accept for now"), RESPONSE_ACCEPT_TEMP);
    if (may_save)
        gtk_dialog_add_button( GTK_DIALOG(dialog), _("Accept Permanently"), RESPONSE_ACCEPT_PERM);

    label = gtk_label_new(question);
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), label);
    gtk_widget_show_all (dialog);

    int response_val;

    gint result = gtk_dialog_run( GTK_DIALOG( dialog));
    g_debug("dialog result = %d", result);
    switch (result)
    {
        case RESPONSE_ACCEPT_TEMP:
            response_val = CERT_RESPONSE_ACCEPT_TEMP;
            g_debug("dialog result: Accept temporarily");
            break;
        case RESPONSE_ACCEPT_PERM:
            response_val = CERT_RESPONSE_ACCEPT_PERM;
            g_debug("dialog result: Accept permanently");
            break;
        default: /* Dialog was cancelled, or closed */
            response_val = CERT_RESPONSE_REJECT;
            g_debug("dialog result: cancel");
            break;
    }

    g_debug("Hiding dialog");
    gtk_widget_hide(dialog);
    g_debug("Destroying dialog");
    gtk_widget_destroy(dialog);
//    g_debug("Unreffing dialog");
//    g_object_unref(G_OBJECT(dialog));

    g_debug("Creating response");
    GnubVersionThreadMessage *response; 
    response = gnubversion_threadmessage_new_cert_response( response_val );

    g_debug("Sending response");
    g_async_queue_push(gnubversion_threadmessage_get_response_queue(msg), response);
    g_debug("done");
}
