#define APP_NAME "gnubversion-auth"

#include <assert.h>
#include <glib/gi18n.h>

#include "gnubversion-auth.h"
#include "gnubversion-init.h"
#include "gnubversion-progresswindow.h"

/* TODO: Get inspiration from subversion/libsvn_subr/prompt.c */

svn_error_t *
gnubversion_auth_simple_prompt ( svn_auth_cred_simple_t **cred_p,
                                void *baton,
                                const char *realm,
                                const char *username,
                                svn_boolean_t may_save,
                                apr_pool_t *pool)
{
    g_debug("gnubversion_auth_simple_prompt");
    if (baton == NULL)
        return svn_error_create(SVN_ERR_CANCELLED, NULL, _("Internal Error: Cannot ask for credentials - no baton"));

    GnubVersionProgressWindow *pw = gnubversion_auth_baton->progressWindow;

    svn_error_t *err = SVN_NO_ERROR;

    GAsyncQueue *queue = g_async_queue_new();
    GnubVersionThreadMessage *msg = gnubversion_threadmessage_new_ask_username_password(queue, realm, username, may_save);
    gnubversion_progresswindow_send_message(pw, msg);

    g_debug("Sent message to gtk thread - awaiting response");
    GnubVersionThreadMessage *response = GNUBVERSION_THREADMESSAGE(g_async_queue_pop(queue));
    g_debug("got response");

    switch (gnubversion_threadmessage_get_message_type(response))
    {
        case MESSAGE_TYPE_USERNAME_PASSWORD_RESPONSE:
            {
                svn_auth_cred_simple_t *ret = apr_pcalloc(pool, sizeof(*ret));
                const char *username = gnubversion_threadmessage_get_username(response)->str;
                const char *password = gnubversion_threadmessage_get_password(response)->str;

                g_debug("Entered username=%s, password=%s", username, password);

                ret->username = svn_stringbuf_create(username, gnubversion_apr_pool)->data;
                ret->password = svn_stringbuf_create(password, gnubversion_apr_pool)->data;
                ret->may_save = gnubversion_threadmessage_get_may_save(response);

                g_debug("may save = %d", ret->may_save);
                *cred_p = ret;
                err = SVN_NO_ERROR;
            }
            break;
            
        case MESSAGE_TYPE_CANCEL_RESPONSE:
            err = svn_error_create(SVN_ERR_CANCELLED, NULL, _("User cancelled password prompt"));
            break;

        default:
            g_error("Unexpected response message type %d", gnubversion_threadmessage_get_message_type(response));
            err = svn_error_create(SVN_ERR_CANCELLED, NULL, _("Internal Error: Unexpected response message type"));
            break;
    }

    g_debug("Unreffing response");
    g_object_unref(G_OBJECT(response));

    g_debug("Unreffing queue");
    g_async_queue_unref(queue);

    g_debug("done");
    return err;
}

svn_error_t *
gnubversion_auth_username_prompt (  svn_auth_cred_username_t **cred_p,
                                    void *baton,
                                    const char *realm,
                                    svn_boolean_t may_save,
                                    apr_pool_t *pool)
{
    g_debug("gnubversion_auth_username_prompt");
    if (baton == NULL)
        return svn_error_create(SVN_ERR_CANCELLED, NULL, _("Cannot ask for credentials - no baton"));

    GnubVersionProgressWindow *pw = gnubversion_auth_baton->progressWindow;

    svn_error_t *err = SVN_NO_ERROR;

    GAsyncQueue *queue = g_async_queue_new();
    GnubVersionThreadMessage *msg = gnubversion_threadmessage_new_ask_username(queue, may_save);
    gnubversion_progresswindow_send_message(pw, msg);

    GnubVersionThreadMessage *response = GNUBVERSION_THREADMESSAGE(g_async_queue_pop(queue));

    switch (gnubversion_threadmessage_get_message_type(response))
    {
        case MESSAGE_TYPE_USERNAME_RESPONSE:
            {
                svn_auth_cred_username_t *ret = apr_pcalloc(pool, sizeof(*ret));
                ret->username = svn_stringbuf_create(gnubversion_threadmessage_get_username(response)->str, gnubversion_apr_pool)->data;
                ret->may_save = gnubversion_threadmessage_get_may_save(response);
                *cred_p = ret;
                err = SVN_NO_ERROR;
            }
            break;
            
        case MESSAGE_TYPE_CANCEL_RESPONSE:
            err = svn_error_create(SVN_ERR_CANCELLED, NULL, _("User cancelled password prompt"));
            break;

        default:
            g_error("Unexpected response message type %d", gnubversion_threadmessage_get_message_type(response));
            err = svn_error_create(SVN_ERR_CANCELLED, NULL, _("Internal Error: Unexpected response message type"));
    }

    g_debug("unreffing response");
    g_object_unref(G_OBJECT(response));

    g_debug("unreffing queue");
    g_async_queue_unref(queue);
    g_debug("done");

    return err;
}

svn_error_t *
gnubversion_auth_ssl_server_trust_prompt(
    svn_auth_cred_ssl_server_trust_t **cred_p,
    void *baton,
    const char *realm,
    apr_uint32_t failures,
    const svn_auth_ssl_server_cert_info_t *cert_info,
    svn_boolean_t may_save,
    apr_pool_t *pool)
{
    svn_error_t *err = SVN_NO_ERROR;
    int choice;
    svn_stringbuf_t *msg;
    GnubVersionProgressWindow *win = gnubversion_auth_baton->progressWindow;
    svn_stringbuf_t *buf = svn_stringbuf_createf (pool, _("Error validating server certificate for '%s':\n"), realm);

    if (failures & SVN_AUTH_SSL_UNKNOWNCA)
    {
        svn_stringbuf_appendcstr
            (buf,
            _(" - The certificate is not issued by a trusted authority. Use the\n"
            "   fingerprint to validate the certificate manually!\n"));
    }

    if (failures & SVN_AUTH_SSL_CNMISMATCH)
    {
        svn_stringbuf_appendcstr (buf, _(" - The certificate hostname does not match.\n"));
    } 

    if (failures & SVN_AUTH_SSL_NOTYETVALID)
    {
        svn_stringbuf_appendcstr (buf, _(" - The certificate is not yet valid.\n"));
    }

    if (failures & SVN_AUTH_SSL_EXPIRED)
    {
        svn_stringbuf_appendcstr (buf, _(" - The certificate has expired.\n"));
    }

    if (failures & SVN_AUTH_SSL_OTHER)
    {
        svn_stringbuf_appendcstr (buf, _(" - The certificate has an unknown error.\n"));
    }

    msg = svn_stringbuf_createf (pool,
                                _("Certificate information:\n"
                                " - Hostname: %s\n"
                                " - Valid: from %s until %s\n"
                                " - Issuer: %s\n"
                                " - Fingerprint: %s\n"),
                                cert_info->hostname,
                                cert_info->valid_from,
                                cert_info->valid_until,
                                cert_info->issuer_dname,
                                cert_info->fingerprint);
    svn_stringbuf_appendstr(buf, msg);
    {
        GAsyncQueue *queue = g_async_queue_new();
        GnubVersionThreadMessage *msg = gnubversion_threadmessage_new_ask_cert(queue, buf->data, may_save);
        GnubVersionThreadMessage *response;

        gnubversion_progresswindow_send_message(win, msg);
        response = GNUBVERSION_THREADMESSAGE(g_async_queue_pop(queue));

        assert(gnubversion_threadmessage_get_message_type(response) == MESSAGE_TYPE_CERT_RESPONSE);
        choice = gnubversion_threadmessage_get_cert_response(response);

        g_async_queue_unref(queue);
        g_object_unref(G_OBJECT(response));
    }

    if (choice == CERT_RESPONSE_ACCEPT_TEMP )
    {
        g_debug("auth response was: accept temp");
        *cred_p = apr_pcalloc(pool, sizeof(**cred_p));
        (*cred_p)->may_save = FALSE;
        (*cred_p)->accepted_failures = failures;
    }
    else if (may_save && ( choice == CERT_RESPONSE_ACCEPT_PERM ) )
    {
        g_debug("auth response was: accept perm");
        *cred_p = apr_pcalloc(pool, sizeof(**cred_p));
        (*cred_p)->may_save = TRUE;
        (*cred_p)->accepted_failures = failures;
    }
    else
    {
        g_debug("auth response was: cancel");
        *cred_p = NULL;
        err = svn_error_create(SVN_ERR_CANCELLED, NULL, _("Cancelled by user"));
    }

    return err;
}
