/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Vivien Chappelier

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*************************** reconstruct reference frame *********************/

/*  reconstruct                                                            */
/*                                                                           */
/*  Description:                                                             */
/*    Reconstruct a block in reference plane.                              */
/*                                                                           */
/*  Arguments:                                                               */
/*    unsigned char *plane: the reference plane                              */
/*    short width: width of the plane                                        */
/*    short height: height of the plane                                      */
/*    dct_t *block: the block to reconstruct                                 */
/*    short x: x position of the block in pixel units.                       */
/*    short y: y position of the block in pixel units.                       */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void inline reconstruct(unsigned char *plane,
			       dct_t *block,
			       int pitch)
{
  int i, j;
  int v;

  for(i = 0; i < 8; i++)
  {
    for(j = 0; j < 8; j++)
    {
      v = 0;
      if(block[(i<<3)+j] > 0) v = (int) (block[(i<<3)+j] + 0.5);
      if(v > 255) v = 255;
      plane[j] = (unsigned char) v;
    }
    plane += pitch;
  }
}


/*  sum                                                                      */
/*                                                                           */
/*  Description:                                                             */
/*    Sums block data with reference to get new reference block              */
/*                                                                           */
/*  Arguments:                                                               */
/*    unsigned char *plane: the new reference plane                          */
/*    unsigned char *ref: the old reference plane                            */
/*    unsigned int *sum: indicates if the block is all zero                  */
/*    dct_t *block: the error data                                           */
/*    int pitch: number of pixels to get to the next line.                   */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void inline sum(unsigned char *plane,
		       unsigned char *ref,
		       unsigned int *sum,
		       dct_t *block,
		       int pitch)
{
  int i, j;
  int v;

  /* the only thing that matters is wether the sum */
  /* is zero or not, so we sum unsigned values instead */
  /* of absolute difference */
  *sum = 0;

  for(i = 0; i < 8; i++) {
    for(j = 0; j < 8; j++) {
      v = 0;
      if(block[(i<<3)+j] > 0) v = (int) (block[(i<<3)+j] + 0.5);
      if(block[(i<<3)+j] < 0) v = (int) (block[(i<<3)+j] - 0.5);
      *sum |= (unsigned) v; 
      v += (int) ref[j];
      if(v < 0) v = 0;
      if(v > 255) v = 255;
      plane[j] = v;
    }
    plane += pitch;
    ref += pitch;
  }
}

/*  move                                                                     */
/*                                                                           */
/*  Description:                                                             */
/*    Moves past reference block according to motion. This is the same as    */
/*    sum with all-zero block, but accelerated.                              */
/*                                                                           */
/*  Arguments:                                                               */
/*    unsigned char *plane: the new reference plane                          */
/*    unsigned char *ref: the old reference plane                            */
/*    unsigned int *sum: indicates if the block is all zero                  */
/*    int pitch: number of pixels to get to the next line.                   */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void inline move(unsigned char *plane,
			unsigned char *ref,
			int pitch)
{
  int i;

  for(i = 0; i < 8; i++) {
    memcpy(plane, ref, 8);
    plane += pitch;
    ref += pitch;
  }
}

