/*
** PHREL
** $Id: phrel.c,v 1.35 2006/10/27 03:17:54 sella Exp $
** Copyright (c) 2004-2006 James M. Sella. All Rights Reserved.
** Released under the GPL Version 2 License.
** http://www.digitalgenesis.com
**
** PHREL - Per Host RatE Limiter
*/

static const char rcsid[] = "$Id: phrel.c,v 1.35 2006/10/27 03:17:54 sella Exp $";

#include "phrel.h"

#include "trap.h"
#include "prefs.h"
#include "thread.h"
#include "capture.h"
#include "data_hash.h"

#include <snmptrap.h>
#include <configuration.h>

#include <pwd.h>
#include <grp.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>
#include <signal.h>
#include <pthread.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/resource.h>

#include <errno.h>
#include <assert.h>

//#define HASH_TEST

/* Global variables */
struct data_t data;
struct args_t args;
int running = 1, restart = 0;
pid_t pid = 0;
time_t startup_time = 0;
pthread_mutex_t data_mutex;

int main(int argc, char **argv) {
	int ret_val;
	char stats[1024];
	struct stat stat_buf;
	pthread_t data_tid = 0;
	struct config_node *node = NULL;
	struct passwd *ent = NULL;
	struct group *gent = NULL;

	memset(&args, 0, sizeof(struct args_t));

#ifdef HASH_TEST
	return hash_test();
#endif

	if ((ret_val = get_args(argc, argv)) != EXIT_SUCCESS) {
		goto exit;
	}

	/* Reopen the log based on args. */
#ifdef HAVE_NET_SNMP
	snmp_enable_syslog();
#endif
	openlog(PROGRAM_NAME, (args.debug) ? LOG_PID | LOG_PERROR : LOG_PID, DEF_SYSLOG_FACILITY);

	if (args.config) {
		read_conf_file(args.config, &node);

		if (args.debug >= DEBUG_INTERNAL) {
			print_config_nodes(node, 0);
		}

		process_config(node);
		free_config_nodes(node);
	}

	if (args.debug >= DEBUG_INTERNAL) {
		display_args();
	}

   if (args.threshold == NULL) {
      fprintf(stderr, "You must specify at least one threshold.\n");
		ret_val = EXIT_FAILURE;

      goto exit;
   }

	if (geteuid() != 0) { /* libpcap needs root to capture packets */
		fprintf(stderr, "Root privileges are required to run this program.\n");
		ret_val = EXIT_FAILURE;

		goto exit;
	}

	if (stat(args.iptables, &stat_buf) == -1) {
		fprintf(stderr, "iptables binary '%s' is not accessable/found.\n", args.iptables);
		ret_val = EXIT_FAILURE;

		goto exit;
	} else if (S_ISREG(stat_buf.st_mode) == 0) {
		fprintf(stderr, "iptables binary '%s' is not a regular file.\n", args.iptables);
		ret_val = EXIT_FAILURE;

		goto exit;
	} else if ((stat_buf.st_mode & (S_IXUSR | S_IXGRP | S_IXOTH)) == 0) {
		fprintf(stderr, "iptables binary '%s' is not executable.\n", args.iptables);
		ret_val = EXIT_FAILURE;

		goto exit;
	}

	/* Install signal handlers. */
	if (signal(SIGINT, &sighandler) == SIG_ERR) {
		fprintf(stderr, "Failed to install signal handler for SIGINT.\n");
	}
	if (signal(SIGTERM, &sighandler) == SIG_ERR) {
		fprintf(stderr, "Failed to install signal handler for SIGTERM.\n");
	}
	if (signal(SIGHUP, &sighandler) == SIG_ERR) {
		fprintf(stderr, "Failed to install signal handler for SIGTERM.\n");
	}
	if (signal(SIGUSR1, &sighandler) == SIG_ERR) {
		fprintf(stderr, "Failed to install signal handler for SIGUSR1.\n");
	}
	if (signal(SIGUSR2, &sighandler) == SIG_ERR) {
		fprintf(stderr, "Failed to install signal handler for SIGUSR2.\n");
	}

	/* Set our niceness */
	setpriority(PRIO_PROCESS, 0, args.nice);

	/* Read in the group, if needed. */
	if (args.group) {
		fprintf(stderr, "Support for setting group not implemented correctly yet.\n");

		if (!(gent = getgrnam(args.group))) {
			fprintf(stderr, "Group '%s' was not found.\n", args.group);
		}
	}

	/* Read in the user, if needed. */
	if (args.user) {
		fprintf(stderr, "Support for setting user is not implemented correctly yet.\n");

		if (!(ent = getpwnam(args.user))) {
			fprintf(stderr, "User '%s' was not found.\n", args.user);
		}
	}

	/* Lock ourselves into a chroot jail, if requested. */
	if (args.chroot) {
		if (chroot(args.chroot) == -1) {
			fprintf(stderr, "Failed to chroot to '%s'\n", args.chroot);
		}
	}

	/* Set our new group. */
	if (gent && setegid(gent->gr_gid)) {
		fprintf(stderr, "Failed to set GID to '%d'.\n", gent->gr_gid);
	}

	/* Set our new user. */
	if (ent && seteuid(ent->pw_uid)) {
		fprintf(stderr, "Failed to set GID to '%d'.\n", ent->pw_uid);
	}

	pthread_mutex_init(&data_mutex, NULL);

	if (args.debug == 0) {
		if ((pid = fork()) != 0) {
			return EXIT_SUCCESS;
		} else if (pid == -1) {
			fprintf(stderr, "failed to fork");
			ret_val = EXIT_FAILURE;

			goto exit;
		} else if (pid == 0) {
			/* Only a forked child gets to this block. */
			fclose(stdin); /* Will no longer accept input (ie: Ctrl-C) */
			fclose(stdout); /* Our output to stdout will go unheard. */
			fclose(stderr); /* Our output to stderr will go unheard. */
		}
	}

	syslog(LOG_INFO, "%s v%s", PROGRAM_NAME, PROGRAM_VERS);
	syslog(LOG_INFO, "%s", pcap_lib_version());

	/* Capture PID and startup time */
	pid = getpid();
	startup_time = time(NULL);

	/* Send startup trap */
	snmptrap_process_startup();

	while (running) {
		if (args.debug >= DEBUG_INTERNAL) {
			syslog(LOG_INFO, "staring up threads...");
		}

		/* Initialize our hast */
		hash_init();

		pthread_create(&data_tid, NULL, (void* (*)(void*))data_thread, (void*) NULL);

		if (args.debug >= DEBUG_INTERNAL) {
			syslog(LOG_INFO, "threads are created");
		}

		/* Capture packets. */
		syslog(LOG_INFO, "starting to monitor traffic");
		if (capture() != 0) {
			running = 0;
		}
		syslog(LOG_INFO, "stopped monitoring traffic");

		if (args.debug >= DEBUG_INTERNAL) {
			syslog(LOG_INFO, "joining thread");
		}

		pthread_join(data_tid, NULL);

		/* Stop filtering. */
		cleanup();

		if (restart == 1) { /* HUP */
			/* Send restart trap */
			snmptrap_process_restart();

			running = 1;
			restart = 0;

			/* Free and re-init hash */
			pthread_mutex_lock(&data_mutex);
			hash_stats(stats, sizeof(stats));
			hash_free();
			hash_init();
			pthread_mutex_unlock(&data_mutex);
			syslog(LOG_INFO, stats);

         syslog(LOG_INFO, "cleared hash and iptables entries");
		} else { /* Shutdown. */
			/* Send shutdown trap */
			snmptrap_process_shutdown();

			/* Free hash */
			pthread_mutex_lock(&data_mutex);
			hash_stats(stats, sizeof(stats));
			hash_free();
			pthread_mutex_unlock(&data_mutex);
			syslog(LOG_INFO, stats);
		}
	}

exit:
	free_args();

	pthread_mutex_destroy(&data_mutex);

	closelog();

	return (ret_val == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}

void sighandler(int sig) {
	char stats[1024];

   switch (sig) {
      case SIGINT:
      case SIGTERM:
        	syslog(LOG_INFO, "SIGTERM/SIGINT - exiting");

			running = 0;
			restart = 0;

         break;

      case SIGHUP:
        	syslog(LOG_INFO, "SIGHUP - restarting");

			running = 0;
			restart = 1;

         break;

      case SIGUSR1:
			hash_stats(stats, sizeof(stats));
			syslog(LOG_INFO, stats);

			if (args.debug >= DEBUG_MAXVAL) {
				args.debug = DEBUG_MAXVAL;
			} else {
       	  args.debug++;
			}
         syslog(LOG_INFO, "debug level now set to '%d'", args.debug);
         break;

      case SIGUSR2:
			if (args.debug <= 0) {
				args.debug = 0;
			} else {
       	  args.debug--;
			}
         syslog(LOG_INFO, "debug level now set to '%d'", args.debug);
         break;

      default:
			if (args.debug >= DEBUG_INTERNAL) {
         	syslog(LOG_INFO, "caught unused signal '%d'", sig);
			}
   }
}

int hash_test() {
	struct in_addr ip;
	struct data_t data;
	int s, save = 0;

	hash_init();

	for (s = 0; s < 500; s++) {
		ip.s_addr = s;
		data.ip = ip;

		hash_add(&ip, &data);
		if (hash_find(&ip) == (int) NULL) {
			abort();
		}

		if (s % 3 == 0) {
			ip.s_addr = save;
			data.ip = ip;
			save = s;
			
			if (hash_del(&ip) == 0) {
				printf("was able to Deleted - s: %u\n", save);
			} else {
				printf("failed to Deleted - s: %u\n", save);
			}
		}
	}

	hash_stats_print();

	printf("starting delete\n");
	hash_free();

	return EXIT_SUCCESS;
}

/*
** Local Variables:
** c-basic-offset: 3
** tab-width: 3
** End:
** vim: noet ts=3 sw=3
*/
