/*
 * gdchart.c
 *
 * Ruby extension module for the Bruce Verderaime's GDCHART charting library.
 *
 * Copyright (C) 2000  Arjen Laarhoven <arjen@aragorn.demon.nl>
 * Copyright (C) 2002  Takehiro Yonekura <ceramic@heart.104.net>
 *
 * $Id: gdchart.c,v 1.11 2002/09/12 08:19:43 c-heart Exp $
 */

#include <stdio.h>
#include <string.h>

#include "gdc.h"
#include "gdchart.h"
#include "gdcpie.h"

#include "ruby.h"
#include "rubyio.h"

#define TRUE  1
#define FALSE 0


VALUE cGDChart;
VALUE cGDChartPie;
VALUE cGDChartAnnotation;
VALUE cGDChartScatter;


/*
 * These structures are used by Ruby to do GC.
 * Pointes indicate dynamically allocated memory in class `GDChart' and
 * 'GDChartPie'.
 */
struct gdc_alloc_t {
  GDC_ANNOTATION_T *annotation;
  GDC_SCATTER_T *scatter;
  unsigned long *ExtVolColor;
  unsigned long *SetColor;
  unsigned long *ExtColor;
};

struct gdc_pie_alloc_t {
  int *explode;
  unsigned long *Color;
  unsigned char *missing;
};


/*
 * If pointers indicate allocated memory (pointers not indicate NULL),
 * free it.
 */
#define Free_alloc_t(alloc_t, var) \
if (alloc_t->var != NULL) free(alloc_t->var)

static void free_gdc_alloc_t(struct gdc_alloc_t *t)
{
  Free_alloc_t(t, annotation);
  Free_alloc_t(t, scatter);
  Free_alloc_t(t, ExtVolColor);
  Free_alloc_t(t, SetColor);
  Free_alloc_t(t, ExtColor);
  free(t);
}

static void free_gdc_pie_alloc_t(struct gdc_pie_alloc_t *t)
{
  Free_alloc_t(t, explode);
  Free_alloc_t(t, Color);
  Free_alloc_t(t, missing);
  free(t);
}


/*
 * GDC_ANNOTATION_T structure class definition
 */
static VALUE gdc_annotation_new(VALUE klass)
{
  VALUE obj;

  obj = rb_obj_alloc(klass);
  return obj;
}

static VALUE gdc_annotation_init(VALUE obj)
{
  rb_iv_set(obj, "@point", rb_float_new(0.0));
  rb_iv_set(obj, "@color", INT2FIX(0));
  rb_iv_set(obj, "@note", rb_str_new2(""));
  return obj;
}


/*
 * GDC_SCATTER_T structure class definition
 */
static VALUE gdc_scatter_new(VALUE klass)
{
  VALUE obj;

  obj = rb_obj_alloc(klass);
  return obj;
}

static VALUE gdc_scatter_init(VALUE obj)
{
  rb_iv_set(obj, "@point", rb_float_new(0.0));
  rb_iv_set(obj, "@val", rb_float_new(0.0));
  rb_iv_set(obj, "@width", INT2NUM(0));
  rb_iv_set(obj, "@color", INT2NUM(0));
  rb_iv_set(obj, "@ind", rb_const_get(obj, rb_intern("GDC_SCATTER_TRIANGLE_UP")));
  return obj;
}



/*
 *
 *	Data conversion functions
 *
 */

/*
 * Convert cGDChartAnnotation object to GDC_ANNOTATION_T structure.
 * If argument type is illegal, returns 'FALSE'.
 */
static int value_to_annotation_t(VALUE value, GDC_ANNOTATION_T *an)
{
  VALUE point, color, note;
  char *s;
  int s_len;

  if (rb_obj_is_instance_of(value, cGDChartAnnotation) == Qfalse)
    return FALSE;

  point = rb_iv_get(value, "@point");
  color = rb_iv_get(value, "@color");
  note  = rb_iv_get(value, "@note");

  an->point = (float)NUM2DBL(point);
  an->color = NUM2ULONG(color);
  Check_SafeStr(note);
  s = rb_str2cstr(note, &s_len);
  if (s_len > MAX_NOTE_LEN)
    rb_raise(rb_eArgError, "annotation note is too long");
  strncpy(an->note, s, s_len+1);

  return TRUE;
}


/*
 * Convert cGDChartScatter object to GDC_SCATTER_T structure.
 * If argument type is illegal, returns 'FALSE'.
 */
static int value_to_scatter_t(VALUE value, GDC_SCATTER_T *sc)
{
  VALUE point, val, width, color, ind;

  if (rb_obj_is_instance_of(value, cGDChartScatter) == Qfalse)
    return FALSE;
  
  point = rb_iv_get(value, "@point");
  val   = rb_iv_get(value, "@val");
  width = rb_iv_get(value, "@width");
  color = rb_iv_get(value, "@color");
  ind   = rb_iv_get(value, "@ind");
  
  sc->point = (float)NUM2DBL(point);
  sc->val   = (float)NUM2DBL(val);
  sc->width = (unsigned short)NUM2UINT(width);
  sc->color = NUM2ULONG(color);
  sc->ind   = (GDC_SCATTER_IND_T)NUM2INT(ind);
  
  return TRUE;
}


/*
 * C parameter types
 */
enum gdc_param_type {
  BOOL,
  BOOL_ARRAY,
  CHAR,
  UCHAR,
  UCHAR_ARRAY,
  INT,
  INT_ARRAY,
  SHORT,
  USHORT,
  ULONG,
  ULONG_ARRAY,
  FLOAT,
  DOUBLE,
  STRING,
  FONT_SIZE,
  STACK_T,
  HLC_STYLE,
  ANNOTATION_T,
  SCATTER_T_ARRAY,
  PCT_TYPE
};

/*
 * Convert Ruby/GDChart parameters to C value.
 * If instance variable is set, corresponding 'var' is initialized by the value
 * and returns TRUE.  If instance variable is not set, just returns FALSE.
 */
static int set_param(VALUE obj, const char *name, const void *var,
		     const enum gdc_param_type type)
{
  VALUE value, entry;
  void *result;
  int i, length;

  value = rb_iv_get(obj, name);
  if (NIL_P(value))
    return FALSE;

  switch (type) {
  case BOOL:
    if (TYPE(value) != T_TRUE && TYPE(value) != T_FALSE)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Bool");
  
    *(char*)var = (char)value;
    break;

  case BOOL_ARRAY:
    if (TYPE(value) != T_ARRAY)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Bool Array");

    length = RARRAY(value)->len;
    (char*)result = ALLOC_N(char, length);

    for (i = 0; i < length; i++) {
      entry = rb_ary_entry(value, i);
      if (TYPE(entry) != T_TRUE && TYPE(entry) != T_FALSE)
	rb_raise(rb_eTypeError, "%s expect %s", name, "Bool Array");
      *((char*)result + i) = (char)entry;
    }
    *(char**)var = (char*)result;
    break;

  case CHAR:
    if (TYPE(value) != T_FIXNUM)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum");

    *(char*)var = NUM2CHR(value);
    break;

  case UCHAR:
    if (TYPE(value) != T_FIXNUM)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum");

    *(unsigned char*)var = (unsigned char)NUM2CHR(value);
    break;

  case INT:
    if (TYPE(value) != T_FIXNUM)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum");

    *(int*)var = FIX2INT(value);
    break;

  case INT_ARRAY:
    if (TYPE(value) != T_ARRAY)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum Array");
    
    length = RARRAY(value)->len;
    (int*)result = ALLOC_N(int, length);

    for (i = 0; i < length; i++) {
      entry = rb_ary_entry(value, i);
      if (TYPE(entry) != T_FIXNUM)
	rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum Array");
      *((int*)result + i) = FIX2INT(entry);
    }
    *(int**)var = (int*)result;
    break;

  case SHORT:
    if (TYPE(value) != T_FIXNUM)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum");

    *(short*)var = (short)FIX2INT(value);
    break;

  case USHORT:
    if (TYPE(value) != T_FIXNUM)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum");

    *(unsigned short*)var = (unsigned short)FIX2UINT(value);
    break;

  case ULONG:
    if (TYPE(value) != T_FIXNUM)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum");

    *(unsigned long*)var = FIX2ULONG(value);
    break;

  case ULONG_ARRAY:
    if (TYPE(value) != T_ARRAY)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum Array");

    length = RARRAY(value)->len;
    (unsigned long*)result = ALLOC_N(unsigned long, length);

    for (i = 0; i < length; i++) {
      entry = rb_ary_entry(value, i);
      if (TYPE(entry) != T_FIXNUM)
	rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum Array");
      *((unsigned long*)result + i) = FIX2ULONG(entry);
    }
    *(unsigned long**)var = (unsigned long*)result;

    break;

  case FLOAT:
    if (TYPE(value) != T_FIXNUM && TYPE(value) != T_FLOAT)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum or Float");

    *(float*)var = (float)NUM2DBL(value);
    break;

  case DOUBLE:
    if (TYPE(value) != T_FIXNUM && TYPE(value) != T_FLOAT)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum or Float");

    *(double*)var = NUM2DBL(value);
    break;

  case STRING:
    if (TYPE(value) != T_STRING)
      rb_raise(rb_eTypeError, "%s expect %s", name, "String"); 
    Check_SafeStr(value);

    *(char**)var = STR2CSTR(value);
    break;

  case FONT_SIZE:
    /* XXX: need more exact type check */
    if (TYPE(value) != T_FIXNUM)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum"); 

    *(enum GDC_font_size *)var = (enum GDC_font_size)FIX2INT(value);
    break;

  case STACK_T:
    /* XXX: need more exact type check */
    if (TYPE(value) != T_FIXNUM)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum"); 

    *(GDC_STACK_T*)var = (GDC_STACK_T)FIX2INT(value);
    break;

  case HLC_STYLE:
    /* XXX: need more exact type check */
    if (TYPE(value) != T_FIXNUM)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum"); 

    *(GDC_HLC_STYLE_T*)var = (GDC_HLC_STYLE_T)FIX2INT(value);
    break;

  case ANNOTATION_T:
    /* Because only 1 annotatin is allowd. */
    (GDC_ANNOTATION_T*)result = ALLOC(GDC_ANNOTATION_T);
    
    if (value_to_annotation_t(value, (GDC_ANNOTATION_T*)result) == FALSE) {
      free(result);
      rb_raise(rb_eTypeError, "%s expect %s", name, rb_class2name(cGDChartAnnotation));
    }

    *(GDC_ANNOTATION_T**)var = (GDC_ANNOTATION_T*)result;
    break;

  case SCATTER_T_ARRAY:
    if (TYPE(value) != T_ARRAY)
      rb_raise(rb_eTypeError, "%s expect %s Array", name, rb_class2name(cGDChartScatter));

    length = RARRAY(value)->len;
    (GDC_SCATTER_T*)result = ALLOC_N(GDC_SCATTER_T, length);

    for (i = 0; i < length; i++) {
      entry = rb_ary_entry(value, i);
      if (value_to_scatter_t(entry, (GDC_SCATTER_T*)(result+i)) == FALSE) {
	free(result);
	rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum Array");
      }
    }

    *(GDC_SCATTER_T**)var = (GDC_SCATTER_T*)result;
    break;

  case PCT_TYPE:
    /* XXX: need more exact type check */
    if (TYPE(value) != T_FIXNUM)
      rb_raise(rb_eTypeError, "%s expect %s", name, "Fixnum"); 

    *(GDCPIE_PCT_TYPE*)var = (GDCPIE_PCT_TYPE)FIX2INT(value);
    break;

  default:
    rb_raise(rb_eTypeError, "type = %d", type);
    break;
  }

  return TRUE;
}



/*
 * GDChart class definition
 */
VALUE gdc_new(VALUE klass)
{
  struct gdc_alloc_t *gdc_t;
  VALUE obj;

  obj = Data_Make_Struct(klass, struct gdc_alloc_t, 0, free_gdc_alloc_t, gdc_t);
  gdc_t->annotation  = NULL;
  gdc_t->scatter     = NULL;
  gdc_t->ExtVolColor = NULL;
  gdc_t->SetColor    = NULL;
  gdc_t->ExtColor    = NULL;

  rb_obj_call_init(obj, 0, NULL);

  return obj;
}

VALUE gdc_initialize(VALUE obj)
{
  return obj;
}


/*
 * Set GDChart option variable 
 */
static VALUE gdc_set_params(VALUE obj)
{
  struct gdc_alloc_t *gdc_t;

  set_param(obj, "@ytitle", &GDC_ytitle, STRING);
  set_param(obj, "@xtitle", &GDC_xtitle, STRING);
  set_param(obj, "@ytitle2", &GDC_ytitle2, STRING);
  set_param(obj, "@title", &GDC_title, STRING);
  set_param(obj, "@title_size", &GDC_title_size, FONT_SIZE);
  set_param(obj, "@ytitle_size", &GDC_ytitle_size, FONT_SIZE);
  set_param(obj, "@xtitle_size", &GDC_xtitle_size, FONT_SIZE);
  set_param(obj, "@yaxisfont_size", &GDC_yaxisfont_size, FONT_SIZE);
  set_param(obj, "@xaxisfont_size", &GDC_xaxisfont_size, FONT_SIZE);
  set_param(obj, "@xaxis_angle", &GDC_xaxis_angle, DOUBLE);
#ifdef HAVE_LIBFREETYPE
  set_param(obj, "@title_font", &GDC_title_font, STRING);
  set_param(obj, "@ytitle_font", &GDC_ytitle_font, STRING);
  set_param(obj, "@xtitle_font", &GDC_xtitle_font, STRING);
  set_param(obj, "@xaxis_font", &GDC_xaxis_font, STRING);
  set_param(obj, "@yaxis_font", &GDC_yaxis_font, STRING);
  set_param(obj, "@title_ptsize", &GDC_title_ptsize, DOUBLE);
  set_param(obj, "@ytitle_ptsize", &GDC_ytitle_ptsize, DOUBLE);
  set_param(obj, "@xtitle_ptsize", &GDC_xtitle_ptsize, DOUBLE);
  set_param(obj, "@xaxis_ptsize", &GDC_xaxis_ptsize, DOUBLE);
  set_param(obj, "@yaxis_ptsize", &GDC_yaxis_ptsize, DOUBLE);
#endif
  set_param(obj, "@ylabel_fmt", &GDC_ylabel_fmt, STRING);
  set_param(obj, "@ylabel2_fmt", &GDC_ylabel2_fmt, STRING);
  set_param(obj, "@xlabel_ctl", &GDC_xlabel_ctl, BOOL_ARRAY);
  set_param(obj, "@xlabel_spacing", &GDC_xlabel_spacing, SHORT);
  set_param(obj, "@ylabel_density", &GDC_ylabel_density, CHAR);
  set_param(obj, "@interpolations", &GDC_interpolations, BOOL);
  set_param(obj, "@requested_ymin", &GDC_requested_ymin, FLOAT);
  set_param(obj, "@requested_ymax", &GDC_requested_ymax, FLOAT);
  set_param(obj, "@requested_yinterval", &GDC_requested_yinterval, FLOAT);
  set_param(obj, "@Shelf0", &GDC_0Shelf, BOOL);
  set_param(obj, "@grid", &GDC_grid, BOOL);
  set_param(obj, "@ticks", &GDC_ticks, INT);
  set_param(obj, "@xaxis", &GDC_xaxis, BOOL);
  set_param(obj, "@yaxis", &GDC_yaxis, BOOL);
  set_param(obj, "@yaxis2", &GDC_yaxis2, BOOL);
  set_param(obj, "@yval_style", &GDC_yval_style, BOOL);
  set_param(obj, "@stack_type", &GDC_stack_type, STACK_T);
  set_param(obj, "@depth_3d", &GDC_3d_depth, FLOAT);
  set_param(obj, "@angle_3d", &GDC_3d_angle, UCHAR);
  set_param(obj, "@bar_width", &GDC_bar_width, UCHAR);
  set_param(obj, "@HLC_style", &GDC_HLC_style, HLC_STYLE);
  set_param(obj, "@HLC_cap_width", &GDC_HLC_cap_width, UCHAR);
  set_param(obj, "@annotation", &GDC_annotation, ANNOTATION_T);
  set_param(obj, "@annotation_font_size", &GDC_annotation_font_size, FONT_SIZE);
#ifdef HAVE_LIBFREETYPE
  set_param(obj, "@annotation_font", &GDC_annotation_font, FONT_SIZE);
  set_param(obj, "@annotation_ptsize", &GDC_annotation_ptsize, DOUBLE);
#endif
  set_param(obj, "@num_scatter_pts", &GDC_num_scatter_pts, INT);
  set_param(obj, "@scatter", &GDC_scatter, SCATTER_T_ARRAY);
  set_param(obj, "@thumbnail", &GDC_thumbnail, BOOL);
  set_param(obj, "@thumblabel", &GDC_thumblabel, STRING);
  set_param(obj, "@thumbval", &GDC_thumbval, FLOAT);
  set_param(obj, "@border", &GDC_border, BOOL);
  set_param(obj, "@BGColor", &GDC_BGColor, ULONG);
  set_param(obj, "@GridColor", &GDC_GridColor, ULONG);
  set_param(obj, "@LineColor", &GDC_LineColor, ULONG);
  set_param(obj, "@PlotColor", &GDC_PlotColor, ULONG);
  set_param(obj, "@VolColor", &GDC_VolColor, ULONG);
  set_param(obj, "@TitleColor", &GDC_TitleColor, ULONG);
  set_param(obj, "@XTitleColor", &GDC_XTitleColor, ULONG);
  set_param(obj, "@YTitleColor", &GDC_YTitleColor, ULONG);
  set_param(obj, "@YTitle2Color", &GDC_YTitle2Color, ULONG);
  set_param(obj, "@XLabelColor", &GDC_XLabelColor, ULONG);
  set_param(obj, "@YLabelColor", &GDC_YLabelColor, ULONG);
  set_param(obj, "@YLabel2Color", &GDC_YLabel2Color, ULONG);
  set_param(obj, "@ExtVolColor", &GDC_ExtVolColor, ULONG_ARRAY);
  set_param(obj, "@SetColor", &GDC_SetColor, ULONG_ARRAY);
  set_param(obj, "@ExtColor", &GDC_ExtColor, ULONG_ARRAY);
  set_param(obj, "@transparent_bg", &GDC_transparent_bg, CHAR);
  set_param(obj, "@BGImage", &GDC_BGImage, STRING);

  set_param(obj, "@hard_size", &GDC_hard_size, BOOL);
  set_param(obj, "@hard_xorig", &GDC_hard_xorig, INT);
  set_param(obj, "@hard_graphwidth", &GDC_hard_graphwidth, INT);
  set_param(obj, "@hard_yorig", &GDC_hard_yorig, INT);
  set_param(obj, "@hard_grapheight", &GDC_hard_grapheight, INT);

  set_param(obj, "@image_type", &GDC_image_type, INT);


  Data_Get_Struct(obj, struct gdc_alloc_t, gdc_t);

  /*
   * Before set new data, free allocated old data if it exist.
   */
  Free_alloc_t(gdc_t, annotation);
  Free_alloc_t(gdc_t, scatter);
  Free_alloc_t(gdc_t, ExtVolColor);
  Free_alloc_t(gdc_t, SetColor);
  Free_alloc_t(gdc_t, ExtColor);

  /*
   * Copy pointers which indicate dynamically allocated structure.
   * It is used to do GC by Ruby.
   *
   * ANNOTATION_T, SCATTER_T, ULONG_ARRAY parameters allocate memory
   * dynamically.
   */
  gdc_t->annotation  = GDC_annotation;
  gdc_t->scatter     = GDC_scatter;
  gdc_t->ExtVolColor = GDC_ExtVolColor;
  gdc_t->SetColor    = GDC_SetColor;
  gdc_t->ExtColor    = GDC_ExtColor;

  return obj;
}


/*
 * GDChartPie class definition
 */
VALUE gdc_pie_new(VALUE klass)
{
  struct gdc_pie_alloc_t *gdc_pie_t;
  VALUE obj;

  obj = Data_Make_Struct(klass, struct gdc_pie_alloc_t, 0, free_gdc_pie_alloc_t, gdc_pie_t);
  gdc_pie_t->explode = NULL;
  gdc_pie_t->Color   = NULL;
  gdc_pie_t->missing = NULL;

  rb_obj_call_init(obj, 0, NULL);

  return obj;
}

VALUE gdc_pie_initialize(VALUE obj)
{
  return obj;
}

/*
 * Set GDChart option variable 
 */
static VALUE gdc_pie_set_params(VALUE obj)
{
  struct gdc_pie_alloc_t *gdc_t;

  set_param(obj, "@BGColor", &GDCPIE_BGColor, ULONG);
  set_param(obj, "@PlotColor", &GDCPIE_PlotColor, ULONG);
  set_param(obj, "@LineColor", &GDCPIE_LineColor, ULONG);
  set_param(obj, "@EdgeColor", &GDCPIE_EdgeColor, ULONG);
  set_param(obj, "@other_threshold", &GDCPIE_other_threshold, CHAR);
  set_param(obj, "@angle_3d", &GDCPIE_3d_angle, USHORT);
  set_param(obj, "@depth_3d", &GDCPIE_3d_depth, USHORT);
  set_param(obj, "@perspective", &GDCPIE_perspective, USHORT);
  set_param(obj, "@title", &GDCPIE_title, STRING);
  set_param(obj, "@title_size", &GDCPIE_title_size, FONT_SIZE);
  set_param(obj, "@label_size", &GDCPIE_label_size, FONT_SIZE);
#ifdef HAVE_LIBFREETYPE
  set_param(obj, "@title_font", &GDCPIE_title_font, STRING);
  set_param(obj, "@label_font", &GDCPIE_label_font, STRING);
  set_param(obj, "@title_ptsize", &GDCPIE_title_ptsize, DOUBLE);
  set_param(obj, "@label_ptsize", &GDCPIE_label_ptsize, DOUBLE);
#endif
  set_param(obj, "@label_dist", &GDCPIE_label_dist, INT);
  set_param(obj, "@label_line", &GDCPIE_label_line, BOOL);
  set_param(obj, "@explode", &GDCPIE_explode, INT_ARRAY);
  set_param(obj, "@Color", &GDCPIE_Color, ULONG_ARRAY);
  set_param(obj, "@missing", &GDCPIE_missing, BOOL_ARRAY);
  set_param(obj, "@percent_labels", &GDCPIE_percent_labels, PCT_TYPE);
  set_param(obj, "@percent_fmt", &GDCPIE_percent_fmt, STRING);

  set_param(obj, "@image_type", &GDC_image_type, INT);

  Data_Get_Struct(obj, struct gdc_pie_alloc_t, gdc_t);

  /*
   * Before set new data, free allocated old data if it exist.
   */
  Free_alloc_t(gdc_t, explode);
  Free_alloc_t(gdc_t, Color);
  Free_alloc_t(gdc_t, missing);

  /*
   * Copy pointers which indicate dynamically allocated structure.
   * It is used to do GC by Ruby.
   *
   * INT_ARRAY, ULONG_ARRAY, UCHAR_ARRAY parameters allocate memory
   * dynamically.
   */
  gdc_t->explode = GDCPIE_explode;
  gdc_t->Color   = GDCPIE_Color;
  gdc_t->missing = GDCPIE_missing;

  return obj;
}



/*
 * output graph image
 */
static VALUE gdc_out_graph(int argc, VALUE *argv, VALUE self)
{
    VALUE imgwidth, imgheight;	/* output image size */
    VALUE out;			/* output file descriptor */
    VALUE type;			/* chart type */
    VALUE num_points;		/* x axis points */
    VALUE labels;
    VALUE num_sets;		/* number of data sets */
    VALUE data;
    VALUE combodata;
    
    OpenFile *fptr;
    FILE *f;

    int i;
    int label_num;
    int all_points;
    char **c_labels;
    float *fdata, *fcombodata;

    rb_scan_args(argc, argv, "81", &imgwidth, &imgheight,
		 &out, &type, &num_points, &labels, &num_sets,
		 &data, &combodata);

    /* Do type check */
    Check_Type(imgwidth, T_FIXNUM);
    Check_Type(imgheight, T_FIXNUM);
    Check_Type(out, T_FILE);
    Check_Type(type, T_FIXNUM);	/* XXX: 'type' is constant? */
    Check_Type(num_points, T_FIXNUM);
    Check_Type(labels, T_ARRAY);
    Check_Type(num_sets, T_FIXNUM);
    Check_Type(data, T_ARRAY);
    if (combodata != Qnil)
      Check_Type(combodata, T_ARRAY);

    /* Set GDChart options */
    gdc_set_params(self);

    /* prepare output */
    rb_io_binmode(out);
    GetOpenFile(out, fptr);
    rb_io_check_writable(fptr);
    f = GetWriteFile(fptr);

    label_num = RARRAY(labels)->len;
    if (label_num < NUM2INT(num_points))
      label_num = NUM2INT(num_points);

    c_labels = ALLOC_N(char *, label_num);
    for (i = 0; i < label_num; i++) {
      switch (TYPE(rb_ary_entry(labels, i))) {
      case T_NIL:
	c_labels[i] = "";
	break;

      case T_STRING:
	c_labels[i] = STR2CSTR(rb_ary_entry(labels, i));
	break;

      default:
	rb_raise(rb_eTypeError, "type error");
	break;
      }
    }

    all_points = FIX2INT(num_points) * FIX2INT(num_sets);
    fdata = ALLOC_N(float, all_points);
    for (i = 0; i < all_points; i++) {
      fdata[i] = NUM2DBL(rb_ary_entry(data, i));
    }     

    if (combodata == Qnil) {
      fcombodata = NULL;
    }
    else {
      fcombodata = ALLOC_N(float, FIX2INT(num_points));
      for (i = 0; i < FIX2INT(num_points); i++) {
	fcombodata[i] =
	  NUM2DBL(rb_ary_entry(combodata, i));
      }
    }
    
#ifdef DEBUG
    for (i = 0; i < FIX2INT(num_points) * FIX2INT(num_sets); i++) {
	rb_warn("fdata[%d] = %f", i, fdata[i]);
    }

    rb_warn("Going to call GDC_out_graph...");
    rb_warn("Arguments to GDC_outgraph:");
    rb_warn("%d, %d, %x, %d, %d, %p, %d, %p, %p",FIX2INT(imgwidth),
	    FIX2INT(imgheight),
	    f,
	    FIX2INT(type),
	    FIX2INT(num_points),
	    clabels,
	    FIX2INT(num_sets),
	    fdata,
	    0
	);
#endif

    GDC_out_graph(FIX2INT(imgwidth),
		  FIX2INT(imgheight),
		  f,
		  FIX2INT(type),
		  FIX2INT(num_points),
		  c_labels,
		  FIX2INT(num_sets),
		  fdata,
		  fcombodata
		  );

    if (c_labels != NULL)   free(c_labels);
    if (fdata != NULL)      free(fdata);
    if (fcombodata != NULL) free(fcombodata);

    return Qnil;
} 


static VALUE 
gdc_pie_out_graph(int argc, VALUE *argv, VALUE self)
{
    VALUE imgwidth, imgheight;	/* output image size */
    VALUE out;			/* output file descriptor */
    VALUE type;			/* chart type */
    VALUE num_points;		/* x axis points */
    VALUE labels;
    VALUE data;
    
    OpenFile *fptr;
    FILE *f;

    int i;
    int label_num;
    char **c_labels;
    float *fdata;

    rb_scan_args(argc, argv, "7", &imgwidth, &imgheight,
		 &out, &type, &num_points, &labels, &data);

    /* Do some sanity check */
    Check_Type(imgwidth, T_FIXNUM);
    Check_Type(imgheight, T_FIXNUM);
    Check_Type(out, T_FILE);
    Check_Type(type, T_FIXNUM);	/* XXX: 'type' is constant? */
    Check_Type(num_points, T_FIXNUM);
    Check_Type(labels, T_ARRAY);
    Check_Type(data, T_ARRAY);

    /* Set GDChart options */
    gdc_pie_set_params(self);

    /* prepare output */
    rb_io_binmode(out);
    GetOpenFile(out, fptr);
    rb_io_check_writable(fptr);
    f = GetWriteFile(fptr);

    label_num = RARRAY(labels)->len;
    if (label_num < NUM2INT(num_points))
      label_num = NUM2INT(num_points);

    c_labels = ALLOC_N(char *, label_num);
    for (i = 0; i < label_num; i++) {
      switch (TYPE(rb_ary_entry(labels, i))) {
      case T_NIL:
	c_labels[i] = "";
	break;

      case T_STRING:
	c_labels[i] = STR2CSTR(rb_ary_entry(labels, i));
	break;

      default:
	rb_raise(rb_eTypeError, "type error");
	break;
      }
    }

    fdata = ALLOC_N(float, FIX2INT(num_points));
    for (i = 0; i < FIX2INT(num_points); i++) {
      fdata[i] = NUM2DBL(rb_ary_entry(data, i));
    }     

    GDC_out_pie(FIX2INT(imgwidth),
		FIX2INT(imgheight),
		f,
		FIX2INT(type),
		FIX2INT(num_points),
		c_labels,
		fdata
	);

    if (c_labels != NULL)   free(c_labels);
    if (fdata != NULL)      free(fdata);

    return Qnil;
} 


void Init_GDChart()
{	
  cGDChart = rb_define_class("GDChart", rb_cObject);
  cGDChartPie = rb_define_class("GDChartPie", rb_cObject);
  cGDChartAnnotation = rb_define_class_under(cGDChart, "Annotation", rb_cObject);
  cGDChartScatter = rb_define_class_under(cGDChart, "Scatter", rb_cObject);

  /* Chart types */
  rb_define_const(cGDChart, "LINE", INT2FIX(GDC_LINE));
  rb_define_const(cGDChart, "AREA", INT2FIX(GDC_AREA));
  rb_define_const(cGDChart, "BAR",  INT2FIX(GDC_BAR));
  rb_define_const(cGDChart, "FLOATINGBAR", INT2FIX(GDC_FLOATINGBAR));
  rb_define_const(cGDChart, "HILOCLOSE", INT2FIX(GDC_HILOCLOSE));
  rb_define_const(cGDChart, "COMBO_LINE_BAR", INT2FIX(GDC_COMBO_LINE_BAR));
  rb_define_const(cGDChart, "COMBO_HLC_BAR", INT2FIX(GDC_COMBO_HLC_BAR));
  rb_define_const(cGDChart, "COMBO_LINE_AREA", INT2FIX(GDC_COMBO_LINE_AREA));
  rb_define_const(cGDChart, "COMBO_LINE_LINE", INT2FIX(GDC_COMBO_LINE_LINE));
  rb_define_const(cGDChart, "COMBO_HLC_AREA", INT2FIX(GDC_COMBO_HLC_AREA));
  rb_define_const(cGDChart, "HILOCLOSE3D", INT2FIX(GDC_3DHILOCLOSE));
  rb_define_const(cGDChart, "COMBO_LINE_BAR3D", INT2FIX(GDC_3DCOMBO_LINE_BAR));
  rb_define_const(cGDChart, "COMBO_LINE_AREA3D", INT2FIX(GDC_3DCOMBO_LINE_AREA));
  rb_define_const(cGDChart, "COMBO_LINE_LINE3D", INT2FIX(GDC_3DCOMBO_LINE_LINE));
  rb_define_const(cGDChart, "COMBO_HLC_BAR3D", INT2FIX(GDC_3DCOMBO_HLC_BAR));
  rb_define_const(cGDChart, "COMBO_HLC_AREA3D", INT2FIX(GDC_3DCOMBO_HLC_AREA));
  rb_define_const(cGDChart, "BAR3D", INT2FIX(GDC_3DBAR));
  rb_define_const(cGDChart, "FLOATINGBAR3D", INT2FIX(GDC_3DFLOATINGBAR));
  rb_define_const(cGDChart, "AREA3D", INT2FIX(GDC_3DAREA));
  rb_define_const(cGDChart, "LINE3D", INT2FIX(GDC_3DLINE));

  /* Stack types */
  rb_define_const(cGDChart, "STACK_DEPTH", INT2FIX(GDC_STACK_DEPTH));
  rb_define_const(cGDChart, "STACK_SUM", INT2FIX(GDC_STACK_SUM));
  rb_define_const(cGDChart, "STACK_BESIDE", INT2FIX(GDC_STACK_BESIDE));
  rb_define_const(cGDChart, "STACK_LAYER", INT2FIX(GDC_STACK_LAYER));

  /* HiLoClose chart styles */
  rb_define_const(cGDChart, "HLC_DIAMOND", INT2FIX(GDC_HLC_DIAMOND));
  rb_define_const(cGDChart, "HLC_CLOSE_CONNECTED", INT2FIX(GDC_HLC_CLOSE_CONNECTED));
  rb_define_const(cGDChart, "HLC_CONNECTING", INT2FIX(GDC_HLC_CONNECTING));
  rb_define_const(cGDChart, "HLC_I_CAP", INT2FIX(GDC_HLC_I_CAP));

  /* Axis tick types */
  rb_define_const(cGDChart, "TICK_LABELS", INT2FIX(GDC_TICK_LABELS));
  rb_define_const(cGDChart, "TICK_POINTS", INT2FIX(GDC_TICK_POINTS));
  rb_define_const(cGDChart, "TICK_NONE", INT2FIX(GDC_TICK_NONE));

  /* Border types */
  rb_define_const(cGDChart, "BORDER_NONE", INT2FIX(GDC_BORDER_NONE));
  rb_define_const(cGDChart, "BORDER_ALL", INT2FIX(GDC_BORDER_ALL));
  rb_define_const(cGDChart, "BORDER_X", INT2FIX(GDC_BORDER_X));
  rb_define_const(cGDChart, "BORDER_Y", INT2FIX(GDC_BORDER_Y));
  rb_define_const(cGDChart, "BORDER_Y2", INT2FIX(GDC_BORDER_Y2));
  rb_define_const(cGDChart, "BORDER_TOP", INT2FIX(GDC_BORDER_TOP));

  /* Output types */
  rb_define_const(cGDChart, "GIF", INT2FIX(GDC_GIF));
  rb_define_const(cGDChart, "JPEG", INT2FIX(GDC_JPEG));
  rb_define_const(cGDChart, "PNG", INT2FIX(GDC_PNG));
  rb_define_const(cGDChart, "WBMP", INT2FIX(GDC_WBMP));

  /* Font sizes */
  rb_define_const(cGDChart, "TINY", INT2FIX(GDC_TINY));
  rb_define_const(cGDChart, "SMALL", INT2FIX(GDC_SMALL));
  rb_define_const(cGDChart, "MEDBOLD", INT2FIX(GDC_MEDBOLD));
  rb_define_const(cGDChart, "LARGE", INT2FIX(GDC_LARGE));
  rb_define_const(cGDChart, "GIANT", INT2FIX(GDC_GIANT));

#if 0
  /* Text justifications */
  rb_define_const(cGDChart, "JUSTIFY_RIGHT", INT2FIX(GDC_JUSTIFY_RIGHT));
  rb_define_const(cGDChart, "JUSTIFY_CENTER", INT2FIX(GDC_JUSTIFY_CENTER));
  rb_define_const(cGDChart, "JUSTIFY_LEFT", INT2FIX(GDC_JUSTIFY_LEFT));
#endif
  rb_define_const(cGDChart, "INTERP_VALUE", rb_float_new(GDC_INTERP_VALUE));

  /* method definitions */
  rb_define_singleton_method(cGDChart, "new", gdc_new, 0);
  rb_define_method(cGDChart, "initialize", gdc_initialize, 0);
  rb_define_method(cGDChart, "set_param", gdc_set_params, 0);

  rb_define_method(cGDChart, "out_graph", gdc_out_graph, -1);

  rb_define_attr(cGDChart, "ytitle", 1, 1);
  rb_define_attr(cGDChart, "xtitle", 1, 1);
  rb_define_attr(cGDChart, "ytitle2", 1, 1);
  rb_define_attr(cGDChart, "title", 1, 1);
  rb_define_attr(cGDChart, "title_size", 1, 1);
  rb_define_attr(cGDChart, "ytitle_size", 1, 1);
  rb_define_attr(cGDChart, "xtitle_size", 1, 1);
  rb_define_attr(cGDChart, "yaxisfont_size", 1, 1);
  rb_define_attr(cGDChart, "xaxisfont_size", 1, 1);
  rb_define_attr(cGDChart, "xaxis_angle", 1, 1);
#ifdef HAVE_LIBFREETYPE
  rb_define_attr(cGDChart, "title_font", 1, 1);
  rb_define_attr(cGDChart, "ytitle_font", 1, 1);
  rb_define_attr(cGDChart, "xtitle_font", 1, 1);
  rb_define_attr(cGDChart, "xaxis_font", 1, 1);
  rb_define_attr(cGDChart, "yaxis_font", 1, 1);
  rb_define_attr(cGDChart, "title_ptsize", 1, 1);
  rb_define_attr(cGDChart, "ytitle_ptsize", 1, 1);
  rb_define_attr(cGDChart, "xtitle_ptsize", 1, 1);
  rb_define_attr(cGDChart, "xaxis_ptsize", 1, 1);
  rb_define_attr(cGDChart, "yaxis_ptsize", 1, 1);
#endif
  rb_define_attr(cGDChart, "ylabel_fmt", 1, 1);
  rb_define_attr(cGDChart, "ylabel2_fmt", 1, 1);
  rb_define_attr(cGDChart, "xlabel_ctl", 1, 1);
  rb_define_attr(cGDChart, "xlabel_spacing", 1, 1);
  rb_define_attr(cGDChart, "ylabel_density", 1, 1);
  rb_define_attr(cGDChart, "interpolations", 1, 1);
  rb_define_attr(cGDChart, "requested_ymin", 1, 1);
  rb_define_attr(cGDChart, "requested_ymax", 1, 1);
  rb_define_attr(cGDChart, "requested_yinterval", 1, 1);
  rb_define_attr(cGDChart, "Shelf0", 1, 1);
  rb_define_attr(cGDChart, "grid", 1, 1);
  rb_define_attr(cGDChart, "ticks", 1, 1);
  rb_define_attr(cGDChart, "xaxis", 1, 1);
  rb_define_attr(cGDChart, "yaxis", 1, 1);
  rb_define_attr(cGDChart, "yaxis2", 1, 1);
  rb_define_attr(cGDChart, "yval_style", 1, 1);
  rb_define_attr(cGDChart, "stack_type", 1, 1);
  rb_define_attr(cGDChart, "depth_3d", 1, 1);
  rb_define_attr(cGDChart, "angle_3d", 1, 1);
  rb_define_attr(cGDChart, "bar_width", 1, 1);
  rb_define_attr(cGDChart, "HLC_style", 1, 1);
  rb_define_attr(cGDChart, "HLC_cap_width", 1, 1);
  rb_define_attr(cGDChart, "annotation", 1, 1);
  rb_define_attr(cGDChart, "annotation_font_size", 1, 1);
#ifdef HAVE_LIBFREETYPE
  rb_define_attr(cGDChart, "annotation_font", 1, 1);
  rb_define_attr(cGDChart, "annotation_ptsize", 1, 1);
#endif
  rb_define_attr(cGDChart, "num_scatter_pts", 1, 1);
  rb_define_attr(cGDChart, "scatter", 1, 1);
  rb_define_attr(cGDChart, "thumbnail", 1, 1);
  rb_define_attr(cGDChart, "thumblabel", 1, 1);
  rb_define_attr(cGDChart, "thumbval", 1, 1);
  rb_define_attr(cGDChart, "border", 1, 1);
  rb_define_attr(cGDChart, "BGColor", 1, 1);
  rb_define_attr(cGDChart, "GridColor", 1, 1);
  rb_define_attr(cGDChart, "LineColor", 1, 1);
  rb_define_attr(cGDChart, "PlotColor", 1, 1);
  rb_define_attr(cGDChart, "VolColor", 1, 1);
  rb_define_attr(cGDChart, "TitleColor", 1, 1);
  rb_define_attr(cGDChart, "XTitleColor", 1, 1);
  rb_define_attr(cGDChart, "YTitleColor", 1, 1);
  rb_define_attr(cGDChart, "YTitle2Color", 1, 1);
  rb_define_attr(cGDChart, "XLabelColor", 1, 1);
  rb_define_attr(cGDChart, "YLabelColor", 1, 1);
  rb_define_attr(cGDChart, "YLabel2Color", 1, 1);
  rb_define_attr(cGDChart, "ExtVolColor", 1, 1);
  rb_define_attr(cGDChart, "SetColor", 1, 1);
  rb_define_attr(cGDChart, "ExtColor", 1, 1);
  rb_define_attr(cGDChart, "transparent_bg", 1, 1);
  rb_define_attr(cGDChart, "BGImage", 1, 1);

  rb_define_attr(cGDChart, "hard_size", 1, 1);
  rb_define_attr(cGDChart, "hard_xorig", 1, 1);
  rb_define_attr(cGDChart, "hard_graphwidth", 1, 1);
  rb_define_attr(cGDChart, "hard_yorig", 1, 1);
  rb_define_attr(cGDChart, "hard_grapheight", 1, 1);

  rb_define_attr(cGDChart, "image_type", 1, 1);
 

  /*
   * GDChartAnnotation class
   */
  rb_define_singleton_method(cGDChartAnnotation, "new", gdc_annotation_new, 0);
  rb_define_method(cGDChartAnnotation, "initialize", gdc_annotation_init, 0);
  rb_define_attr(cGDChartAnnotation, "point", 1, 1);
  rb_define_attr(cGDChartAnnotation, "color", 1, 1);
  rb_define_attr(cGDChartAnnotation, "note", 1, 1);

  /*
   * GDChartScatter class
   */
  rb_define_singleton_method(cGDChartScatter, "new", gdc_scatter_new, 0);
  rb_define_method(cGDChartScatter, "initialize", gdc_scatter_init, 0);
  rb_define_attr(cGDChartScatter, "point", 1, 1);
  rb_define_attr(cGDChartScatter, "val", 1, 1);
  rb_define_attr(cGDChartScatter, "width", 1, 1);
  rb_define_attr(cGDChartScatter, "color", 1, 1);
  rb_define_attr(cGDChartScatter, "ind", 1, 1);

  /* Scatter indicators */
  rb_define_const(cGDChartScatter, "SCATTER_TRIANGLE_DOWN",
		  INT2FIX(GDC_SCATTER_TRIANGLE_DOWN));
  rb_define_const(cGDChartScatter, "SCATTER_TRIANGLE_UP",
		  INT2FIX(GDC_SCATTER_TRIANGLE_UP));
  rb_define_const(cGDChartScatter, "SCATTER_CIRCLE",
 		  INT2FIX(GDC_SCATTER_CIRCLE));


  /*
   * GDChart Pie class
   */

  /* instance variable */
  rb_define_attr(cGDChartPie, "BGColor", 1, 1);
  rb_define_attr(cGDChartPie, "PlotColor", 1, 1);
  rb_define_attr(cGDChartPie, "LineColor", 1, 1);
  rb_define_attr(cGDChartPie, "EdgeColor", 1, 1);
  rb_define_attr(cGDChartPie, "other_threshold", 1, 1);
  rb_define_attr(cGDChartPie, "angle_3d", 1, 1);
  rb_define_attr(cGDChartPie, "depth_3d", 1, 1);
  rb_define_attr(cGDChartPie, "perspective", 1, 1);
  rb_define_attr(cGDChartPie, "title", 1, 1);
  rb_define_attr(cGDChartPie, "title_size", 1, 1);
  rb_define_attr(cGDChartPie, "label_size", 1, 1);
#ifdef HAVE_LIBFREETYPE
  rb_define_attr(cGDChartPie, "title_font", 1, 1);
  rb_define_attr(cGDChartPie, "label_font", 1, 1);
  rb_define_attr(cGDChartPie, "title_ptsize", 1, 1);
  rb_define_attr(cGDChartPie, "label_ptsize", 1, 1);
#endif
  rb_define_attr(cGDChartPie, "label_dist", 1, 1);
  rb_define_attr(cGDChartPie, "label_line", 1, 1);
  rb_define_attr(cGDChartPie, "explode", 1, 1);
  rb_define_attr(cGDChartPie, "Color", 1, 1);
  rb_define_attr(cGDChartPie, "missing", 1, 1);
  rb_define_attr(cGDChartPie, "percent_labels", 1, 1);
  rb_define_attr(cGDChartPie, "percent_fmt", 1, 1);

  rb_define_attr(cGDChartPie, "image_type", 1, 1);

  /* method definitions */
  rb_define_singleton_method(cGDChartPie, "new", gdc_new, 0);
  rb_define_method(cGDChartPie, "initialize", gdc_initialize, 0);
  rb_define_method(cGDChartPie, "set_param", gdc_set_params, 0);

  rb_define_method(cGDChartPie, "out_graph", gdc_pie_out_graph, -1);

  /* constant */
  rb_define_const(cGDChartPie, "PIE3D", INT2FIX(GDC_3DPIE));
  rb_define_const(cGDChartPie, "PIE2D", INT2FIX(GDC_2DPIE));

  rb_define_const(cGDChartPie, "PCT_NONE", INT2FIX(GDCPIE_PCT_NONE));
  rb_define_const(cGDChartPie, "PCT_ABOVE", INT2FIX(GDCPIE_PCT_ABOVE));
  rb_define_const(cGDChartPie, "PCT_BELOW", INT2FIX(GDCPIE_PCT_BELOW));
  rb_define_const(cGDChartPie, "PCT_RIGHT", INT2FIX(GDCPIE_PCT_RIGHT));
  rb_define_const(cGDChartPie, "PCT_LEFT", INT2FIX(GDCPIE_PCT_LEFT));
}
