/****************************************************************************
** Simple gestion du drag and drop
** Créé par Anacr0x : anacr0x@free.fr
**
** Sous licence GPL v2 ou plus
****************************************************************************/

#include "mytreewidget.h"

// ////////////////////////////////////////////////////////////////
//
// MyTreeWidget
//
// ////////////////////////////////////////////////////////////////

MyTreeWidget::MyTreeWidget(QWidget * parent):BasicTreeWidget(parent)
{
	// On ajoute les colonnes nécessaires
	setHeaderLabels(QStringList() << tr("Fichier") << tr("Taille") <<
	                trUtf8("Modifié le") << tr("Droits") <<
	                trUtf8("Propriétaire") << tr("Type"));

	QHeaderView *head = header();
	head->setMovable(true);
	head->hideSection(5);

	// On permet de chosir le tri
	setSortingEnabled(true);
	// On accepte les drag&drop
	setAcceptDrops(true);
	setDragEnabled(true);
	// On accepte les sélections multiples
	setSelectionMode(QAbstractItemView::ExtendedSelection);
}

void MyTreeWidget::autoResize()
{
	QHeaderView *head = header();
	head->setResizeMode(QHeaderView::ResizeToContents);
	head->setResizeMode(QHeaderView::Interactive);

	// Bonus : On utilise un tri de liste bien spécial
	sortItems(5, Qt::AscendingOrder);
}

void MyTreeWidget::slotSelect(int select)
{
	clearSelection();
	// On énumère tous les items
	for (int item = 0; item < topLevelItemCount(); item++)
	{
		MyTreeWidgetItem *lvi = (MyTreeWidgetItem *) topLevelItem(item);

		// On sélectionne ou désélectionne en fonction du fichier
		if (lvi->getState() == MyTreeWidgetItem::File)
		{
			if (select == MyTreeWidget::AllFiles
			        || select == MyTreeWidget::AllFilesAndDirs)
				setItemSelected(lvi, true);
			else
				setItemSelected(lvi, false);
		}

		else if (lvi->getState() == MyTreeWidgetItem::Dir)
		{
			if (select == MyTreeWidget::AllDirs
			        || select == MyTreeWidget::AllFilesAndDirs)
				setItemSelected(lvi, true);
			else
				setItemSelected(lvi, false);
		}
	}
}

void MyTreeWidget::mouseMoveEvent(QMouseEvent * e)
{
	// On regarde s'il faut crée un drag
	QList<QTreeWidgetItem*> list = selectedItems();
	if (list.isEmpty() || e->buttons() != Qt::LeftButton)
	{
		e->ignore();
		return;
	}
	// On crée le drag
	QMimeData *mime = new QMimeData;
	mime->setData("application/x-qabstractitemmodeldatalist", NULL);
	QDrag *drag = new QDrag(this);
	drag->setMimeData(mime);
	drag->setPixmap(list.first()->icon(0).pixmap(16, 16));
	drag->start();
}

void MyTreeWidget::dragEnterEvent(QDragEnterEvent * event)
{
	// On accepte uniquement les drag&drop comportant du texte (nom de fichier)
	if (event->source() != this
	        && event->mimeData()->hasFormat("application/x-qabstractitemmodeldatalist"))
		event->acceptProposedAction();
	else
		event->ignore();
}

void MyTreeWidget::dropEvent(QDropEvent * event)
{
	// On accepte uniquement les drag&drop venant du même logiciel
	if (event->source() != this
	        && event->mimeData()->hasFormat("application/x-qabstractitemmodeldatalist"))
	{
		// On regarde s'il faut enregistrer le dossier dans un sous dossier
		QString destDir = "";
		MyTreeWidgetItem *lvi = (MyTreeWidgetItem *) itemAt(event->pos());
		if (lvi)
		{
			// Il faut remonté d'un cran pour avoir l'item réelement cliqué
			// lvi = (MyTreeWidgetItem *) lvi->itemAbove ();
			if (lvi && lvi->isDir() && lvi->isSimple())
			{
				// On sélection le dossier choisi et indique qu'il fodra mettre le
				// ficher dedans
				setCurrentItem(lvi);
				destDir = lvi->text(0);
				switch (QMessageBox::information(NULL, tr("Transfert"),
				                                 trUtf8
				                                 ("Voulez-vous réellement mettre ce(s) fichier(s)/dossier(s) dans le dossier \"")
				                                 + destDir + "\" ?",
				                                 tr("&Oui"),
				                                 tr("&Dossier courant"),
				                                 tr("&Annuler"), 0, 2))
				{
				case 1:
					destDir = "";
					break;
				case 2:
					return;
				}
			}
		}
		emit startDownloadOrUpload(destDir);
	}

	event->ignore();
}

// ////////////////////////////////////////////////////////////////
//
// MyTreeWidgetItem
//
// ////////////////////////////////////////////////////////////////

MyTreeWidgetItem::MyTreeWidgetItem(QTreeWidget * parent, bool dir, const QString & name, const QString & size, const QString &lastModified, const QString & droits, const QString & proprietaire):
		QTreeWidgetItem(parent)
{
	setText(0, name);
	setText(2, lastModified);
	setText(3, droits);
	setText(4, proprietaire);
	// Permet de savoir s'il s'agit d'un fichier ou d'un dossier
	DirOrNot = dir;

	// Initialise le type de fichier
	fileState = None;

	// Si c'est un dossier, icone spéciale
	if (DirOrNot == true)
	{
		if ((name.length()>0 && name[0] == '/') || (name.length()>2 && name[2] == '/'))
		{
			state = Root;
			setIcon(0, QIcon(":/images/dd.png"));
		}
		else if (name == ".")
		{
			state = Refresh;
			setIcon(0, QIcon(":/images/up.png"));
		}
		else if (name == "..")
		{
			state = CdUp;
			setIcon(0, QIcon(":/images/up.png"));
		}
		else
		{
			state = Dir;
			setIcon(0, QIcon(":/images/folder.png"));
		}
	}

	else			// En cas de fichier
	{
		state = File;

		// On prend uniquement l'extension et la compare aux
		// extensions prédéfinies pour sélectionner la bonne icône
		QString ext = QFileInfo(name).suffix();
		if (ext == "jpg" || ext == "jpeg" || ext == "png" || ext == "gif"
		        || ext == "bmp" || ext == "ico" || ext == "svg")
		{
			setIcon(0, QIcon(":/images/imagefile.png"));
			fileState = Image;
		}
		else if (ext == "avi" || ext == "mpg" || ext == "mpeg" || ext == "wmv")
		{
			setIcon(0, QIcon(":/images/videofile.png"));
			fileState = Video;
		}
		else if (ext == "ogg" || ext == "mp3" || ext == "wav" || ext == "wma")
		{
			setIcon(0, QIcon(":/images/soundfile.png"));
			fileState = Audio;
		}
		else if (ext == "tar" || ext == "bz2" || ext == "zip" || ext == "gz"
		         || ext == "rar" || ext == "rpm" || ext == "deb")
		{
			setIcon(0, QIcon(":/images/compressedfile.png"));
			fileState = Archive;
		}
		else if (ext == "txt" || ext == "cpp" || ext == "c" || ext == "h")
		{
			setIcon(0, QIcon(":/images/txtfile.png"));
			fileState = Texte;
		}
		else if (ext == "doc" || ext == "rtf" || ext == "sxw" || ext == "xls"
		         || ext == "sxc" || ext == "odt" || ext == "ods")
		{
			setIcon(0, QIcon(":/images/docfile.png"));
			fileState = Document;
		}
		else if (ext == "html" || ext == "htm" || ext == "php")
		{
			setIcon(0, QIcon(":/images/netfile.png"));
			fileState = Internet;
		}
		else
		{
			setIcon(0, QIcon(":/images/file.png"));
			fileState = Autre;
		}
	}

	// Les dossiers bloqués sont rangé avec les dossiers normaux
	if (state == DirLocked)
		setText(5, QString::number(Dir) + name);
	else
		setText(5, QString::number(state) + name);

	// Affichage de la taille en o/Ko/Mo
	if (size != "")
	{
		QString Size;
		uint taille = size.toUInt();
		// Si l'on doit afficher la taille en Go
		if (taille > 1073741824)
			Size = QString(" %1 Go").arg((double) taille / 1073741824, 0, 'f', 1);
		// En Mo
		else if (taille > 1048576)
			Size = QString(" %1 Mo").arg((double) taille / 1048576, 0, 'f', 1);
		// En Ko
		else if (taille > 1024)
			Size = QString(" %1 Ko").arg(taille / 1024);
		// En octet
		else
			Size = QString(" %1 octets").arg(taille);

		// On affiche la taille
		setText(1, Size);
	}
}

bool MyTreeWidgetItem::isSimple(const QString & filename)
{
	if (filename == "" || filename == "." || filename == ".." ||
	        filename == "/" || (filename.length() == 3 && filename[2] == '/'))
		return false;
	else
		return true;
}
