/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

%{
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "node.h"
#include "entry.h"

typedef char *charstar ;
#define YYSTYPE charstar
extern int yylex(void);
extern int yylineno;
struct node **yynodelist;
struct entry *entry;
char *stryychar( char alternative ) ;
char *scat( char *s, char *t ) ;
char *comment_cat(char*, char*, char*, char*);
void define (char *a, char *b ) ;
char *lookup (char *a) ;
void yysetnodelist(struct node **n) {yynodelist = n;}
int yywrap(void){ return 1;}
int yyerror(const char *s) { fprintf(stderr, "%d : %s\n",yylineno, s); return 1;}
int sense;
%}

%token DEFINESTRING
%token PREAMBLESTRING
%token ATSTRING
%token STRING
%token QUOTE
%token LBRACKET
%token RBRACKET
%token LCBRACKET
%token RCBRACKET
%token EQUAL
%token COMMA
%token HASHSIGN
%token WHITE
%token ENDLINE
%token COMMENT
%token COMMENTSTRING
%token LATEXCOMMAND

%start file

%%

file	:	w entries
	;

entries :	/* empty */
	|	entry entries
	|	comment entries
	;

w	:	/* empty */
	|	WHITE w
	|	ENDLINE w
	;

entry 	: 	ATSTRING w LBRACKET  key COMMA { entry = add_entry(yynodelist, ($1 + sizeof(char)), $4); }
			 w fields RBRACKET  w  { ; }
	|	ATSTRING w LCBRACKET key COMMA { entry = add_entry(yynodelist, ($1 + sizeof(char)), $4); }
			 w fields RCBRACKET w  { ; }
	|	DEFINESTRING w LCBRACKET key EQUAL w body RCBRACKET w
					       { add_string(yynodelist, $4, $7); }
	|	DEFINESTRING w LBRACKET key EQUAL w body RBRACKET w
					       { add_string(yynodelist, $4, $7); }
	|	PREAMBLESTRING w LBRACKET body RBRACKET w {add_preamble(yynodelist, $4); }
	|	PREAMBLESTRING w LCBRACKET body RCBRACKET w {add_preamble(yynodelist, $4); }
	|	COMMENTSTRING w LBRACKET whatever RBRACKET w  {add_comment(yynodelist, comment_cat($1, $3, $4, $5));}
	|	COMMENTSTRING w LCBRACKET whatever RCBRACKET w  {add_comment(yynodelist, comment_cat($1, $3, $4, $5));}
	;
	
comment	:	COMMENT ENDLINE w	{add_comment(yynodelist, $1);}
		|	stringcomment ENDLINE w		{add_comment(yynodelist, $1);}
		;
		
stringcomment : /* empty */	{ $$ = strdup("") ; }
				| symbol stringcomment { $$ = scat($1, $2); }
				;

key	:	w STRING w			{ $$ = $2 ; }
	|	w ATSTRING w			{ $$ = $2 ; }
	;

fields	:	field
	|	field fields
	;

field	:	STRING w EQUAL w body optcomma { add_tag(entry, $1, $5, sense); }
	;

optcomma:	COMMA w optcomma
	|	
	;

body	:	string				{ $$ = $1 ; }
	|	string HASHSIGN w body		{ $$ = scat( $1, $4 ) ; }
	;

string 	:	STRING w 			{ $$ = lookup($1) ; }
	|	ATSTRING w 			{ $$ = $1 ; }
	|	QUOTE noquote QUOTE w 		{ sense = 0; $$ = $2 ; }
	|	LATEXCOMMAND w			{ $$ = $1 ; }
	|	LCBRACKET whatever RCBRACKET w 	{ sense = 1; $$ = $2 ; }
	;

noquote	:	/* Empty */			{ $$ = strdup("") ; }
	|	noqsym noquote			{ $$ = scat( $1, $2 ) ; }
	;

noqsym 	: WHITE | ATSTRING | STRING | LBRACKET | RBRACKET | EQUAL | COMMA | HASHSIGN | ENDLINE | COMMENTSTRING | DEFINESTRING | PREAMBLESTRING | LATEXCOMMAND
	| LCBRACKET w whatever w RCBRACKET { $$ = scat(scat(strdup("{"),$3),strdup("}"));}
	;
	
whatever : /* empty */			{ $$ = strdup("") ; } 
		| noqsym  whatever	{ $$ = scat( $1, $2 ) ; }
		| QUOTE whatever	{ $$ = scat( $1, $2 ) ; }
		;
	
symbol : WHITE | ATSTRING | STRING | LBRACKET | RBRACKET | EQUAL | COMMA | HASHSIGN | QUOTE | COMMENTSTRING | DEFINESTRING | PREAMBLESTRING | LATEXCOMMAND
			| LCBRACKET | RCBRACKET
			;

%%

char *comment_cat(char *atstring, char *s_bracket, char *comment, char *e_bracket)
{
	char *ret = malloc(strlen(atstring) + strlen(s_bracket) + 
			strlen(comment) + strlen(e_bracket) + 1);
	strcpy(ret, atstring);
	strcat(ret, s_bracket);
	strcat(ret, comment);
	strcat(ret, e_bracket);
	free(atstring);
	free(s_bracket);
	free(comment);
	free(e_bracket);
	return ret;
}

char *scat( char *s, char *t )
{
  char *z = malloc( strlen( s ) + strlen( t ) + 1 ) ;
  strcpy( z, s ) ;
  strcat( z, t ) ;
  free( s ) ;
  free( t ) ;
  return z;
}

char *stryychar( char alternative )
{
    char s[2] ;
    s[0] = alternative ;
    s[1] = '\0' ;
    return strdup( s ) ;
}

typedef struct strings {
  struct strings *left, *right ;
  char *identifier ;
  char *contents ;
} tree ;

static tree *root = NULL ;

void insert( tree **root, char *id, char *contents )
{
  if( *root == NULL ) {
    *root = (tree *) malloc( sizeof( tree ) ) ;
    (*root)->left = NULL ;
    (*root)->right = NULL ;
    (*root)->identifier = id ;
    (*root)->contents = contents ;
  } else if( strcasecmp( id, (*root)->identifier) < 0 ) {
    insert( &(*root)->left, id, contents ) ;
  } else {
    insert( &(*root)->right, id, contents ) ;
  }
}

void define( char *id, char *contents )
{
  insert( &root, id, contents ) ;
}

char *lookup( char *id )
{
  tree *s ;
  int comp ;

  s = root ;
  while( s != NULL && (comp = strcasecmp( id, s->identifier ) ) != 0 ) {
    if( comp < 0 ) {
      s = s->left ;
    } else {
      s = s->right ;
    }
  }
  return s == NULL ? id : s->contents ;
}

