/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if defined(HAVE_STDARG_H)
#include <stdarg.h>
#elif defined(HAVE_VARARGS_H)
#include <varargs.h>
#define va_start(v, l) va_start(v)
#else
#error Your system does not support variable-argument functions.
#endif

#ifdef HAVE_DBUS

#ifndef DBUS_API_SUBJECT_TO_CHANGE  /* exo 0.2 sets this, but 0.3 doesn't */
#define DBUS_API_SUBJECT_TO_CHANGE
#endif
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#ifdef DBUS_USE_OLD_API
#define dbus_bus_request_name dbus_bus_acquire_service
#define dbus_bus_start_service_by_name dbus_bus_activate_service
#define dbus_bus_get_unique_name dbus_bus_get_base_service
#define dbus_bus_name_has_owner dbus_bus_service_exists
#define DBUS_NAME_FLAG_PROHIBIT_REPLACEMENT DBUS_SERVICE_FLAG_PROHIBIT_REPLACEMENT
#define DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER DBUS_SERVICE_REPLY_PRIMARY_OWNER
#endif

#endif  /* HAVE_DBUS */

#include <libxfce4util/libxfce4util.h>

#include "xfmedia-remote-client.h"

#ifdef HAVE_DBUS

static DBusGConnection *__dbus_connection = NULL;

DBusConnection *
__xfmedia_dbus_get_bus_connection()
{
    GError *err = NULL;
    
    if(!__dbus_connection) {
        __dbus_connection = dbus_g_bus_get(DBUS_BUS_SESSION, &err);
        if(!__dbus_connection) {
            g_warning("Failed to open a connection to the D-BUS session bus.  "
                    "Please ensure that the session bus is running. (%d: %s)",
                    err->code, err->message);
            g_error_free(err);
            
            return NULL;
        }
    }
    
    return dbus_g_connection_get_connection(__dbus_connection);
}

static XfmediaRemoteStatus
remote_do_ping(DBusConnection *dbus_conn, gint session_id)
{
    XfmediaRemoteStatus ret = XFMEDIA_REMOTE_COMMAND_FAILED;
    DBusMessage *msg, *msg_ret;
    DBusError derr;
    gchar service_name[64], path_name[64];
    
    g_snprintf(service_name, 64, XFMEDIA_DBUS_SERVICE_FMT, session_id);
    g_snprintf(path_name, 64, XFMEDIA_DBUS_PATH_FMT, session_id);
    msg = dbus_message_new_method_call(service_name, path_name,
            XFMEDIA_DBUS_INTERFACE, XFMEDIA_REMOTE_IS_RUNNING);
#ifdef DBUS_USE_OLD_API
    dbus_message_set_auto_activation(msg, FALSE);
#else
    dbus_message_set_auto_start(msg, FALSE);
#endif
    
    dbus_error_init(&derr);
    msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
            msg, 5000, &derr);
    dbus_message_unref(msg);
    
    if(!msg_ret) {
        DBG("Unable to contact Xfmedia instance on id %d", session_id);
        dbus_error_free(&derr);
    } else if(!dbus_message_is_error(msg_ret, XFMEDIA_DBUS_ERROR))
        ret = session_id;
    
    if(msg_ret)
        dbus_message_unref(msg_ret);
    
    return ret;
}

gboolean
xfmedia_remote_client_init()
{
    gboolean ret = FALSE;
    FILE *fp;
    gchar filename[PATH_MAX], buf[2048], *p;
    gint i;
    
    /* check for DBUS env vars */
    if(g_getenv("DBUS_SESSION_BUS_ADDRESS") && g_getenv("DBUS_SESSION_BUS_PID"))
        return TRUE;
    
    for(i = 0; i < 1024; ++i) {
        g_snprintf(filename, PATH_MAX, "%s/xfmedia-remote-env.%d.%d",
                   g_get_tmp_dir(), (gint)getuid(), i);
        fp = fopen(filename, "r");
        if(!fp)
            continue;
        
        while(fgets(buf, 2048, fp)) {
            p = strstr(buf, "=");
            if(!p)
                continue;
            
            while(buf[strlen(buf)-1] == '\n' || buf[strlen(buf)-1] == '\r')
                buf[strlen(buf)-1] = 0;
            
            *p = 0;
            ++p;
            
            g_setenv(buf, p, TRUE);
        }
        
        fclose(fp);
        ret = TRUE;
    }
    
    return ret;
}

XfmediaRemoteStatus
#ifdef HAVE_STDARG_H
xfmedia_remote_client_send_command(gint session_id, const gchar *command, ...)
#else /* ifdef HAVE_VARARGS_H */
xfmedia_remote_client_send_command(va_alist) va_dcl
#endif
{
    XfmediaRemoteStatus ret = XFMEDIA_REMOTE_UNKNOWN_ERROR;
    va_list ap;
    DBusConnection *dbus_conn;
    DBusMessage *msg, *msg_ret = NULL;
    DBusError derr;
    gchar service_name[64], path_name[64];
#ifdef HAVE_VARARGS_H
    gint session_id;
    const gchar *command;
#endif
    
    dbus_conn = __xfmedia_dbus_get_bus_connection();
    if(!dbus_conn)
        return XFMEDIA_REMOTE_NO_SESSION_BUS;
    
#ifdef HAVE_STDARG_H
    va_start(ap, command);
#else /* ifdef HAVE_VARARGS_H */
    va_start(ap);
    session_id = va_arg(ap, gint);
    command = va_arg(ap, const gchar *);
#endif
    
    /* we only accept '-1' as a session id if we're doing a ping */
    g_return_val_if_fail(session_id >= 0 || !strcmp(command, XFMEDIA_REMOTE_IS_RUNNING),
            XFMEDIA_REMOTE_BAD_ARGUMENTS);
    
    g_snprintf(service_name, 64, XFMEDIA_DBUS_SERVICE_FMT, session_id);
    g_snprintf(path_name, 64, XFMEDIA_DBUS_PATH_FMT, session_id);
    
    if(!strcmp(command, XFMEDIA_REMOTE_IS_RUNNING)) {
        if(session_id >= 0)
            ret = remote_do_ping(dbus_conn, session_id);
        else {
            gint i;
            for(i = 0; i < MAX_INSTANCES; i++) {
                ret = remote_do_ping(dbus_conn, i);
                if(ret != XFMEDIA_REMOTE_COMMAND_FAILED)
                    break;
            }
        }
    } else if(!strcmp(command, XFMEDIA_REMOTE_CLEAR_PLAYLIST)) {
        msg = dbus_message_new_method_call(service_name, path_name,
                XFMEDIA_DBUS_INTERFACE, XFMEDIA_REMOTE_CLEAR_PLAYLIST);
#ifdef DBUS_USE_OLD_API
        dbus_message_set_auto_activation(msg, FALSE);
#else
        dbus_message_set_auto_start(msg, FALSE);
#endif
        
        dbus_error_init(&derr);
        msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                msg, 5000, &derr);
        dbus_message_unref(msg);
        
        if(!msg_ret) {
            ret = XFMEDIA_REMOTE_UNKNOWN_ERROR;
            dbus_error_free(&derr);
        } else if(dbus_message_is_error(msg_ret, XFMEDIA_DBUS_ERROR))
            ret = XFMEDIA_REMOTE_COMMAND_FAILED;
        else if(msg_ret)
            ret = XFMEDIA_REMOTE_COMMAND_SUCCEEDED;
        
        if(msg_ret)
            dbus_message_unref(msg_ret);
    } else if(!strcmp(command, XFMEDIA_REMOTE_LOAD_PLAYLIST)
            || !strcmp(command, XFMEDIA_REMOTE_SAVE_PLAYLIST))
    {
        const gchar *filename;
        
        filename = va_arg(ap, const gchar *);
        if(!filename || !*filename)
            ret = XFMEDIA_REMOTE_BAD_ARGUMENTS;
        else {
            gchar *utf8_fn = g_filename_to_utf8(filename, -1, NULL, NULL, NULL);
            
            if(utf8_fn) {
                msg = dbus_message_new_method_call(service_name, path_name,
                        XFMEDIA_DBUS_INTERFACE, command);
#ifdef DBUS_USE_OLD_API
                dbus_message_set_auto_activation(msg, FALSE);
#else
                dbus_message_set_auto_start(msg, FALSE);
#endif
                dbus_message_append_args(msg,
#ifdef DBUS_USE_OLD_API
                        DBUS_TYPE_STRING, utf8_fn,
#else
                        DBUS_TYPE_STRING, &utf8_fn,
#endif
                        DBUS_TYPE_INVALID);

                dbus_error_init(&derr);
                msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                        msg, 5000, &derr);
                dbus_message_unref(msg);
                g_free(utf8_fn);
            }
            
            if(!msg_ret) {
                ret = XFMEDIA_REMOTE_UNKNOWN_ERROR;
                dbus_error_free(&derr);
            } else if(dbus_message_is_error(msg_ret, XFMEDIA_DBUS_ERROR))
                ret = XFMEDIA_REMOTE_COMMAND_FAILED;
            else if(msg_ret)
                ret = XFMEDIA_REMOTE_COMMAND_SUCCEEDED;
            
            if(msg_ret)
                dbus_message_unref(msg_ret);
        }
    } else if(!strcmp(command, XFMEDIA_REMOTE_ADD_FILE)) {
        const gchar *filename;
        gint idx;
        
        filename = va_arg(ap, const gchar *);
        idx = va_arg(ap, gint);
        if(idx < -1)
            idx = -1;
        
        if(!filename || !*filename)
            ret = XFMEDIA_REMOTE_BAD_ARGUMENTS;
        else {
            gchar *utf8_fn = g_filename_to_utf8(filename, -1, NULL, NULL, NULL);
            
            if(utf8_fn) {
                msg = dbus_message_new_method_call(service_name, path_name,
                        XFMEDIA_DBUS_INTERFACE, XFMEDIA_REMOTE_ADD_FILE);
#ifdef DBUS_USE_OLD_API
                dbus_message_set_auto_activation(msg, FALSE);
#else
                dbus_message_set_auto_start(msg, FALSE);
#endif
                dbus_message_append_args(msg,
#ifdef DBUS_USE_OLD_API
                        DBUS_TYPE_STRING, utf8_fn,
                        DBUS_TYPE_INT32, idx,
#else
                        DBUS_TYPE_STRING, &utf8_fn,
                        DBUS_TYPE_INT32, &idx,
#endif
                        DBUS_TYPE_INVALID);

                dbus_error_init(&derr);
                msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                        msg, 5000, &derr);
                dbus_message_unref(msg);
                g_free(utf8_fn);
            }
            
            if(!msg_ret) {
                ret = XFMEDIA_REMOTE_UNKNOWN_ERROR;
                dbus_error_free(&derr);
            } else if(dbus_message_is_error(msg_ret, XFMEDIA_DBUS_ERROR))
                ret = XFMEDIA_REMOTE_COMMAND_FAILED;
            else if(msg_ret) {
                dbus_error_init(&derr);
                if(!dbus_message_get_args(msg_ret, &derr,
                        DBUS_TYPE_INT32, &ret, DBUS_TYPE_INVALID))
                {
                    ret = XFMEDIA_REMOTE_COMMAND_FAILED;
                    dbus_error_free(&derr);
                }
            }
            
            if(msg_ret)
                dbus_message_unref(msg_ret);
        }
    } else if(!strcmp(command, XFMEDIA_REMOTE_REMOVE_FILE)) {
        guint idx;
        
        idx = va_arg(ap, guint);
        
        msg = dbus_message_new_method_call(service_name, path_name,
                XFMEDIA_DBUS_INTERFACE, XFMEDIA_REMOTE_REMOVE_FILE);
#ifdef DBUS_USE_OLD_API
        dbus_message_set_auto_activation(msg, FALSE);
#else
        dbus_message_set_auto_start(msg, FALSE);
#endif
        dbus_message_append_args(msg,
#ifdef DBUS_USE_OLD_API
                DBUS_TYPE_UINT32, idx,
#else
                DBUS_TYPE_UINT32, &idx,
#endif
                DBUS_TYPE_INVALID);
        
        dbus_error_init(&derr);
        msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                msg, 5000, &derr);
        dbus_message_unref(msg);
        
        if(!msg_ret) {
            ret = XFMEDIA_REMOTE_UNKNOWN_ERROR;
            dbus_error_free(&derr);
        } else if(dbus_message_is_error(msg_ret, XFMEDIA_DBUS_ERROR))
            ret = XFMEDIA_REMOTE_COMMAND_FAILED;
        else if(msg_ret)
            ret = XFMEDIA_REMOTE_COMMAND_SUCCEEDED;
        
        if(msg_ret)
            dbus_message_unref(msg_ret);
    } else if(!strcmp(command, XFMEDIA_REMOTE_PLAY)) {
        gint idx;
        
        idx = va_arg(ap, gint);
        if(idx < -1)
            idx = -1;
        
        msg = dbus_message_new_method_call(service_name, path_name,
                XFMEDIA_DBUS_INTERFACE, XFMEDIA_REMOTE_PLAY);
#ifdef DBUS_USE_OLD_API
        dbus_message_set_auto_activation(msg, FALSE);
#else
        dbus_message_set_auto_start(msg, FALSE);
#endif
        dbus_message_append_args(msg,
#ifdef DBUS_USE_OLD_API
                DBUS_TYPE_INT32, idx,
#else
                DBUS_TYPE_INT32, &idx,
#endif
                DBUS_TYPE_INVALID);
        
        dbus_error_init(&derr);
        msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                msg, 5000, &derr);
        dbus_message_unref(msg);
        
        if(!msg_ret) {
            ret = XFMEDIA_REMOTE_UNKNOWN_ERROR;
            dbus_error_free(&derr);
        } else if(dbus_message_is_error(msg_ret, XFMEDIA_DBUS_ERROR))
            ret = XFMEDIA_REMOTE_COMMAND_FAILED;
        else if(msg_ret)
            ret = XFMEDIA_REMOTE_COMMAND_SUCCEEDED;
        
        if(msg_ret)
            dbus_message_unref(msg_ret);
    } else if(!strcmp(command, XFMEDIA_REMOTE_PAUSE)
            || !strcmp(command, XFMEDIA_REMOTE_TOGGLE_PLAY)
            || !strcmp(command, XFMEDIA_REMOTE_STOP)
            || !strcmp(command, XFMEDIA_REMOTE_PREV)
            || !strcmp(command, XFMEDIA_REMOTE_NEXT)
            || !strcmp(command, XFMEDIA_REMOTE_QUIT)
            || !strcmp(command, XFMEDIA_REMOTE_TRIGGER_JTF))
    {
        msg = dbus_message_new_method_call(service_name, path_name,
                XFMEDIA_DBUS_INTERFACE, command);
#ifdef DBUS_USE_OLD_API
        dbus_message_set_auto_activation(msg, FALSE);
#else
        dbus_message_set_auto_start(msg, FALSE);
#endif
        
        dbus_error_init(&derr);
        msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                msg, 5000, &derr);
        dbus_message_unref(msg);
        
        if(!msg_ret) {
            ret = XFMEDIA_REMOTE_UNKNOWN_ERROR;
            dbus_error_free(&derr);
        } else if(dbus_message_is_error(msg_ret, XFMEDIA_DBUS_ERROR))
            ret = XFMEDIA_REMOTE_COMMAND_FAILED;
        else if(msg_ret)
            ret = XFMEDIA_REMOTE_COMMAND_SUCCEEDED;
        
        if(msg_ret)
            dbus_message_unref(msg_ret);
    } else {
        ret = XFMEDIA_REMOTE_BAD_COMMAND;
    }
    
    va_end(ap);
    
    return ret;
}

GList *
#ifdef HAVE_STDARG_H
xfmedia_remote_client_get_info(guint session_id, const gchar *command, ...)
#else
xfmedia_remote_client_get_info(va_alist) va_dcl
#endif
{
    GList *retvals = NULL;
    va_list ap;
    DBusConnection *dbus_conn;
    DBusMessage *msg, *msg_ret;
    DBusError derr;
    gchar service_name[64], path_name[64];
#ifdef HAVE_VARARGS_H
    guint session_id;
    const gchar *command;
#endif
    
    dbus_conn = __xfmedia_dbus_get_bus_connection();
    if(!dbus_conn)
        return NULL;
    
#ifdef HAVE_STDARG_H
    va_start(ap, command);
#else /* ifdef HAVE_VARARGS_H */
    va_start(ap);
    session_id = va_arg(ap, guint);
    command = va_arg(ap, const gchar *);
#endif
    
    g_snprintf(service_name, 64, XFMEDIA_DBUS_SERVICE_FMT, session_id);
    g_snprintf(path_name, 64, XFMEDIA_DBUS_PATH_FMT, session_id);
    
    if(!strcmp(command, XFMEDIA_REMOTE_GET_PLAYLIST)) {
        msg = dbus_message_new_method_call(service_name, path_name,
                XFMEDIA_DBUS_INTERFACE, XFMEDIA_REMOTE_GET_PLAYLIST);
        
        dbus_error_init(&derr);
        msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                msg, 5000, &derr);
        dbus_message_unref(msg);
        
        if(!msg_ret)
            dbus_error_free(&derr);
        else {
            gchar **strv = NULL;
            gint nentries = 0;
            
            dbus_error_init(&derr);
            if(dbus_message_get_args(msg, &derr,
                    DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &strv, &nentries,
                    DBUS_TYPE_INVALID))
            {
                gint i;
                
                for(i = 0; i < nentries; i++)
                    retvals = g_list_prepend(retvals, g_strdup(strv[i]));
                retvals = g_list_reverse(retvals);
                
                if(strv)
                    dbus_free_string_array(strv);
            } else
                dbus_error_free(&derr);
            
            dbus_message_unref(msg_ret);
        }
    } else if(!strcmp(command, XFMEDIA_REMOTE_GET_PLAYLIST_ENTRY)) {
        gint idx = va_arg(ap, guint);
        
        msg = dbus_message_new_method_call(service_name, path_name,
                XFMEDIA_DBUS_INTERFACE, XFMEDIA_REMOTE_GET_PLAYLIST_ENTRY);
        dbus_message_append_args(msg,
#ifdef DBUS_USE_OLD_API
                DBUS_TYPE_UINT32, idx,
#else
                DBUS_TYPE_UINT32, &idx,
#endif
                DBUS_TYPE_INVALID);
        
        dbus_error_init(&derr);
        msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                msg, 5000, &derr);
        dbus_message_unref(msg);
        
        if(!msg_ret)
            dbus_error_free(&derr);
        else {
            gchar *title = NULL, *filename = NULL;
            gint length = -1;
            
            dbus_error_init(&derr);
            if(dbus_message_get_args(msg_ret, &derr,
                    DBUS_TYPE_STRING, &title,
                    DBUS_TYPE_INT32, &length,
                    DBUS_TYPE_STRING, &filename,
                    DBUS_TYPE_INVALID))
            {
                retvals = g_list_prepend(retvals, g_strdup(filename));
                retvals = g_list_prepend(retvals, GINT_TO_POINTER(length));
                retvals = g_list_prepend(retvals, g_strdup(title));
            } else
                dbus_error_free(&derr);
            
            dbus_message_unref(msg_ret);
        }
    } else if(!strcmp(command, XFMEDIA_REMOTE_NOW_PLAYING)) {
        msg = dbus_message_new_method_call(service_name, path_name,
                XFMEDIA_DBUS_INTERFACE, XFMEDIA_REMOTE_NOW_PLAYING);
        
        dbus_error_init(&derr);
        msg_ret = dbus_connection_send_with_reply_and_block(dbus_conn,
                msg, 5000, &derr);
        dbus_message_unref(msg);
        
        if(!msg_ret)
            dbus_error_free(&derr);
        else {
            gchar *title = NULL, *filename = NULL;
            gint length = -1;
            
            dbus_error_init(&derr);
            if(dbus_message_get_args(msg_ret, &derr,
                    DBUS_TYPE_STRING, &title,
                    DBUS_TYPE_INT32, &length,
                    DBUS_TYPE_STRING, &filename,
                    DBUS_TYPE_INVALID))
            {
                retvals = g_list_prepend(retvals, g_strdup(filename));
                retvals = g_list_prepend(retvals, GINT_TO_POINTER(length));
                retvals = g_list_prepend(retvals, g_strdup(title));
            } else
                dbus_error_free(&derr);
            
            dbus_message_unref(msg_ret);
        }
    }
    
    va_end(ap);
    
    return retvals;
}

#else

gboolean
xfmedia_remote_client_init()
{
    g_critical(_("Xfmedia was not compiled with D-BUS support.  Remote control interface is not available."));
    return FALSE;
}

XfmediaRemoteStatus
xfmedia_remote_client_send_command(gint session_id, const gchar *command, ...)
{
    g_critical(_("Xfmedia was not compiled with D-BUS support.  Remote control interface is not available."));
    return XFMEDIA_REMOTE_UNKNOWN_ERROR;
}

GList *
xfmedia_remote_client_get_info(guint session_id, const gchar *command, ...)
{
    g_critical(_("Xfmedia was not compiled with D-BUS support.  Remote control interface is not available."));
    return NULL;
}

#endif /* !defined(HAVE_DBUS) */
