# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2005 Duke University
# Copyright 2009 Rakesh Pandit <rakesh@fedoraproject.org>


import os
import time
import urlgrabber
import inspect

import yum.i18n
_ = yum.i18n._
import yum.Errors as Errors
from yum.plugins import PluginYumExit, TYPE_INTERACTIVE


requires_api_version = '2.1'
plugin_type = (TYPE_INTERACTIVE,)


###################################################################
# Copy paste downloadPkgs from yum-3.2.22-4.fc11.noarch __ini__.py file
# Just one modification .. mediasort
###################################################################
from yum import YumBase

# order can be a: alphanumeric, s: smallest first and l: largest first
class MyYumBase(YumBase):
    def setorder(self, order='s'):
        self.order = order

    def downloadPkgs(self, pkglist, callback=None, callback_total=None):

        """download list of package objects handed to you, output based on
        callback, raise yum.Errors.YumBaseError on problems"""

        def mediasort(apo, bpo):
            # FIXME: we should probably also use the mediaid; else we
            # could conceivably ping-pong between different disc1's
            a = apo.getDiscNum()
            b = bpo.getDiscNum()
            if a is None and b is None:
                if self.order == 's':
                    return apo.size - bpo.size
                elif self.order == 'l':
                    return bpo.size - apo.size
                else:
                    cmp(apo.name, bpo.name)
            if a is None:
                return -1
            if b is None:
                return 1
            if a < b:
                return -1
            elif a > b:
                return 1
            return 0

        errors = {}
        def adderror(po, msg):
            errors.setdefault(po, []).append(msg)

        self.plugins.run('predownload', pkglist=pkglist)
        repo_cached = False
        remote_pkgs = []
        remote_size = 0
        for po in pkglist:
            if hasattr(po, 'pkgtype') and po.pkgtype == 'local':
                continue
                    
            local = po.localPkg()
            if os.path.exists(local):
                if not self.verifyPkg(local, po, False):
                    if po.repo.cache:
                        repo_cached = True
                        adderror(po, _('package fails checksum but caching is '
                            'enabled for %s') % po.repo.id)
                else:
                    self.verbose_logger.debug(_("using local copy of %s") %(po,))
                    continue
                        
            remote_pkgs.append(po)
            remote_size += po.size
            
            # caching is enabled and the package 
            # just failed to check out there's no 
            # way to save this, report the error and return
            if (self.conf.cache or repo_cached) and errors:
                return errors
                

        remote_pkgs.sort(mediasort)
        #  This is kind of a hack and does nothing in non-Fedora versions,
        # we'll fix it one way or anther soon.
        if (hasattr(urlgrabber.progress, 'text_meter_total_size') and
            len(remote_pkgs) > 1):
            urlgrabber.progress.text_meter_total_size(remote_size)
        beg_download = time.time()
        i = 0
        local_size = 0
        for po in remote_pkgs:
            #  Recheck if the file is there, works around a couple of weird
            # edge cases.
            local = po.localPkg()
            i += 1
            if os.path.exists(local):
                if self.verifyPkg(local, po, False):
                    self.verbose_logger.debug(_("using local copy of %s") %(po,))
                    remote_size -= po.size
                    if hasattr(urlgrabber.progress, 'text_meter_total_size'):
                        urlgrabber.progress.text_meter_total_size(remote_size,
                                                                  local_size)
                    continue
                if os.path.getsize(local) >= po.size:
                    os.unlink(local)

            checkfunc = (self.verifyPkg, (po, 1), {})
            dirstat = os.statvfs(po.repo.pkgdir)
            if (dirstat.f_bavail * dirstat.f_bsize) <= long(po.size):
                adderror(po, _('Insufficient space in download directory %s\n'
                        "    * free   %s\n"
                        "    * needed %s") %
                         (po.repo.pkgdir,
                          format_number(dirstat.f_bavail * dirstat.f_bsize),
                          format_number(po.size)))
                continue
            
            try:
                if i == 1 and not local_size and remote_size == po.size:
                    text = os.path.basename(po.relativepath)
                else:
                    text = '(%s/%s): %s' % (i, len(remote_pkgs),
                                            os.path.basename(po.relativepath))
                mylocal = po.repo.getPackage(po,
                                   checkfunc=checkfunc,
                                   text=text,
                                   cache=po.repo.http_caching != 'none',
                                   )
                local_size += po.size
                if hasattr(urlgrabber.progress, 'text_meter_total_size'):
                    urlgrabber.progress.text_meter_total_size(remote_size,
                                                              local_size)
            except Errors.RepoError, e:
                adderror(po, str(e))
            else:
                po.localpath = mylocal
                if errors.has_key(po):
                    del errors[po]

        if hasattr(urlgrabber.progress, 'text_meter_total_size'):
            urlgrabber.progress.text_meter_total_size(0)
        if callback_total is not None and not errors:
            callback_total(remote_pkgs, remote_size, beg_download)

        self.plugins.run('postdownload', pkglist=pkglist, errors=errors)

        return errors

#############################################################################
# Copy paste ends here
#############################################################################


def config_hook(conduit):
    """ """
    parser = conduit.getOptParser()
    if hasattr(parser, 'plugin_option_group'):
        parser = parser.plugin_option_group

    parser.add_option('', '--download-order', dest='dorder', action='store', default=None,
           choices=["smallestfirst", "largestfirst"], help="downlaod order option")


def prereposetup_hook(conduit):
    """ """
    opts, commands = conduit.getCmdLine()
    downloadorder = conduit.confString('main', 'downloadorder', default='smallestfirst')
    yumbase = None

    if downloadorder not in ['default', 'smallestfirst', 'largestfirst']:
        raise PluginYumExit('downloadorder in download-order.conf has bad value.')

    if opts.dorder:
        if opts.dorder == 'smallestfirst':
            yumbase = MyYumBase()
            yumbase.setorder('s')
        else:
            yumbase = MyYumBase()
            yumbase.setorder('l')
        conduit._base.downloadPkgs = yumbase.downloadPkgs
        return

    if downloadorder:
        if downloadorder == 'default':
            yumbase = MyYumBase()
            yumbase.setorder('a')
        elif downloadorder == 'largestfirst':
            yumbase = MyYumBase()
            yumbase.setorder('l')
        else:
            yumbase = MyYumBase()
            yumbase.setorder('s')
        conduit._base.downloadPkgs = yumbase.downloadPkgs
    return

